/*  _______________________________________________________________________

    DAKOTA: Design Analysis Kit for Optimization and Terascale Applications
    Copyright (c) 2006, Sandia National Laboratories.
    This software is distributed under the GNU General Public License.
    For more information, see the README file in the top Dakota directory.
    _______________________________________________________________________ */

//- Class:       PRPCache (not quite a full-fledged class just yet)
//- Description: Common API for lookup of ParamResponsePairs in evaluation cache
//- Checked by:
//- Version: $Id$

#ifndef PRP_CACHE_H
#define PRP_CACHE_H

#ifdef HAVE_CONFIG_H
#include "dakota_config.h"
#endif
#include "system_defs.h"
#include "data_types.h"
#include "DakotaList.H"
#include "ParamResponsePair.H"

#ifdef DAKOTA_BOOST
#include "boost/multi_index_container.hpp"
#include "boost/multi_index/hashed_index.hpp"
#include "boost/multi_index/mem_fun.hpp"
#include "boost/multi_index/ordered_index.hpp"
namespace bmi = boost::multi_index;
#endif // DAKOTA_BOOST

namespace Dakota {


// Interface class for various Dakota evaluation caches to implement ?

/* Establish a common interface for lookups List-based lookups are
   still needed with OR without Boost, so they are defined outside of
   any conditional compilation blocks. */


// -------------------------------------------------------
// Comparison and hashing functions for ParamResponsePairs
// -------------------------------------------------------
#ifdef DAKOTA_BOOST
/// search function for a particular ParamResponsePair within a PRPHashSet

/** a global function to compare the interface id and variables of a
    particular database_pr (presumed to be in the global history list) with
    a passed in key of interface id and variables provided by search_pr. */
inline bool id_vars_exact_compare(const ParamResponsePair& database_pr,
				  const ParamResponsePair& search_pr)
{
  // First check interface id strings.  If a different interface was used, then
  // we must assume that the results are not interchangeable (differing model
  // fidelity).
  if ( search_pr.interface_id() != database_pr.interface_id() )
    return false;

  // For Boost hashing, need exact binary equality instead of the
  // tolerance-based operator== for Variables
  //if ( search_vars != stored_vars )
  if ( !binary_equal_to( search_pr.prp_parameters(),
			 database_pr.prp_parameters() ) )
    return false;

  // For Boost hashing, a post-processing step is used to manage the ActiveSet
  // logic as shown below in id_vars_set_compare

  return true;
}


/// hash_value for ParamResponsePairs stored in a PRPHashSet
inline std::size_t hash_value(const ParamResponsePair& prp)
{
  // this function is a friend of ParamResponsePair

  // hash using interface ID string.
  std::size_t seed = 0;
  boost::hash_combine(seed, prp.interface_id());

  // Now, hash values of variables using Variables hash_value friend function
  boost::hash_combine(seed, prp.prp_parameters());

  return seed;
}
#endif // DAKOTA_BOOST


/// search function for a particular ParamResponsePair within a PRPList based
/// on ActiveSet content (request vector and derivative variables vector)

/** a global function to compare the ActiveSet of a particular database_pr
    (presumed to be in the global history list) with a passed in ActiveSet
    (search_set). */
inline bool 
set_compare(const ParamResponsePair& database_pr, const ActiveSet& search_set)
{
  // Check each entry of ASV for presence of all requests in the stored data.
  // A match is now detected when the search_asv is a SUBSET of the stored_asv
  // (only exact matches were interpreted as duplicates previously).  This 
  // extension is widely applicable, but was first implemented to eliminate 
  // duplication in Model::estimate_derivatives() when the gradient evaluation 
  // contains inactive fns. whereas a previous line search evaluation at the 
  // same X had no inactive fns.
  const ActiveSet&  stored_set = database_pr.active_set();
  const ShortArray& stored_asv = stored_set.request_vector();
  const ShortArray& search_asv = search_set.request_vector();
  size_t i, asv_len = search_asv.length();
  if ( stored_asv.length() != asv_len )
    return false;
  bool deriv_flag = false;
  for (i=0; i<asv_len; i++) {
    short search_bits = search_asv[i];
    if (search_bits & 6)
      deriv_flag = true;
    // bit-wise AND used to check if each of the search bits is
    // present in the stored_asv value
    if ( (stored_asv[i] & search_bits) != search_bits )
      return false;
  }

  // check if each of the search derivative variables is present in stored_dvv
  if (deriv_flag) {
    const UIntArray& stored_dvv = stored_set.derivative_vector();
    const UIntArray& search_dvv = search_set.derivative_vector();
    size_t dvv_len = search_dvv.length();
    for (i=0; i<dvv_len; i++)
      if ( !stored_dvv.contains(search_dvv[i]) )
	return false;
  }

  return true;
}


/// alternate form of ActiveSet-based search function where search_set
/// is passed by void* pointer (as required by Dakota::List::find() API)

/** a global function to compare the ActiveSet of a particular database_pr
    (presumed to be in the global history list) with a passed in ActiveSet
    (search_ptr). */
inline bool set_compare_by_ptr(const ParamResponsePair& database_pr,
			       const void* search_ptr)
{
  const ActiveSet* search_set = (const ActiveSet*)search_ptr;
  return set_compare(database_pr, *search_set);
}


/// search function for a particular ParamResponsePair within a PRPList
/// based on interface id, variables, and ActiveSet

/** a global function to compare the interface id, variables, and ActiveSet
    of a particular database_pr (presumed to be in the global history list)
    with a passed in interface id, variables, and ActiveSet. */
inline bool 
id_vars_set_compare(const ParamResponsePair& database_pr,
		    const String& search_interface_id,
		    const Variables& search_vars, const ActiveSet& search_set)
{
  // First check interface id strings.  If a different interface was used, then
  // we must assume that the results are not interchangeable (differing model
  // fidelity).
  if ( search_interface_id != database_pr.interface_id() )
    return false;

  // Next, check equality of variables using tolerance-based equality operator.
  if ( search_vars != database_pr.prp_parameters() )
    return false;

  // Check ActiveSet based on subset logic.
  if ( !set_compare( database_pr, search_set ) )
    return false;

  return true;
}


/// alternate form of id/set/vars-based search function where search_pr
/// is passed by void* pointer (as required by Dakota::List::find() API)

/** a global function to compare the interface id, variables, and ActiveSet, and
    interface id of a particular database_pr (presumed to be in the
    global history list) with a passed in interface id, variables, and
    ActiveSet provided by search_ptr. */
inline bool id_vars_set_compare_by_ptr(const ParamResponsePair& database_pr,
				       const void* search_ptr)
{
  const ParamResponsePair* search_pr = (const ParamResponsePair*)search_ptr;
  return id_vars_set_compare(database_pr, search_pr->interface_id(),
			     search_pr->prp_parameters(),
			     search_pr->active_set());
}


/// alternate form of id/set/vars-based search function where
/// search_pr is passed by const reference

/** a global function to compare the interface id, variables, and
    ActiveSet of a particular database_pr (presumed to be in the
    global history list) with a passed in interface id, variables, and
    ActiveSet provided by search_pr. */
inline bool id_vars_set_compare(const ParamResponsePair& database_pr,
				const ParamResponsePair& search_pr)
{
  return id_vars_set_compare(database_pr, search_pr.interface_id(),
			     search_pr.prp_parameters(),
			     search_pr.active_set());
}


/// search function for a particular ParamResponsePair within a
/// PRPList/PRPHashSet based on evaluation id, where the id is passed
/// by const reference

/** a global function to compare the evalId of a particular
    ParamResponsePair (from a container) with a passed in evaluation id. */
inline bool eval_id_compare(const ParamResponsePair& pair, const int& id)
{ return ( pair.eval_id() == id ) ? true : false; }


/// search function for a particular ParamResponsePair within a
/// PRPList/PRPHashSet based on evaluation id, where the id is passed by void*

/** a global function to compare the evalId of a particular
    ParamResponsePair (from a container) with a passed in evaluation id. */
inline bool 
eval_id_compare_by_ptr(const ParamResponsePair& pair, const void* id)
{ return ( pair.eval_id() == *(const int*)id ) ? true : false; }


/// sort function for ParamResponsePair

/** a global function used to sort a PRPList by evalId's. */
inline bool eval_id_sort_fn(const ParamResponsePair& pr1,
                            const ParamResponsePair& pr2)
{
  return ( pr1.interface_id() < pr2.interface_id() ||
           ( pr1.interface_id() == pr2.interface_id() &&
             pr1.eval_id() < pr2.eval_id() ) ) ? true : false;
}


// search function for a particular ParamResponsePair within a
// PRPList/PRPHashSet based on interface id, where this id is passed
// by const reference

/* a global function to compare the idInterface of a particular
   ParamResponsePair (from a container) with a passed in interface id. */
//inline bool 
//interface_id_compare(const ParamResponsePair& pair, const String& id)
//{ return ( pair.interface_id() == id ) ? true : false; }


// search function for a particular ParamResponsePair within a
// PRPList/PRPHashSet based on interface id, where this id is passed by void*

/* a global function to compare the idInterface of a particular
   ParamResponsePair (from a container) with a passed in interface id. */
//inline bool 
//interface_id_compare_by_ptr(const ParamResponsePair& pair, const void* id)
//{ return ( pair.interface_id() == *(const String*)id ) ? true : false; }


// --------------------------
// lookup_by_val for PRPLists
// --------------------------
/// find the iterator of a ParamResponsePair within a PRPList based on
/// interface id, variables, and ActiveSet search data
inline PRPLIter 
lookup_by_val(PRPList& prp_list, const String& search_interface_id,
	      const Variables& search_vars, const ActiveSet& search_set)
{
  // Note: could use List::find(id_vars_set_compare_by_ptr, &search_data) if
  //       id/vars/set are collected within a tuple and passed as void*

  PRPLIter it;
  for (it = prp_list.begin(); it != prp_list.end(); it++)
    if (id_vars_set_compare(*it, search_interface_id, search_vars, search_set))
      return it;
  return prp_list.end();
}


/// find the evaluation id of a ParamResponsePair within a PRPList
/// based on interface id, variables, and ActiveSet search data
inline bool
lookup_by_val(PRPList& prp_list, const String& search_interface_id,
	      const Variables& search_vars, const ActiveSet& search_set,
	      int& found_eval_id)
{
  PRPLIter it
    = lookup_by_val(prp_list, search_interface_id, search_vars, search_set);
  if (it != prp_list.end()) {
    found_eval_id = it->eval_id();
    return true;
  }
  return false;
}


/// find the response of a ParamResponsePair within a PRPList
/// based on interface id, variables, and ActiveSet search data
inline bool 
lookup_by_val(PRPList& prp_list, const String& search_interface_id,
	      const Variables& search_vars, const ActiveSet& search_set,
	      Response& found_resp)
{
  PRPLIter it
    = lookup_by_val(prp_list, search_interface_id, search_vars, search_set);
  if (it != prp_list.end()) {
    found_resp = it->prp_response();
    return true;
  }
  return false;
}


/// find a ParamResponsePair within a PRPList based on interface id,
/// variables, and ActiveSet search data
inline bool 
lookup_by_val(PRPList& prp_list, const String& search_interface_id,
	      const Variables& search_vars, const ActiveSet& search_set,
	      ParamResponsePair& found_pr)
{
  PRPLIter it
    = lookup_by_val(prp_list, search_interface_id, search_vars, search_set);
  if (it != prp_list.end()) {
    found_pr = *it;
    return true;
  }
  return false;
}


// ------------------------------
// lookup_by_eval_id for PRPLists
// ------------------------------
/// find the iterator of a ParamResponsePair within a PRPList based on
/// evaluation id search data
inline PRPLIter lookup_by_eval_id(PRPList& prp_list, const int& search_eval_id)
{ return prp_list.find(eval_id_compare_by_ptr, &search_eval_id); }


/// find the response of a ParamResponsePair within a PRPList based on
/// evaluation id search data
inline bool 
lookup_by_eval_id(PRPList& prp_list, const int& search_eval_id,
		  Response& found_resp)
{
  PRPLIter it = lookup_by_eval_id(prp_list, search_eval_id);
  if (it != prp_list.end()) {
    found_resp = it->prp_response();
    return true;
  }
  return false;
}


/// find a ParamResponsePair within a PRPList based on evaluation id
/// search data
inline bool 
lookup_by_eval_id(PRPList& prp_list, const int& search_eval_id,
		  ParamResponsePair& found_pr)
{
  PRPLIter it = lookup_by_eval_id(prp_list, search_eval_id);
  if (it != prp_list.end()) {
    found_pr = *it;
    return true;
  }
  return false;
}


// --------------------------
// lookup_by_set for PRPLists
// --------------------------
/// find the iterator of a ParamResponsePair within a PRPList based on
/// ActiveSet search data
inline PRPLIter lookup_by_set(PRPList& prp_list, const ActiveSet& search_set)
{ return prp_list.find(set_compare_by_ptr, &search_set); }


/// find a ParamResponsePair within a PRPList based on ActiveSet search data
inline bool 
lookup_by_set(PRPList& prp_list, const ActiveSet& search_set,
	      ParamResponsePair& found_pr)
{
  PRPLIter it = lookup_by_set(prp_list, search_set);
  if (it != prp_list.end()) {
    found_pr = *it;
    return true;
  }
  return false;
}


// -----------------------------------
// lookup_by_interface_id for PRPLists
// -----------------------------------
// find the iterator of a ParamResponsePair within a PRPList based on
// interface id search data
//inline PRPLIter 
//lookup_by_interface_id(PRPList& prp_list, const String& interface_id)
//{ return prp_list.find(interface_id_compare_by_ptr, &interface_id); }


// find a ParamResponsePair within a PRPList based on interface id search data
//inline bool 
//lookup_by_interface_id(PRPList& prp_list, const String& interface_id,
//		       ParamResponsePair& found_pr)
//{
//  PRPLIter it = lookup_by_interface_id(prp_list, interface_id);
//  if (it != prp_list.end()) {
//    found_pr = *it;
//    return true;
//  }
//  return false;
//}


#ifdef DAKOTA_BOOST
// ------------------------------------
// structs and typedefs for PRPHashSets
// ------------------------------------
// define structs used below in PRPHashSet typedef

/// wrapper to delegate to the ParamResponsePair hash_value function
struct partial_prp_hash {
  /// access operator
  std::size_t operator()(const ParamResponsePair& prp) const
  { return hash_value(prp); } // ONLY idInterface & Vars used for hash_value
};

/// predicate for comparing ONLY the idInterface and Vars attributes of PRPair
struct partial_prp_equality {
  /// access operator
  bool operator()(const ParamResponsePair& database_pr,
                  const ParamResponsePair& search_pr) const
  { return id_vars_exact_compare(database_pr, search_pr); }
};

/// Boost Multi-Index Container for fast lookup of ParamResponsePairs
typedef bmi::multi_index_container<ParamResponsePair, bmi::indexed_by<
  // sorted beginning with lowest ID value
  bmi::ordered_non_unique<
    BOOST_MULTI_INDEX_CONST_MEM_FUN(Dakota::ParamResponsePair, int, eval_id),
    std::less<int> >,
  // hashed using partial_prp_hash and compared using partial_prp_equality
  bmi::hashed_non_unique<bmi::identity<ParamResponsePair>, partial_prp_hash,
    partial_prp_equality> > > PRPHashSet;


// -----------------------------
// lookup_by_val for PRPHashSets
// -----------------------------

/// find a ParamResponsePair based on the interface id, variables, and
/// ActiveSet search data within search_pr.

/** Lookup occurs in two steps: (1) PRPHashSet lookup based on strict
    equality in interface id and variables, and (2) PRPList
    post-processing based on ActiveSet subset logic. */
inline bool 
lookup_by_val(PRPHashSet& prp_cache, const ParamResponsePair& search_pr,
	      ParamResponsePair& found_pr)
{
  PRPHashSet::nth_index<1>::type& prp_hash_index = prp_cache.get<1>();
  PRPHashSet::nth_index_iterator<1>::type prp_hash_iter0, prp_hash_iter1;
  boost::tuples::tie(prp_hash_iter0, prp_hash_iter1)
    = prp_hash_index.equal_range(search_pr);

  // equal_range returns a short list of possibilities resulting from hashing
  // with ONLY interfaceId and variables.  Post-processing is applied with the
  // PRPList-based ActiveSet lookup.
  List<ParamResponsePair> pairs_to_postproc;
  while (prp_hash_iter0 != prp_hash_iter1) {
    pairs_to_postproc.insert(*prp_hash_iter0);
    prp_hash_iter0++;
  }
  return lookup_by_set(pairs_to_postproc, search_pr.active_set(), found_pr);
}


/// find the evaluation id of a ParamResponsePair within a PRPHashSet
/// based on interface id, variables, and ActiveSet search data
inline bool
lookup_by_val(PRPHashSet& prp_cache, const String& search_interface_id,
	      const Variables& search_vars, const ActiveSet& search_set,
	      int& found_eval_id)
{
  Response search_resp(search_set);
  ParamResponsePair search_pr(search_vars, search_interface_id, search_resp),
    found_pr;
  bool found = lookup_by_val(prp_cache, search_pr, found_pr);
  if (found)
    found_eval_id = found_pr.eval_id();
  return found;
}


/// find the response of a ParamResponsePair within a PRPHashSet
/// based on interface id, variables, and ActiveSet search data
inline bool 
lookup_by_val(PRPHashSet& prp_cache, const String& search_interface_id,
	      const Variables& search_vars, const ActiveSet& search_set,
	      Response& found_resp)
{
  Response search_resp(search_set);
  ParamResponsePair search_pr(search_vars, search_interface_id, search_resp),
    found_pr;
  bool found = lookup_by_val(prp_cache, search_pr, found_pr);
  if (found)
    found_resp = found_pr.prp_response();
  return found;
}


/// find a ParamResponsePair within a PRPHashSet based on interface id,
/// variables, and ActiveSet search data
inline bool 
lookup_by_val(PRPHashSet& prp_cache, const String& search_interface_id,
	      const Variables& search_vars, const ActiveSet& search_set,
	      ParamResponsePair& found_pr)
{
  Response search_resp(search_set);
  ParamResponsePair search_pr(search_vars, search_interface_id, search_resp);
  return lookup_by_val(prp_cache, search_pr, found_pr);
}


// ---------------------------------
// lookup_by_eval_id for PRPHashSets
// ---------------------------------
/*
TO DO: use a hashed_unique construced from the eval_id and the interface_id,
       or perhaps a ordered_unique based on std::pair<eval_id,interface_id>?
       Once successful, remove lookup_by_interface_id() and
       interface_id_compare() fns.

// find a ParamResponsePair within a PRPHashSet based on evaluation id
// search data
inline bool 
lookup_by_eval_id(PRPHashSet& prp_cache, const ParamResponsePair& search_pr,
		  ParamResponsePair& found_pr)
{
  PRPHashSet::nth_index<0>::type& prp_order_index = prp_cache.get<0>();
  PRPHashSet::nth_index_iterator<0>::type prp_order_iter0, prp_order_iter1;
  boost::tuples::tie(prp_order_iter0, prp_order_iter1)
    = prp_order_index.equal_range(search_pr);

  // equal_range returns a short list of possibilities resulting from ordering
  // based only on eval id.  Post-processing is applied with the PRPList-based
  // interfaceId comparison.
  List<ParamResponsePair> pairs_to_postproc;
  while (prp_order_iter0 != prp_order_iter1) {
    pairs_to_postproc.insert(*prp_order_iter0);
    prp_order_iter0++;
  }
  return lookup_by_interface_id(pairs_to_postproc, search_pr.interface_id(),
				found_pr);
}
*/
#endif // DAKOTA_BOOST


// ---------------------------------
// structs and typedefs for PRPCache
// ---------------------------------
// Establish a common interface for PRPCache, PRPCache iterators, and the
// beginning/end of the Cache
#ifdef DAKOTA_BOOST
typedef PRPHashSet PRPCache;
typedef PRPCache::nth_index_const_iterator<0>::type PRPCacheOrderedConstIter;

/// PRPHashSet definition of prpCacheBegin
inline PRPCacheOrderedConstIter prpCacheBegin(const PRPCache& prp_cache)
{ return prp_cache.get<0>().begin(); }

/// PRPHashSet definition of prpCacheEnd
inline PRPCacheOrderedConstIter prpCacheEnd(const PRPCache& prp_cache)
{ return prp_cache.get<0>().end(); }

#else  // not DAKOTA_BOOST

typedef PRPList PRPCache;
typedef PRPCache::const_iterator PRPCacheOrderedConstIter;

/// PRPList definition of prpCacheBegin
inline PRPCacheOrderedConstIter prpCacheBegin(const PRPCache& prp_cache)
{ return prp_cache.begin(); }

/// PRPList definition of prpCacheEnd
inline PRPCacheOrderedConstIter prpCacheEnd(const PRPCache& prp_cache)
{ return prp_cache.end(); }

#endif // DAKOTA_BOOST

} // namespace Dakota

#endif // PRP_CACHE_H

