/*  _______________________________________________________________________

    DAKOTA: Design Analysis Kit for Optimization and Terascale Applications
    Copyright (c) 2001, Sandia National Laboratories.
    This software is distributed under the GNU General Public License.
    For more information, see the README file in the top Dakota directory.
    _______________________________________________________________________ */

//- Class:	 NonDSparseGrid
//- Description: Implementation code for NonDSparseGrid class
//- Owner:       Mike Eldred
//- Revised by:  
//- Version:

#include "data_types.h"
#include "system_defs.h"
#include "NonDSparseGrid.H"
#include "sparse_grid_mixed.H"
#include "DakotaModel.H"
#include "ProblemDescDB.H"
#include "BasisPolynomial.H"

static const char rcsId[]="@(#) $Id: NonDSparseGrid.C,v 1.57 2004/06/21 19:57:32 mseldre Exp $";

//#define DEBUG

namespace Dakota {


/** This constructor is called for a standard letter-envelope iterator
    instantiation.  In this case, set_db_list_nodes has been called
    and probDescDB can be queried for settings from the method
    specification.  It is not currently used, as there is not yet a
    separate nond_sparse_grid method specification. */
NonDSparseGrid::NonDSparseGrid(Model& model): NonDIntegration(model),
  sparseGridLevelSpec(probDescDB.get_dusa("method.nond.sparse_grid_level")),
  sparseGridLevel(sparseGridLevelSpec)
{
  // pure virtual fn cannot be called from NonDIntegration ctor
  check_input();
}


/** This alternate constructor is used for on-the-fly generation and
    evaluation of sparse grids. */
NonDSparseGrid::NonDSparseGrid(Model& model, const UShortArray& levels): 
  NonDIntegration(NoDBBaseConstructor(), model), sparseGridLevelSpec(levels),
  sparseGridLevel(levels)
{
  // ProbabilityTransformation::ranVarTypesX not yet available: defer until
  // quantify_uncertainty()
  //check_input();
}


NonDSparseGrid::~NonDSparseGrid()
{ }


/** Called from probDescDB-based constructors and from
    NonDIntegration::quantify_uncertainty() */
void NonDSparseGrid::check_input()
{
  const Pecos::ShortArray& x_types = natafTransform.x_types();
  numDesignVars = std::count(x_types.begin(), x_types.end(), (short)DESIGN);
  numStateVars  = std::count(x_types.begin(), x_types.end(), (short)STATE);
  numContinuousVars = numDesignVars + numUncertainVars + numStateVars;
  if (x_types.size() != numContinuousVars) {
    Cerr << "Error: Bad variable counts in NonDSparseGrid::check_input()." 
	 << endl;
    abort_handler(-1);
  }

  const Pecos::ShortArray& u_types = natafTransform.u_types();
  initialize_rules(u_types, integrationRules);
}


void NonDSparseGrid::
initialize_rules(const Pecos::ShortArray& u_types, IntArray& rules)
{
  size_t i, num_u_vars = u_types.size();
  rules.reshape(num_u_vars);
  for (i=0; i<num_u_vars; i++) {
    switch (u_types[i]) {
    case NORMAL:
      rules[i] = 5; break; // Gauss-Hermite open weakly nested
    case UNIFORM:
      rules[i] = 1; break; // Clenshaw-Curtis closed fully nested
      //rules[i] = 4; break;// Gauss-Legendre open weakly nested
    case EXPONENTIAL:
      rules[i] = 7; break; // Gauss-Laguerre open non-nested
    case BETA:
      rules[i] = 9; break; // Gauss-Jacobi open non-nested
    case GAMMA:
      rules[i] = 8; break; // Gen. Gauss-Laguerre open non-nested
    //case USER_DEFINED:
    //  rules[i] = 10; break; // Golub-Welsch open non-nested
    }
    // generalized Gauss-Laguerre, Gauss-Jacobi, and Golub-Welsch are not yet
    // supported in webbur sparse grid routines
    if (rules[i] > 7) {
      Cerr << "Error: generalized Gauss-Laguerre, Gauss-Jacobi, and "
	   << "Golub-Welsch not yet supported in NonDSparseGrid." << endl;
      abort_handler(-1);
    }
  }
}


void NonDSparseGrid::get_parameter_sets(const Model& model)
{
  unsigned short sp_grid_level = sparseGridLevel[0]; // isotropic for now

  // --------------------------------
  // Get number of collocation points
  // --------------------------------
  int num_colloc_pts = webbur::sparse_grid_mixed_size(numContinuousVars,
    sp_grid_level, integrationRules);
  Cout << "Total number of sparse grid integration points: "
       << num_colloc_pts << '\n';

  // ----------------------------------------------
  // Get collocation points and integration weights
  // ----------------------------------------------
  double* weight_sets   = new double [num_colloc_pts];
  double* variable_sets = new double [num_colloc_pts*numContinuousVars];
  webbur::sparse_grid_mixed_weight(numContinuousVars, sp_grid_level,
    integrationRules, num_colloc_pts, weight_sets);
  webbur::sparse_grid_mixed_point(numContinuousVars, sp_grid_level,
    integrationRules, num_colloc_pts, variable_sets);

  size_t i, j, index = 0;
  Real wt_factor = 1.;
  RealVector pt_factor(numContinuousVars, 1.);
  for (i=0; i<numContinuousVars; i++) {
    if (integrationRules[i] == 5) {
      // transform from the Abramowitz-Stegun H(x) weighting fn to an
      // n-dimensional normal PDF weighting fn. (see HermiteOrthogPolynomial.C)
      wt_factor   /= sqrt(Pi);
      pt_factor[i] = sqrt(2.);
    }
    else if (integrationRules[i] == 1)
      // transform from the Abramowitz-Stegun Legendre weighting fn to an
      // n-dimensional uniform PDF weight fn. (see LegendreOrthogPolynomial.C)
      wt_factor /= 2.;
    // TO DO: Gen Laguerre & Jacobi weight factors: see corresponding .C
  }
  if (weightProducts.length() != num_colloc_pts)
    weightProducts.reshape(num_colloc_pts);
  for (i=0; i<num_colloc_pts; i++)
    weightProducts[i] = weight_sets[i] * wt_factor;

  if (allVariables.length() != num_colloc_pts)
    allVariables.reshape(num_colloc_pts);
  const pair<short,short>& view = iteratedModel.current_variables().view();
  RealVector c_vars(numContinuousVars);
  for (i=0; i<num_colloc_pts; i++) {
    if (allVariables[i].is_null())
      allVariables[i] = Variables(view); // minimal instantiate-on-the-fly
    for (j=0; j<numContinuousVars; j++)
      c_vars[j] = variable_sets[index++] * pt_factor[j];
    allVariables[i].continuous_variables(c_vars);
#ifdef DEBUG
    Cout << "Point " << i+1 << ":\n" << c_vars
	 << "Weight " << i+1 << ": " << weightProducts[i] << '\n';
#endif
  }
  delete [] weight_sets;
  delete [] variable_sets;

  // ----------------------------
  // Define 1-D point/weight sets
  // ----------------------------
  BasisPolynomial hermite_poly, legendre_poly, laguerre_poly;
  if (numNormalVars)
    hermite_poly  = BasisPolynomial(HERMITE);
  if (numDesignVars || numUniformVars || numStateVars)
    legendre_poly = BasisPolynomial(LEGENDRE);
  if (numExponentialVars)
    laguerre_poly = BasisPolynomial(LAGUERRE);
  if (gaussPts1D.empty() || gaussWts1D.empty()) {
    gaussPts1D.reshape(numContinuousVars);
    gaussWts1D.reshape(numContinuousVars);
  }
  size_t num_levels_per_var = sp_grid_level + 1;
  // level_index (j indexing) range is 0:w, level (i indexing) range is 1:w+1
  unsigned short level_index, order;
  const Pecos::ShortArray& u_types = natafTransform.u_types();
  for (i=0; i<numContinuousVars; i++) {
    gaussPts1D[i].reshape(num_levels_per_var);
    gaussWts1D[i].reshape(num_levels_per_var);
    switch (u_types[i]) {
    case NORMAL:
      for (level_index=0; level_index<num_levels_per_var; level_index++) {
	level_to_order_open(level_index, order); // j = i - 1 -> order
	gaussPts1D[i][level_index] = hermite_poly.gauss_points(order);
	gaussWts1D[i][level_index] = hermite_poly.gauss_weights(order);
      }
      break;
    case UNIFORM:
      for (level_index=0; level_index<num_levels_per_var; level_index++) {
	// Clenshaw-Curtis pts are extrema (not roots) of Chebyshev polynomials
	level_to_order_closed(level_index, order); // j = i - 1 -> order
	gaussPts1D[i][level_index].reshape(order);
	gaussWts1D[i][level_index].reshape(order);
	double* cc_wts = new double [order];
	webbur::cc_weights(order, cc_wts);
	for (j=0; j<order; j++) {
	  gaussPts1D[i][level_index][j] = webbur::cc_point(order, j+1);
	  gaussWts1D[i][level_index][j] = cc_wts[j]/2.;// scale to PDF weighting
	}
	delete [] cc_wts;
	// Gauss-Legendre points are roots of Legendre polynomials
	//level_to_order_open(level_index, order); // j = i - 1 -> order
	//gaussPts1D[i][level_index] = legendre_poly.gauss_points(order);
	//gaussWts1D[i][level_index] = legendre_poly.gauss_weights(order);
      }
      break;
    case EXPONENTIAL:
      for (level_index=0; level_index<num_levels_per_var; level_index++) {
	level_to_order_open(level_index, order); // j = i - 1 -> order
	gaussPts1D[i][level_index] = laguerre_poly.gauss_points(order);
	gaussWts1D[i][level_index] = laguerre_poly.gauss_weights(order);
      }
      break;
    }
  }

  /*
  // -----------------------------------
  // Get sparse grid index/base mappings
  // -----------------------------------
  size_t size = numContinuousVars*num_colloc_pts, cntr = 0;
  int* indices = new int [size];
  int* bases   = new int [size];

  webbur::sparse_grid_mixed_index(numContinuousVars, sp_grid_level,
    integrationRules, num_colloc_pts, bases, indices);

  IntArray key(2*numContinuousVars);
  unsigned short closed_order_max;
  level_to_order_closed(sp_grid_level, closed_order_max);
  for (i=0; i<num_colloc_pts; i++) {
    for (j=0; j<numContinuousVars; j++, cntr++) {
      switch (u_types[j]) {
      case NORMAL:      // Gauss-Hermite
      //case UNIFORM:   // Gauss-Legendre
	key[j] = 2 * bases[cntr] + 1;             // map to quad order
	key[j+numContinuousVars] = indices[cntr] + bases[cntr]; // 0-based index
	break;
      case UNIFORM:     // Clenshaw-Curtis
	key[j] = closed_order_max;                // promotion to highest grid
	key[j+numContinuousVars] = indices[cntr]; // already 0-based
	break;
      case EXPONENTIAL: // Gauss-Laguerre
	key[j] = bases[cntr];                         // already quad order
	key[j+numContinuousVars] = indices[cntr] - 1; // map to 0-based
	break;
      }
    }
    sparseGridIndexMap[key] = i;
#ifdef DEBUG
    Cout << "i = " << i << " key =\n" << key << endl;
#endif // DEBUG
  }
  delete [] indices;
  delete [] bases;
  */
}


/** used by DataFitSurrModel::build_global() to publish the minimum
    number of points needed from the sparse grid routine in order to
    build a particular global approximation. */
void NonDSparseGrid::
sampling_reset(int min_samples,  int rec_samples, bool all_data_flag,
	       bool stats_flag)
{
  // determine the minimum sparse grid level that provides at least min_samples
  unsigned short min_level = 0;
  int sparse_grid_samples = 1;
  while (sparse_grid_samples < min_samples)
    sparse_grid_samples = webbur::sparse_grid_mixed_size(numContinuousVars,
      ++min_level, integrationRules);

  // sparse grid level may be increased or decreased to provide at least
  // min_samples, but the original user specification (sparseGridLevelSpec) is
  // a hard lower bound.  maxConcurrency must not be updated since parallel
  // config management depends on having the same value at ctor/run/dtor times.
  sparseGridLevel[0] = (min_level > sparseGridLevelSpec[0]) ?
    min_level : sparseGridLevelSpec[0]; // isotropic for now

  // not currently used by this class:
  //allDataFlag = all_data_flag;
  //statsFlag   = stats_flag;
}

} // namespace Dakota
