/*  _______________________________________________________________________

    DAKOTA: Design Analysis Kit for Optimization and Terascale Applications
    Copyright (c) 2006, Sandia National Laboratories.
    This software is distributed under the GNU General Public License.
    For more information, see the README file in the top Dakota directory.
    _______________________________________________________________________ */

//- Class:	 NonDSampling
//- Description: Implementation code for NonDSampling class
//- Owner:       Mike Eldred
//- Checked by:
//- Version:

#include "system_defs.h"
#include "DakotaModel.H"
#include "DakotaResponse.H"
#include "NonDLHSSampling.H"
#include "ProblemDescDB.H"

static const char rcsId[]="@(#) $Id: NonDLHSSampling.C 5234 2008-09-04 19:19:28Z mseldre $";


namespace Dakota {

/** This constructor is called for a standard letter-envelope iterator 
    instantiation.  In this case, set_db_list_nodes has been called and 
    probDescDB can be queried for settings from the method specification. */
NonDLHSSampling::NonDLHSSampling(Model& model): NonDSampling(model),
  varBasedDecompFlag(probDescDB.get_bool("method.variance_based_decomp"))
{
  samplingVarsMode = ACTIVE;

  const short& active_view = iteratedModel.current_variables().view().first;
  bool all_vars = false;
  if (active_view == MERGED_ALL || active_view == MIXED_ALL) {
    // DACE: variables type = "All"
    all_vars = true;
    numDesignVars = probDescDB.get_sizet("variables.continuous_design");
    numStateVars  = probDescDB.get_sizet("variables.continuous_state");
    if (active_view == MERGED_ALL) {
      numDesignVars += probDescDB.get_sizet("variables.discrete_design");
      numStateVars  += probDescDB.get_sizet("variables.discrete_state");
    }
    if (numContinuousVars != numDesignVars + numUncertainVars + numStateVars) {
      Cerr << "\nError: bad number of active variables in nond_sampling."<<endl;
      abort_handler(-1);
    }
    // not needed since no x/u transformations:
    //natafTransform.reshape_correlation_matrix(numDesignVars, numUncertainVars,
    //                                          numStateVars);
  }

  // Check for proper variables and response function definitions
  if ( !numContinuousVars || ( !all_vars && !numUncertainVars ) ) {
    Cerr << "\nError: number of active input variables must be nonzero in "
	 << "nond_sampling." << endl;
    abort_handler(-1);
  }
  // NOTE: numResponseFunctions implies UQ usage.  numFunctions has a broader
  // interpretation (e.g., general DACE usage of NonDLHSSampling iterator).
  if ( !numFunctions || ( !all_vars && !numResponseFunctions ) ) {
    Cerr << "\nError: number of response functions must be nonzero in "
	 << "nond_sampling." << endl;
    abort_handler(-1);
  }
}


/** This alternate constructor is used for generation and evaluation
    of Model-based sample sets.  A set_db_list_nodes has not been
    performed so required data must be passed through the constructor.
    It's purpose is to avoid the need for a separate LHS specification
    within methods that use LHS sampling. */
NonDLHSSampling::
NonDLHSSampling(Model& model, int samples, int seed, short sampling_vars_mode): 
  NonDSampling(NoDBBaseConstructor(), model, samples, seed),
  varBasedDecompFlag(false)
{
  samplingVarsMode = sampling_vars_mode;

  const short& active_view = model.current_variables().view().first;
  if (active_view == MERGED_ALL || active_view == MIXED_ALL) {
    const StringArray& acv_types = model.all_continuous_variable_types();
    numDesignVars = acv_types.count("continuous_design");
    numStateVars  = acv_types.count("continuous_state");
    if (active_view == MERGED_ALL) {
      numDesignVars += acv_types.count("discrete_design");
      numStateVars  += acv_types.count("discrete_state");
    }
  }
}


/** This alternate constructor is used by ConcurrentStrategy for
    generation of uniform, uncorrelated sample sets.  It is _not_ a
    letter-envelope instantiation and a set_db_list_nodes has not been
    performed.  It is called with all needed data passed through the
    constructor and is designed to allow more flexibility in variables
    set definition (i.e., relax connection to a variables
    specification and allow sampling over parameter sets such as
    multiobjective weights).  In this case, a Model is not used and 
    the object must only be used for sample generation (no evaluation). */
NonDLHSSampling::
NonDLHSSampling(int samples, int seed, const RealVector& lower_bnds,
		const RealVector& upper_bnds): 
  NonDSampling(NoDBBaseConstructor(), samples, seed, lower_bnds, upper_bnds),
  varBasedDecompFlag(false)
{
  samplingVarsMode = ACTIVE_UNIFORM; // not used but included for completeness

  // since there will be no late data updates to capture in this case
  // (no sampling_reset()), go ahead and get the parameter sets.
  get_parameter_sets(lower_bnds, upper_bnds);
}


NonDLHSSampling::~NonDLHSSampling()
{ }


/** Loop over the set of samples and compute responses.  Compute
    statistics on the set of responses if statsFlag is set. */
void NonDLHSSampling::quantify_uncertainty()
{
  // if VBD has been selected, evaluate a series of parameter sets
  // (each of the size specified by the user) in order to compute VBD metrics.
  if (varBasedDecompFlag)
    var_based_decomp(numContinuousVars, numSamples);
  // if VBD has not been selected, evaluate a single parameter set of the size
  // specified by the user
  else {
    // run LHS
    get_parameter_sets(iteratedModel);

    // convenience function from Analyzer for evaluating parameter sets
    bool log_resp_flag = (allDataFlag || statsFlag);
    evaluate_parameter_sets(iteratedModel, log_resp_flag, false);

    // Statistics are generated here and output in NonDLHSSampling's
    // redefinition of print_results().
    if (statsFlag) // calculate statistics on the set of responses
      compute_statistics(allVariables, allResponses);
  }
}


void NonDLHSSampling::print_results(ostream& s)
{
  if (statsFlag && !varBasedDecompFlag) {
    s << "\nStatistics based on " << numSamples << " samples:\n";
    print_statistics(s);
  }

  // TO DO: if DACE usage, catalog/print bestVariables/bestResponse as done
  // with PStudyDACE for use within optimization strategies
}

} // namespace Dakota
