/*  _______________________________________________________________________

    DAKOTA: Design Analysis Kit for Optimization and Terascale Applications
    Copyright (c) 2006, Sandia National Laboratories.
    This software is distributed under the GNU General Public License.
    For more information, see the README file in the top Dakota directory.
    _______________________________________________________________________ */

//- Class:        MergedConstraints
//- Description:  Class implementation
//- Owner:        Mike Eldred

#include "MergedConstraints.H"
#include "ProblemDescDB.H"

static const char rcsId[]="@(#) $Id";


namespace Dakota {

/** In this class, a merged data approach is used in which continuous
    and discrete arrays are combined into a single continuous array
    (integrality is relaxed; the converse of truncating reals is not
    currently supported but could be in the future if needed).
    Iterators/strategies which use this class include:
    BranchBndOptimizer.  Extract fundamental lower and upper bounds
    and merge continuous and discrete domains to create
    mergedDesignLowerBnds, mergedDesignUpperBnds,
    mergedStateLowerBnds, and mergedStateUpperBnds. */
MergedConstraints::
MergedConstraints(const ProblemDescDB& problem_db,
		  const pair<short,short>& view):
  Constraints(BaseConstructor(), problem_db, view),
  uncertainLowerBnds(problem_db.get_drv("variables.uncertain.lower_bounds")),
  uncertainUpperBnds(problem_db.get_drv("variables.uncertain.upper_bounds"))
{
#ifdef REFCOUNT_DEBUG
  Cout << "Letter instantiated: variablesView active = " << variablesView.first
       << " inactive = " << variablesView.second << endl;
#endif

  // Get fundamental arrays from problem_db
  const RealVector& cdv_lower_bnds
    = problem_db.get_drv("variables.continuous_design.lower_bounds");
  const RealVector& cdv_upper_bnds
    = problem_db.get_drv("variables.continuous_design.upper_bounds");
  const IntVector& ddv_lower_bnds
    = problem_db.get_div("variables.discrete_design.lower_bounds");
  const IntVector& ddv_upper_bnds
    = problem_db.get_div("variables.discrete_design.upper_bounds");
  const RealVector& csv_lower_bnds
    = problem_db.get_drv("variables.continuous_state.lower_bounds");
  const RealVector& csv_upper_bnds
    = problem_db.get_drv("variables.continuous_state.upper_bounds");
  const IntVector& dsv_lower_bnds
    = problem_db.get_div("variables.discrete_state.lower_bounds");
  const IntVector& dsv_upper_bnds
    = problem_db.get_div("variables.discrete_state.upper_bounds");

  // Initialize derived arrays for this class.
  aggregate_merged(cdv_lower_bnds, ddv_lower_bnds, mergedDesignLowerBnds);
  aggregate_merged(cdv_upper_bnds, ddv_upper_bnds, mergedDesignUpperBnds);
  aggregate_merged(csv_lower_bnds, dsv_lower_bnds, mergedStateLowerBnds);
  aggregate_merged(csv_upper_bnds, dsv_upper_bnds, mergedStateUpperBnds);

  // Manage linear constraints.
  manage_linear_constraints(problem_db);
}


void MergedConstraints::read(istream& s)
{
  mergedDesignLowerBnds.read(s);
  uncertainLowerBnds.read(s);
  mergedStateLowerBnds.read(s);

  mergedDesignUpperBnds.read(s);
  uncertainUpperBnds.read(s);
  mergedStateUpperBnds.read(s);
}


void MergedConstraints::write(ostream& s) const
{
  mergedDesignLowerBnds.write(s);
  uncertainLowerBnds.write(s);
  mergedStateLowerBnds.write(s);

  mergedDesignUpperBnds.write(s);
  uncertainUpperBnds.write(s);
  mergedStateUpperBnds.write(s);
}


void MergedConstraints::copy_rep(const Constraints* con_rep)
{
  // copy derived class attributes
  mergedDesignLowerBnds = ((MergedConstraints*)con_rep)->mergedDesignLowerBnds;
  mergedDesignUpperBnds = ((MergedConstraints*)con_rep)->mergedDesignUpperBnds;
  uncertainLowerBnds    = ((MergedConstraints*)con_rep)->uncertainLowerBnds;
  uncertainUpperBnds    = ((MergedConstraints*)con_rep)->uncertainUpperBnds;
  mergedStateLowerBnds  = ((MergedConstraints*)con_rep)->mergedStateLowerBnds;
  mergedStateUpperBnds  = ((MergedConstraints*)con_rep)->mergedStateUpperBnds;
}


void MergedConstraints::reshape_rep(const Sizet2DArray& vars_comps)
{
  size_t num_mdv = vars_comps[0][0], num_uv = vars_comps[1][0],
         num_msv = vars_comps[2][0];
  mergedDesignLowerBnds.reshape(num_mdv);
  mergedDesignUpperBnds.reshape(num_mdv);
  uncertainLowerBnds.reshape(num_uv);
  uncertainUpperBnds.reshape(num_uv);
  mergedStateLowerBnds.reshape(num_msv);
  mergedStateUpperBnds.reshape(num_msv);
}

} // namespace Dakota
