/*
===============================================================================
    PROJECT:

        Genetic Algorithm for Sandia National Laboratories

    CONTENTS:

        Implementation of class JEGAOptimizer.

    NOTES:

        See notes of JEGAOptimizer.H.

    PROGRAMMERS:

        John Eddy (jpeddy@sandia.gov) (JE)

    ORGANIZATION:

        Sandia National Laboratories

    COPYRIGHT:

        This library is free software; you can redistribute it and/or
        modify it under the terms of the GNU Lesser General Public
        License as published by the Free Software Foundation; either
        version 2.1 of the License, or (at your option) any later version.

        This library is distributed in the hope that it will be useful,
        but WITHOUT ANY WARRANTY; without even the implied warranty of
        MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
        GNU Lesser General Public License for more details.

        For a copy of the GNU Lesser General Public License, write to:
            Free Software Foundation, Inc.
            59 Temple Place, Suite 330
            Boston, MA 02111-1307 USA

    VERSION:

        2.0.0

    CHANGES:

        Mon Jun 09 09:48:34 2003 - Original Version (JE)
        Wed Dec 07 15:00:00 2005 - Added ParameterDatabase subclass to wrap
                                   ProblemDescDB for dependency removal (JE)
        Wed Mar 29 12:00:00 2006 - Adopted use of JEGA front end project to
                                   reduce code duplication and keep Dakota
                                   up to speed with the latest in JEGA.

===============================================================================
*/




/*
===============================================================================
Document This File
===============================================================================
*/
/** \file
 * \brief Contains the implementation of the JEGAOptimizer class.
 */




/*
===============================================================================
Includes
===============================================================================
*/
// JEGAConfig.hpp should be the first include in all JEGA files.
#include <../Utilities/include/JEGAConfig.hpp>
#include <../Utilities/include/Logging.hpp>

// Standard includes.

// JEGA core includes.
#include <GeneticAlgorithm.hpp>
#include <GeneticAlgorithmEvaluator.hpp>
#include <GeneticAlgorithmInitializer.hpp>
#include <OperatorGroups/AllOperators.hpp>

// JEGA front end includes.
#include <../FrontEnd/Core/include/Driver.hpp>
#include <../FrontEnd/Core/include/ProblemConfig.hpp>
#include <../FrontEnd/Core/include/AlgorithmConfig.hpp>
#include <../FrontEnd/Core/include/EvaluatorCreator.hpp>

// Dakota includes.
#include <JEGAOptimizer.H>
#include <ProblemDescDB.H>

// Eddy utility includes.
#include <utilities/include/EDDY_DebugScope.hpp>

// JEGA utility includes.
#include <../Utilities/include/DesignGroup.hpp>
#include <../Utilities/include/ConstraintInfo.hpp>
#include <../Utilities/include/ParameterExtractor.hpp>
#include <../Utilities/include/BasicParameterDatabaseImpl.hpp>
#include <../Utilities/include/MultiObjectiveStatistician.hpp>
#include <../Utilities/include/SingleObjectiveStatistician.hpp>

/*
===============================================================================
Namespace Using Directives
===============================================================================
*/
using namespace std;
using namespace JEGA::Logging;
using namespace JEGA::FrontEnd;
using namespace eddy::utilities;
using namespace JEGA::Utilities;
using namespace JEGA::Algorithms;



/*
===============================================================================
Begin Namespace
===============================================================================
*/
namespace Dakota {




/*
===============================================================================
File Scope Helper Functions
===============================================================================
*/
/// Creates a string from the argument \a val using an ostringstream.
/**
 * This only gets used in this file and is only ever called with ints so no
 * error checking is in place.
 *
 * \param val The value of type T to convert to a string.
 * \return The string representation of \a val created using an ostringstream.
 */
template <typename T>
string
asstring(
    const T& val
    )
{
    EDDY_FUNC_DEBUGSCOPE
    ostringstream ostr;
    ostr << val;
    return ostr.str();
}

/// An evaluator specialization that knows how to interact with Dakota.
/**
 * This evaluator knows how to use the model to do evaluations both in
 * synchronous and asynchronous modes.
 */
class JEGAOptimizer::Evaluator :
    public GeneticAlgorithmEvaluator
{
    /*
    ===========================================================================
    Member Data Declarations
    ===========================================================================
    */
    private:

        /// The Model known by this evaluator.
        /**
         * It is through this model that evaluations will take place.
         */
        Model& _model;

    /*
    ===========================================================================
    Public Methods
    ===========================================================================
    */
    public:

         /// Returns the proper name of this operator.
        /**
         * \return The string "DAKOTA JEGA Evaluator".
         */
        static
        const std::string&
        Name(
            )
        {
            EDDY_FUNC_DEBUGSCOPE
            static const string ret("DAKOTA JEGA Evaluator");
            return ret;
        }

        /// Returns a full description of what this operator does and how.
        /**
         * The returned text is:
         * \verbatim
            This evaluator uses Sandia's DAKOTA optimization
            software to evaluate the passed in Designs.  This
            makes it possible to take advantage of the fact that
            DAKOTA is designed to run on massively parallel machines.
           \endverbatim.
         *
         * \return A description of the operation of this operator.
         */
        static
        const std::string&
        Description(
            )
        {
            EDDY_FUNC_DEBUGSCOPE
            static const string ret(
                "This evaluator uses Sandia's DAKOTA optimization software to "
                "evaluate the passed in Designs.  This makes it possible to "
                "take advantage of the fact that DAKOTA is designed to run on "
                "massively parallel machines."
                );
            return ret;
        }

    /*
    ===========================================================================
    Subclass Visible Methods
    ===========================================================================
    */
    protected:

        /**
         * \brief Returns the continuous Design variable values held in Design
         *        \a from.
         *
         * It returns them as a RealVector for use in the Dakota interface.
         * The values in the returned vector will be the actual values intended
         * for use in the evaluation functions.
         *
         * \param from The Design class object from which to extract the
         *             continuous design variable values.
         * \return A vector of the continuous design variable values associated
         *         with \a from.
         */
        RealVector
        GetContinuumVariableValues(
            const Design& from
            ) const
        {
            EDDY_FUNC_DEBUGSCOPE

            RealVector ret;
            GetContinuumVariableValues(from, ret);
            return ret;
        }

        /// Returns the discrete Design variable values held in Design \a from.
        /**
         * It returns them as a IntVector for use in the Dakota interface.
         * The values in the returned vector will be the values for the design
         * variables as far as JEGA knows.  However, in actuality, the values
         * are the representations due to the way that Dakota manages discrete
         * variables.
         *
         * \param from The Design class object from which to extract the
         *             discrete design variable values.
         * \return A vector of the discrete design variable values associated
         *         with \a from.
         */
        IntVector
        GetDiscreteVariableValues(
            const Design& from
            ) const
        {
            EDDY_FUNC_DEBUGSCOPE

            IntVector ret;
            GetDiscreteVariableValues(from, ret);
            return ret;
        }

        /**
         * \brief Places the continuous Design variable values from Design
         *        \a from into RealVector \a into.
         *
         * The values in the returned vector will be the actual values intended
         * for use in the evaluation functions.
         *
         * \param from The Design class object from which to extract the
         *             continuous design variable values.
         * \param into The vector into which to place the extracted values.
         */
        void
        GetContinuumVariableValues(
            const Design& from,
            RealVector& into
            ) const;

        /**
         * \brief Places the discrete Design variable values from Design
         *        \a from into IntVector \a into.
         *
         * The values placed in the vector will be the values for the design
         * variables as far as JEGA knows.  However, in actuality, the values
         * are the representations due to the way that Dakota manages discrete
         * variables.
         *
         * \param from The Design class object from which to extract the
         *             discrete design variable values.
         * \param into The vector into which to place the extracted values.
         */
        void
        GetDiscreteVariableValues(
            const Design& from,
            IntVector& into
            ) const;

        /**
         * \brief This method fills \a intoDisc and \a intoCont appropriately
         *        using the values of \a from.
         *
         * The discrete design variable values are placed in \a intoDisc and
         * the continuum are placed into \a intoCont.
         *
         * It is more efficient to use this method than to use
         * GetDiscreateVariableValues and GetContinuumVariableValues separately
         * if you want both.
         *
         * \param from The Design class object from which to extract the
         *             discrete design variable values.
         * \param intoDisc The vector into which to place the extracted
         *                 discrete values.
         * \param intoCont The vector into which to place the extracted
         *                 continuous values.
         */
        void
        SeparateVariables(
            const Design& from,
            IntVector& intoDisc,
            RealVector& intoCont
            ) const;

        /**
         * \brief Records the computed objective and constraint function values
         *        into \a into.
         *
         * This method takes the response values stored in \a from and properly
         * transfers them into the \a into design.
         *
         * The response vector \a from is expected to contain values for each
         * objective function followed by values for each non-linear constraint
         * in the order in which the info objects were loaded into the target
         * by the optimizer class.
         *
         * \param from The vector of responses to install into \a into.
         * \param into The Design to which the responses belong and into which
         *             they must be written.
         */
        void
        RecordResponses(
            const RealVector& from,
            Design& into
            ) const;

        /// Returns the number of non-linear constraints for the problem.
        /**
         * This is computed by adding the number of non-linear equality
         * constraints to the number of non-linear inequality constraints.
         * These values are obtained from the model.
         *
         * \return The total number of non-linear constraints.
         */
        std::size_t
        GetNumberNonLinearConstraints(
            ) const
        {
            EDDY_FUNC_DEBUGSCOPE
            return _model.num_nonlinear_eq_constraints() +
                   _model.num_nonlinear_ineq_constraints();
        }

        /// Returns the number of linear constraints for the problem.
        /**
         * This is computed by adding the number of linear equality
         * constraints to the number of linear inequality constraints.
         * These values are obtained from the model.
         *
         * \return The total number of linear constraints.
         */
        std::size_t
        GetNumberLinearConstraints(
            ) const
        {
            EDDY_FUNC_DEBUGSCOPE
            return _model.num_linear_eq_constraints() +
                   _model.num_linear_ineq_constraints();
        }

    /*
    ===========================================================================
    Subclass Overridable Methods
    ===========================================================================
    */
    public:

        /// Does evaluation of each design in \a group.
        /**
         * This method uses the Model known by this class to get Designs
         * evaluated.  It properly formats the Design class information in a
         * way that Dakota will understand and then interprets the Dakota
         * results and puts them back into the Design class object.  It
         * respects the asynchronous flag in the Model so evaluations may
         * occur synchronously or asynchronously.
         *
         * Prior to evaluating a Design, this class checks to see if it
         * is marked as already evaluated.  If it is, then the evaluation
         * of that Design is not carried out.  This is not strictly
         * necessary because Dakota keeps track of evaluated designs and
         * does not re-evaluate.  An exception is the case of a population
         * read in from a file complete with responses where Dakota is
         * unaware of the evaluations.
         *
         * \param group The group of Design class objects to be evaluated.
         * \return true if all evaluations completed and false otherwise.
         */
        virtual
        bool
        Evaluate(
            DesignGroup& group
            );

        /// This method cannot be used!!
        /**
         * This method does nothing and cannot be called.  This is because in
         * the case of asynchronous evaluation, this method would be unable
         * to conform.  It would require that each evaluation be done in a
         * synchronous fashion.
         *
         * \param eJob A job holding the Design that would be evaluated if this
         *             method worked.
         * \return Would return true if the Design were evaluated and false
         *         otherwise.  Never actually returns here.  Issues a fatal
         *         error.  Otherwise, it would always return false.
         */
        virtual
        bool
        Evaluate(
            EvaluationJob& eJob
            )
        {
            EDDY_FUNC_DEBUGSCOPE

            JEGALOG_II_F(GetLogger(), this,
                text_entry(lfatal(),
                    GetName() + 
                    ":You cannot use Evaluate(EvaluationJob&)...ever.")
                )
            return false;
        }

        /// Returns the proper name of this operator.
        /**
         * \return See Name().
         */
        virtual
        std::string
        GetName(
            ) const
        {
            EDDY_FUNC_DEBUGSCOPE
            return Evaluator::Name();
        }

        /// Returns a full description of what this operator does and how.
        /**
         * \return See Description().
         */
        virtual
        std::string
        GetDescription(
            ) const
        {
            EDDY_FUNC_DEBUGSCOPE
            return Evaluator::Description();
        }

        /**
         * \brief Creates and returns a pointer to an exact duplicate of this
         *        operator.
         *
         * \param algorithm The GA for which the clone is being created.
         * \return A clone of this operator.
         */
        virtual
        GeneticAlgorithmOperator*
        Clone(
            GeneticAlgorithm& algorithm
            ) const
        {
            EDDY_FUNC_DEBUGSCOPE
            return new Evaluator(*this, algorithm, _model);
        }


    /*
    ===========================================================================
    Structors
    ===========================================================================
    */
    public:

        /**
         * \brief Constructs a Evaluator for use by \a  algorithm.
         *
         * The optimizer is needed for purposes of variable scaling.
         *
         * \param algorithm The GA for which the new evaluator is to be used.
         * \param model The model through which evaluations will be done.
         */
        Evaluator(
            GeneticAlgorithm& algorithm,
            Model& model
            ) :
                GeneticAlgorithmEvaluator(algorithm),
                _model(model)
        {
            EDDY_FUNC_DEBUGSCOPE
        }

        /**
         * \brief Copy constructs a Evaluator.
         *
         * \param copy The evaluator from which properties are to be duplicated
         *             into this.
         */
        Evaluator(
            const Evaluator& copy
            ) :
                GeneticAlgorithmEvaluator(copy),
                _model(copy._model)
        {
            EDDY_FUNC_DEBUGSCOPE
        }

        /**
         * \brief Copy constructs a Evaluator for use by \a algorithm.
         *
         * The optimizer is needed for purposes of variable scaling.
         *
         * \param copy The existing Evaluator from which to retrieve
         *             properties.
         * \param algorithm The GA for which the new evaluator is to be used.
         * \param model The model through which evaluations will be done.
         */
        Evaluator(
            const Evaluator& copy,
            GeneticAlgorithm& algorithm,
            Model& model
            ) :
                GeneticAlgorithmEvaluator(copy, algorithm),
                _model(model)
        {
            EDDY_FUNC_DEBUGSCOPE
        }


    private:

        /// This constructor has no implementation and cannot be used.
        /**
         * This constructor can never be used.  It is provided so that this
         * operator can still be registered in an operator registry even though
         * it can never be instantiated from there.
         *
         * \param algorithm The GA for which the new evaluator is to be used.
         */
        Evaluator(
            GeneticAlgorithm& algorithm
            );


}; // class JEGAOptimizer::Evaluator

/**
 * \brief A specialization of the JEGA::FrontEnd::EvaluatorCreator that
 *        creates a new instance of a Evaluator.
 */
class JEGAOptimizer::EvaluatorCreator :
    public JEGA::FrontEnd::EvaluatorCreator
{
    /*
    ===========================================================================
    Member Data Declarations
    ===========================================================================
    */
    private:

        /**
         * \brief The user defined model to be passed to the constructor of the
         *        Evaluator.
         */
        Model& _theModel;

    /*
    ===========================================================================
    Subclass Overridable Methods
    ===========================================================================
    */
    public:

        /// Overriden to return a newly created Evaluator.
        /**
         * The GA will assume ownership of the evaluator so we needn't worry
         * about keeping track of it for destruction.  The additional
         * parameters needed by the Evaluator are stored as members of this
         * class at construction time.
         *
         * \param alg The GA for which the evaluator is to be created.
         * \return A pointer to a newly created Evaluator.
         */
        virtual
        GeneticAlgorithmEvaluator*
        CreateEvaluator(
            GeneticAlgorithm& alg
            )
        {
            EDDY_FUNC_DEBUGSCOPE
            return new Evaluator(alg, _theModel);
        }

    /*
    ===========================================================================
    Structors
    ===========================================================================
    */
    public:

        /**
         * \brief Constructs an EvaluatorCreator using the supplied model.
         *
         * \param theModel The Dakota::Model this creator will pass to the
         *                 created evaluator.
         */
        EvaluatorCreator(
            Model& theModel
            ) :
                _theModel(theModel)
        {
            EDDY_FUNC_DEBUGSCOPE
        }

}; // class JEGAOptimizer::EvaluatorCreator

/**
 * \brief A subclass of the JEGA front end driver that exposes the
 *        individual protected methods to execute the algorithm.
 *
 * This is necessary because DAKOTA requires that all problem
 * information be extracted from the problem description DB at the
 * time of Optimizer construction and the front end does it all in
 * the execute algorithm method which must be called in find_optimum.
 */
class JEGAOptimizer::Driver :
    public JEGA::FrontEnd::Driver
{
    /*
    ===========================================================================
    Member Data Declarations
    ===========================================================================
    */
    private:

    /*
    ===========================================================================
    Public Methods
    ===========================================================================
    */
    public:

        /**
         * \brief Reads all required data from the problem description database
         *        stored in the supplied algorithm config.
         *
         * The returned GA is fully configured and ready to be run.  It must
         * also be destroyed at some later time.  You MUST call
         * DestroyAlgorithm for this purpose.  Failure to do so could result
         * in a memory leak and an eventual segmentation fault!  Be sure to
         * call DestroyAlgorithm prior to destroying the algorithm config that
         * was used to create it!
         *
         * This is just here to expose the base class method to users.
         *
         * \param algConfig The fully loaded configuration object containing
         *                  the database of parameters for the algorithm to be
         *                  run on the known problem.
         * \return The fully configured and loaded GA ready to be run using
         *         the PerformIterations method.
         */
        GeneticAlgorithm*
        ExtractAllData(
            const AlgorithmConfig& algConfig
            )
        {
            return JEGA::FrontEnd::Driver::ExtractAllData(algConfig);
        }

        /**
         * \brief Performs the required iterations on the supplied GA.
         *
         * This includes the calls to AlgorithmInitialize and AlgorithmFinalize
         * and logs some information if appropriate.
         *
         * This is just here to expose the base class method to users.
         *
         * \param theGA The GA on which to perform iterations.  This parameter
         *              must be non-null.
         * \return The final solutions reported by the supplied GA after all
         *         iterations and call to AlgorithmFinalize.
         */
        DesignOFSortSet
        PerformIterations(
            GeneticAlgorithm* theGA
            )
        {
            return JEGA::FrontEnd::Driver::PerformIterations(theGA);
        }

        /**
         * \brief Deletes the supplied GA.
         *
         * Use this method to destroy a GA after all iterations have been run.
         * This method knows if the log associated with the GA was created here
         * and needs to be destroyed as well or not.
         *
         * This is just here to expose the base class method to users.
         *
         * Be sure to use this prior to destoying the algorithm config object
         * which contains the target.  The GA destructor needs the target to
         * be in tact.
         *
         * \param theGA The algorithm that is no longer needed and thus must be
         *              destroyed.
         */
        void
        DestroyAlgorithm(
            GeneticAlgorithm* theGA
            )
        {
            JEGA::FrontEnd::Driver::DestroyAlgorithm(theGA);
        }

    /*
    ===========================================================================
    Structors
    ===========================================================================
    */
    public:

        /// Default constructs a Driver
        /**
         * \param probConfig The definition of the problem to be solved by this
         *                   Driver whenever ExecuteAlgorithm is called.
         *
         * The problem can be solved in multiple ways by multiple algorithms
         * even using multiple different evaluators by issuing multiple calls
         * to ExecuteAlgorithm with different AlgorithmConfigs.
         */
        Driver(
            const ProblemConfig& probConfig
            ) :
                JEGA::FrontEnd::Driver(probConfig)
        {
            EDDY_FUNC_DEBUGSCOPE
        }

}; // class JEGAOptimizer::Driver


/*
===============================================================================
Static Member Data Definitions
===============================================================================
*/
const string JEGAOptimizer::_sogaMethodText("soga");
const string JEGAOptimizer::_mogaMethodText("moga");





/*
===============================================================================
Mutators
===============================================================================
*/








/*
===============================================================================
Accessors
===============================================================================
*/





/*
===============================================================================
Public Methods
===============================================================================
*/
void
JEGAOptimizer::find_optimum(
    )
{
    EDDY_FUNC_DEBUGSCOPE

    // Load up an algorithm config and a problem config.
    ProblemConfig pConfig;
    LoadProblemConfig(pConfig);

    AlgorithmConfig aConfig(*_theEvalCreator, *this->_theParamDB);
    LoadAlgorithmConfig(aConfig);

    // retrieve parameter database for repeated use (is actaully _theParamDB)
    ParameterDatabase& pdb = aConfig.GetParameterDB();

    // Create a new driver for JEGA.
    JEGAOptimizer::Driver driver(pConfig);

    // Get the algorithm separately (rather than simply running the current
    // configuration) in case we need to change the initializer.
    GeneticAlgorithm* theGA = driver.ExtractAllData(aConfig);

    // Get the name of the GA for repeated use below.  We need this regardless
    // of whether or not logging b/c it is used in a fatal error.
    const string& name = theGA->GetName();

    // The initializer requires some additional logic to account for the
    // possibility that JEGA is being used in a Dakota strategy.  If that is
    // the case, the _initPts array will be non-empty and we will use them
    // instead of whatever initialization has been specified by the user.
    if(!_initPts.empty())
    {
        const GeneticAlgorithmInitializer& oldInit =
            theGA->GetOperatorSet().GetInitializer();

        JEGALOG_II_G(lquiet(), this,
            text_entry(lquiet(), name + ": discovered multiple initial "
                "points presumably supplied by a previous iterator in a "
                "strategy.  The \"" + oldInit.GetName() + "\" initializer "
                "will not be used and instead will be replaced with the "
                "double_matrix initializer which will read the supplied "
                "initial points."
                )
            )

        pdb.AddIntegralParam(
            "method.population_size", static_cast<int>(oldInit.GetSize())
            );

        pdb.AddDoubleMatrixParam(
            "method.jega.design_matrix", ToDoubleMatrix(initial_points())
            );

        GeneticAlgorithmInitializer* newInit =
            AllOperators::FullInstance().GetInitializer(
                "double_matrix", *theGA
                );

        JEGAIFLOG_II_G_F(newInit == 0x0, this,
            text_entry(lfatal(), name + ": Unable to resolve "
                "Initializer \"double_matrix\".")
            );

        JEGAIFLOG_II_F(!theGA->SetInitializer(newInit),
            theGA->GetLogger(), this,
            text_entry(lfatal(), name + ": Unable to set the initializer to "
                "double_matrix because it is incompatible with the other "
                "operators."
                )
            )

        JEGAIFLOG_II_F(
            !newInit->ExtractParameters(pdb), theGA->GetLogger(), this,
            text_entry(lfatal(),
                name + ": Failed to retrieve the parameters for \"" +
                newInit->GetName() + "\".")
            );

    }

    JEGALOG_II_G(lverbose(), this,
        text_entry(lverbose(),
            name + ": About to perform algorithm execution.")
            )

    DesignOFSortSet bests(driver.PerformIterations(theGA));

    JEGALOG_II_G(lverbose(), this,
        ostream_entry(lverbose(), name + ": algorithm execution completed. ")
            << bests.size() << " solutions found. Passing them back to DAKOTA."
        )

    // Return the correct values to DAKOTA

    // Begin by returning the "Best" solution as the single best iterator
    // solution.
    const Design* bestOne = GetBestSolution(bests);

    // If we were unable to find a best for some reason, issue a warning and
    // skip the dakota load.
    JEGAIFLOG_II_G(bestOne == 0x0, lquiet(), this,
        text_entry(lquiet(), name + ": was unable to identify a best "
            "solution.  The Dakota best variables and best responses "
            "objects will not be loaded.\n\n")
        )

    if(bestOne != 0x0)
        this->LoadDakotaResponses(*bestOne, bestVariables, bestResponse);

    // Now add the entire solution set into the arrays of best responses and
    // variables.  If this is the MOGA, the array will then contain the Pareto
    // set.  If it is a SOGA, it will contain all the solutions with the same
    // best "fitness".
    resize_response_results_array(bests.size());
    resize_variables_results_array(bests.size());

    ResponseArray::size_type index = 0;
    for(DesignOFSortSet::const_iterator it(bests.begin()); it!=bests.end();
        ++it, ++index)
    {
        LoadDakotaResponses(
            **it, bestVariablesArray.at(index), bestResponseArray.at(index)
            );
    }

    // now we are done with our solution set so we can flush it
    // per Driver rules.
    bests.flush();

    JEGALOG_II_G(lquiet(), this,
        text_entry(lquiet(), name + ": find optimum completed and all "
            "results have been passed back to DAKOTA.\n\n")
        )

    // We can not destroy our GA.
    driver.DestroyAlgorithm(theGA);
}

bool
JEGAOptimizer::accepts_multiple_points(
    ) const
{
    return true;
}

bool
JEGAOptimizer::returns_multiple_points(
    ) const
{
    return true;
}

void
JEGAOptimizer::initial_points(
    const VariablesArray& pts
    )
{
    _initPts = pts;
}

const VariablesArray&
JEGAOptimizer::initial_points(
    ) const
{
    return _initPts;
}


/*
===============================================================================
Subclass Visible Methods
===============================================================================
*/

void
JEGAOptimizer::resize_variables_results_array(
    std::size_t newsize
    )
{
    // If there is no change in size, we needn't continue.
    if(newsize == bestVariablesArray.size()) return;

    // If this is a reduction in size, we can use the standard resize method
    if(newsize < bestVariablesArray.size())
    { bestVariablesArray.resize(newsize); return; }

    // Otherwise, we have to do the iteration ourselves so that we make use
    // of the model's current variables for envelope-letter requirements.
    bestVariablesArray.reserve(newsize);
    for(std::size_t i=bestVariablesArray.size(); i<newsize; ++i)
        bestVariablesArray.push_back(
            iteratedModel.current_variables().copy()
            );
}

void
JEGAOptimizer::resize_response_results_array(
    std::size_t newsize
    )
{
    // If there is no change in size, we needn't continue.
    if(newsize == bestResponseArray.size()) return;

    // If this is a reduction in size, we can use the standard resize method
    if(newsize < bestResponseArray.size())
    { bestResponseArray.resize(newsize); return; }

    // Otherwise, we have to do the iteration ourselves so that we make use
    // of the model's current response for envelope-letter requirements.
    bestResponseArray.reserve(newsize);
    for(std::size_t i=bestResponseArray.size(); i<newsize; ++i)
        bestResponseArray.push_back(
            iteratedModel.current_response().copy()
            );
}

void
JEGAOptimizer::LoadDakotaResponses(
    const JEGA::Utilities::Design& des,
    Dakota::Variables& vars,
    Dakota::Response& resp
    ) const
{
    RealVector c_vars(numContinuousVars);
    IntVector  d_vars(numDiscreteVars);

    // The first numContinuousVars of a design will be all the continuous
    // variables of the problem (see LoadTheDesignVariables).
    for(size_t i=0; i<numContinuousVars; ++i)
        c_vars[i] = des.GetVariableValue(i);

    // The remaining variables represent the discrete variables.
    for(size_t i=0; i<numDiscreteVars; ++i)
      d_vars[i] = static_cast<int>(
         des.GetVariableValue(i+numContinuousVars)
         );

    vars.continuous_variables(c_vars);
    vars.discrete_variables(d_vars);

    RealVector fn_vals(numFunctions);
    for(size_t i=0; i<numObjectiveFns; i++)
      fn_vals[i]= des.GetObjective(i);

    // JEGA constraint ordering is nonlinear inequality, nonlinear equality,
    // linear inequality, linear equality (see
    // JEGAOptimizer::LoadTheConstraints()).
    for (size_t i=0; i<static_cast<size_t>(numNonlinearConstraints); i++)
      fn_vals[i+numObjectiveFns] = des.GetConstraint(i);

    resp.function_values(fn_vals);
}

void
JEGAOptimizer::ReCreateTheParameterDatabase(
    )
{
    EDDY_FUNC_DEBUGSCOPE
    delete this->_theParamDB;
    this->_theParamDB = new BasicParameterDatabaseImpl();
}

void
JEGAOptimizer::LoadTheParameterDatabase(
    )
{
    EDDY_FUNC_DEBUGSCOPE

    ReCreateTheParameterDatabase();

    // Duplicate in all the integral parameters.
    this->_theParamDB->AddIntegralParam(
        "method.random_seed",
        probDescDB.get_int("method.random_seed")
        );

    // now all the reals
    this->_theParamDB->AddDoubleParam(
        "method.constraint_penalty",
        probDescDB.get_real("method.constraint_penalty")
        );
    this->_theParamDB->AddDoubleParam(
        "method.crossover_rate",
        probDescDB.get_real("method.crossover_rate")
        );
    this->_theParamDB->AddDoubleParam(
        "method.mutation_rate",
        probDescDB.get_real("method.mutation_rate")
        );
    this->_theParamDB->AddDoubleParam(
        "method.mutation_scale",
        probDescDB.get_real("method.mutation_scale")
        );
    this->_theParamDB->AddDoubleParam(
        "method.jega.percent_change",
        probDescDB.get_real("method.jega.percent_change")
        );
    this->_theParamDB->AddDoubleParam(
        "method.convergence_tolerance",
        probDescDB.get_real("method.convergence_tolerance")
        );
    this->_theParamDB->AddDoubleParam(
        "method.jega.shrinkage_percentage",
        probDescDB.get_real("method.jega.shrinkage_percentage")
        );
    this->_theParamDB->AddDoubleParam(
        "method.jega.fitness_limit",
        probDescDB.get_real("method.jega.fitness_limit")
        );

    // now get all the size_t's
    this->_theParamDB->AddSizeTypeParam(
        "method.jega.num_cross_points",
        probDescDB.get_sizet("method.jega.num_cross_points")
        );
    this->_theParamDB->AddSizeTypeParam(
        "method.jega.num_parents",
        probDescDB.get_sizet("method.jega.num_parents")
        );
    this->_theParamDB->AddSizeTypeParam(
        "method.jega.num_offspring",
        probDescDB.get_sizet("method.jega.num_offspring")
        );
    this->_theParamDB->AddSizeTypeParam(
        "method.jega.num_generations",
        probDescDB.get_sizet("method.jega.num_generations")
        );
    // Note that the population size, max evals, and max gens are in as ints.
    // Do a conversion for each here.
    this->_theParamDB->AddSizeTypeParam(
        "method.population_size",
        static_cast<size_t>(probDescDB.get_int("method.population_size"))
        );
    this->_theParamDB->AddSizeTypeParam(
        "method.max_iterations",
        static_cast<size_t>(probDescDB.get_int("method.max_iterations"))
        );
    this->_theParamDB->AddSizeTypeParam(
        "method.max_function_evaluations",
        static_cast<size_t>(
            probDescDB.get_int("method.max_function_evaluations")
            )
        );

    // Dakota does not currently expose the evaluation concurrency flag
    // through the interface nor should it b/c Dakota handles evaluations.
    this->_theParamDB->AddSizeTypeParam("method.jega.eval_concurrency", 1);

    // Now get all the booleans
    this->_theParamDB->AddBooleanParam(
        "method.print_each_pop",
        probDescDB.get_bool("method.print_each_pop")
        );

    // Dakota does not currently expose the flag to instruct the GA whether or
    // not to write the final data file and discards file.  Put those in here
    // to avoid warnings about them missing.
    this->_theParamDB->AddBooleanParam("method.print_final_data", true);
    this->_theParamDB->AddBooleanParam("method.print_discards", true);

    // Dakota does not currently expose the flag to instruct a nicher
    // as to whether or not to cache niched designs.  So put that flag in here
    // with a default true value.  It won't hurt anything if it is not needed.
    this->_theParamDB->AddBooleanParam("method.jega.cache_niched_designs", true);

    // now get all the strings.
    
    // Dakota does not expose the ability to specify the ability to specify the
    // weighted sum only fitness assessor b/c it is only for use with the
    // favor feasible selector.  Likewise, the favor feasible selector can only
    // be used with the weighted sum fitness asessor.  Because of this, we will
    // detect use of the the favor feasible and enforce the use of the weighted
    // sum only.  We will write a log message about it.
    const string& selector = probDescDB.get_string("method.replacement_type");
    this->_theParamDB->AddStringParam(
        "method.replacement_type", selector
        );
    
    const string& fitness = probDescDB.get_string("method.fitness_type");
    if(selector == "favor_feasible")
    {
        JEGALOG_II_G(lquiet(), this,
            text_entry(lquiet(),
            "Use of the favor_feasible selector has been detected.  Use of "
            "this selector type requires use of the \"weighted_sum_only\" "
            "fitness assessor.  Therefore, use of the \"" + fitness +
            "\" will be changed to use of the \"weighted_sum_only\".")
            )

        this->_theParamDB->AddStringParam(
            "method.fitness_type", "weighted_sum_only"
            );
    }
    else
    {
        this->_theParamDB->AddStringParam("method.fitness_type", fitness);
    }

    this->_theParamDB->AddStringParam(
        "method.crossover_type",
        probDescDB.get_string("method.crossover_type")
        );
    this->_theParamDB->AddStringParam(
        "method.mutation_type",
        probDescDB.get_string("method.mutation_type")
        );
    this->_theParamDB->AddIntegralParam(
        "method.output",
        probDescDB.get_short("method.output")
        );
    this->_theParamDB->AddStringParam(
        "method.initialization_type",
        probDescDB.get_string("method.initialization_type")
        );
    this->_theParamDB->AddStringParam(
        "method.flat_file",
        probDescDB.get_string("method.flat_file")
        );

    // Dakota does not currently expose the final data file name pattern
    // through the interface.
    this->_theParamDB->AddStringParam(
        "method.jega.final_data_filename", "finaldata#.dat"
        );

    // Dakota does not currently expose the main loop operator selection
    // through the interface.
    this->_theParamDB->AddStringParam(
        "method.jega.mainloop_type", "duplicate_free"
        );

    // The log file gets special attention.  If it is the default global log
    // file name, we will replace it with an empty string b/c we don't want the
    // created GA to think it owns the global log file.
    string log_file = probDescDB.get_string("method.log_file");
    this->_theParamDB->AddStringParam(
        "method.log_file",
        log_file == "JEGAGlobal.log" ? "" : log_file
        );
    this->_theParamDB->AddStringParam(
        "method.jega.convergence_type",
        probDescDB.get_string("method.jega.convergence_type")
        );
    this->_theParamDB->AddStringParam(
        "method.jega.niching_type",
        probDescDB.get_string("method.jega.niching_type")
        );
    this->_theParamDB->AddStringParam(
        "method.jega.postprocessor_type",
        probDescDB.get_string("method.jega.postprocessor_type")
        );

    // Dakota does not expose a flat file delimiter for the case where we
    // are using the flat file initializer but the initializer is going to
    // be looking for one if it is in use.
    this->_theParamDB->AddStringParam("method.jega.initializer_delimiter", "");

    // now get all vector of doubles.
    this->_theParamDB->AddDoubleVectorParam(
        "method.jega.niche_vector",
        probDescDB.get_drv("method.jega.niche_vector")
        );
    this->_theParamDB->AddDoubleVectorParam(
        "method.jega.distance_vector",
        probDescDB.get_drv("method.jega.distance_vector")
        );

    // Dakota does not expose a capability to enter multiple flat file names
    // as a vector (can delimit them in the single string and JEGA will parse
    // them).  We will add an empty vector to prevent the warning from JEGA.
    this->_theParamDB->AddStringVectorParam(
        "method.flat_files", vector<string>()
        );



    // now get all int vector params.
    // nothing to do here.

    // now get all the double matrices.
    // nothing to do here.

    // now get all integer list params.
    // nothing to do here.

    // now get all string vectors
    // nothing to do here.

    // now get all string lists
    // nothing to do here.
}

void
JEGAOptimizer::LoadAlgorithmConfig(
    JEGA::FrontEnd::AlgorithmConfig& aConfig
    )
{
    EDDY_FUNC_DEBUGSCOPE

    ParameterDatabase& pdb = aConfig.GetParameterDB();

    // Determine what kind of algorithm we are creating (MOGA or SOGA)
    // based on the methodName base class variable.
    AlgorithmConfig::AlgType algType;

    if(methodName == _mogaMethodText)
        algType = AlgorithmConfig::MOGA;

    else if(methodName == _sogaMethodText)
        algType = AlgorithmConfig::SOGA;

    else
        JEGALOG_II_G_F(this,
            text_entry(lfatal(), "JEGA Error: \"" + methodName +
                "\" is an invalid method specification.")
            )

    aConfig.SetAlgorithmType(algType);

    // We will use the method id as the algorithm name if it is non-empty and
    // we will otherwise use the method name.
    aConfig.SetAlgorithmName(
        method_id().empty() ? methodName : method_id()
        );
}

void
JEGAOptimizer::LoadProblemConfig(
    JEGA::FrontEnd::ProblemConfig& pConfig
    )
{
    EDDY_FUNC_DEBUGSCOPE

    // This is carried out by loading the design variables, objective
    // functions, and constraints.
    LoadTheDesignVariables(pConfig);
    LoadTheObjectiveFunctions(pConfig);
    LoadTheConstraints(pConfig);
}

void
JEGAOptimizer::LoadTheDesignVariables(
    JEGA::FrontEnd::ProblemConfig& pConfig
    )
{
    EDDY_FUNC_DEBUGSCOPE

    // The information needed to create the design variable infos
    // is contained in the data structures of the base classes.
    // In particular, the Model (iteratedModel) has most of the
    // info.  We will create a shorthand for it here to ease syntax.
    const Model& m = iteratedModel;

    // We will start with the continuous variables.  The data for
    // those is in the Model as continuous_lower_bounds,
    // continuous_upper_bounds, and continuous_variable_labels.
    const RealVector& clbs = m.continuous_lower_bounds();
    const RealVector& cubs = m.continuous_upper_bounds();
    const StringArray& clabels = m.continuous_variable_labels();

    // Loop over all continuous variables and add an info object.  Don't worry
    // about the precision so much.  It is only considered by the operators
    // that do binary encoding such as the NPointParameterizedBinaryCrosser and
    // the RandomBitMutator.  Other than that, it is largely ignored by this
    // implementation.  It can have a fairly profound effect on the preformance
    // of those operators that use it to encode to binary.
    for(size_t i=0; i<numContinuousVars; ++i)
        pConfig.AddContinuumRealVariable(
            clabels.at(i), clbs.at(i), cubs.at(i), 6
            );

    // now move on to the discrete variables.  The data for those is in the
    // Model as discrete_lower_bounds, discrete_upper_bounds, and
    // discrete_variable_labels.
    const IntVector& dlbs = m.discrete_lower_bounds();
    const IntVector& dubs = m.discrete_upper_bounds();
    const StringArray& dlabels = m.discrete_variable_labels();

    for(size_t i=0; i<numDiscreteVars; ++i)
    {
        // The first thing we have to do here is to create and load the
        // discrete values vector.  It is a vector of real variables.
        JEGA::DoubleVector disVals;
        disVals.reserve(dubs.at(i) - dlbs.at(i) + 1);
        for(int j=dlbs.at(i); j<=dubs.at(i); ++j)
            disVals.push_back(static_cast<double>(j));

/*
*****************************************************************
Mike,

If DAKOTA ever changes its handling of discrete variables,
the following code in this comment block will have to replace
the existing code below.  The unknown information in quotes
will depend on your implementation but when the time comes
it should be easy to fill in.

        for(int j=0; j<"num discrete values [i]"; ++j)
        {
            (*it)->AddDiscreteValue("j'th discrete value");
        }

This is conceptually the same as what is happening now except
that we are treating the consecutive integers as values instead
of representations (which is what they are according to my system).
*****************************************************************
*/

        // This is a RealDesignVariableType because hopefully, someday
        // DAKOTA will be more capable with discrete values
        // such that they are not limited to consecutive integral
        // values.  Otherwise it would be better to use
        // the IntegerDesignVariableType with a continuum nature.
        pConfig.AddDiscreteRealVariable(dlabels.at(i), disVals);
    }

    // Now make sure that an info was created for each variable.
    EDDY_ASSERT(pConfig.GetDesignTarget().GetNDV() ==
        (numContinuousVars + numDiscreteVars));
}

void
JEGAOptimizer::LoadTheObjectiveFunctions(
    JEGA::FrontEnd::ProblemConfig& pConfig
    )
{
    EDDY_FUNC_DEBUGSCOPE

    // For now, all objectives will be of type minimize.  Hopefully,
    // Dakota will soon support mixed extremization schemes.
    // Dakota does not support labeling objectives.  Until it does,
    // we will create a label that looks like "Nature Type Index".
    for(size_t i=0; i<numObjectiveFns; ++i)
        pConfig.AddNonlinearMinimizeObjective(
            "Non-Linear Minimize " + asstring(i)
            );

    // see to it that the numbers match up.
    EDDY_ASSERT(pConfig.GetDesignTarget().GetNOF() ==
        numObjectiveFns);
}

void
JEGAOptimizer::LoadTheConstraints(
    JEGA::FrontEnd::ProblemConfig& pConfig
    )
{
    EDDY_FUNC_DEBUGSCOPE

    // The information needed to create the constraint infos
    // is contained in the data structures of the base classes.
    // In particular, the Model (iteratedModel) has most of the
    // info.  We will create a shorthand for it here to ease syntax.
    const Model& m = iteratedModel;

    /**************************************************************************

    Note the order in which these are created.  Do not change this order.  It
    is this way because of the order in which responses are returned out of the
    Model.  Technically, it only involves the first two blocks which create the
    non-linear constraints.  But don't mess with any of it anyway.

    **************************************************************************/


    // start with non-linear (2-sided) inequality constraints.
    // The information we need for these is in
    // nonlinear_ineq_constraint_lower_bounds and
    // nonlinear_ineq_constraint_upper_bounds.  As with objective
    // functions, Dakota does not allow labeling of constraints.
    // we will create a label that looks like "Nature Type Index".
    const RealVector& nln_ineq_lwr_bnds
      = m.nonlinear_ineq_constraint_lower_bounds();
    const RealVector& nln_ineq_upr_bnds
      = m.nonlinear_ineq_constraint_upper_bounds();

    // Loop over all two sided non linear inequality constraitns and add an
    // info object for each.
    for(size_t i=0; i<numNonlinearIneqConstraints; ++i)
        pConfig.AddNonlinearTwoSidedInequalityConstraint(
            "Non-Linear Two-Sided Inequality " + asstring(i),
            nln_ineq_lwr_bnds.at(i), nln_ineq_upr_bnds.at(i)
            );

    // now do non-linear equality constraints.  The information we need for
    // these is in nonlinear_eq_constraint_targets.
    const RealVector& nln_eq_targets = m.nonlinear_eq_constraint_targets();
    for(size_t i=0; i<numNonlinearEqConstraints; ++i)
        pConfig.AddNonlinearEqualityConstraint(
            "Non-Linear Equality " + asstring(i), nln_eq_targets.at(i)
            );

    // now do linear (2-sided) inequality constraints  The information we need
    // for these is in linear_ineq_constraint_lower_bounds and
    // linear_ineq_constraint_upper_bounds.
    // In addition to bounds, these accept coefficients for possible shortcut
    // evaluation.  That information is in linear_ineq_constraint_coeffs.
    const RealVector& lin_ineq_lwr_bnds
      = m.linear_ineq_constraint_lower_bounds();
    const RealVector& lin_ineq_upr_bnds
      = m.linear_ineq_constraint_upper_bounds();
    const RealMatrix& lin_ineq_coeffs
      = m.linear_ineq_constraint_coeffs();

    for(size_t i=0; i<numLinearIneqConstraints; ++i)
        pConfig.AddLinearTwoSidedInequalityConstraint(
            "Linear Two-Sided Inequality " + asstring(i),
            lin_ineq_lwr_bnds.at(i), lin_ineq_upr_bnds.at(i),
            lin_ineq_coeffs(i)
            );

    // now do linear equality constraints.  The information we need for these
    // is in lin_eq_targets. In addition to targets, these accept coefficients
    // for possible shortcut evaluation.  That information is in
    // linear_eq_constraint_coeffs.
    const RealVector& lin_eq_targets = m.linear_eq_constraint_targets();
    const RealMatrix& lin_eq_coeffs = m.linear_eq_constraint_coeffs();

    for(size_t i=0; i<numLinearEqConstraints; ++i)
        pConfig.AddLinearEqualityConstraint(
            "Linear Equality " + asstring(i),
            lin_eq_targets(i), 0.0, lin_eq_coeffs(i)
            );

    // see to it that the numbers match up.
    EDDY_ASSERT(pConfig.GetDesignTarget().GetNCN() ==
                (numNonlinearIneqConstraints + numLinearIneqConstraints +
                 numNonlinearEqConstraints +numLinearEqConstraints));
}

const Design*
JEGAOptimizer::GetBestSolution(
    const JEGA::Utilities::DesignOFSortSet& from
    )
{
    EDDY_FUNC_DEBUGSCOPE

    if(methodName == _mogaMethodText)
        return GetBestMOSolution(from);

    else if(methodName == _sogaMethodText)
        return GetBestSOSolution(from);

    else
    {
        JEGALOG_II_G_F(this,
            text_entry(lfatal(), "JEGA Error: \"" + methodName +
                       "\" is an invalid method specification.")
            )
        return 0x0;
    }
}

const Design*
JEGAOptimizer::GetBestMOSolution(
    const JEGA::Utilities::DesignOFSortSet& from
    )
{
    EDDY_FUNC_DEBUGSCOPE

    // Start by removing any infeasible from "from".
    DesignOFSortSet feasible(DesignStatistician::GetFeasible(from));

    // If we were unable to find a feasible solution, return a null pointer.
    if(feasible.empty()) return 0x0;

    // We now need the extremes of the feasible solutions.
    DoubleExtremes extremeSet(
        MultiObjectiveStatistician::FindParetoExtremes(feasible)
        );

    //get number of objective functions
    const eddy::utilities::uint64_t nof =
        from.front()->GetDesignTarget().GetNOF();
    EDDY_ASSERT(nof == extremeSet.size());

    // This will store the minimum squared deviation from the
    // utopia point.  Initialize it to a very large number.
    double minDistance = DBL_MAX;

    // Create a pointer to store the best design as each is
    // considered.  Initialize it to the null pointer.
    const Design* bestDesign = 0x0;

    // prepare to iterate the designs by objective function
    // and find the best.
    DesignOFSortSet::const_iterator it(feasible.begin());
    const DesignOFSortSet::const_iterator e(feasible.end());

    for (; it!=e; ++it)
    {
        // prepare a place to store sum-of-squared distance
        // for this Design as we traverse the objective functions.
        double tempDistance = 0.0;

        // compute the sum-of-squares between the current point and
        // the utopia point.  Store it as tempDistance.
        for (size_t i=0; i<nof;++i)
            tempDistance +=
                Math::Pow(((*it)->GetObjective(i)-extremeSet.get_min(i)),2);

        // now, if this Design turns out to be closer to the utopia
        // point than any previously considered Design, take it as
        // the best and update our current minimum distance.
        if (tempDistance < minDistance)
        {
            bestDesign = *it;
            minDistance = tempDistance;
        }
    }

    // now return our best Design.  The only way this value
    // could be null at this point is if every Design in our feasible
    // set has sum-of-squares distance from the utopia point == DBL_MAX.
    return bestDesign;
}

const Design*
JEGAOptimizer::GetBestSOSolution(
    const JEGA::Utilities::DesignOFSortSet& from
    )
{
    EDDY_FUNC_DEBUGSCOPE

    if(from.empty()) return 0x0;

    // This is the case where there are some feasible designs.
    if(DesignStatistician::FindFeasible(from) != from.end())
    {
        // in order to get the best, we need the weights.
        JEGA::DoubleVector weights;

        bool success = ParameterExtractor::GetDoubleVectorFromDB(
            *this->_theParamDB,
            "responses.multi_objective_weights",
            weights
            );

        // if we could not get them and there are multiple objectives,
        // then we have to abort without an answer.
        // Otherwise if we did not succeed and there is a single objective,
        // then we will assume a single weight of value 1.

        // get number of objective functions
        const eddy::utilities::uint64_t nof =
            from.front()->GetDesignTarget().GetNOF();
        if(!success)
        {
            if(nof > 1) return 0x0;
            weights.assign(1, 1.0);
        }

        // Now we want to locate the design(s) with the best weighted sum.
        pair<double, vector<DesignOFSortSet::const_iterator> > bests(
            SingleObjectiveStatistician::FindMinSumFeasibleDesigns(
                from, weights
                )
            );

        return bests.second.empty() ? 0x0 : *bests.second.front();
    }
    return 0x0;
}

JEGA::DoubleMatrix
JEGAOptimizer::ToDoubleMatrix(
    const VariablesArray& variables
    ) const
{
    EDDY_FUNC_DEBUGSCOPE

    // Prepare the resultant matrix with proper initial capacity
    JEGA::DoubleMatrix ret(variables.size());

    // Iterate the variables objects and create entries in the new matrix
    size_t i = 0;
    for(VariablesArray::const_iterator it(variables.begin());
        it!=variables.end(); ++it, ++i)
    {
        // Store the continuous and discrete variables arrays for use below.
        const RealVector& cvs = (*it).continuous_variables();
        const IntVector& dvs = (*it).discrete_variables();

        // Prepare the row we are working with to hold all variable values.
        ret[i].reserve(cvs.size() + dvs.size());

        // Copy in first the continuous followed by the discrete values.
        ret[i].insert(ret[i].end(), cvs.begin(), cvs.end());
        ret[i].insert(ret[i].end(), dvs.begin(), dvs.end());
    }

    return ret;
}


/*
===============================================================================
Subclass Overridable Methods
===============================================================================
*/




/*
===============================================================================
Private Methods
===============================================================================
*/




/*
===============================================================================
Structors
===============================================================================
*/
JEGAOptimizer::JEGAOptimizer(
    Model& model
    ) :
        Optimizer(model),
        _theParamDB(0x0),
        _theEvalCreator(0x0)
{
    EDDY_FUNC_DEBUGSCOPE

    // JEGAOptimizer now makes use of the JEGA front end core project to run
    // an algorithm.  In order to do this, it creates and loads a DesignTarget,
    // a ProblemConfig, and an AlgorithmConfig.

    // The first step is to initialize JEGA via the front end Driver
    // class.  The data needed is available from the problem description
    // database.  This should only happen once in any run of Dakota regardless
    // of how many JEGAOptimizers are used.
    if(!JEGA::FrontEnd::Driver::IsJEGAInitialized())
    {
        // The random seed must be handled separately because the sentry value
        // for JEGA (0) is not the same as the sentry value for Dakota (-1).
        int rseed_temp = this->probDescDB.get_int("method.random_seed");

        // if the rseed is negative, it is the sentry value and we will use the
        // JEGA sentry value of 0.
        unsigned int rSeed = (rseed_temp < 0) ? 0 :
            static_cast<unsigned int>(rseed_temp);

        // For now, we will use the level of the first instance of an optimizer
        // as the level for the global log.  This is only potentially not ideal
        // in the case of strategies.  The 4 - below is to account for the fact
        // that the actual dakota levels count upwards by increasing amount of
        // output while the dakota_levels must count downwards in order to be
        // compatable with the logging library code.
        short dakLev = this->probDescDB.get_short("method.output");
        LogLevel jegaLev;

        switch (dakLev)
        {
            case SILENT_OUTPUT: jegaLev = lsilent(); break;
            case NORMAL_OUTPUT: jegaLev = lnormal(); break;
            case DEBUG_OUTPUT: jegaLev = ldebug(); break;
            case QUIET_OUTPUT: jegaLev = lquiet(); break;
            case VERBOSE_OUTPUT: jegaLev = lverbose(); break;
            default: jegaLev = ldefault();
        }

        JEGA::FrontEnd::Driver::InitializeJEGA(
            "JEGAGlobal.log", jegaLev, rSeed
            );
    }

    // If we failed to init, we cannot continue.
    JEGAIFLOG_II_G_F(!JEGA::FrontEnd::Driver::IsJEGAInitialized(), this,
        text_entry(lfatal(), "JEGAOptimizer Error: Unable to initialize JEGA")
        );

    // we only need to load up the parameter database at this point.
    this->LoadTheParameterDatabase();

	// population_size is extracted by JEGA in
    // GeneticAlgorithmInitializer::PollForParameters(), but it is
    // also needed here to specify the algorithmic concurrency.  Note
    // that the JEGA population size may grow or shrink during its
    // iterations, so this is only an initial estimate.
    const int& pop_size = this->probDescDB.get_int("method.population_size");
    this->maxConcurrency *= pop_size;

    // We only ever need one EvaluatorCreator so we can create it now.
    this->_theEvalCreator = new EvaluatorCreator(iteratedModel);

    // The following is not performed in the Optimizer constructor since
    // maxConcurrency is updated above. The matching free_communicators()
    // appears in the Optimizer destructor.
    if (this->scaleFlag || this->multiObjFlag)
      this->iteratedModel.init_communicators(this->maxConcurrency);
}

JEGAOptimizer::~JEGAOptimizer(
    )
{
    EDDY_FUNC_DEBUGSCOPE

    delete this->_theEvalCreator;
    delete this->_theParamDB;
}


/*
===============================================================================
Inner Class Implementations
===============================================================================
*/

void
JEGAOptimizer::Evaluator::GetContinuumVariableValues(
    const Design& from,
    RealVector& into
    ) const
{
    EDDY_FUNC_DEBUGSCOPE

    // make sure that the "into" vector is empty.
    into.clear();

    // We need the target to get information about the
    // design variables.
    const DesignTarget& target = this->GetDesignTarget();
    const DesignVariableInfoVector& dvis =
        target.GetDesignVariableInfos();

    // prepare to iterate over the design variable information
    // objects
    DesignVariableInfoVector::const_iterator it(dvis.begin());
    const DesignVariableInfoVector::const_iterator e(dvis.end());

    // we are looking for continuum variables.
    for(; it!=e; ++it)
        if((*it)->IsContinuum())
            // The double value stored in the Design class is the
            // representation. What has to be used in the evaluations is the
            // value.  In the case of a continuum variable, they are probably
            // the same thing but not necessarily.  That is why you see the
            // "GetValueOf" method call.
            into.push_back((*it)->WhichValue(from));
}

void
JEGAOptimizer::Evaluator::GetDiscreteVariableValues(
    const Design& from,
    IntVector& into
    ) const
{
    EDDY_FUNC_DEBUGSCOPE

    // make sure that the "into" vector is empty.
    into.clear();

    // We need the target to get information about the
    // design variables.
    const DesignTarget& target = from.GetDesignTarget();
    const DesignVariableInfoVector& dvis = target.GetDesignVariableInfos();

    // prepare to iterate over the design variable information
    // objects
    DesignVariableInfoVector::const_iterator it(dvis.begin());
    const DesignVariableInfoVector::const_iterator  e(dvis.end());

    // we are looking for discrete variables.
    for(; it!=e; ++it)
        // The double value stored in the Design class is the representation.
        // What has to be used in the evaluations is the value.  In the case
        // of a discrete natured variable, they are not the same thing.  That
        // is why you see the "GetValueOf" method call.
        if((*it)->IsDiscrete())
            into.push_back(
                static_cast<int>((*it)->WhichValue(from))
                );
}

void
JEGAOptimizer::Evaluator::SeparateVariables(
    const Design& from,
    IntVector& intoDisc,
    RealVector& intoCont
    ) const
{
    EDDY_FUNC_DEBUGSCOPE

    // make sure that the "into" vectors are empty.
    intoDisc.clear();
    intoCont.clear();

    // We need the target to get information about the
    // design variables.
    const DesignTarget& target = from.GetDesignTarget();
    const DesignVariableInfoVector& dvis =
        target.GetDesignVariableInfos();

    // prepare to iterate over the design variable information
    // objects
    DesignVariableInfoVector::const_iterator it(dvis.begin());
    DesignVariableInfoVector::const_iterator  e(dvis.end());

    for(; it!=e; ++it)
    {
        // Check for continuum first since it is most likely
        if((*it)->IsContinuum())
            intoCont.push_back((*it)->WhichValue(from));

        // now check for discrete
        else if((*it)->IsDiscrete())
            intoDisc.push_back(
                static_cast<int>((*it)->WhichValue(from))
                );

        else
            JEGALOG_II_F(GetLogger(), this, text_entry(lfatal(),
                "JEGA Evaluator: Design variable " + (*it)->GetLabel() +
                " neither continuum nor discrete!"
                )
            )
    }
}

void
JEGAOptimizer::Evaluator::RecordResponses(
    const RealVector& from,
    Design& into
    ) const
{
    EDDY_FUNC_DEBUGSCOPE

    // get the target for information.
    const DesignTarget& target = GetDesignTarget();

    // get the information about the constraints.
    const ConstraintInfoVector& cnis = target.GetConstraintInfos();

    // prepare to store the location in the responses vector.
    RealVector::size_type loc = 0;

    // find out how many objective functions there are.
    const size_t nof = target.GetNOF();

    // record the objective functions first.
    for(size_t i=0; i<nof; ++i) into.SetObjective(i, from.at(loc++));

    // now record the nonlinear constraints.  To do this,
    // we will need to know how many there are.  They will be the
    // first of the constraints in the design.
    const size_t num_nonlin_cn = GetNumberNonLinearConstraints();
    size_t cn = 0;
    for(ConstraintInfoVector::const_iterator it(cnis.begin());
        cn<num_nonlin_cn && it!=cnis.end(); ++cn)
    {
        into.SetConstraint(cn, from.at(loc++));
        (*it)->RecordViolation(into);
    }
}

bool
JEGAOptimizer::Evaluator::Evaluate(
    DesignGroup& group
    )
{
    EDDY_FUNC_DEBUGSCOPE

    JEGALOG_II(GetLogger(), ldebug(), this,
        text_entry(ldebug(), GetName() + ": Performing group evaluation.")
        )

    // check for trivial abort conditions
    if(group.IsEmpty()) return true;

    // first, let's see if we can avoid any evaluations.
    ResolveClones(group);

    // we'll prepare Vectors for repeated use without re-construction
    RealVector contVars;
    IntVector  discVars;

    // prepare to iterate over the group
    DesignDVSortSet::const_iterator it(group.BeginDV());
    const DesignDVSortSet::const_iterator e(group.EndDV());

    // these quantities will be used below
    const DesignTarget& target = GetDesignTarget();

    // Find out the counts on the different types of constraints
    const size_t num_nonlin_cn = GetNumberNonLinearConstraints();
//    const size_t num_lin_cn = GetNumberLinearConstraints();

    // Get the information about the constraints.
    const ConstraintInfoVector& cninfos = target.GetConstraintInfos();

    // Prepare to store the number of evaluations + requests in order
    // to avoid overshooting the limit.
    eddy::utilities::uint64_t numEvalReqs = GetNumberEvaluations();

    // store an iterator to the first linear constraint so that
    // we can evaluate it using the cinfo objects.

    // prepare to iterate
    ConstraintInfoVector::const_iterator flincn(
        cninfos.begin() + num_nonlin_cn
        );
    ConstraintInfoVector::const_iterator cit;

    for(; it!=e; ++it)
    {
        // If this Design is evaluated, let's skip it.
        if((*it)->IsEvaluated()) continue;

        // If we've reached our maximum allowable number of
        // evaluations, tag remaining as evaluated and illconditioned.
        // By doing so, they will be flushed from the algorithm.
        if(numEvalReqs >= GetMaxEvaluations())
        {
            (*it)->SetEvaluated(true);
            (*it)->SetIllconditioned(true);
            continue;
        }

        // extract the real and continuous variables
        // from the current Design
        SeparateVariables(**it, discVars, contVars);

        // send this guy out for evaluation using the _model.

        // first, set the current values of the variables in the model
        _model.continuous_variables(contVars);
        _model.discrete_variables(discVars);

        // now request the evaluation in synchronous or asyncronous mode.
        if (_model.asynch_flag())
        {
            // The following method call will use the default
            // Active set vector which is to just compute
            // function values, no gradients or hessians.
            _model.asynch_compute_response();
        }
        else
        {
            // The following method call will use the default
            // Active set vector which is to just compute
            // function values, no gradients or hessians.
            _model.compute_response();

            // increment the number of performed evaluations by 1
            IncrementNumberEvaluations();

            // Record the responses back into the Design
            const RealVector& ftn_vals =
                _model.current_response().function_values();

            RecordResponses(ftn_vals, **it);

            // Label this guy as now being evaluated.
            (*it)->SetEvaluated(true);

            // now check the feasibility of this design
            target.CheckFeasibility(**it);
        }

        // no matter what, we want to increment the number of evaluations
        // or evaluation requests.
        ++numEvalReqs;

        // The responses do not (or will not) include the linear
        // constraint values. We have to compute them ourselves.
        // we can do it using the "EvaluateConstraint" method which
        // will only succeed for linear constraints for which
        // coefficients have been supplied.
        for(cit=flincn; cit!=cninfos.end(); ++cit)
        {
            (*cit)->EvaluateConstraint(**it);
            (*cit)->RecordViolation(**it);
        }
    }


    // If we did our evaluations asynchronously, we did not yet record
    // the results (because they were not available).  We need to do so
    // now.  The call to _model.synchronize causes the program to block
    // until all the results are available.  We can then record the
    // responses in the same fashion as above.  Note that the linear
    // constraints have already been computed!!
    if (_model.asynch_flag())
    {
        // Wait for the responses.
        const ResponseArray& response_array = _model.synchronize();

        // increment the number of evaluations by the number of responses
        IncrementNumberEvaluations(response_array.length());

        EDDY_ASSERT(response_array.length() == group.GetSize());

        // prepare to access the elements of the response_array by index.
        size_t i = 0;

        // Record the set of responses in the DesignGroup
        for(it=group.BeginDV(); it!=e; ++it)
        {
            // we didn't send already-evaluated Designs out for evaluation
            // so skip them here as well.
            if((*it)->IsEvaluated()) continue;

            // Put the responses into the Design properly.
            RecordResponses(response_array[i++].function_values(), **it);

            // Label this guy as now being evaluated.
            (*it)->SetEvaluated(true);

            // now check the feasibility of this design
            target.CheckFeasibility(**it);
        }
    }

    return true;
}

/*
===============================================================================
End Namespace
===============================================================================
*/
} // namespace Dakota
