/*  _______________________________________________________________________

    DAKOTA: Design Analysis Kit for Optimization and Terascale Applications
    Copyright (c) 2006, Sandia National Laboratories.
    This software is distributed under the GNU General Public License.
    For more information, see the README file in the top Dakota directory.
    _______________________________________________________________________ */

//- Class:	 GaussProcApproximation
//- Description: Class implementation for GaussianProcess Approximation
//- Owner:       Laura Swiler
//- Checked by:
//- Version:

#include "data_types.h"
#include "system_defs.h"
#include "GaussProcApproximation.H"
#include "DakotaIterator.H"
#include "DakotaResponse.H"
//#include "NPSOLOptimizer.H"
#ifdef DAKOTA_NCSU
#include "NCSUOptimizer.H"
#endif
#include "ProblemDescDB.H"

#ifdef DAKOTA_OPTPP
#include "SNLLOptimizer.H"
using OPTPP::NLPFunction;
using OPTPP::NLPGradient;
#endif

#include "Teuchos_LAPACK.hpp"
#include "Teuchos_SerialDenseHelpers.hpp"

//#define DEBUG
//#define DEBUG_FULL


namespace Dakota {
  
using Teuchos::rcp;

//initialization of statics
GaussProcApproximation* GaussProcApproximation::GPinstance(NULL);

// --------------- Distance routines for point selection -----------
// These functions are external to GaussProcApproximation, and are used by the
// point selection algorithm to determine whether distance criteria are met.

/** Gets the Euclidean distance between x1 and x2 */
Real getdist(RealVector& x1, RealVector& x2)
{
  int j,d;
  d = x1.size();
  if (d!=x2.size()) 
    Cerr << "Size mismatch in getdist in GaussProcApproximation\n"; 
  if (d==0) 
    Cerr << "Zero dimension in getdist in GaussProcApproximation\n"; 
  Real result;
  result = 0.0;
  for (j=0; j<d; j++) {
    result += (x1[j]-x2[j])*(x1[j]-x2[j]);
  }
  return sqrt(result);
}


/** Returns the minimum distance between the point x and the points in
    the set xset (compares against all points in xset except point
    "except"): if except is not needed, pass 0. */
Real mindist(RealVector& x, RealMatrix& xset, int except) 
{
  int i,j,n,d;
  d = x.size();
  if (d!= xset.num_columns()) Cout << "Dimension mismatch in mindist";
  n = xset.num_rows();
  Real dist, result;
  RealVector xp(d);
  for (i=0; i<n; i++) {
    for (j=0; j<d; j++) xp[j]=xset[i][j];
    dist = getdist(x,xp);
    if ((dist < result && i != except) || i==0) result=dist;
  }
  return result;
}


/** Gets the min distance between x and points in the set xset defined
    by the nindx values in indx. */
Real mindistindx(RealVector& x, RealMatrix& xset, IntVector& indx)
{
  int i,j,d,n,nindx;
  d = x.size();
  n = xset.num_rows();
  nindx = indx.size();
  if (nindx > n) 
    Cerr << "Size error in mindistinx in GaussProcApproximation\n";
  Real dist,result;
  RealVector xp(d);
  for (i=0; i<nindx; i++) {
    for (j=0; j<d; j++) xp[j]=xset[indx[i]][j];
    dist = getdist(x,xp);
    if (dist < result || i==0) result = dist;
  }
  return result;
}


/** Gets the maximum of the min distance between each point and the 
    rest of the set. */
Real getRmax(RealMatrix& xset)
{
  int i,j,d,n;
  d = xset.num_columns();
  n = xset.num_rows();
  if (n==0 || d==0) 
    Cerr << "Zero size in getRmax in GaussProcApproximation.  n:" 
	 << n << "  d:" << d << endl;
  Real mini,result;
  RealVector xp(d);
  for (i=0; i<n; i++) {
    for (j=0; j<d; j++) xp[j] = xset[i][j];
    mini = mindist(xp,xset,i);
    if (mini>result || i==0) result=mini;
  }
  return result;
}
// ----------------------------------------------------------------


GaussProcApproximation::
GaussProcApproximation(const ProblemDescDB& problem_db, const size_t& num_acv):
  Approximation(BaseConstructor(), problem_db, num_acv),
  usePointSelection(problem_db.get_bool("model.surrogate.point_selection")),
  trendOrder(problem_db.get_short("model.surrogate.trend_order"))
{ 
#ifdef DAKOTA_NCSU
	Cout << "Using NCSU DIRECT to optimize correlation coefficients."<<endl;
#else
        Cerr << "NCSU DIRECT Optimizer is not available to calculate " 
             << "the correlation coefficients governing the Gaussian process." 
             << "Aborting process. " << endl;
        abort_handler(-1);
#endif
}


// alternate constructor used by EffGlobalOptimization and
// NonDGlobalReliability that does not use a problem database
// defaults here are no point selectinn and quadratic trend function
GaussProcApproximation::GaussProcApproximation():
  Approximation(),trendOrder(2),usePointSelection(false)
{ }


int GaussProcApproximation::min_coefficients() const
{
  // min number of samples required to build the network is equal to
  // the number of design variables + 1

  // Note: Often this is too few samples.  It is better to have about
  // O(n^2) samples, where 'n' is the number of variables.

  return numVars+1;
}


int GaussProcApproximation::num_constraints() const
{ return (anchorPoint.is_null()) ? 0 : 1; }


void GaussProcApproximation::find_coefficients()
{
  size_t i, j, offset = 0, numObs = currentPoints.entries();
  // GaussProcApproximation does not directly handle anchorPoint
  // -> treat it as another currentPoint
  if (!anchorPoint.is_null()) {
    offset  = 1;
    numObs += 1;
  }

  // Transfer the training data to the Teuchos arrays used by the GP
  trainPoints.shapeUninitialized(numObs, numVars);
  trainValues.shapeUninitialized(numObs, 1);
  // process anchorPoint, if present
  if (!anchorPoint.is_null()) {
    const RealVector& c_vars = anchorPoint.continuous_variables();
    for (j=0; j<numVars; j++)
      trainPoints(0,j) = c_vars[j];
    trainValues(0,0) = anchorPoint.response_function();
  }
  // process currentPoints
  List<SurrogateDataPoint>::iterator it = currentPoints.begin();
  for (i=offset; i<numObs; i++, it++) {
    const RealVector& c_vars = it->continuous_variables();
    for (j=0; j<numVars; j++)
      trainPoints(i,j) = c_vars[j];
    trainValues(i,0) = it->response_function();
  }

  // Build a GP covariance model using the sampled data
  GPmodel_build();
}


const Real& GaussProcApproximation::get_value(const RealVector& x)
{ GPmodel_apply(x,false,false); return approxValue; }


const Real& GaussProcApproximation::get_variance(const RealVector& x)
{ GPmodel_apply(x,true,false); return approxVariance;}

const RealBaseVector& GaussProcApproximation::get_gradient(const RealVector& x)
{ GPmodel_apply(x,false,true); return approxGradient;}

void GaussProcApproximation::GPmodel_build()
{
  // Point selection is off by default, but will be forced if a large
  //   number of points are present
  // Note that even if the point selector uses all the points, it is not 
  //   equivalent to no point selection, b/c the point selector will 
  //   probably use only a subset of the points for MLE, so the theta 
  //   parameters may be different.
  numObs = trainValues.numRows();
  // bool use_point_selection = false;
  //if (someLogicToTurnOnPointSelection) 
  //  use_point_selection = true;

  // Normalize the inputs and output
  normalize_training_data();

  // Initialize the GP parameters
  //trendOrder = 2; // 0-constant, 1-linear, 2-quadratic

  switch (trendOrder) {
  case 0: 
    betaCoeffs.shape(1,1);
    break;
  case 1:  
    betaCoeffs.shape(numVars+1,1);
    break;
  case 2:  
    betaCoeffs.shape(2*numVars+1,1);
    break;
  }

  thetaParams.reshape(numVars);
  
  // Calculate the trend function
  get_trend();

  if (usePointSelection) {
    // Make copies of training data
    numObsAll          = numObs;
    normTrainPointsAll = normTrainPoints;
    trainValuesAll     = trainValues;
    trendFunctionAll   = trendFunction; 
    run_point_selection();
  }
  else {
    Cout << "\nBuilding GP using all " << numObs <<" training points...\n"; 
    //optimize_theta_multipoint();
    optimize_theta_global();
    get_cov_matrix();
    get_cholesky_factor();
    get_beta_coefficients();
    get_process_variance();
  }

#ifdef DEBUG
  Cout << "Theta:" << endl;
  for (size_t i=0; i<numVars; i++)
    Cout << thetaParams[i] << endl;
#endif //DEBUG
}


void GaussProcApproximation::GPmodel_apply(const RealVector& approx_pt, 
				   bool variance_flag, bool gradients_flag)
{
  if (approx_pt.size() != numVars) {
    Cout << "Dimension mismatch in GPmodel_apply" << endl;
    abort_handler(-1);
  }

  approxPoint.shapeUninitialized(1,numVars);
  for (size_t i=0; i<numVars; i++)
    approxPoint(0,i) = (approx_pt(i)-trainMeans(i))/trainStdvs(i);

  get_cov_vector();
  predict(variance_flag,gradients_flag);
}


void GaussProcApproximation::normalize_training_data()
{
  size_t i,j;
  trainMeans.sizeUninitialized(numVars);
  trainStdvs.sizeUninitialized(numVars);
  normTrainPoints = trainPoints;

 //get means of each column for X input
  for (i=0; i<numVars; i++) {
    Real sum = 0.;
    for (j=0; j<numObs; j++)
      sum += normTrainPoints(j,i);
    trainMeans(i) = sum/(double(numObs));
  }

  //subtract means from each value in each column
  for (i=0; i<numVars; i++) {
    trainStdvs(i)=0.;
    for (j=0; j<numObs; j++) {
      normTrainPoints(j,i) -= trainMeans(i);
      trainStdvs(i) += pow(normTrainPoints(j,i),2);
    }
    trainStdvs(i) = sqrt(trainStdvs(i)/double(numObs-1));
  }

  //divide by standard deviation
  for (i=0; i<numVars; i++)
    for (j=0; j<numObs; j++)
      normTrainPoints(j,i) /= trainStdvs(i);
 
#ifdef DEBUG_FULL
  for (j=0; j<numObs; j++) {
    for (i=0; i<numVars; i++)
      Cout <<" input  " << normTrainPoints(j,i) << '\n';
    Cout <<" output " <<     trainValues(j,0) << '\n';
  }
#endif // DEBUG_FULL
}


void GaussProcApproximation::get_trend()
{
  switch (trendOrder) {
  case 0: 
    trendFunction.shapeUninitialized(numObs,1);
    break;
  case 1: 
    trendFunction.shapeUninitialized(numObs,numVars+1);
    break;
  case 2: 
    trendFunction.shapeUninitialized(numObs,2*numVars+1);
    break;
  }

  // all orders require ones in first column
  for (size_t i=0; i<numObs; i++)
    trendFunction(i,0) = 1.;
 
  if (trendOrder > 0) {
    for (size_t j=0; j<numVars; j++) {
      for (size_t i=0; i<numObs; i++) {
        trendFunction(i,j+1) = normTrainPoints(i,j);
        if (trendOrder == 2) 
          trendFunction(i,numVars+j+1)
	    = normTrainPoints(i,j)*normTrainPoints(i,j);
      }
    }
  }
}


void GaussProcApproximation::optimize_theta_global()
{
  GPinstance = this;
  Iterator nll_optimizer; // empty envelope

  // bounds for non-log transformation - ie, no exp(theta)
  //RealVector theta_lbnds(numVars,1.e-5), theta_ubnds(numVars,150.);
  // bounds for log transformation of correlation parameters
  RealVector theta_lbnds(numVars,-9.), theta_ubnds(numVars,5.);
#ifdef DAKOTA_NCSU  
  //NCSU DIRECT optimize of Negative Log Likelihood
  nll_optimizer.assign_rep(
    new NCSUOptimizer(theta_lbnds,theta_ubnds,1000,10000,negloglikNCSU),false);
  nll_optimizer.run_iterator();
  const Variables& vars_star = nll_optimizer.variables_results();
  const Response&  resp_star = nll_optimizer.response_results();
  thetaParams = vars_star.continuous_variables();

#ifdef DEBUG
  Cout << "Optimal NLL = " << resp_star.function_values()[0] << '\n';
#endif //DEBUG
#endif //DAKOTA_NCSU
}


void GaussProcApproximation::get_cov_matrix()
{
  // Note, this gets only the lower triangle of covmatrix, which is all
  // that is needed by the Teuchos SPD solvers.  However, if this matrix
  // is to be used in full matrix multiplication, or with a non-SPD
  // solver, then the lower part should be copied to the upper part.

  size_t i,j,k;
  covMatrix.shape(numObs);
  Real sume, delta;
  for (j=0; j<numObs; j++){
    for (k=j; k<numObs; k++){
      sume = 0.;
      for (i=0; i<numVars; i++){
	Real pt_diff = normTrainPoints(j,i) - normTrainPoints(k,i);
	//sume += thetaParams[i]*pt_diff*pt_diff;
	sume += exp(thetaParams[i])*pt_diff*pt_diff;
      }
      covMatrix(k,j) = exp(-1.*sume);
    }
  }

#ifdef DEBUG_FULL
  Cout << "covariance matrix" << '\n';
  for (j=0; j<numObs; j++){
    for (k=0; k<numObs; k++)
      Cout << covMatrix(j,k) << " ";
    Cout << '\n';
  }
  Cout << '\n';
#endif //DEBUG_FULL
}


int GaussProcApproximation::get_cholesky_factor()
// Gets the Cholesky factorization of the covariance matrix.  If this
// is called by prediction routines, then they are basically "stuck"
// with the current covariance matrix, so if it turns out to be
// singular to wp, then we iteratively condition it with very small
// values until it is no longer singular.  However, if this routine is
// called by the likelihood estimation procedures, we want them to
// "know" about singular covariance as opposed to "working around it"
// by conditioning.  Thus, this routine returns an integer of 0 if the
// covariance matrix is positive definite, and returns 1 if it is
// singular to wp.
{
  int ok,i;
  Real nugget = 1e-15;
  covSlvr.setMatrix( rcp(&covMatrix, false) );
  covSlvr.factorWithEquilibration(true);
  ok = covSlvr.factor();
  if (ok > 0) {
    do {
      get_cov_matrix();
      for (i=0; i<numObs; i++) covMatrix(i,i) += nugget;
      covSlvr.setMatrix( rcp(&covMatrix, false) );
      covSlvr.factorWithEquilibration(true);
      ok = covSlvr.factor();
      nugget *= 3.0;
    } while (ok > 0);
#ifdef DEBUG_FULL
    Cout << "COV matrix corrected with nugget: " << nugget/3. << endl;
#endif
    cholFlag = 1;
    return 1;
  }
  else {
    cholFlag = 0;
    return 0;
  }
}
  

void GaussProcApproximation::get_beta_coefficients()
{
  // Default is generalized least squares, but ordinary least squares is
  //    available if desired
  bool ordinary = false;

  size_t trend_dim = 1 + trendOrder * numVars;
  // Solver for the identity matrix - only used if OLS requested
  RealSpdDenseSolver eye_slvr;
  RealSymDenseMatrix identity_matrix;
  if (ordinary) {
    // Construct the identity matrix:
    identity_matrix.shape(numObs); // inits to 0
    for (size_t i=0; i<numObs; i++)
      identity_matrix(i,i) = 1.;
    eye_slvr.setMatrix( rcp(&identity_matrix, false) );
  }

  RealDenseMatrix Rinv_Y(numObs, 1, false);
  if (ordinary) {
    eye_slvr.setVectors( rcp(&Rinv_Y, false), rcp(&trainValues, false) );
    eye_slvr.solve();
  }
  else {
    covSlvr.setVectors( rcp(&Rinv_Y, false), rcp(&trainValues, false) );
    covSlvr.solve();
  }

  RealDenseMatrix FT_Rinv_Y(trend_dim, 1, false);
  FT_Rinv_Y.multiply(Teuchos::TRANS, Teuchos::NO_TRANS, 1.0, trendFunction,
		     Rinv_Y, 0.0);
  
  RealDenseMatrix Rinv_F(numObs, trend_dim, false);
  if (ordinary) {
    eye_slvr.setVectors( rcp(&Rinv_F, false), rcp(&trendFunction, false) );
    eye_slvr.solve();
  }
  else {
    covSlvr.setVectors( rcp(&Rinv_F, false), rcp(&trendFunction, false) );
    covSlvr.solve();
  }

  RealDenseMatrix FT_Rinv_F(trend_dim, trend_dim, false);
  FT_Rinv_F.multiply(Teuchos::TRANS, Teuchos::NO_TRANS, 1., trendFunction,
		     Rinv_F, 0.);
  // LPS TO DO:
  //RealSymDenseMatrix FT_Rinv_F(trend_dim, false);
  //Teuchos::symMatTripleProduct(Teuchos::TRANS, 1., trendFunction,
  //                             Rinv, FT_Rinv_F);
  //RealSpdDenseSolver Temp_slvr;
 
  RealDenseMatrix temphold5(trend_dim, 1, false);
  RealDenseSolver Temp_slvr;
  Temp_slvr.setMatrix( rcp(&FT_Rinv_F, false) );
  Temp_slvr.setVectors( rcp(&temphold5, false), rcp(&FT_Rinv_Y, false) );
  Temp_slvr.factorWithEquilibration(true);
  Temp_slvr.factor();
  Temp_slvr.solve();

  for (size_t i=0; i<trend_dim; i++)
    betaCoeffs(i,0) = temphold5(i,0);

  if (betaCoeffs[0][0] != betaCoeffs[0][0]) 
    Cerr << "Nan for beta at exit of get_beta in GaussProcApproximation\n";
}


void GaussProcApproximation::get_process_variance()
{
  RealDenseMatrix YFb(numObs, 1, false), Rinv_YFb(numObs, 1, false),
    temphold3(1, 1, false);

  YFb.multiply(Teuchos::NO_TRANS, Teuchos::NO_TRANS, 1., trendFunction,
	       betaCoeffs, 0.);
  YFb.scale(-1);
  YFb += trainValues;

  covSlvr.setVectors( rcp(&Rinv_YFb, false), rcp(&YFb, false) );
  covSlvr.solve();

  temphold3.multiply(Teuchos::TRANS, Teuchos::NO_TRANS, 1., YFb, Rinv_YFb, 0.);
 
  procVar = temphold3(0,0)/double(numObs);
}


void GaussProcApproximation::get_cov_vector()
{
  covVector.shapeUninitialized(numObs,1);

  for (size_t j=0; j<numObs; j++){
    Real sume = 0.;    
    for (size_t i=0; i<numVars; i++){
      //sume += thetaParams[i]*pow((normTrainPoints(j,i)-approxPoint(0,i)),2);
      sume +=
	exp(thetaParams[i])*pow((normTrainPoints(j,i)-approxPoint(0,i)),2);
    }
    covVector(j,0) = exp(-1.*sume);
  }

#ifdef DEBUG_FULL
  Cout << "covariance vector" << '\n';
  for (size_t j=0; j<numObs; j++)
    Cout << covVector(j,0) << " ";
  Cout << '\n';
#endif //DEBUG_FULL
}

void GaussProcApproximation::predict(bool variance_flag, bool gradients_flag)
{
  size_t i,j,k;
  RealDenseMatrix f_xstar;
  switch (trendOrder) {
  case 0:
    f_xstar.shapeUninitialized(1,1);
    f_xstar(0,0) = 1.;
    break;
  case 1:
    f_xstar.shapeUninitialized(1,numVars+1);
    f_xstar(0,0) = 1.;
    for (j=0; j<numVars; j++)
      f_xstar(0,j+1) = approxPoint(0,j);
    break;
  case 2:
    f_xstar.shapeUninitialized(1,2*numVars+1);
    f_xstar(0,0) = 1.;
    for (j=0; j<numVars; j++) {
      f_xstar(0,j+1) = approxPoint(0,j);
      f_xstar(0,numVars+j+1) = approxPoint(0,j)*approxPoint(0,j);
    }
    break;
  }

  RealDenseMatrix YFb(numObs, 1, false);
  YFb.multiply(Teuchos::NO_TRANS, Teuchos::NO_TRANS, 1., trendFunction,
	       betaCoeffs, 0.);
  YFb.scale(-1);
  YFb += trainValues;

  RealDenseMatrix Rinv_covvector(numObs, 1, false);
  covSlvr.setVectors( rcp(&Rinv_covvector, false), rcp(&covVector, false) );
  covSlvr.solve();

  RealDenseMatrix approx_val(1, 1, false);
  approx_val.multiply(Teuchos::TRANS, Teuchos::NO_TRANS, 1., Rinv_covvector,
		      YFb, 0.);

  RealDenseMatrix f_beta(1, 1, false);
  f_beta.multiply(Teuchos::NO_TRANS, Teuchos::NO_TRANS, 1., f_xstar, 
		  betaCoeffs, 0.);

  approxValue = approx_val(0,0) + f_beta(0,0);

#ifdef DEBUG_FULL
  Cout << "prediction " << approxValue << '\n';
#endif //DEBUG_FULL

  // Gradient of prediction with respect to input
  if (gradients_flag) {
    // Currently, these gradients are only valid for constant trend
    Cout << "\n<<<<< Entering gradient prediction in GP code";
    if (trendOrder!=0) {
      Cout << "GP gradients currently only valid for constant trend\n";
      abort_handler(-1);
    }
    // Construct the global matrix of derivatives, gradCovVector
    get_grad_cov_vector();

    // I'll reuse Rinv_covvector to store inv(R)*YFb
    covSlvr.setVectors( rcp(&Rinv_covvector, false), rcp(&YFb, false) );
    covSlvr.solve();

    RealDenseMatrix gradPred(numVars, 1, false), dotProd(1, 1, false),
      gradCovVector_i(numObs, 1, false);
    approxGradient.reshape(numVars);
    for (i=0; i<numVars; i++) {
      // First set up gradCovVector_i to be the ith column of the matrix
      // gradCovVector -- Teuchos allow for a more elegant way of doing this...
      for (j=0; j<numObs; j++) 
	gradCovVector_i(j,0) = gradCovVector(j,i);
      // Grad_i = dot_prod(gradCovVector_i, inv(R)*YFb)
      dotProd.multiply(Teuchos::TRANS, Teuchos::NO_TRANS, 1., Rinv_covvector,
		       gradCovVector_i, 0.);
      approxGradient[i] = gradPred(i,0) = dotProd(0,0);
    }
  }
  // End of gradient computations
  

  if (variance_flag) {
    //RealDenseMatrix var_predicted(1,1);

    RealDenseMatrix Rinv_covvec(numObs, 1, false), rT_Rinv_r(1, 1, false);

    covSlvr.setVectors( rcp(&Rinv_covvec, false), rcp(&covVector, false) );
    covSlvr.solve();

    rT_Rinv_r.multiply(Teuchos::TRANS, Teuchos::NO_TRANS, 1., covVector,
		       Rinv_covvec, 0.);

    //var_predicted(0,0) = procVar*(1.0-rT_Rinv_r(0,0));
    //approxVariance = var_predicted(0,0);
    approxVariance = procVar*(1.- rT_Rinv_r(0,0));

#ifdef DEBUG_FULL
      Cout << "variance " << approxVariance << '\n';
      Cout << '\n';
#endif //DEBUG_FULL

    // set this flag to true to use longer equation
    bool full_var_flag = true;
    if (full_var_flag) {
      size_t trend_dim = 1 + trendOrder * numVars;
      
      //RealDenseMatrix full_variance(1,1);
      RealDenseMatrix f_FT_Rinv_r(trend_dim, 1, false),
	f_xstar_T(trend_dim, 1, false),	Rinv_F(numObs, trend_dim, false),
	temp9(trend_dim, 1, false), temp10(1,1, false);
      f_FT_Rinv_r.multiply(Teuchos::TRANS, Teuchos::NO_TRANS, 1., trendFunction,
			   Rinv_covvec, 0.); 
      f_FT_Rinv_r.scale(-1);

      for (size_t i=0; i<trend_dim; i++) 
	f_xstar_T(i,0) = f_xstar(0,i);
      f_FT_Rinv_r += f_xstar_T;
      
      covSlvr.setVectors( rcp(&Rinv_F, false), rcp(&trendFunction, false) );
      covSlvr.solve();

      RealDenseMatrix FT_Rinv_F(trend_dim, trend_dim, false);
      FT_Rinv_F.multiply(Teuchos::TRANS, Teuchos::NO_TRANS, 1., trendFunction,
			 Rinv_F, 0.);
      // LPS TO DO:
      //RealSymDenseMatrix FT_Rinv_F(trend_dim, false);
      //Teuchos::symMatTripleProduct(Teuchos::TRANS, 1., trendFunction,
      //                             Rinv, FT_Rinv_F);
      //RealSpdDenseSolver Temp_slvr;

      RealDenseSolver Temp_slvr;
      Temp_slvr.setMatrix( rcp(&FT_Rinv_F, false) );
      Temp_slvr.setVectors( rcp(&temp9, false), rcp(&f_FT_Rinv_r, false) );
      Temp_slvr.factorWithEquilibration(true);
      Temp_slvr.factor();
      Temp_slvr.solve();

      temp10.multiply(Teuchos::TRANS, Teuchos::NO_TRANS, 1., temp9, 
		      f_FT_Rinv_r, 0.);

      //full_variance(0,0)=procVar*(1.0-rT_Rinv_r(0,0)+temp10(0,0));
      //approxVariance = full_variance(0,0);
      approxVariance = procVar*(1.- rT_Rinv_r(0,0) + temp10(0,0));

#ifdef DEBUG_FULL
      Cout << "full variance " << approxVariance << "\n\n";
#endif //DEBUG_FULL
    }
 
    // check for small (possibly negative) variance
    if (approxVariance < 1.e-9)
      approxVariance = 1.e-9;
  }
}


void GaussProcApproximation::get_grad_cov_vector()
// Gets the derivatives of the covVector with respect to each
// component in X (note that the derivatives are taken with respect to
// the pre-transformed values of X).  Each column of gradCovVector is
// set to one of the derivatives.  covVector will be accessed, so it
// must be set prior to calling get_grad_cov_vector, and approxPoint
// will be accessed as well.  Also note that this is currently only
// valid for the squared-exponential correlation function.
{
  gradCovVector.shapeUninitialized(numObs,numVars);
  for (size_t i=0; i<numObs; i++)
    for (size_t j=0; j<numVars; j++)
      gradCovVector(i,j) = covVector(i,0) * -2. * exp(thetaParams[j]) *
	(approxPoint(0,j) - normTrainPoints(i,j)) / trainStdvs(j);
      // division by trainStdvs(j) accounts for standardization transformation:
}


Real GaussProcApproximation::calc_nll()
// This routine is called only after the covariance matrix has been
// computed and factored, so that we work with the global solver
// Cov_slvr
{
  Real det = 1., nll;
  // determinant based on Cholesky factor is simply the square of
  // the product of the diagonals

  for (size_t i=0; i<numObs; i++)
    det = det*(*covSlvr.getFactoredMatrix())(i, i);
  det = det*det;
  if (det <= 0.) cholFlag = -1; // indicates singular matrix

  if (cholFlag == 0) { //not singular
    get_beta_coefficients();
    get_process_variance();

    nll = numObs*log(procVar) + log(det);
  }
  else { 
#ifdef DEBUG_FULL
    Cout << "Singular covariance encountered" << endl;
#endif
    nll = 5.e100; 
  }
  return (nll);
}

// function maps a single point x to function value f
double GaussProcApproximation::negloglikNCSU(const RealVector &x)
{
  for (size_t i=0; i<x.size(); i++)
    GPinstance->thetaParams[i] = x[i];
  GPinstance->get_cov_matrix();
  GPinstance->get_cholesky_factor(); //get the factored matrix only once
  return(GPinstance->calc_nll());
}

void GaussProcApproximation::optimize_theta_multipoint()
{
  GPinstance = this;
  Iterator nll_optimizer; // empty envelope

  // bounds for non-log transformation - ie, no exp(theta)
  //RealVector theta_lbnds(numVars,1.e-5), theta_ubnds(numVars,150.);
  // bounds for log transformation of correlation parameters
  RealVector theta_lbnds(numVars,-9.), theta_ubnds(numVars,5.);
  RealMatrix lin_ineq_coeffs, lin_eq_coeffs, constraint_evals;
  RealVector lin_ineq_lower_bnds, lin_ineq_upper_bnds, lin_eq_targets,
  nln_ineq_lower_bnds, nln_ineq_upper_bnds, nln_eq_targets; 
  
  size_t i,j;
  RealVector THETA_INIT_STARTS(3);
  THETA_INIT_STARTS[0] = log(0.1);
  THETA_INIT_STARTS[1] = log(1.0);
  THETA_INIT_STARTS[2] = log(4.0);
  Real nll, nll_best = DBL_MAX;
  RealVector theta_best(numVars);

#ifdef DAKOTA_OPTPP
  for (i=0; i<3; i++) {
    for (j=0; j<numVars; j++)
      thetaParams[j] = THETA_INIT_STARTS[i];
    nll_optimizer.assign_rep(
      new SNLLOptimizer(
        thetaParams, theta_lbnds, theta_ubnds, lin_ineq_coeffs,
	lin_ineq_lower_bnds, lin_ineq_lower_bnds, lin_eq_coeffs,
	lin_eq_targets, nln_ineq_lower_bnds, nln_ineq_upper_bnds,
	nln_eq_targets, negloglik, constraint_eval), false);
    nll_optimizer.run_iterator();
    const Variables& vars_star = nll_optimizer.variables_results();
    const Response&  resp_star = nll_optimizer.response_results();
    thetaParams = vars_star.continuous_variables();
    nll = resp_star.function_values()[0];
    if (nll < nll_best) {
      nll_best = nll;
      for (j=0; j<numVars; j++) 
	theta_best[j] = thetaParams[j];
    }
  }
  // Now put the best run back into thetaParams and nll
  for (j=0; j<numVars; j++)
    thetaParams[j] = theta_best[j];
  nll = nll_best;
#endif //OPTPP
  
#ifdef DEBUG
  Cout << "Optimal NLL = " << nll << '\n';
#endif //DEBUG
}


void GaussProcApproximation::calc_grad_nll()
{
  size_t i,j,k;
  double trace_i;

  // First check if the covariance is singular.
  Real det = 1.;
  for (i=0; i<numObs; i++)
    det = det*(*covSlvr.getFactoredMatrix())(i, i);
  det = det*det;
  if (det <= 0.) cholFlag = -1;  //Singular matrix

  gradNegLogLikTheta.shapeUninitialized(numVars,1);

  if (cholFlag == 0) { // ok to go ahead with gradient calcs

    RealDenseMatrix YFb(numObs, 1, false), Rinv_Y(numObs, 1, false),
      Rk(numObs, numObs, false), trace(numObs, numObs, false),
      temphold5(numObs, 1, false), temphold6(1, 1, false);
    YFb.multiply(Teuchos::NO_TRANS, Teuchos::NO_TRANS, 1., trendFunction,
		 betaCoeffs, 0.);
    YFb.scale(-1);
    YFb += trainValues;

    covSlvr.setVectors( rcp(&Rinv_Y, false), rcp(&YFb, false) );
    covSlvr.solve();
    // Note that trans(Y)*inv(R) = transpose(inv(R)*Y)

    // Note: Rk is symmetric, but trace=inv(R)*Rk is not.  This is the
    // only time we need to do a matrix fill of a symmetric matrix
 
    for (i=0; i<numVars; i++){
      for (k=0; k<numObs; k++){
	for (j=k; j<numObs; j++){
	  Real pt_diff = normTrainPoints(j,i)-normTrainPoints(k,i);
	  //Rk(j,k) = -1*thetaParams[i]*pt_diff*pt_diff*covMatrix(j,k);
	  Rk(j,k) = -exp(thetaParams[i])*pt_diff*pt_diff*covMatrix(j,k);
	  Rk(k,j) = Rk(j,k);  //fill upper part without redoing the calcs
	}
      }
  
      covSlvr.setVectors( rcp(&trace, false), rcp(&Rk, false) );
      covSlvr.solve();

      trace_i = 0.;
      for (j=0; j<numObs; j++)
	trace_i += trace(j,j);
    
      temphold5.multiply(Teuchos::NO_TRANS, Teuchos::NO_TRANS, 1., Rk,
			 Rinv_Y, 0.);
      temphold6.multiply(Teuchos::TRANS, Teuchos::NO_TRANS, 1., Rinv_Y, 
			 temphold5, 0.);

      gradNegLogLikTheta(i,0) = trace_i - ((1./procVar)*temphold6(0,0));
    }
  }
  else {
#ifdef DEBUG_FULL
    Cout << "Singular covariance encountered in gradient routine" << endl;
#endif
    //encourage thetas to increase, but not too much
    for (i=0; i<numVars; i++)
      gradNegLogLikTheta(i,0) = -1.e3;
  }
}


#ifdef DAKOTA_OPTPP
void GaussProcApproximation::
negloglik(int mode, int n, const NEWMAT::ColumnVector& x, NEWMAT::Real& fx, 
	  NEWMAT::ColumnVector& grad_f, int& result_mode)
{
  for (size_t i=0; i<n; i++)
    GPinstance->thetaParams[i] = x(i+1);
  GPinstance->get_cov_matrix();
  GPinstance->get_cholesky_factor(); //get the factored matrix only once
  fx = GPinstance->calc_nll();
  GPinstance->calc_grad_nll();
  if (mode & NLPFunction){
    result_mode = NLPFunction;
  }
  if (mode & NLPGradient){
    for (size_t i=0; i<n; i++)
      grad_f(i+1)=GPinstance->gradNegLogLikTheta(i,0);
    result_mode = NLPGradient;
  }
}


void GaussProcApproximation::
constraint_eval(int mode, int n, const NEWMAT::ColumnVector& x, 
		NEWMAT::ColumnVector& g, NEWMAT::Matrix& grad_c, 
		int& res_mode)
{ }
#endif //DAKOTA_OPTPP


void GaussProcApproximation::run_point_selection()
// Runs the point selection algorithm, which may modify numObs,
// normTrain, and trainVals accordingly.  The point selector works by starting
// out with an initial subset of the training data of size
// numVars^2+1 and iteratively adding points in an optimal fashion.
// This incurs additional computational costs associated with building
// the GP model, but it has the advantage that it will usually avoid
// situations where clusters of points or very large data sets result
// in a singular covariance matrix (for any reasonable values of
// theta).
{
  const int N_MAX_MLE = 35, N_MAX = 500, LOOP_MAX = 100;
  int counter = 0, Chol_return=0, delta_increase_count = 0, i, j, nadded;
  const Real TOLDELTA = 1e-2;
  RealVector delta(numObsAll);
  Real dmaxprev, maxdelta;

  Cout << "\nUsing point selection routine..." << endl;

  pointsAddedIndex.reshape(0);  // initialize
  initialize_point_selection();

  do {
    dmaxprev = maxdelta;
    //Cout << "\n\n\nNew point_sel iteration, n = " << numObs << endl;
    if (numObs<N_MAX_MLE) {
      // Note that within optimize_nll(), before each call to the
      // objective function, get_cov_matrix() and get_cholesky_factor()
      // are both called
      optimize_theta_global();
      //optimize_theta_multipoint();
#ifdef DEBUG_FULL
      Cout << "New theta values:  " << thetaParams[0] << "  " 
	   << thetaParams[1] << endl;
#endif
    }
    get_cov_matrix();
    get_cholesky_factor();
    pointsel_get_errors(delta);
    nadded = pointsel_add_sel(delta);
    maxdelta = maxval(delta);
    Cout << "Points: " << numObs << "  Maxdelta: " << maxdelta << endl;
    if (maxdelta >= dmaxprev && counter!=0) delta_increase_count++;
    if (maxdelta < dmaxprev) delta_increase_count=0;
    counter ++;

    // Need to allow maxdelta to go up one or two times to avoid stopping too early
  } while(maxdelta > TOLDELTA && numObs < N_MAX && counter < LOOP_MAX 
	  && numObs< numObsAll && delta_increase_count<=5);

  get_process_variance(); //in case variance estimates are needed

  Cout << "Number of points used:  " << numObs << endl;
  Cout << "Maximum CV error at next to last iteration:  " << maxdelta;
  if (numObsAll-(numObs-nadded)<=5) 
    Cout << "  (only " << numObsAll-(numObs-nadded) << " CV test point(s))" 
	 << endl;
  else Cout << endl;

#ifdef DEBUG_FULL
  Cout << "Final theta values:\n";
  for (i=0;i<numVars;i++) Cout << i+1 << ":  " << thetaParams[i] << endl;
#endif //DEBUG_FULL

  if (numObs < numObsAll/2.5 && numObs < 100*numVars) {
    Cerr << "***Possible early termination of point selection in " 
	 << "GaussProcApproximation***" << endl;
    Cerr << "***Only " << numObs << " of " << numObsAll 
	 << " points were used" << endl;
  }
}


void GaussProcApproximation::initialize_point_selection()
// This function initializes the point selection algorithm by choosing
// a small initial subset of the training points with which to start.
// A possible future improvement might be to make sure this initial
// set isn't a "cluster" -- this could make use of the various
// distance calculation routines
{
  size_t i,ninit;
  if (numVars==1) 
    ninit = (5 <= numObs) ? 5 : numObs;
  else 
    ninit = (numVars*numVars+1 <= numObs) ? numVars*numVars+1 : numObs;

  size_t q; // dimension of trend function
  q = trendFunction.numCols();

  numObs = ninit;

  normTrainPoints.reshape(numObs,numVars);
  trainValues.reshape(numObs,1);
  trendFunction.reshape(numObs,q);

  for (i=0;i<numObs;i++)
    pointsAddedIndex.push_back(i);
}


void GaussProcApproximation::pointsel_get_errors(RealVector& delta)
// Uses the current GP model to compute predictions at all of the
// training points and find the errors
{
  size_t i,j;
  RealVector xpred(numVars);

  for (i=0; i<numObsAll; i++) {
    for (j=0; j<numVars; j++) 
      xpred[j] = trainPoints(i,j);  
    //Here we want to pass un-normalized values to the predictor
    GPmodel_apply(xpred,false,false);
    delta[i] = approxValue;
    delta[i] -= trainValuesAll(i,0);
    delta[i] = fabs(delta[i]);
    //Cout << i << "  abs error: " << delta[i] << endl;
  }
}


namespace idx_table
// stuff for making a sort index.  Simply call
// idx_table:indexx(x.begin(),x.end(),indx.begin()).  x will not be
// altered, and the integer vector indx will be such that x[indx[i]]
// is in ascending order
{
   template<class T, class U>
   struct ComparePair1stDeref
   {
      bool operator()( const std::pair<T,U>& a,
		       const std::pair<T,U>& b ) const
      { return *a.first < *b.first; }
   };

   template<class IterIn, class IterOut>
   void indexx( IterIn first, IterIn last, IterOut out )
   {
      std::vector< std::pair<IterIn,int> > s( last-first );
      for( int i=0; i < s.size(); ++i )
	 s[i] = std::make_pair( first+i, i );
      std::sort( s.begin(), s.end(),
		 ComparePair1stDeref<IterIn,int>() );
      for( int i=0; i < s.size(); ++i, ++out )
	 *out = s[i].second;
   }
}


int GaussProcApproximation::pointsel_add_sel(RealVector& delta)
// Uses unsorted errors in delta, finds which points to add, and adds
// them
{
  size_t ntest, i, j, itest, nadded;
  int chol_return;
  IntVector added_index(0), indx(numObsAll);
  const Real alph = 0.05;
  Real Rmax, dist;
  RealVector xp(numVars);

  // Stuff for copying Teuchos data:
  RealMatrix xmat, xallmat;
  copy_data(normTrainPoints,    xmat);
  copy_data(normTrainPointsAll, xallmat);

  idx_table::indexx(delta.begin(),delta.end(),indx.begin());

  ntest = int(alph*(numObsAll-numObs));
#ifdef DEBUG_FULL
  Cout << "Max delta:  " << delta[indx[numObsAll-1]] << endl;
#endif

  addpoint(indx[numObsAll-1],added_index);
  Rmax = getRmax(xmat);

  //Now test remaining 100*alph % points for closeness criterion
  for (i=3; i<ntest+3; i++) {
    itest = indx[numObsAll-i+1];
    for (j=0; j<numVars; j++) xp[j] = normTrainPointsAll(itest,j);
    dist = mindistindx(xp,xallmat,added_index);
    if (dist > 0.5*Rmax || added_index.size()==0) addpoint(itest,added_index);
  }

  if (added_index.size()>0)
    nadded = added_index.size()-1;
  else 
    nadded = 0;

#ifdef DEBUG_FULL
  Cout << "Added " << nadded << " of " << ntest << " new points" << endl; 
  // The - 1 above is to not count the first point added (the maxdelta point)
  Cout << "Total points:  " << numObs << endl << endl << endl;
#endif

  get_cov_matrix();
  chol_return = get_cholesky_factor();
  get_beta_coefficients();

  return nadded+1;
}


Real GaussProcApproximation::maxval(RealVector& x)  
{
  Real result;
  for (size_t i=0; i<x.size(); i++) {
    if (x[i]>result || i==0) 
      result = x[i];
  }
  return result;
}


int GaussProcApproximation::addpoint(int pnum, IntVector& added_index)
// Adds a point to the effective training set.  Returns 1 on success.
{
  size_t i, q = trendFunction.numCols();
  bool already_in = false;
  // JMM: shouldn't need this already_in check b/c delta should be 0
  // (dist criteria only checks against points in the local addindx
  // set).  However, delta!=0 may indicate numerical problems with the
  // covariance

  for (i=0; i<numObs; i++) {
    if (pnum==pointsAddedIndex.at(i)) 
      already_in = true;
  }

  if (!already_in) {

    numObs++;

    normTrainPoints.reshape(numObs,numVars);
    trendFunction.reshape(numObs,q);
    trainValues.reshape(numObs,1);

    for (i=0; i<numVars; i++) 
      normTrainPoints(numObs-1,i) = normTrainPointsAll(pnum,i);

    for (i=0; i<q; i++) 
      trendFunction(numObs-1,i) = trendFunctionAll(pnum,i);

    trainValues(numObs-1,0) = trainValuesAll(pnum,0);

    added_index.push_back(pnum);
    pointsAddedIndex.push_back(pnum);

    return 1;
  }
  else {
#ifdef DEBUG_FULL
    Cout << "Attempted to add a point already in the set: " << pnum << endl;
#endif
    return 0;
  }
}


void GaussProcApproximation::pointsel_write_points()
{
// Writes the training set before and after point selection, so they
// may be easily compared
  writex("daktx.txt");
  run_point_selection();
  writex("daktxsel.txt");
}


void GaussProcApproximation::lhood_2d_grid_eval()
{
  // Can be used to evaluate the likelihood on a grid (for problems
  // with two dimensional input
  Real lb = 1e-9, ub = 1.0, nll, eps = 0.005;
  if (numVars != 2) {
    Cerr << "lhood_2d_grid_eval is only for two-dimensional problems\n";
    return;
  }
  thetaParams[0] = lb;
  thetaParams[1] = lb;
  ofstream fout("lhood.txt");
  do {
    do {
      get_cov_matrix();
      get_cholesky_factor();
      nll = calc_nll();
      if (nll > 10000) 
	nll = 1000.0;
      fout << thetaParams[0] << " " << thetaParams[1] << " " << nll << endl;
      thetaParams[0] += eps;
    } while (thetaParams[0] < ub);
    thetaParams[1] += eps;
    thetaParams[0] = lb;
  } while (thetaParams[1] < ub);
  fout.close();
}


void GaussProcApproximation::writeCovMat(char filename[])
{
  ofstream fout(filename);
  size_t i,j;
  for (i=0; i<numObs; i++) {
    for (j=0; j<numObs; j++) 
      fout << covMatrix(i,j) << "\t";
    fout << endl;
  }
  fout.close();
}
  

void GaussProcApproximation::writex(char filename[])
{
  // writes out the current training set in original units.  This can
  // be used to compare the selected subset to the entire set by
  // calling this before and after point selection with different
  // filenames.  Also, we convert back to original units before
  // writing.
  ofstream fout(filename);
  size_t i,j;
  //RealDenseMatrix x_orig_units(normTrainPoints);
  Real xij;
  for (i=0; i<numObs; i++) {
    for (j=0; j<numVars; j++) {
      xij = normTrainPoints(i,j)*trainStdvs(j) + trainMeans(j);
      fout << xij << "\t";
    }
    fout << endl;
  }
  fout.close();
}

} // namespace Dakota
