/*  _______________________________________________________________________

    DAKOTA: Design Analysis Kit for Optimization and Terascale Applications
    Copyright (c) 2006, Sandia National Laboratories.
    This software is distributed under the GNU General Public License.
    For more information, see the README file in the top Dakota directory.
    _______________________________________________________________________ */

//- Class:        DirectApplicInterface
//- Description:  Class implementation
//- Owner:        Mike Eldred

#include "ParamResponsePair.H"
#include "DirectApplicInterface.H"
#include "system_defs.h"
#include "ProblemDescDB.H"
#include "ParallelLibrary.H"
//#include <unistd.h> // for sleep(int)
#ifdef DAKOTA_MODELCENTER
#include "PHXCppApi.h"
#endif
#ifdef DAKOTA_MATLAB
#include "engine.h"
#endif
#ifdef DAKOTA_PYTHON
#include <Python.h>
#ifdef DAKOTA_PYTHON_NUMPY
#include <arrayobject.h>
#endif
#endif

namespace Dakota {


#define POW_VAL 1.0 // text_book: 1.0 is nominal, 1.4 used for B&B testing

#ifdef DAKOTA_SALINAS
/// subroutine interface to SALINAS simulation code
int salinas_main(int argc, char *argv[], MPI_Comm* comm);
#endif // DAKOTA_SALINAS

#ifdef DAKOTA_MATLAB
#ifndef MWSIZE_MAX
// Older Matlab versions used int for sizes.  Newer versions
// (7.3/R2006b and later) typedef mwSize as int or size_t in
// tmwtypes.h.  Provide definition for mwSize if necessary:
#define mwSize int
#endif
/* fields to pass to Matlab in Dakota structure */
// TODO: resolve number of feidls / remove failure
const char *FIELD_NAMES[] = { "numFns", "numVars", "numACV", "numADV",  // 0
			      "numDerivVars", "xC", "xD", "xCLabels",   // 4
			      "xDLabels", "directFnASV", "directFnASM", // 8
			      "directFnDVV", "directFnDVV_bool",        // 11
			      "fnFlag", "gradFlag", "hessFlag",         // 13
			      "fnVals",  "fnGrads",  "fnHessians",      // 16
			      "fnLabels", "failure" };                  // 19
const int NUMBER_OF_FIELDS = 21;
#endif

DirectApplicInterface::
DirectApplicInterface(const ProblemDescDB& problem_db):
  ApplicationInterface(problem_db),
  iFilterName(problem_db.get_string("interface.application.input_filter")),
  oFilterName(problem_db.get_string("interface.application.output_filter")),
  gradFlag(false), hessFlag(false), numFns(0), numVars(0), numDerivVars(0),
  analysisDrivers(problem_db.get_dsa("interface.application.analysis_drivers")),
  analysisComponents(
    problem_db.get_ds2a("interface.application.analysis_components"))
{ 
#ifdef DAKOTA_MATLAB
  if (analysisDrivers.contains("matlab")) {

    String engine_cmd;
    const char* env_engine_cmd = getenv("DAKOTA_MATLAB_ENGINE_CMD");
#if defined(__CYGWIN__)
    engine_cmd = "\0";
    if (env_engine_cmd)
      Cerr << "\nWarning: DAKOTA_MATLAB_ENGINE_CMD ignored on Windows." << endl;
#else
    engine_cmd = env_engine_cmd ? env_engine_cmd :
      "matlab -nodesktop -nosplash";
#endif
    if (matlabEngine = engOpen(engine_cmd)) {
      Cout << "Matlab engine initialized for direct function evaluation."
	   << endl;
      const char * env_matlab_startup = getenv("DAKOTA_MATLAB_STARTUP");
      if (env_matlab_startup)
	engEvalString(matlabEngine, env_matlab_startup);      
    }
    else {
      Cerr << "Error (Direct:Matlab): Could not initialize Matlab engine "
	   << "for direct fn. eval." << endl;
      abort_handler(-1);
    }
    mxArray *dakota_matlab = NULL;  // MATLAB Dakota structure
    /* fields to pass to Matlab in Dakota structure */
    const mwSize ndim = 2;
    const mwSize dims[2] = {1, 1};
    /* Create Dakota variable structure in the engine workspace */
    dakota_matlab = mxCreateStructArray(ndim, dims, NUMBER_OF_FIELDS,
					FIELD_NAMES);
    engPutVariable(matlabEngine, "Dakota", dakota_matlab);
  }
#endif

#ifdef DAKOTA_PYTHON
  if (analysisDrivers.contains("python")) {
    Py_Initialize();
    if (Py_IsInitialized())
      Cout << "Python interpreter initialized for direct function evaluation."
	   << endl;
    else {
      Cerr << "Error: Could not initialize Python for direct function "
	   << "evaluation." << endl;
      abort_handler(-1);
    }
#ifdef DAKOTA_PYTHON_NUMPY
    import_array();
    //      userNumpyFlag = problem_db.get_bool("python_numpy");
    userNumpyFlag = true;
#else
    //      if (problem_db.get_bool("python_numpy")) {
    Cout << "Warning: Python numpy not available, ignoring user request."
	 << endl;
    userNumpyFlag = false;
    //}
#endif
  }
#endif
}

DirectApplicInterface::~DirectApplicInterface()
{ 
#ifdef DAKOTA_MATLAB
  if (analysisDrivers.contains("matlab") && matlabEngine)
    if ( engClose(matlabEngine) ) {
      Cerr << "Error (Direct:Matlab): Couldn't terminate Matlab engine post "
	   << "direct fn. eval." << endl;
      abort_handler(-1);
    } else 
      Cout << "Matlab engine terminated." << endl;
#endif
#ifdef DAKOTA_PYTHON
  if (Py_IsInitialized()) {
    Py_Finalize();
    Cout << "Python interpreter terminated." << endl;
  }
#endif
}

void DirectApplicInterface::
derived_map(const Variables& vars, const ActiveSet& set, Response& response,
	    int fn_eval_id)
{
  // Check for erroneous concurrent analysis specification:
  if (asynchLocalAnalysisFlag && evalCommRank == 0 && evalServerId == 1)
    Cerr << "Warning: multiple threads not yet supported in direct interfaces."
	 << "\n         Asynchronous analysis request will be ignored.\n";

  if (evalCommRank == 0 && !suppressOutput && outputLevel > SILENT_OUTPUT) {
    bool curly_braces = ( numAnalysisDrivers > 1 || !iFilterName.empty() ||
                          !oFilterName.empty() ) ? true : false;
    if (eaDedMasterFlag)
      Cout << "Direct function: self-scheduling ";
    else if (numAnalysisServers > 1)
      Cout << "Direct function: static scheduling ";
    else
      Cout << "Direct function: invoking ";
    if (curly_braces)
      Cout << "{ ";
    if (!iFilterName.empty())
      Cout << iFilterName << ' ';
    for (size_t i=0; i<numAnalysisDrivers; i++)
      Cout << analysisDrivers[i] << ' ';
    if (!oFilterName.empty())
      Cout << oFilterName << ' ';
    if (curly_braces)
      Cout << "} ";
    if (numAnalysisServers > 1)
      Cout << "among " << numAnalysisServers << " analysis servers.";
    Cout << endl;
  }

  // Goes before input filter to set up variables data:
  set_local_data(vars, set, response);

  // --------------------
  // Input filter portion
  // --------------------
  // Use ifilter just once per evaluation.  This provides a separate identity
  // for the ifilter: instead of just being part of each analysis process (and
  // therefore not having much justification for being separate from the
  // analysis), it is now the *non-replicated* portion of setting up the
  // analysis drivers.  For example, remeshing might be performed once per
  // evaluation and would be part of derived_map_if, whereas aprepro might be
  // used for each analysis and would be part of derived_map_ac.
  if (!iFilterName.empty() && evalCommRank == 0)
    derived_map_if(iFilterName);

  // ----------------
  // Analysis portion
  // ----------------
  if (eaDedMasterFlag) { // set up master-slave
    if (evalCommRank == 0)
      self_schedule_analyses();
    else
      serve_analyses_synch();
  }
  else // simple static schedule (all peer cases including single analysis)
    for (analysisDriverIndex =  analysisServerId-1;
	 analysisDriverIndex <  numAnalysisDrivers;
	 analysisDriverIndex += numAnalysisServers)
      derived_map_ac(analysisDrivers[analysisDriverIndex]);
  // NOTE: no synchronization enforced in static case (some processors may lag)

  // ---------------------
  // Output filter portion
  // ---------------------
  // As with ifilter, use ofilter just once per evaluation to accomodate any
  // non-replicated portions of post-processing (e.g., max response, sqrt of
  // sum of squares, other reductions which don't fit the overlay concept).
  if (!oFilterName.empty()) {
    // Provide synchronization if there's an oFilter and a static schedule
    // was used since evalCommRank 0 must postprocess all results (MPI_Reduce
    // provides synchronization in overlay() for the case of no oFilter).
    if (evalCommSize > 1 && !eaDedMasterFlag) // static schedule
      parallelLib.barrier_e(); // evalCommRank 0 waits for analyses to complete
    if (evalCommRank == 0)
      derived_map_of(oFilterName);
  }
  // OLD: Goes after ofilter since ofilter maps raw analysis data to
  // Response data, which is then overlaid into a combined response below.
  // NEW: in the concurrent analysis fork case, ofilter and overlay are
  // mutually exclusive, i.e., the nonreplicated operations of the ofilter
  // can be used to enact nonstandard overlays (e.g., sqrt(sum_sq)) and are
  // therefore responsible for mapping results.out.[eval#].[1->numPrograms] to
  // results.out.[eval#]
  else // mutually exclusive overlay/ofilter
    overlay_response(response); // MPI_Reduce provides synchronization
}


int DirectApplicInterface::derived_map_if(const String& if_name)
{
  int fail_code = 0;
  //if (if_name == "text_book_if") {
    //fail_code = text_book_if();
  //}
  //else {
    Cerr << if_name << " is not available as an input filter within "
         << "DirectApplicInterface." << endl;
    abort_handler(-1);
  //}

  // Failure capturing
  if (fail_code)
    throw fail_code;

  return 0;
}


/** When a direct analysis/filter is a member function, the (vars,set,response)
    data does not need to be passed through the API.  If, however, non-member
    analysis/filter functions are added, then pass (vars,set,response) through
    to the non-member fns:
    \code
    // API declaration
    int sim(const Variables& vars, const ActiveSet& set, Response& response);
    // use of API within derived_map_ac()
    if (ac_name == "sim")
      fail_code = sim(directFnVars, directFnActSet, directFnResponse);
    \endcode */
int DirectApplicInterface::derived_map_ac(const String& ac_name)
{
  // NOTE: a Factory pattern might be appropriate in the future to manage the
  // conditional presence of linked subroutines in DirectApplicInterface.

#ifdef MPI_DEBUG
    Cout << "analysis server " << analysisServerId << " invoking " << ac_name
         << " within DirectApplicInterface." << endl;
#endif // MPI_DEBUG
  int fail_code = 0;
  if (ac_name == "cantilever")
    fail_code = cantilever();
  else if (ac_name == "cyl_head")
    fail_code = cyl_head();
  else if (ac_name == "rosenbrock")
    fail_code = rosenbrock();
  else if (ac_name == "generalized_rosenbrock")
    fail_code = generalized_rosenbrock();
  else if (ac_name == "extended_rosenbrock")
    fail_code = extended_rosenbrock();
  else if (ac_name == "log_ratio")
    fail_code = log_ratio();
  else if (ac_name == "short_column")
    fail_code = short_column();
  else if (ac_name == "steel_column_cost")
    fail_code = steel_column_cost();
  else if (ac_name == "steel_column_perf")
    fail_code = steel_column_perf();
  else if (ac_name == "text_book")
    fail_code = text_book();
  else if (ac_name == "text_book1") // for testing concurrent analyses
    fail_code = text_book1();
  else if (ac_name == "text_book2") // for testing concurrent analyses
    fail_code = text_book2();
  else if (ac_name == "text_book3") // for testing concurrent analyses
    fail_code = text_book3();
  else if (ac_name == "text_book_ouu")
    fail_code = text_book_ouu();
  else if (ac_name == "multimodal")
    fail_code = multimodal();
#ifdef DAKOTA_SALINAS
  else if (ac_name == "salinas")
    fail_code = salinas();
#endif // DAKOTA_SALINAS
#ifdef DAKOTA_MODELCENTER
  else if (ac_name == "mc_api_run" || toLower(ac_name) == "modelcenter")
    fail_code = mc_api_run();
#endif // DAKOTA_MODELCENTER
#ifdef DAKOTA_MATLAB
  else if (ac_name == "matlab")
    fail_code = matlab_engine_run();
#endif // DAKOTA_MATLAB
#ifdef DAKOTA_PYTHON
  else if (ac_name == "python")
    fail_code = python_run();
#endif // DAKOTA_PYTHON
  else {
    Cerr << ac_name << " is not available as an analysis within "
         << "DirectApplicInterface." << endl;
    abort_handler(-1);
  }

  // Failure capturing
  if (fail_code)
    throw fail_code;

  return 0;
}


int DirectApplicInterface::derived_map_of(const String& of_name)
{
  int fail_code = 0;
  //if (of_name == "text_book_of") {
    //fail_code = text_book_of();
  //}
  //else {
    Cerr << of_name << " is not available as an output filter within "
         << "DirectApplicInterface." << endl;
    abort_handler(-1);
  //}

  // Failure capturing
  if (fail_code)
    throw fail_code;

  return 0;
}


void DirectApplicInterface::derived_map_asynch(const ParamResponsePair& pair)
{
  Cerr << "Error: asynchronous capability (multiple threads) not installed in"
       << "\nDirectApplicInterface." << endl;
  abort_handler(-1);

  //pthread_create/thr_create(derived_map(...)) launches a new thread
  //threadIdMap[tid] = fn_eval_id;
}


void DirectApplicInterface::derived_synch(PRPList& prp_list)
{
  Cerr << "Error: asynchronous capability (multiple threads) not installed in"
       << "\nDirectApplicInterface." << endl;
  abort_handler(-1);

/*
  int fail_code = 0, id = prp_list[i].eval_id();
  Variables vars    = prp_list[i].prp_parameters(); // copy
  Response response = prp_list[i].prp_response();   // copy

  // pthread_join/thr_join(target_thread, ..., status) recovers threads.
  // status provides a mechanism to return failure codes from analyses.
  // The Solaris thr_join allows completion of any thread in the set if 
  // target_thread is set to 0; unfortunately, this does not appear to be part
  // of the POSIX standard (pthread_join must complete a valid target_thread).

  // Possible better solution: look at OpenMP ??

  // For the asynch case, Direct (unlike SysCall) can manage failures w/o 
  // throwing exceptions.  See ApplicationInterface::manage_failure for notes.
  if (fail_code)
    manage_failure(vars, response.active_set(), response, id);

  Cout << "Thread for evaluation " << id << " captured.\n";
  prp_list[i].prp_response(response);
*/
}


void DirectApplicInterface::derived_synch_nowait(PRPList& prp_list)
{
  Cerr << "Error: asynchronous capability (multiple threads) not installed in"
       << "\nDirectApplicInterface." << endl;
  abort_handler(-1);
}


// -----------------------------------
// Begin utilities used by derived_map
// -----------------------------------
void DirectApplicInterface::
set_local_data(const Variables& vars, const ActiveSet& set,
	       const Response& response)
{
  // This function is performed once per evaluation, which may involve multiple
  // analyses.  Since the data has class scope, it has persistence from one
  // function evaluation to the next.  Old data must be zeroed, current 
  // variable values must be assigned to xC/xD, and variable and response 
  // arrays are resized (if necessary).

  // ------------------------
  // Set local variables data
  // ------------------------

  // As for ApproximationInterface::map(), it does not make sense to evaluate
  // the direct functions on an active subset of vars (for which the effect of
  // inactive vars would not be properly captured); rather, all of vars must be
  // mapped through.  This is important in particular for OUU since the inactive
  // variables are carrying data from the outer loop.

  // Initialize copies of incoming data
  //directFnVars = vars; // shared rep
  numVars = vars.tv(); // total number of vars
  size_t i, num_cv = vars.cv(), num_dv = vars.dv();
  if (numVars == num_cv + num_dv) {
    // no inactive vars: avoid some copies for this common case
    xC = vars.continuous_variables();
    xD = vars.discrete_variables();
    // labels should not change, but need initialization on all processors 
    if (xCLabels.empty())
      xCLabels = vars.continuous_variable_labels();
    if (xDLabels.empty())
      xDLabels = vars.discrete_variable_labels();
  }
  else {
    // inactive vars are present: more general case (e.g., OUU); copies
    // required to create/return "all" vectors
    xC = vars.all_continuous_variables();
    xD = vars.all_discrete_variables();
    if (xCLabels.empty())
      xCLabels = vars.all_continuous_variable_labels();
    if (xDLabels.empty())
      xDLabels = vars.all_discrete_variable_labels();
  }
  numACV = xC.length();
  numADV = xD.length();

  // -------------------------
  // Set local active set data
  // -------------------------
  //directFnActSet = set;                 // copy
  directFnASV  = set.request_vector();    // copy
  directFnDVV  = set.derivative_vector(); // copy
  numDerivVars = directFnDVV.length();

  // -----------------------
  // Set local response data
  // -----------------------
  //directFnResponse = response; // shared rep
  numFns = directFnASV.length();
  // labels should not change, but need initialization on all processors 
  if (fnLabels.empty())
    fnLabels = response.function_labels();
  gradFlag = false;
  hessFlag = false;
  for (i=0; i<numFns; i++) {
    if (directFnASV[i] & 2)
      gradFlag = true;
    if (directFnASV[i] & 4)
      hessFlag = true;
  }

  // Resize and clear required data constructs
  if (fnVals.length() != numFns)
    fnVals.reshape(numFns);
  fnVals = 0.;

  if (gradFlag) {
    if (fnGrads.num_rows() != numFns || fnGrads.num_columns() != numDerivVars)
      fnGrads.reshape_2d(numFns, numDerivVars);
    fnGrads = 0.;
  }

  if (hessFlag) {
    if (fnHessians.length() != numFns)
      fnHessians.reshape(numFns);
    for (i=0; i<numFns; i++) {
      if (fnHessians[i].num_rows()    != numDerivVars || 
          fnHessians[i].num_columns() != numDerivVars)
        fnHessians[i].reshape_2d(numDerivVars, numDerivVars);
      fnHessians[i] = 0.;
    }
  }
}


void DirectApplicInterface::overlay_response(Response& response)
{
  // Individual analysis servers are allowed to divide up the function 
  // evaluation in whatever way is convenient.  It need not be by function 
  // number (although this simple approach is used in text_book1/2/3).  The 
  // overlay_response function uses MPI_Reduce over evalAnalysisIntraComm to
  // add all contributions to the response object from analysisComm leaders.

  // If not rank 0 within an analysisComm, then nothing to contribute to the
  // total response.  Note that an evaluation dedicated master must participate
  // in the reduction (even though it contributes no response data) since it is
  // the final destination of the evaluation result.
  if (analysisCommRank)
    return;

  // set response data for analysisComm leaders (excluding master if present)
  if (analysisServerId) {
    ActiveSet set;
    set.request_vector(directFnASV);
    set.derivative_vector(directFnDVV);
    response.copy_results(fnVals, fnGrads, fnHessians, set);
  }

  // For all master-slave cases & for peer cases in which numAnalysisServers>1,
  // response components from analysis servers must be overlaid using 
  // MPI_Reduce(..., MPI_SUM, ...).  This is performed using a mapping of 
  // response->double*->Reduce(double*)->response.
  if (numAnalysisServers > 1 || eaDedMasterFlag) {
    int num_doubles   = response.data_size();
    double* local_fns = new double [num_doubles];
    if (analysisServerId) // analysis leaders
      response.write_data(local_fns);
    else { // evaluation dedicated master (if present)
      for (size_t i=0; i<num_doubles; i++)
        local_fns[i] = 0.0;
    }
    // sum response data over evalAnalysisIntraComm.  This is more efficient
    // than performing the reduction over evalComm since only analysisComm
    // leaders have data to reduce.  evalCommRank 0 then returns the results
    // to the iterator in ApplicationInterface::serve_evaluations().
    double* sum_fns = (evalCommRank) ? NULL : new double [num_doubles];
    parallelLib.reduce_sum_ea(local_fns, sum_fns, num_doubles);
    delete [] local_fns;
    if (evalCommRank == 0) {
      response.read_data(sum_fns);
      delete [] sum_fns;
    }
  }
}


// -----------------------------------------
// Begin direct interfaces to test functions
// -----------------------------------------
int DirectApplicInterface::cantilever()
{
  if (multiProcAnalysisFlag) {
    Cerr << "Error: cantilever direct fn does not support multiprocessor "
	 << "analyses." << endl;
    abort_handler(-1);
  }
  // cantilever normally has 6 variables: 2 design + 4 uncertain
  // If, however, design variables are _inserted_ into the uncertain variable
  // distribution parameters (e.g., dakota_rbdo_cantilever_mapvars.in) instead
  // of augmenting the uncertain variables, then the number of variables is 4.
  // Design gradients are not supported for the case of design var insertion.
  if ( (numVars != 4 && numVars != 6) || numADV || // total variables, no dv
       (gradFlag && numVars == 4 && numDerivVars != 4) ) { // design insertion
    Cerr << "Error: Bad number of variables in cantilever direct fn." << endl;
    abort_handler(-1);
  }
   if (numFns != 3) {
    Cerr << "Error: Bad number of functions in cantilever direct fn." << endl;
    abort_handler(-1);
  }

  // Compute the cross-sectional area, stress, and displacement of the
  // cantilever beam.  This simulator is unusual in that it must support both
  // the case of design variable insertion and the case of design variable
  // augmentation.  It does not support mixed insertion/augmentation.
  double w = (numVars == 4) ? 2.5   : xC[0]; // beam width
  double t = (numVars == 4) ? 2.5   : xC[1]; // beam thickness
  double R = (numVars == 4) ? xC[0] : xC[2]; // yield strength
  double E = (numVars == 4) ? xC[1] : xC[3]; // Young's modulus
  double X = (numVars == 4) ? xC[2] : xC[4]; // horizontal load
  double Y = (numVars == 4) ? xC[3] : xC[5]; // vertical load

  double D0 = 2.2535, L = 100., area = w*t, w_sq = w*w, t_sq = t*t,
    R_sq = R*R, X_sq = X*X, Y_sq = Y*Y;
  double stress = 600.*Y/w/t_sq + 600.*X/w_sq/t;
  double D1 = 4.*pow(L,3)/E/area, D2 = pow(Y/t_sq, 2)+pow(X/w_sq, 2);
  double D3 = D1/sqrt(D2)/D0,     D4 = D1*sqrt(D2)/D0;

  // limit state >= 0
  //double g_stress = R  - stress;
  //double g_disp   = D0 - disp;

  // inequality constraint <= 0
  //double g_stress = stress/R - 1.0;
  //double g_disp   = disp/D0  - 1.0;

  // **** f:
  if (directFnASV[0] & 1)
    fnVals[0] = area;

  // **** c1:
  if (directFnASV[1] & 1)
    fnVals[1] = stress/R - 1.;

  // **** c2:
  if (directFnASV[2] & 1)
    fnVals[2] = D4 - 1.;

  // **** df/dx:
  if (directFnASV[0] & 2) {
    if (numVars == 6) { // normal design + uncertain case
      for (size_t i=0; i<numDerivVars; i++) {
	switch (directFnDVV[i]) {
	case 1:  fnGrads[0][i] = t;  break; // design var derivative
	case 2:  fnGrads[0][i] = w;  break; // design var derivative
	default: fnGrads[0][i] = 0.; break; // uncertain var derivative
	}
      }
    }
    else if (numVars == 4) // uncertain vars only
      fnGrads[0] = 0.; // uncertain var derivative
  }

  // **** dc1/dx:
  if (directFnASV[1] & 2) {
    if (numVars == 6) { // normal design + uncertain case
      for (size_t i=0; i<numDerivVars; i++) {
	switch (directFnDVV[i]) {
	case 1: fnGrads[1][i] = -600.*(Y/t + 2.*X/w)/w_sq/t/R; break; // des var
	case 2: fnGrads[1][i] = -600.*(2.*Y/t + X/w)/w/t_sq/R; break; // des var
	case 3: fnGrads[1][i] = -stress/R_sq;  break; // uncertain var deriv
	case 4: fnGrads[1][i] = 0.;            break; // uncertain var deriv
	case 5: fnGrads[1][i] = 600./w_sq/t/R; break; // uncertain var deriv
	case 6: fnGrads[1][i] = 600./w/t_sq/R; break; // uncertain var deriv
	}
      }
    }
    else if (numVars == 4) { // uncertain vars only
      for (size_t i=0; i<numDerivVars; i++) {
	switch (directFnDVV[i]) {
	case 1: fnGrads[1][i] = -stress/R_sq;  break; // uncertain var deriv
	case 2: fnGrads[1][i] = 0.;            break; // uncertain var deriv
	case 3: fnGrads[1][i] = 600./w_sq/t/R; break; // uncertain var deriv
	case 4: fnGrads[1][i] = 600./w/t_sq/R; break; // uncertain var deriv
	}
      }
    }
  }

  // **** dc2/dx:
  if (directFnASV[2] & 2) {
    if (numVars == 6) { // normal design + uncertain case
      for (size_t i=0; i<numDerivVars; i++) {
	switch (directFnDVV[i]) {
	case 1: fnGrads[2][i] = -D3*2.*X_sq/w_sq/w_sq/w - D4/w; break;// des var
	case 2: fnGrads[2][i] = -D3*2.*Y_sq/t_sq/t_sq/t - D4/t; break;// des var
	case 3: fnGrads[2][i] = 0.;             break; // unc var deriv
	case 4: fnGrads[2][i] = -D4/E;          break; // unc var deriv
	case 5: fnGrads[2][i] = D3*X/w_sq/w_sq; break; // unc var deriv
	case 6: fnGrads[2][i] = D3*Y/t_sq/t_sq; break; // unc var deriv
	}
      }
    }
    else if (numVars == 4) { // uncertain vars only
      for (size_t i=0; i<numDerivVars; i++) {
	switch (directFnDVV[i]) {
	case 1: fnGrads[2][i] = 0.;             break; // unc var deriv
	case 2: fnGrads[2][i] = -D4/E;          break; // unc var deriv
	case 3: fnGrads[2][i] = D3*X/w_sq/w_sq; break; // unc var deriv
	case 4: fnGrads[2][i] = D3*Y/t_sq/t_sq; break; // unc var deriv
	}
      }
    }
  }

  // **** d^2f/dx^2:
  if (directFnASV[0] & 4) {
    if (numVars == 6) { // normal design + uncertain case
      for (size_t i=0; i<numDerivVars; i++) {
	size_t var_index_i = directFnDVV[i] - 1;
	for (size_t j=0; j<numDerivVars; j++) {
	  size_t var_index_j = directFnDVV[j] - 1;
	  fnHessians[0][i][j]
	    = ( (var_index_i == 0 && var_index_j == 1) ||
		(var_index_i == 1 && var_index_j == 0) ) ? 1. : 0.;
	}
      }
    }
    else if (numVars == 4) // uncertain vars only
      fnHessians[0] = 0.;
  }

  // **** d^2c1/dx^2:
  if (directFnASV[1] & 4) {
    if (numVars == 6) { // normal design + uncertain case
      for (size_t i=0; i<numDerivVars; i++) {
	size_t var_index_i = directFnDVV[i] - 1;
	for (size_t j=0; j<numDerivVars; j++) {
	  size_t var_index_j = directFnDVV[j] - 1;
	  if (var_index_i == 0 && var_index_j == 0)          // d^2g/dw^2
	    fnHessians[1][i][j] = 1200.*(Y/t + 3.*X/w)/w_sq/area/R;
	  else if (var_index_i == 1 && var_index_j == 1)     // d^2g/dt^2
	    fnHessians[1][i][j] = 1200.*(3.*Y/t + X/w)/t_sq/area/R;
	  else if (var_index_i == 2 && var_index_j == 2)     // d^2g/dR^2
	    fnHessians[1][i][j] = 2.*stress/pow(R, 3);
	  else if ( (var_index_i == 0 && var_index_j == 1) ||
		    (var_index_i == 1 && var_index_j == 0) ) // d^2g/dwdt
	    fnHessians[1][i][j] = 1200.*(Y/t + X/w)/w_sq/t_sq/R;
	  else if ( (var_index_i == 0 && var_index_j == 2) ||
		    (var_index_i == 2 && var_index_j == 0) ) // d^2g/dwdR
	    fnHessians[1][i][j] = 600.*(Y/t + 2.*X/w)/w_sq/t/R_sq;
	  else if ( (var_index_i == 0 && var_index_j == 4) ||
		    (var_index_i == 4 && var_index_j == 0) ) // d^2g/dwdX
	    fnHessians[1][i][j] = -1200./w_sq/w/t/R;
	  else if ( (var_index_i == 0 && var_index_j == 5) ||
		    (var_index_i == 5 && var_index_j == 0) ) // d^2g/dwdY
	    fnHessians[1][i][j] = -600./w_sq/t_sq/R;
	  else if ( (var_index_i == 1 && var_index_j == 2) ||
		    (var_index_i == 2 && var_index_j == 1) ) // d^2g/dtdR
	    fnHessians[1][i][j] = 600.*(2.*Y/t + X/w)/w/t_sq/R_sq;
	  else if ( (var_index_i == 1 && var_index_j == 4) ||
		    (var_index_i == 4 && var_index_j == 1) ) // d^2g/dtdX
	    fnHessians[1][i][j] = -600./w_sq/t_sq/R;
	  else if ( (var_index_i == 1 && var_index_j == 5) ||
		    (var_index_i == 5 && var_index_j == 1) ) // d^2g/dtdY
	    fnHessians[1][i][j] = -1200./w/t_sq/t/R;
	  else if ( (var_index_i == 2 && var_index_j == 4) ||
		    (var_index_i == 4 && var_index_j == 2) ) // d^2g/dRdX
	    fnHessians[1][i][j] = -600./w_sq/t/R_sq;
	  else if ( (var_index_i == 2 && var_index_j == 5) ||
		    (var_index_i == 5 && var_index_j == 2) ) // d^2g/dRdY
	    fnHessians[1][i][j] = -600./w/t_sq/R_sq;
	  else
	    fnHessians[1][i][j] = 0.;
	}
      }
    }
    else if (numVars == 4) { // uncertain vars only
      for (size_t i=0; i<numDerivVars; i++) {
	size_t var_index_i = directFnDVV[i] - 1;
	for (size_t j=0; j<numDerivVars; j++) {
	  size_t var_index_j = directFnDVV[j] - 1;
	  if (var_index_i == 0 && var_index_j == 0)          // d^2g/dR^2
	    fnHessians[1][i][j] = 2.*stress/pow(R, 3);
	  else if ( (var_index_i == 0 && var_index_j == 2) ||
		    (var_index_i == 2 && var_index_j == 0) ) // d^2g/dRdX
	    fnHessians[1][i][j] = -600./w_sq/t/R_sq;
	  else if ( (var_index_i == 0 && var_index_j == 3) ||
		    (var_index_i == 3 && var_index_j == 0) ) // d^2g/dRdY
	    fnHessians[1][i][j] = -600./w/t_sq/R_sq;
	  else
	    fnHessians[1][i][j] = 0.;
	}
      }
    }
  }

  // **** d^2c2/dx^2:
  if (directFnASV[2] & 4) {
    double D5 = 1./sqrt(D2)/D0, D6 = -D1/2./D0/pow(D2,1.5);
    double D7 = sqrt(D2)/D0,    D8 =  D1/2./D0/sqrt(D2);
    double dD2_dX = 2.*X/w_sq/w_sq, dD3_dX = D6*dD2_dX, dD4_dX = D8*dD2_dX;
    double dD2_dY = 2.*Y/t_sq/t_sq, dD3_dY = D6*dD2_dY, dD4_dY = D8*dD2_dY;
    if (numVars == 6) { // normal design + uncertain case
      double dD1_dw = -D1/w, dD2_dw = -4.*X_sq/w_sq/w_sq/w,
	dD3_dw = D5*dD1_dw + D6*dD2_dw, dD4_dw = D7*dD1_dw + D8*dD2_dw;
      double dD1_dt = -D1/t, dD2_dt = -4.*Y_sq/t_sq/t_sq/t,
	dD3_dt = D5*dD1_dt + D6*dD2_dt, dD4_dt = D7*dD1_dt + D8*dD2_dt;
      for (size_t i=0; i<numDerivVars; i++) {
	size_t var_index_i = directFnDVV[i] - 1;
	for (size_t j=0; j<numDerivVars; j++) {
	  size_t var_index_j = directFnDVV[j] - 1;
	  if (var_index_i == 0 && var_index_j == 0)            // d^2g/dw^2
	    fnHessians[2][i][j] = D3*10.*X_sq/pow(w_sq,3)
	      - 2.*X_sq/w_sq/w_sq/w*dD3_dw + D4/w_sq - dD4_dw/w;
	  else if (var_index_i == 1 && var_index_j == 1)       // d^2g/dt^2
	    fnHessians[2][i][j] = D3*10.*Y_sq/pow(t_sq,3)
	      - 2.*Y_sq/t_sq/t_sq/t*dD3_dt + D4/t_sq - dD4_dt/t;
	  else if (var_index_i == 3 && var_index_j == 3) {     // d^2g/dE^2
	    double dD1_dE = -D1/E, dD4_dE = D7*dD1_dE;
	    fnHessians[2][i][j] = D4/E/E - dD4_dE/E;
	  }
	  else if (var_index_i == 4 && var_index_j == 4)       // d^2g/dX^2
	    fnHessians[2][i][j] = D3/w_sq/w_sq + X/w_sq/w_sq*dD3_dX;
	  else if (var_index_i == 5 && var_index_j == 5)       // d^2g/dY^2
	    fnHessians[2][i][j] = D3/t_sq/t_sq + Y/t_sq/t_sq*dD3_dY;
	  else if ( (var_index_i == 0 && var_index_j == 1) ||
		    (var_index_i == 1 && var_index_j == 0) )   // d^2g/dwdt
	    fnHessians[2][i][j] = -2.*X_sq/w_sq/w_sq/w*dD3_dt - dD4_dt/w;
	  else if ( (var_index_i == 0 && var_index_j == 3) ||
		    (var_index_i == 3 && var_index_j == 0) )   // d^2g/dwdE
	    fnHessians[2][i][j] = -dD4_dw/E;
	  else if ( (var_index_i == 0 && var_index_j == 4) ||
		    (var_index_i == 4 && var_index_j == 0) )   // d^2g/dwdX
	    fnHessians[2][i][j] = -4.*X*D3/w_sq/w_sq/w + X/w_sq/w_sq*dD3_dw;
	  else if ( (var_index_i == 0 && var_index_j == 5) ||
		    (var_index_i == 5 && var_index_j == 0) )   // d^2g/dwdY
	    fnHessians[2][i][j] = Y/t_sq/t_sq*dD3_dw;
	  else if ( (var_index_i == 1 && var_index_j == 3) ||
		    (var_index_i == 3 && var_index_j == 1) )   // d^2g/dtdE
	    fnHessians[2][i][j] = -dD4_dt/E;
	  else if ( (var_index_i == 1 && var_index_j == 4) ||
		    (var_index_i == 4 && var_index_j == 1) )   // d^2g/dtdX
	    fnHessians[2][i][j] = X/w_sq/w_sq*dD3_dt;
	  else if ( (var_index_i == 1 && var_index_j == 5) ||
		    (var_index_i == 5 && var_index_j == 1) )   // d^2g/dtdY
	    fnHessians[2][i][j] = -4.*Y*D3/t_sq/t_sq/t + Y/t_sq/t_sq*dD3_dt;
	  else if ( (var_index_i == 3 && var_index_j == 4) ||
		    (var_index_i == 4 && var_index_j == 3) )   // d^2g/dEdX
	    fnHessians[2][i][j] = -dD4_dX/E;
	  else if ( (var_index_i == 3 && var_index_j == 5) ||
		    (var_index_i == 5 && var_index_j == 3) )   // d^2g/dEdY
	    fnHessians[2][i][j] = -dD4_dY/E;
	  else if ( (var_index_i == 4 && var_index_j == 5) ||
		    (var_index_i == 5 && var_index_j == 4) )   // d^2g/dXdY
	    fnHessians[2][i][j] = X/w_sq/w_sq*dD3_dY;
	  else
	    fnHessians[2][i][j] = 0.;
	}
      }
    }
    else if (numVars == 4) { // uncertain vars only
      for (size_t i=0; i<numDerivVars; i++) {
	size_t var_index_i = directFnDVV[i] - 1;
	for (size_t j=0; j<numDerivVars; j++) {
	  size_t var_index_j = directFnDVV[j] - 1;
	  if (var_index_i == 1 && var_index_j == 1) {          // d^2g/dE^2
	    double dD1_dE = -D1/E, dD4_dE = D7*dD1_dE;
	    fnHessians[2][i][j] = D4/E/E - dD4_dE/E;
	  }
	  else if (var_index_i == 2 && var_index_j == 2)       // d^2g/dX^2
	    fnHessians[2][i][j] = D3/w_sq/w_sq + X/w_sq/w_sq*dD3_dX;
	  else if (var_index_i == 3 && var_index_j == 3)       // d^2g/dY^2
	    fnHessians[2][i][j] = D3/t_sq/t_sq + Y/t_sq/t_sq*dD3_dY;
	  else if ( (var_index_i == 1 && var_index_j == 2) ||
		    (var_index_i == 2 && var_index_j == 1) )   // d^2g/dEdX
	    fnHessians[2][i][j] = -dD4_dX/E;
	  else if ( (var_index_i == 1 && var_index_j == 3) ||
		    (var_index_i == 3 && var_index_j == 1) )   // d^2g/dEdY
	    fnHessians[2][i][j] = -dD4_dY/E;
	  else if ( (var_index_i == 2 && var_index_j == 3) ||
		    (var_index_i == 3 && var_index_j == 2) )   // d^2g/dXdY
	    fnHessians[2][i][j] = X/w_sq/w_sq*dD3_dY;
	  else
	    fnHessians[2][i][j] = 0.;
	}
      }
    }
  }

  return 0; // no failure
}


int DirectApplicInterface::cyl_head()
{
  if (multiProcAnalysisFlag) {
    Cerr << "Error: cyl_head direct fn does not yet support "
	 << "multiprocessor analyses." << endl;
    abort_handler(-1);
  }
  if (numVars != 2 || numADV || (gradFlag && numDerivVars != 2)) {
    Cerr << "Error: Bad number of variables in cyl_head direct fn." << endl;
    abort_handler(-1);
  }
  if (numFns != 4) {
    Cerr << "Error: Bad number of functions in cyl_head direct fn." << endl;
    abort_handler(-1);
  }
  if (hessFlag) {
    Cerr << "Error: Hessians not supported in cyl_head direct fn." << endl;
    abort_handler(-1);
  }

  double exhaust_offset = 1.34;
  double exhaust_dia    = 1.556;
  double intake_offset  = 3.25;
  // Use nondimensional xC[1]: 
  // (0. <= nondimensional <= 4.), (0. in <= dimensional <= 0.004 in)
  double warranty       = 100000. + 15000. * (4. - xC[1]);
  double cycle_time     = 45. + 4.5*pow(4. - xC[1], 1.5);
  double wall_thickness = intake_offset - exhaust_offset
                        - (xC[0]+exhaust_dia)/2.;
  double horse_power    = 250.+200.*(xC[0]/1.833-1.);
  double max_stress     = 750. + pow(fabs(wall_thickness),-2.5);

  // **** f:
  if (directFnASV[0] & 1)
    fnVals[0] =  -1.*(horse_power/250.+warranty/100000.);

  // **** c1:
  if (directFnASV[1] & 1)
    fnVals[1] = max_stress/1500.-1.;

  // **** c2:
  if (directFnASV[2] & 1)
    fnVals[2] = 1.-warranty/100000.;

  // **** c3:
  if (directFnASV[3] & 1)
    fnVals[3] = cycle_time/60. - 1.;

  // **** c4: (Unnecessary if intake_dia upper bound reduced to 2.164)
  //if (directFnASV[4] & 1)
  //  fnVals[4] = 1.-20.*wall_thickness;

  // **** df/dx:
  if (directFnASV[0] & 2) {
    fnGrads[0][0] = -.8/1.833;
    fnGrads[0][1] = 0.15;
  }

  // **** dc1/dx:
  if (directFnASV[1] & 2) {
    fnGrads[1][0] = 1.25/1500*pow(wall_thickness, -3.5);
    fnGrads[1][1] = 0.;
  }

  // **** dc2/dx:
  if (directFnASV[2] & 2) {
    fnGrads[2][0] = 0.;
    fnGrads[2][1] = 0.15;
  }

  // **** dc3/dx:
  if (directFnASV[3] & 2) {
    fnGrads[3][0] = 0.;
    fnGrads[3][1] = -0.1125*sqrt(4. - xC[1]);
  }

  //sleep(5); // for faking a more expensive evaluation
  return 0; // no failure
}


int DirectApplicInterface::rosenbrock()
{
  if (multiProcAnalysisFlag) {
    Cerr << "Error: rosenbrock direct fn does not yet support multiprocessor "
	 << "analyses." << endl;
    abort_handler(-1);
  }
  if (numVars != 2 || numADV) {
    Cerr << "Error: Bad number of variables in rosenbrock direct fn." << endl;
    abort_handler(-1);
  }
  if (numFns > 2) { // 1 fn -> opt, 2 fns -> least sq
    Cerr << "Error: Bad number of functions in rosenbrock direct fn." << endl;
    abort_handler(-1);
  }

  bool least_sq_flag = (numFns > 1) ? true : false;

  double f0 = xC[1]-xC[0]*xC[0];
  double f1 = 1.-xC[0];

  if (least_sq_flag) {
    // **** Residual R1:
    if (directFnASV[0] & 1)
      fnVals[0] = 10*f0;
    // **** Residual R2:
    if (directFnASV[1] & 1)
      fnVals[1] = f1;

    // **** dR1/dx:
    if (directFnASV[0] & 2) {
      for (size_t i=0; i<numDerivVars; i++) {
	size_t var_index = directFnDVV[i] - 1;
	switch (var_index) {
	case 0: fnGrads[0][i] = -20.*xC[0]; break;
	case 1: fnGrads[0][i] =  10.;       break;
	}
      }
    }
    // **** dR2/dx:
    if (directFnASV[1] & 2) {
      for (size_t i=0; i<numDerivVars; i++) {
	size_t var_index = directFnDVV[i] - 1;
	switch (var_index) {
	case 0: fnGrads[1][i] = -1.; break;
	case 1: fnGrads[1][i] =  0.; break;
	}
      }
    }

    // **** d^2R1/dx^2:
    if (directFnASV[0] & 4) {
      for (size_t i=0; i<numDerivVars; i++) {
	size_t var_index_i = directFnDVV[i] - 1;
	for (size_t j=0; j<numDerivVars; j++) {
	  size_t var_index_j = directFnDVV[j] - 1;
	  if (var_index_i == 0 && var_index_j == 0)
	    fnHessians[0][i][j] = -20.;
	  else
	    fnHessians[0][i][j] =   0.;
	}
      }
    }
    // **** d^2R2/dx^2:
    if (directFnASV[1] & 4)
      fnHessians[1] = 0.;
  }
  else {
    // **** f:
    if (directFnASV[0] & 1)
      fnVals[0] = 100.*f0*f0+f1*f1;

    // **** df/dx:
    if (directFnASV[0] & 2) {
      for (size_t i=0; i<numDerivVars; i++) {
	size_t var_index = directFnDVV[i] - 1;
	switch (var_index) {
	case 0: fnGrads[0][i] = -400.*f0*xC[0] - 2.*f1; break;
	case 1: fnGrads[0][i] =  200.*f0;               break;
	}
      }
    }

    // **** d^2f/dx^2:
    if (directFnASV[0] & 4) {
      for (size_t i=0; i<numDerivVars; i++) {
	size_t var_index_i = directFnDVV[i] - 1;
	for (size_t j=0; j<numDerivVars; j++) {
	  size_t var_index_j = directFnDVV[j] - 1;
	  if (var_index_i == 0 && var_index_j == 0)
	    fnHessians[0][i][j] = -400.*(xC[1] - 3.*xC[0]*xC[0]) + 2.;
	  else if ( (var_index_i == 0 && var_index_j == 1) ||
		    (var_index_i == 1 && var_index_j == 0) )
	    fnHessians[0][i][j] = -400.*xC[0];
	  else if (var_index_i == 1 && var_index_j == 1)
	    fnHessians[0][i][j] =  200.;
	}
      }
    }
  }

  //sleep(5); // for faking a more expensive evaluation
  return 0; // no failure
}


int DirectApplicInterface::generalized_rosenbrock()
{
  if (multiProcAnalysisFlag) {
    Cerr << "Error: generalized_rosenbrock direct fn does not support "
	 << "multiprocessor analyses." << endl;
    abort_handler(-1);
  }
  if (numADV) {
    Cerr << "Error: discrete variables not supported in generalized_rosenbrock "
	 << "direct fn." << endl;
    abort_handler(-1);
  }
  if ( (directFnASV[0] & 6) && numVars != numDerivVars ) {
    Cerr << "Error: DVV subsets not supported in generalized_rosenbrock direct "
	 << "fn." << endl;
    abort_handler(-1);
  }
  if (numFns > 1) {
    Cerr << "Error: Bad number of functions in generalized_rosenbrock direct "
	 << "fn." << endl;
    abort_handler(-1);
  }

  for (size_t i=1; i<numVars; i++) {
    size_t index_ip1 = i, index_i = i-1; // offset by 1
    const double& x_ip1 = xC[index_ip1];
    const double& x_i   = xC[index_i];
    double f0 = x_ip1 - x_i*x_i, f1 = 1. - x_i;

    // **** f:
    if (directFnASV[0] & 1)
      fnVals[0] += 100.*f0*f0 + f1*f1;

    // **** df/dx:
    if (directFnASV[0] & 2) {
      fnGrads[0][index_i]   += -400.*f0*x_i - 2.*f1;
      fnGrads[0][index_ip1] +=  200.*f0;
    }

    // **** d^2f/dx^2:
    if (directFnASV[0] & 4) {
      double fx = x_ip1 - 3.*x_i*x_i;
      fnHessians[0][index_i][index_i]     += -400.*fx + 2.0;
      fnHessians[0][index_i][index_ip1]   += -400.*x_i;
      fnHessians[0][index_ip1][index_i]   += -400.*x_i;
      fnHessians[0][index_ip1][index_ip1] +=  200.;
    }
  }

  //sleep(5); // for faking a more expensive evaluation
  return 0; // no failure
}


int DirectApplicInterface::extended_rosenbrock()
{
  if (multiProcAnalysisFlag) {
    Cerr << "Error: extended_rosenbrock direct fn does not support "
	 << "multiprocessor analyses." << endl;
    abort_handler(-1);
  }
  if (numADV) {
    Cerr << "Error: discrete variables not supported in extended_rosenbrock "
	 << "direct fn." << endl;
    abort_handler(-1);
  }
  if ( (directFnASV[0] & 6) && numVars != numDerivVars ) {
    Cerr << "Error: DVV subsets not supported in extended_rosenbrock direct fn."
	 << endl;
    abort_handler(-1);
  }
  if (numFns > 1) {
    Cerr << "Error: Bad number of functions in extended_rosenbrock direct fn."
	 << endl;
    abort_handler(-1);
  }

  const double alpha = 100.;
  for (size_t i=1; i<=numVars/2; i++) {
    size_t index_2i = 2*i-1, index_2im1 = 2*i-2; // offset by 1
    const double& x_2i   = xC[index_2i];
    const double& x_2im1 = xC[index_2im1];
    double f0 = x_2i - x_2im1*x_2im1, f1 = 1. - x_2im1;

    // **** f:
    if (directFnASV[0] & 1)
      fnVals[0] += alpha*f0*f0 + f1*f1;

    // **** df/dx:
    if (directFnASV[0] & 2) {
      fnGrads[0][index_2im1] += -4.*alpha*f0*x_2im1 - 2.*f1;
      fnGrads[0][index_2i]   +=  2.*alpha*f0;
    }

    // **** d^2f/dx^2:
    if (directFnASV[0] & 4) {
      double fx = x_2i - 3.*x_2im1*x_2im1;
      fnHessians[0][index_2im1][index_2im1] += -4.*alpha*fx + 2.0;
      fnHessians[0][index_2im1][index_2i]   += -4.*alpha*x_2im1;
      fnHessians[0][index_2i][index_2im1]   += -4.*alpha*x_2im1;
      fnHessians[0][index_2i][index_2i]     +=  2.*alpha;
    }
  }

  //sleep(5); // for faking a more expensive evaluation
  return 0; // no failure
}


int DirectApplicInterface::log_ratio()
{
  if (multiProcAnalysisFlag) {
    Cerr << "Error: log_ratio direct fn does not support multiprocessor "
	 << "analyses." << endl;
    abort_handler(-1);
  }
  if (numVars != 2 || numADV || ((gradFlag || hessFlag) && numDerivVars != 2)) {
    Cerr << "Error: Bad number of variables in log_ratio direct fn." << endl;
    abort_handler(-1);
  }
  if (numFns != 1) {
    Cerr << "Error: Bad number of functions in log_ratio direct fn." << endl;
    abort_handler(-1);
  }

  // **** f:
  if (directFnASV[0] & 1)
    fnVals[0] = xC[0]/xC[1];

  // **** df/dx:
  if (directFnASV[0] & 2) {
    fnGrads[0][0] = 1./xC[1];
    fnGrads[0][1] = -xC[0]/(xC[1]*xC[1]);
  }

  // **** d^2f/dx^2:
  if (directFnASV[0] & 4) {
    fnHessians[0][0][0] = 0.0;
    fnHessians[0][0][1] = fnHessians[0][1][0] = -1./(xC[1]*xC[1]);
    fnHessians[0][1][1] = 2.*xC[0]/pow(xC[1],3);
  }

  return 0; // no failure
}


int DirectApplicInterface::short_column()
{
  if (multiProcAnalysisFlag) {
    Cerr << "Error: short_column direct fn does not support multiprocessor "
	 << "analyses." << endl;
    abort_handler(-1);
  }
  if (numVars != 5 || numADV) {
    Cerr << "Error: Bad number of variables in short_column direct fn." << endl;
    abort_handler(-1);
  }
  if (numFns != 2) {
    Cerr << "Error: Bad number of functions in short_column direct fn." << endl;
    abort_handler(-1);
  }

  // b = xC[0] = column base   (design var.)
  // h = xC[1] = column height (design var.)
  // P = xC[2] (normal uncertain var.)
  // M = xC[3] (normal uncertain var.)
  // Y = xC[4] (lognormal uncertain var.)
  double b = xC[0], h = xC[1], P = xC[2], M = xC[3], Y = xC[4],
         b_sq = b*b, h_sq = h*h, P_sq = P*P, Y_sq = Y*Y;

  // **** f (objective = bh = cross sectional area):
  if (directFnASV[0] & 1)
    fnVals[0] = b*h;

  // **** g (limit state = short column response):
  if (directFnASV[1] & 1)
    fnVals[1] = 1. - 4.*M/(b*h_sq*Y) - P_sq/(b_sq*h_sq*Y_sq);

  // **** df/dx (w.r.t. active variables):
  if (directFnASV[0] & 2) {
    for (size_t i=0; i<numDerivVars; i++) {
      size_t var_index = directFnDVV[i] - 1;
      switch (var_index) {
      case 0: // design variable derivative
	fnGrads[0][i] = h;  break;
      case 1: // design variable derivative
	fnGrads[0][i] = b;  break;
      default: // uncertain variable derivative
	fnGrads[0][i] = 0.; break;
      }
    }
  }

  // **** dg/dx (w.r.t. active variables):
  if (directFnASV[1] & 2) {
    for (size_t i=0; i<numDerivVars; i++) {
      size_t var_index = directFnDVV[i] - 1;
      switch (var_index) {
      case 0: // design variable derivative
	fnGrads[1][i] = 4.*M/(b_sq*h_sq*Y) + 2.*P_sq/(b_sq*b*h_sq*Y_sq); break;
      case 1: // design variable derivative
	fnGrads[1][i] = 8.*M/(b*h_sq*h*Y)  + 2.*P_sq/(b_sq*h_sq*h*Y_sq); break;
      case 2: // uncertain variable derivative
	fnGrads[1][i] = -2.*P/(b_sq*h_sq*Y_sq);                          break;
      case 3: // uncertain variable derivative
	fnGrads[1][i] = -4./(b*h_sq*Y);                                  break;
      case 4: // uncertain variable derivative
	fnGrads[1][i] = 4.*M/(b*h_sq*Y_sq) + 2.*P_sq/(b_sq*h_sq*Y_sq*Y); break;
      }
    }
  }

  // **** d^2f/dx^2: (SORM)
  if (directFnASV[0] & 4) {
    for (size_t i=0; i<numDerivVars; i++) {
      size_t var_index_i = directFnDVV[i] - 1;
      for (size_t j=0; j<numDerivVars; j++) {
	size_t var_index_j = directFnDVV[j] - 1;
	fnHessians[0][i][j]
	  = ( (var_index_i == 0 && var_index_j == 1) ||
	      (var_index_i == 1 && var_index_j == 0) ) ? 1. : 0.;
      }
    }
  }

  // **** d^2g/dx^2: (SORM)
  if (directFnASV[1] & 4) {
    for (size_t i=0; i<numDerivVars; i++) {
      size_t var_index_i = directFnDVV[i] - 1;
      for (size_t j=0; j<numDerivVars; j++) {
	size_t var_index_j = directFnDVV[j] - 1;
	if (var_index_i == 0 && var_index_j == 0)          // d^2g/db^2
	  fnHessians[1][i][j] = -8.*M/(b_sq*b*h_sq*Y)
	    - 6.*P_sq/(b_sq*b_sq*h_sq*Y_sq);
	else if ( (var_index_i == 0 && var_index_j == 1) ||
		  (var_index_i == 1 && var_index_j == 0) ) // d^2g/dbdh
	  fnHessians[1][i][j] = -8.*M/(b_sq*h_sq*h*Y)
	    - 4.*P_sq/(b_sq*b*h_sq*h*Y_sq);
	else if (var_index_i == 1 && var_index_j == 1)     // d^2g/dh^2
	  fnHessians[1][i][j] = -24.*M/(b*h_sq*h_sq*Y)
	    - 6.*P_sq/(b_sq*h_sq*h_sq*Y_sq);
	else if (var_index_i == 2 && var_index_j == 2)     // d^2g/dP^2
	  fnHessians[1][i][j] = -2./(b_sq*h_sq*Y_sq);
	else if ( (var_index_i == 2 && var_index_j == 3) ||
		  (var_index_i == 3 && var_index_j == 2) ) // d^2g/dPdM
	  fnHessians[1][i][j] = 0.;
	else if ( (var_index_i == 2 && var_index_j == 4) ||
		  (var_index_i == 4 && var_index_j == 2) ) // d^2g/dPdY
	  fnHessians[1][i][j] = 4.*P/(b_sq*h_sq*Y_sq*Y);
	else if (var_index_i == 3 && var_index_j == 3)     // d^2g/dM^2
	  fnHessians[1][i][j] = 0.;
	else if ( (var_index_i == 3 && var_index_j == 4) ||
		  (var_index_i == 4 && var_index_j == 3) ) // d^2g/dMdY
	  fnHessians[1][i][j] = 4./(b*h_sq*Y_sq);
	else if (var_index_i == 4 && var_index_j == 4)     // d^2g/dY^2
	  fnHessians[1][i][j] = -8.*M/(b*h_sq*Y_sq*Y)
	    - 6.*P_sq/(b_sq*h_sq*Y_sq*Y_sq);
	else { // unsupported cross-derivative
	  Cerr << "Error: unsupported Hessian cross term in short_column."
	       << endl;
	  abort_handler(-1);
	}
      }
    }
  }

  return 0; // no failure
}


int DirectApplicInterface::steel_column_cost()
{
  if (numVars != 3 || numFns != 1) {
    Cerr << "Error: wrong number of inputs/outputs in steel_column_cost."<<endl;
    abort_handler(-1);
  }

  // In the steel column description in Kuschel & Rackwitz, Cost is _not_
  // defined as a random variable.  That is Cost is not a fn(B, D, H), but
  // is rather defined as a fn(b, d, h).  Since dCost/dX|_{X=mean} is not the
  // same as dCost/dmean for non-normal X (jacobian_dX_dS is not 1), dCost/dX
  // may not be used and an optional interface must be defined for Cost.

  // b  = xC[0] = flange breadth   (lognormal unc. var., mean is design var.)
  // d  = xC[1] = flange thickness (lognormal unc. var., mean is design var.)
  // h  = xC[2] = profile height   (lognormal unc. var., mean is design var.)

  double b = xC[0], d = xC[1], h = xC[2];

  // **** f (objective = bd + 5h = cost of column):
  if (directFnASV[0] & 1)
    fnVals[0] = b*d + 5.*h;

  // **** df/dx:
  if (directFnASV[0] & 2) {
    for (size_t i=0; i<numDerivVars; i++) {
      size_t var_index = directFnDVV[i] - 1;
      switch (var_index) {
      case 0: // b
	fnGrads[0][i] = d;  break;
      case 1: // d
	fnGrads[0][i] = b;  break;
      case 2: // h
	fnGrads[0][i] = 5.; break;
      }
    }
  }

  // **** d^2f/dx^2:
  if (directFnASV[0] & 4) {
    for (size_t i=0; i<numDerivVars; i++) {
      size_t var_index_i = directFnDVV[i] - 1;
      for (size_t j=0; j<numDerivVars; j++) {
	size_t var_index_j = directFnDVV[j] - 1;
	fnHessians[0][i][j]
	  = ( (var_index_i == 0 && var_index_j == 1) ||
	      (var_index_i == 1 && var_index_j == 0) ) ? 1. : 0.;
      }
    }
  }

  return 0; // no failure
}


int DirectApplicInterface::steel_column_perf()
{
  if (numVars != 9 || numFns != 1) {
    Cerr << "Error: wrong number of inputs/outputs in steel_column_perf."<<endl;
    abort_handler(-1);
  }

  // In the steel column description in Kuschel & Rackwitz, Cost is _not_
  // defined as a random variable.  That is Cost is not a fn(B, D, H), but
  // is rather defined as a fn(b, d, h).  Since dCost/dX|_{X=mean} is not the
  // same as dCost/dmean for non-normal X (jacobian_dX_dS is not 1), dCost/dX
  // may not be used and an optional interface must be defined for Cost.

  // set effective length s based on assumed boundary conditions
  // actual length of the column is 7500 mm
  double s = 7500;

  // F0 = xC[0] = init. deflection (normal    unc. var.)
  // P1 = xC[1] = dead weight load (normal    unc. var.)
  // B  = xC[2] = flange breadth   (lognormal unc. var., mean is design var.)
  // D  = xC[3] = flange thickness (lognormal unc. var., mean is design var.)
  // H  = xC[4] = profile height   (lognormal unc. var., mean is design var.)
  // Fs = xC[5] = yield stress     (lognormal unc. var.)
  // P2 = xC[6] = variable load    (gumbel    unc. var.)
  // P3 = xC[7] = variable load    (gumbel    unc. var.)
  // E  = xC[8] = elastic modulus  (weibull   unc. var.)

  double F0 = xC[0], B = xC[2], D = xC[3], H = xC[4], Fs = xC[5], E = xC[8],
         P = xC[1]+xC[6]+xC[7], Pi = 3.1415926535897932385, Pi2 = Pi*Pi,
         Pi4 = Pi2*Pi2, Pi6 = Pi2*Pi4, B2 = B*B, D2 = D*D,
         H2 = H*H, H3 = H*H2, H5 = H2*H3, E2 = E*E, E3 = E*E2, s2 = s*s,
         X = Pi2*E*B*D*H2 - 2.*s2*P, X2 = X*X, X3 = X*X2;

  // **** g (limit state):
  if (directFnASV[0] & 1)
    fnVals[0] = Fs - P*(1./2./B/D + Pi2*F0*E*H/X);

  // **** dg/dx (w.r.t. active/uncertain variables):
  if (directFnASV[0] & 2) {
    for (size_t i=0; i<numDerivVars; i++) {
      size_t var_index = directFnDVV[i] - 1;
      switch (var_index) {
      case 0: // df/dF0
	fnGrads[0][i] = -E*H*P*Pi2/X;                       break;
      case 1: case 6: case 7: // df/dP1, df/dP2, df/dP3
	fnGrads[0][i] = -1./2./B/D - B*D*E2*F0*H3*Pi4/X2;   break;
      case 2: // df/dB
	fnGrads[0][i] = P*(1./2./B2/D + D*E2*F0*H3*Pi4/X2); break;
      case 3: // df/dD
	fnGrads[0][i] = P*(1./2./B/D2 + B*E2*F0*H3*Pi4/X2); break;
      case 4: // df/dH
	fnGrads[0][i] = E*F0*P*Pi2*(X + 4.*P*s2)/X2;        break;
      case 5: // df/dFs
	fnGrads[0][i] = 1.;                                 break;
      case 8: // df/dE
	fnGrads[0][i] = 2.*F0*H*P*P*Pi2*s2/X2;              break;
      }
    }
  }

  // **** d^2g/dx^2: (SORM)
  if (directFnASV[0] & 4) {
    for (size_t i=0; i<numDerivVars; i++) {
      size_t var_index_i = directFnDVV[i] - 1;
      for (size_t j=0; j<numDerivVars; j++) {
	size_t var_index_j = directFnDVV[j] - 1;
	if (var_index_i == 5 || var_index_j == 5)          // d^2g/dFsdx
	  fnHessians[0][i][j] = 0.;
	else if ( (var_index_i == 1 && var_index_j == 1) ||
                  (var_index_i == 1 && var_index_j == 6) ||
                  (var_index_i == 1 && var_index_j == 7) ||
		  (var_index_i == 6 && var_index_j == 1) ||
		  (var_index_i == 6 && var_index_j == 6) ||
		  (var_index_i == 6 && var_index_j == 7) ||
		  (var_index_i == 7 && var_index_j == 1) ||
		  (var_index_i == 7 && var_index_j == 6) ||
		  (var_index_i == 7 && var_index_j == 7) ) // d^2g/dPdP
	  fnHessians[0][i][j] = -4.*B*D*E2*F0*H3*Pi4*s2/X3;
	else if ( (var_index_i == 1 && var_index_j == 2) ||
 		  (var_index_i == 6 && var_index_j == 2) ||
		  (var_index_i == 7 && var_index_j == 2) ||
		  (var_index_i == 2 && var_index_j == 1) ||
		  (var_index_i == 2 && var_index_j == 6) ||
		  (var_index_i == 2 && var_index_j == 7) ) // d^2g/dPdB
	  fnHessians[0][i][j] = 1./2./B2/D
	    + D*E2*F0*H3*Pi4/X2*(2.*B*D*E*H2*Pi2/X - 1.);
	else if ( (var_index_i == 1 && var_index_j == 3) ||
 		  (var_index_i == 6 && var_index_j == 3) ||
		  (var_index_i == 7 && var_index_j == 3) ||
		  (var_index_i == 3 && var_index_j == 1) ||
		  (var_index_i == 3 && var_index_j == 6) ||
		  (var_index_i == 3 && var_index_j == 7) ) // d^2g/dPdD
	  fnHessians[0][i][j] = 1./2./B/D2
	    + B*E2*F0*H3*Pi4/X2*(2.*B*D*E*H2*Pi2/X - 1.);
	else if ( (var_index_i == 1 && var_index_j == 4) ||
 		  (var_index_i == 6 && var_index_j == 4) ||
		  (var_index_i == 7 && var_index_j == 4) ||
		  (var_index_i == 4 && var_index_j == 1) ||
		  (var_index_i == 4 && var_index_j == 6) ||
		  (var_index_i == 4 && var_index_j == 7) ) // d^2g/dPdH
	  fnHessians[0][i][j] = B*D*E2*F0*H2*Pi4*(X+8.*P*s2)/X3;
	else if ( (var_index_i == 1 && var_index_j == 0) ||
 		  (var_index_i == 6 && var_index_j == 0) ||
		  (var_index_i == 7 && var_index_j == 0) ||
		  (var_index_i == 0 && var_index_j == 1) ||
		  (var_index_i == 0 && var_index_j == 6) ||
		  (var_index_i == 0 && var_index_j == 7) ) // d^2g/dPdF0
	  fnHessians[0][i][j] = -B*D*E2*H3*Pi4/X2;
	else if ( (var_index_i == 1 && var_index_j == 8) ||
 		  (var_index_i == 6 && var_index_j == 8) ||
		  (var_index_i == 7 && var_index_j == 8) ||
		  (var_index_i == 8 && var_index_j == 1) ||
		  (var_index_i == 8 && var_index_j == 6) ||
		  (var_index_i == 8 && var_index_j == 7) ) // d^2g/dPdE
	  fnHessians[0][i][j] = 4.*B*D*E*F0*H3*P*Pi4*s2/X3;
	else if (var_index_i == 2 && var_index_j == 2)     // d^2g/dB^2
	  fnHessians[0][i][j] = -P*(1./B2/B/D + 2.*D2*E3*F0*H5*Pi6/X3);
	else if ( (var_index_i == 2 && var_index_j == 3) ||
		  (var_index_i == 3 && var_index_j == 2) ) // d^2g/dBdD
	  fnHessians[0][i][j] = -P*(1./2./B2/D2
	    + E2*F0*H3*Pi4/X2*(2.*B*D*E*H2*Pi2/X - 1.));
	else if ( (var_index_i == 2 && var_index_j == 4) ||
		  (var_index_i == 4 && var_index_j == 2) ) // d^2g/dBdH
	  fnHessians[0][i][j] = -D*E2*F0*H2*P*Pi4*(X + 8.*P*s2)/X3;
	else if ( (var_index_i == 0 && var_index_j == 2) ||
		  (var_index_i == 2 && var_index_j == 0) ) // d^2g/dBdF0
	  fnHessians[0][i][j] = D*E2*H3*P*Pi4/X2;
	else if ( (var_index_i == 2 && var_index_j == 8) ||
		  (var_index_i == 8 && var_index_j == 2) ) // d^2g/dBdE
	  fnHessians[0][i][j] = -4.*D*E*F0*H3*P*P*Pi4*s2/X3;
	else if (var_index_i == 3 && var_index_j == 3)     // d^2g/dD^2
	  fnHessians[0][i][j] = -P*(1./B/D2/D + 2.*B2*E3*F0*H5*Pi6/X3);
	else if ( (var_index_i == 3 && var_index_j == 4) ||
		  (var_index_i == 4 && var_index_j == 3) ) // d^2g/dDdH
	  fnHessians[0][i][j] = -B*E2*F0*H2*P*Pi4*(X + 8.*P*s2)/X3;
	else if ( (var_index_i == 0 && var_index_j == 3) ||
		  (var_index_i == 3 && var_index_j == 0) ) // d^2g/dDdF0
	  fnHessians[0][i][j] = B*E2*H3*P*Pi4/X2;
	else if ( (var_index_i == 3 && var_index_j == 8) ||
		  (var_index_i == 8 && var_index_j == 3) ) // d^2g/dDdE
	  fnHessians[0][i][j] = -4.*B*E*F0*H3*P*P*Pi4*s2/X3;
	else if (var_index_i == 4 && var_index_j == 4)     // d^2g/dH^2
	  fnHessians[0][i][j] = -2.*B*D*E2*F0*H*P*Pi4*(X + 8.*P*s2)/X3;
	else if ( (var_index_i == 0 && var_index_j == 4) ||
		  (var_index_i == 4 && var_index_j == 0) ) // d^2g/dHdF0
	  fnHessians[0][i][j] = E*P*Pi2*(X + 4.*P*s2)/X2;
	else if ( (var_index_i == 4 && var_index_j == 8) ||
		  (var_index_i == 8 && var_index_j == 4) ) // d^2g/dHdE
	  fnHessians[0][i][j] = -2.*F0*P*P*Pi2*s2*(3.*X + 8.*P*s2)/X3;
	else if (var_index_i == 0 && var_index_j == 0)     // d^2g/dF0^2
	  fnHessians[0][i][j] = 0.;
	else if ( (var_index_i == 0 && var_index_j == 8) ||
		  (var_index_i == 8 && var_index_j == 0) ) // d^2g/dF0dE
	  fnHessians[0][i][j] = 2.*H*P*P*Pi2*s2/X2;
	else if (var_index_i == 8 && var_index_j == 8)     // d^2g/dE^2
	  fnHessians[0][i][j] = -4.*B*D*F0*H3*P*P*Pi4*s2/X3;
	else { // unsupported derivative
	  Cerr << "Error: unsupported Hessian cross term in steel_column."
	       << endl;
	  abort_handler(-1);
	}
      }
    }
  }

  return 0; // no failure
}


int DirectApplicInterface::text_book()
{
  // This version is used when evalComm is not split into multiple analysis
  // servers.  In this case, execute the 3 portions serially.
  text_book1(); // objective fn val/grad/Hessian
  text_book2(); // constraint 1 val/grad/Hessian
  text_book3(); // constraint 2 val/grad/Hessian

  // Test failure capturing for Direct case
  //int r = rand();
  //if (r < 10000) // RAND_MAX = 32767
  //  return 1; // failure

  //sleep(5); // for faking a more expensive evaluation
  return 0; // no failure
}


// text_book1/2/3 are used when evalComm is split into multiple analysis
// servers.  In this case, the 3 portions are executed in parallel.
int DirectApplicInterface::text_book1()
{
  if (numFns > 3) {
    Cerr << "Error: Bad number of functions in text_book direct fn." << endl;
    abort_handler(-1);
  }
  // The presence of discrete variables can cause offsets in directFnDVV which
  // the text_book derivative logic does not currently account for.
  if ( (gradFlag || hessFlag) && numADV ) {
    Cerr << "Error: text_book direct fn assumes no discrete variables in "
	 << "derivative mode." << endl;
    abort_handler(-1);
  }

  // **********************************
  // **** f: sum (x[i] - POWVAL)^4 ****
  // **********************************
  size_t i;
  if (directFnASV[0] & 1) {
    double local_val = 0.0;
    for (i=analysisCommRank; i<numVars; i+=analysisCommSize) {
      // orders all continuous vars followed by all discrete vars.  This is 
      // fine in the direct case so long as everything is self-consistent.
      double x_i = (i<numACV) ? xC[i] : (double)xD[i-numACV];
      local_val += pow(x_i-POW_VAL, 4);
#ifdef TB_EXPENSIVE
      // MDO98/WCSMO99 TFLOP/NOW timings: j<=15000 used for fnVals[0] only
      //   (text_book1 only)
      // StrOpt_2002 TFLOP timings: j<=5000 used for all fnVals, fnGrads, and
      //   fnHessians that are present (text_book1/2/3)
      for (size_t j=1; j<=5000; j++)
        local_val += 1./(pow(x_i-POW_VAL,4)+j/100.)/(pow(x_i-POW_VAL,4)+j/100.);
#endif // TB_EXPENSIVE
    }

    if (multiProcAnalysisFlag) {
      double global_val = 0.0;
      parallelLib.reduce_sum_a(&local_val, &global_val, 1);
      // only analysisCommRank 0 has the correct sum.  This is OK (MPI_Allreduce
      // not needed) since only analysisCommRank 0 updates response for 
      // evalCommRank 0 in overlay_response.  evalCommRank 0 then returns the 
      // results to the iterator in ApplicationInterface::serve_evaluations().
      if (analysisCommRank == 0)
	fnVals[0] = global_val;
    }
    else
      fnVals[0] = local_val;
  }

  // ****************
  // **** df/dx: ****
  // ****************
  if (directFnASV[0] & 2) {
    //for (i=0; i<numDerivVars; i++)
    //  fnGrads[0][i] = 4.*pow(xC[i]-POW_VAL,3);
    fnGrads[0] = 0.;
    for (i=analysisCommRank; i<numDerivVars; i+=analysisCommSize) {
      size_t var_index = directFnDVV[i] - 1; // assumes no discrete vars
      double x_i = xC[var_index]; // assumes no discrete vars
      fnGrads[0][i] = 4.*pow(x_i-POW_VAL,3);
#ifdef TB_EXPENSIVE
      for (size_t j=1; j<=5000; j++)
        fnGrads[0][i] += 1./(pow(x_i-POW_VAL,3)+j/100.)
                           /(pow(x_i-POW_VAL,3)+j/100.);
#endif // TB_EXPENSIVE
    }

    if (multiProcAnalysisFlag) {
      double* sum_fns = (analysisCommRank) ? NULL : new double [numDerivVars];
      parallelLib.reduce_sum_a((double*)fnGrads[0].data(), sum_fns, 
			       numDerivVars);
      if (analysisCommRank == 0) {
	copy_data(sum_fns, numDerivVars, fnGrads[0]);
	delete [] sum_fns;
      }
    }
  }

  // ********************
  // **** d^2f/dx^2: ****
  // ********************
  if (directFnASV[0] & 4) {
    fnHessians[0] = 0.;
    //for (i=0; i<numDerivVars; i++)
    //  fnHessians[0][i][i] = 12.*pow(xC[i]-POW_VAL,2);
    for (i=analysisCommRank; i<numDerivVars; i+=analysisCommSize) {
      size_t var_index = directFnDVV[i] - 1; // assumes no discrete vars
      double x_i = xC[var_index]; // assumes no discrete vars
      fnHessians[0][i][i] = 12.*pow(x_i-POW_VAL,2);
#ifdef TB_EXPENSIVE
      for (size_t j=0; j<numDerivVars; j++)
	for (size_t k=1; k<=5000; k++)
	  fnHessians[0][i][j] += 1./(pow(x_i-POW_VAL,2)+k/100.)
                                   /(pow(x_i-POW_VAL,2)+k/100.);
#endif // TB_EXPENSIVE
    }

    if (multiProcAnalysisFlag) {
      int num_doubles = numDerivVars * numDerivVars;
      double* local_fns = new double [num_doubles];
      copy_data(fnHessians[0], local_fns, num_doubles, "c");
      double* sum_fns = (analysisCommRank) ? NULL : new double [num_doubles];
      parallelLib.reduce_sum_a(local_fns, sum_fns, num_doubles);
      delete [] local_fns;
      if (analysisCommRank == 0) {
	copy_data(sum_fns, num_doubles, "c", fnHessians[0], numDerivVars,
		  numDerivVars);
	delete [] sum_fns;
      }
    }
  }

  //sleep(2);
  return 0;
}


int DirectApplicInterface::text_book2()
{
  // **********************************
  // **** c1: x[0]*x[0] - 0.5*x[1] ****
  // **********************************
  size_t i;
  if (numFns > 1 && (directFnASV[1] & 1)) {
    double local_val = 0.0;
    // Definitely not the most efficient way to do this, but the point is to
    // demonstrate Comm communication.
    for (i=analysisCommRank; i<numVars; i+=analysisCommSize) {
      // orders all continuous vars followed by all discrete vars.  This is 
      // fine in the direct case so long as everything is self-consistent.
      double x_i = (i<numACV) ? xC[i] : (double)xD[i-numACV];
      if (i==0) // could be changed to i % 2 == 0 to get even vars.
        local_val += x_i*x_i;
      else if (i==1) // could be changed to i % 2 == 1 to get odd vars
        local_val -= 0.5*x_i;
#ifdef TB_EXPENSIVE
      for (size_t j=1; j<=5000; j++)
        local_val += 1./(pow(x_i-POW_VAL,4)+j/100.)/(pow(x_i-POW_VAL,4)+j/100.);
#endif // TB_EXPENSIVE
    }

    if (multiProcAnalysisFlag) {
      double global_val = 0.0;
      parallelLib.reduce_sum_a(&local_val, &global_val, 1);
      // only analysisCommRank 0 has the correct sum.  This is OK (MPI_Allreduce
      // not needed) since only analysisCommRank 0 updates response for 
      // evalCommRank 0 in overlay_response.  evalCommRank 0 then returns the 
      // results to the iterator in ApplicationInterface::serve_evaluations().
      if (analysisCommRank == 0)
	fnVals[1] = global_val;
    }
    else
      fnVals[1] = local_val;
  }

  // *****************
  // **** dc1/dx: ****
  // *****************
  if (numFns > 1 && (directFnASV[1] & 2)) {
    fnGrads[1] = 0.;
    //fnGrads[1][0] = 2.*xC[0];
    //fnGrads[1][1] = -0.5;
    for (i=analysisCommRank; i<numDerivVars; i+=analysisCommSize) {
      size_t var_index = directFnDVV[i] - 1; // assumes no discrete vars
      if (var_index == 0)
        fnGrads[1][i] = 2.*xC[0];
      else if (var_index == 1)
        fnGrads[1][i] = -0.5;
#ifdef TB_EXPENSIVE
      double x_i = xC[var_index];
      for (size_t j=1; j<=5000; j++)
        fnGrads[1][i] += 1./(pow(x_i-POW_VAL,3)+j/100.)
                           /(pow(x_i-POW_VAL,3)+j/100.);
#endif // TB_EXPENSIVE
    }

    if (multiProcAnalysisFlag) {
      double* sum_fns = (analysisCommRank) ? NULL : new double [numDerivVars];
      parallelLib.reduce_sum_a((double*)fnGrads[1].data(), sum_fns,
			       numDerivVars);
      if (analysisCommRank == 0) {
	copy_data(sum_fns, numDerivVars, fnGrads[1]);
	delete [] sum_fns;
      }
    }
  }

  // *********************
  // **** d^2c1/dx^2: ****
  // *********************
  if (numFns > 1 && (directFnASV[1] & 4)) {
    fnHessians[1] = 0.;
    //fnHessians[1][0][0] = 2.;
    for (i=analysisCommRank; i<numDerivVars; i+=analysisCommSize) {
      size_t var_index = directFnDVV[i] - 1; // assumes no discrete vars
      if (var_index == 0)
	fnHessians[1][i][i] = 2.;
#ifdef TB_EXPENSIVE
      double x_i = xC[var_index];
      for (size_t j=0; j<numDerivVars; j++)
	for (size_t k=1; k<=5000; k++)
	  fnHessians[1][i][j] += 1./(pow(x_i-POW_VAL,2)+k/100.)
                                   /(pow(x_i-POW_VAL,2)+k/100.);
#endif // TB_EXPENSIVE
    }

    if (multiProcAnalysisFlag) {
      int num_doubles = numDerivVars * numDerivVars;
      double* local_fns = new double [num_doubles];
      copy_data(fnHessians[1], local_fns, num_doubles, "c");
      double* sum_fns = (analysisCommRank) ? NULL : new double [num_doubles];
      parallelLib.reduce_sum_a(local_fns, sum_fns, num_doubles);
      delete [] local_fns;
      if (analysisCommRank == 0) {
	copy_data(sum_fns, num_doubles, "c", fnHessians[1], numDerivVars,
		  numDerivVars);
	delete [] sum_fns;
      }
    }
  }

  //sleep(2);
  return 0;
}


int DirectApplicInterface::text_book3()
{
  // **********************************
  // **** c2: x[1]*x[1] - 0.5*x[0] ****
  // **********************************
  size_t i;
  if (numFns > 2 && (directFnASV[2] & 1)) {
    double local_val = 0.0;
    // Definitely not the most efficient way to do this, but the point is to
    // demonstrate Comm communication.
    for (i=analysisCommRank; i<numVars; i+=analysisCommSize) {
      // orders all continuous vars followed by all discrete vars.  This is 
      // fine in the direct case so long as everything is self-consistent.
      double x_i = (i<numACV) ? xC[i] : (double)xD[i-numACV];
      if (i==0) // could be changed to i % 2 == 0 to get even vars.
        local_val -= 0.5*x_i;
      else if (i==1) // could be changed to i % 2 == 1 to get odd vars
        local_val += x_i*x_i;
#ifdef TB_EXPENSIVE
      for (size_t j=1; j<=5000; j++)
        local_val += 1./(pow(x_i-POW_VAL,4)+j/100.)/(pow(x_i-POW_VAL,4)+j/100.);
#endif // TB_EXPENSIVE
    }

    if (multiProcAnalysisFlag) {
      double global_val = 0.0;
      parallelLib.reduce_sum_a(&local_val, &global_val, 1);
      // only analysisCommRank 0 has the correct sum.  This is OK (MPI_Allreduce
      // not needed) since only analysisCommRank 0 updates response for 
      // evalCommRank 0 in overlay_response.  evalCommRank 0 then returns the 
      // results to the iterator in ApplicationInterface::serve_evaluations().
      if (analysisCommRank == 0)
	fnVals[2] = global_val;
    }
    else
      fnVals[2] = local_val;
  }

  // *****************
  // **** dc2/dx: ****
  // *****************
  if (numFns > 2 && (directFnASV[2] & 2)) {
    fnGrads[2] = 0.;
    //fnGrads[2][0] = -0.5;
    //fnGrads[2][1] = 2.*xC[1];
    for (i=analysisCommRank; i<numDerivVars; i+=analysisCommSize) {
      size_t var_index = directFnDVV[i] - 1; // assumes no discrete vars
      if (var_index == 0)
        fnGrads[2][i] = -0.5;
      else if (var_index == 1)
        fnGrads[2][i] = 2.*xC[1];
#ifdef TB_EXPENSIVE
      double x_i = xC[var_index];
      for (size_t j=1; j<=5000; j++)
        fnGrads[2][i] += 1./(pow(x_i-POW_VAL,3)+j/100.)
                           /(pow(x_i-POW_VAL,3)+j/100.);
#endif // TB_EXPENSIVE
    }

    if (multiProcAnalysisFlag) {
      double* sum_fns = (analysisCommRank) ? NULL : new double [numDerivVars];
      parallelLib.reduce_sum_a((double*)fnGrads[2].data(), sum_fns,
			       numDerivVars);
      if (analysisCommRank == 0) {
	copy_data(sum_fns, numDerivVars, fnGrads[2]);
	delete [] sum_fns;
      }
    }
  }

  // *********************
  // **** d^2c2/dx^2: ****
  // *********************
  if (numFns > 2 && (directFnASV[2] & 4)) {
    fnHessians[2] = 0.;
    //fnHessians[2][1][1] = 2.;
    for (i=analysisCommRank; i<numDerivVars; i+=analysisCommSize) {
      size_t var_index = directFnDVV[i] - 1; // assumes no discrete vars
      if (var_index == 1)
	fnHessians[2][i][i] = 2.;
#ifdef TB_EXPENSIVE
      double x_i = xC[var_index];
      for (size_t j=0; j<numDerivVars; j++)
	for (size_t k=1; k<=5000; k++)
	  fnHessians[2][i][j] += 1./(pow(x_i-POW_VAL,2)+k/100.)
                                   /(pow(x_i-POW_VAL,2)+k/100.);
#endif // TB_EXPENSIVE
    }

    if (multiProcAnalysisFlag) {
      int num_doubles = numDerivVars * numDerivVars;
      double* local_fns = new double [num_doubles];
      copy_data(fnHessians[2], local_fns, num_doubles, "c");
      double* sum_fns = (analysisCommRank) ? NULL : new double [num_doubles];
      parallelLib.reduce_sum_a(local_fns, sum_fns, num_doubles);
      delete [] local_fns;
      if (analysisCommRank == 0) {
	copy_data(sum_fns, num_doubles, "c", fnHessians[2], numDerivVars,
		  numDerivVars);
	delete [] sum_fns;
      }
    }
  }

  //sleep(2);
  return 0;
}


int DirectApplicInterface::text_book_ouu()
{
  if (multiProcAnalysisFlag) {
    Cerr << "Error: text_book_ouu direct fn does not support multiprocessor "
	 << "analyses." << endl;
    abort_handler(-1);
  }
  // typical usage is 2 design vars + 6 uncertain variables, although the
  // number of uncertain variables can be any factor of two.
  if (numVars < 4 || numVars % 2 || numADV) {
    Cerr << "Error: Bad number of variables in text_book_ouu direct fn." <<endl;
    abort_handler(-1);
  }
  if (numFns > 3) {
    Cerr << "Error: Bad number of functions in text_book_ouu direct fn." <<endl;
    abort_handler(-1);
  }
  if (hessFlag) {
    Cerr << "Error: Hessians not supported in text_book_ouu direct fn." << endl;
    abort_handler(-1);
  }

  size_t i, split = 2 + (numVars - 2)/2; // split the uncertain vars among d1,d2
  // xC[0], xC[1]     = design
  // xC[2], xC[3],... = uncertain

  // **** f:
  if (directFnASV[0] & 1) {
    double f = 0.;
    for(i=2; i<split; i++)
      f += pow(xC[i]-10.*xC[0], 4.0);
    for(i=split; i<numVars; i++)
      f += pow(xC[i]-10.*xC[1], 4.0);
    fnVals[0] = f;
  }

  // **** c1:
  if (numFns>1 && (directFnASV[1] & 1))
    fnVals[1] = xC[0]*(xC[2]*xC[2] - 0.5*xC[3]);

  // **** c2:
  if (numFns>2 && (directFnASV[2] & 1))
    fnVals[2] = xC[1]*(xC[3]*xC[3] - 0.5*xC[2]);

  // **** df/dx:
  if (directFnASV[0] & 2) {
    for (i=0; i<numDerivVars; i++) {
      size_t var_index = directFnDVV[i] - 1;
      double f0, f1, xC0, xC1, xCvi;
      switch (var_index) {
      case 0: // design variable derivative
	f0 = 0.; xC0 = xC[0];
	for (size_t j=2; j<split; j++)
	  f0 += -40.*pow(xC[j]-10.*xC0, 3.0);
	fnGrads[0][i] = f0;
	break;
      case 1: // design variable derivative
	f1 = 0.; xC1 = xC[1];
	for (size_t j=split; j<numVars; j++)
	  f1 += -40.*pow(xC[j]-10.*xC1, 3.0);
	fnGrads[0][i] = f1;
	break;
      default: // uncertain variable derivative
	xCvi = xC[var_index];
	fnGrads[0][i] = (var_index<split) ? 4*pow(xCvi-10.*xC[0], 3.0)
                                          : 4*pow(xCvi-10.*xC[1], 3.0);
	break;
      }
    }
  }

  // **** dc1/dx:
  if (numFns>1 && (directFnASV[1] & 2)) {
    for (i=0; i<numDerivVars; i++) {
      size_t var_index = directFnDVV[i] - 1;
      switch (var_index) {
      case 0: // design variable derivative
	fnGrads[1][i] = xC[2]*xC[2] - 0.5*xC[3]; break;
      case 2: // uncertain variable derivative
	fnGrads[1][i] = 2*xC[0]*xC[2];           break;
      case 3: // uncertain variable derivative
	fnGrads[1][i] = -0.5*xC[0];              break;
      default: // all other derivatives
	fnGrads[1][i] = 0.;                      break;
      }
    }
  }

  // **** dc2/dx:
  if (numFns>2 && (directFnASV[2] & 2)) {
    for (i=0; i<numDerivVars; i++) {
      size_t var_index = directFnDVV[i] - 1;
      switch (var_index) {
      case 1: // design variable derivative
	fnGrads[2][i] = xC[3]*xC[3] - 0.5*xC[2]; break;
      case 2: // uncertain variable derivative
	fnGrads[2][i] = -0.5*xC[1];              break;
      case 3: // uncertain variable derivative
	fnGrads[2][i] = 2*xC[1]*xC[3];           break;
      default: // all other derivative
	fnGrads[2][i] = 0.;                      break;
      }
    }
  }

  //sleep(5); // for faking a more expensive evaluation
  return 0; // no failure
}


int DirectApplicInterface::multimodal()
{
  if (multiProcAnalysisFlag) {
    Cerr << "Error: multimodal direct fn does not support multiprocessor "
	 << "analyses." << endl;
    abort_handler(-1);
  }
  if (numVars != 2 || numADV || ((gradFlag || hessFlag) && numDerivVars != 2)) {
    Cerr << "Error: Bad number of variables in multimodal direct fn." << endl;
    abort_handler(-1);
  }
  if (numFns != 1) {
    Cerr << "Error: Bad number of functions in multimodal direct fn." << endl;
    abort_handler(-1);
  }

  // **** f:
  if (directFnASV[0] & 1)
    fnVals[0] = (xC[0]*xC[0]+4)*(xC[1]-1)/20 - sin(5*xC[0]/2) - 2;

  // **** df/dx:
  if (directFnASV[0] & 2) {
    fnGrads[0][0] = xC[0]*(xC[1]-1)/10 - (5/2)*cos(5*xC[0]/2);
    fnGrads[0][1] = (xC[0]*xC[0]+4)/20;
  }

  // **** d^2f/dx^2:
  if (directFnASV[0] & 4) {
    fnHessians[0][0][0] = (xC[1]-1)/10 + (25/4)*sin(5*xC[0]/2);
    fnHessians[0][0][1] = fnHessians[0][1][0] = xC[0]/10;
    fnHessians[0][1][1] = 0.0;
  }

  return 0; // no failure
}


// -------------------------------------------
// Begin direct interfaces to simulation codes
// -------------------------------------------
#ifdef DAKOTA_SALINAS
int DirectApplicInterface::salinas() 
{
  if (numFns < 1 || numFns > 3) {
    Cerr << "Error: Bad number of functions in salinas direct fn." << endl;
    abort_handler(-1);
  }
  if (numVars < 1) {
    Cerr << "Error: Bad number of variables in salinas direct fn." << endl;
    abort_handler(-1);
  }
  if (gradFlag || hessFlag) {
    Cerr << "Error: analytic derivatives not yet supported in salinas direct "
	 << "fn." <<endl;
    abort_handler(-1);
  }

  // ------------------------
  // Salinas input processing
  // ------------------------

  // Set up dummy argc and argv with name of modified input file.
  // NOTE: for concurrent analyses within each fn eval, may want something like
  //       salinas.<eval>.<analysis>.inp (e.g., salinas.3.2.inp)
  int argc = 2;
  char* argv[3];
  argv[0] = "salinas"; // should be ignored
  char si[32];
  sprintf(si,"salinas%d.inp", evalServerId); // tag root name in root.inp
  argv[1] = si;
  argv[2] = NULL; // standard requires this, see Kern.&Ritchie, p. 115

  // Insert vars into Salinas input file (Exodus model file avoided for now).
  // Set up loop to process input file and match tokens to variable tags.  The 
  // Salinas parser is not dependent on new lines, so don't worry about 
  // formatting.
  const StringArray& cdv_labels = vars.continuous_variable_labels();
  const StringArray& ddv_labels = vars.discrete_variable_labels();
  size_t i, num_cdv = cdv_labels.length(), num_ddv = ddv_labels.length();
  StringArray x_vect_labels(numVars);
  for (i=0; i<num_cdv; i++)
    x_vect_labels[i] = cdv_labels[i];
  for (i=0; i<num_ddv; i++)
    x_vect_labels[i+num_cdv] = ddv_labels[i];
  ifstream fin("salinas.inp.template");
  ofstream fout(argv[1]);
  String token;
  while (fin) {
    fin >> token;
    if (token=="//")
      fin.ignore(256, '\n'); // comments will not be replicated in fout
    else if (token=="'./tagged.exo'") {
      // **** Issues with the Exodus input file.  Exodus input files must be 
      //   tagged because the Exodus output uses the same name and must be 
      //   protected from conflict with other concurrent simulations.  This 
      //   requires the creation of these tagged files by DAKOTA or their 
      //   existence a priori (which is a problem when tagging with open ended
      //   indices like evaluation id). A few approaches to handling this:
      // 1.) could use system("cp root.exo root#.exo"), but no good on TFLOP
      // 2.) could tag w/ evalServerId & put sal[0-9].exo out before launching,
      //   but Salinas must overwrite properly (it does) & data loss must be OK
      // 3.) could modify salinas to use (tagged) root.inp i/o root.exo in 
      //   creating root-out.exo, thereby removing the need to tag Exodus input
      char se[32];
      sprintf(se,"'./salinas%d.exo' ", evalServerId); // tag root in root.exo
      fout << se;
    }
    else {
      int found = 0;
      for (i=0; i<numVars; i++) { // loop to remove any order dependency
        if (token == x_vect_labels[i]) { // detect variable label
          fout << xC[i] << ' ';
          found=1;
          break;
        }
      }
      if (!found)
        fout << token << ' ';
    }
  }
  fout << flush;

  // -----------------
  // Salinas execution
  // -----------------

  // salinas_main is a bare bones wrapper for salinas.  It is provided to 
  // permit calling salinas as a subroutine.

  // analysis_comm may be invalid if multiProcAnalysisFlag is not true!
  const ParallelLevel& ea_pl
    = parallelLib.parallel_configuration().ea_parallel_level();
  MPI_Comm analysis_comm = ea_pl.server_intra_communicator();
  int fail_code = salinas_main(argc, argv, &analysis_comm);
  if (fail_code)
    return fail_code;

  // ---------------------------
  // Salinas response processing
  // ---------------------------

  // Compute margins and return lowest margin as objective function to be 
  // _maximized_ (minimize opposite sign).  Constrain mass to be: 
  // mass_low_bnd <= mass <= mass_upp_bnd
  //Real min_margin = 0.;
  Real lambda1, mass, mass_low_bnd=3., mass_upp_bnd=6.; // 4.608 nominal mass

  // Call EXODUS utilities to retrieve stress & g data 

  // Retrieve data from salinas#.rslt
  char so[32];
  sprintf(so,"salinas%d.rslt",evalServerId);
  ifstream f2in(so);
  while (f2in) {
    f2in >> token;
    if (token=="Total") {
      for (i=0; i<4; i++) // After "Total", parse "Mass of Structure is"
        f2in >> token;
      f2in >> mass;
    }
    else if (token=="1:") {
      f2in >> lambda1;
    }
    else 
      f2in.ignore(256, '\n');
  }

  // **** f:
  if (directFnASV[0] & 1)
    fnVals[0] = -lambda1; // max fundamental freq. s.t. mass bounds
    //fnVals[0] = -min_margin;

  // **** c1:
  if (numFns > 1 && (directFnASV[1] & 1))
    fnVals[1] = (mass_low_bnd - mass)/mass_low_bnd;

  // **** c2:
  if (numFns > 2 && (directFnASV[2] & 1))
    fnVals[2] = (mass - mass_upp_bnd)/mass_upp_bnd;

  // **** df/dx:
  //if (directFnASV[0] & 2) {
  //}

  // **** dc1/dx:
  //if (numFns > 1 && (directFnASV[1] & 2)) {
  //}

  // **** dc2/dx:
  //if (numFns > 2 && (directFnASV[2] & 2)) {
  //}

  return 0;
}
#endif // DAKOTA_SALINAS


#ifdef DAKOTA_MODELCENTER
int DirectApplicInterface::mc_api_run()
{
  // ModelCenter interface through direct Dakota interface, HKIM 4/3/03
  // Modified to replace pxcFile with analysisComponents,   MSE 6/20/05

  if (multiProcAnalysisFlag) {
    Cerr << "Error: mc_api_run direct fn does not yet support multiprocessor "
	 << "analyses." << endl;
    abort_handler(-1);
  }

  int i, j, ireturn, iprint = 1;

  if(!mc_ptr_int) { // If null, instantiate ModelCenter
    // the pxcFile (Phoenix configuration file) is passed through the
    // analysis_components specification
    if (!analysisComponents.empty() &&
	!analysisComponents[analysisDriverIndex].empty())
      mc_load_model(ireturn,iprint,mc_ptr_int,
		    analysisComponents[analysisDriverIndex][0].c_str());
    else
      ireturn = -1;
    if(ireturn == -1) abort_handler(-1);
  }

  for(i=0; i<numACV; i++) {
    double input = xC[i];
    const char* inStr = xCLabels[i].c_str();
    if (mc_ptr_int) {
      mc_set_value(ireturn,iprint,mc_ptr_int,xC[i],inStr);
      if(ireturn == -1) abort_handler(-1);
    }
    else {
      Cerr << "Error: null ModelCenter pointer for mc_set_value" << endl;
      abort_handler(-1);
    }
  }

  for(i=0; i<numADV; i++) {
    int input = xD[i];
    int iType, enumLen;
    const char* inStr = xDLabels[i].c_str();
    if (mc_ptr_int) {
      mc_get_var_type(ireturn,iprint,mc_ptr_int,iType,inStr);
      if(ireturn == -1) abort_handler(-1);
      mc_get_enum_len(ireturn,iprint,mc_ptr_int,enumLen,inStr);
      if(ireturn == -1) abort_handler(-1);

      if (iType == 2) { // integer type
	if(enumLen > 0) {
	  int* enumVal = new int[enumLen]; 
	  mc_get_enum_int(ireturn,iprint,mc_ptr_int,enumVal,enumLen,inStr);
	  if(ireturn == -1) abort_handler(-1);
	  mc_set_value_int(ireturn,iprint,mc_ptr_int,enumVal[xD[i]],inStr);
	  if(ireturn == -1) abort_handler(-1);
	  delete [] enumVal;
	}
	else {
	  mc_set_value_int(ireturn,iprint,mc_ptr_int,xD[i],inStr);
	  if(ireturn == -1) abort_handler(-1);
	}
      }
      else if(iType == 1) { // double type
	if(enumLen > 0) {
	  double* enumVal = new double[enumLen];
	  mc_get_enum_dbl(ireturn,iprint,mc_ptr_int,enumVal,enumLen,inStr);
	  if(ireturn == -1) abort_handler(-1);
	  double e1 = enumVal[0], e2 = enumVal[1], e3 = enumVal[2], 
	    e4 = enumVal[3], e5 = enumVal[4], inFF = enumVal[xD[i]];
	  mc_set_value_dbl(ireturn,iprint,mc_ptr_int,enumVal[xD[i]],inStr);
	  if(ireturn == -1) abort_handler(-1);
	  delete [] enumVal;
	}
	else {
	  Cerr << "Error: cannot assign a DAKOTA integer variable to double "
	       << "variable in ModelCenter" << endl;
	  abort_handler(-1);
	}
      }
    }
    else {
      Cerr << "Error: null ModelCenter pointer for mc_set_value_int" << endl;
      abort_handler(-1);
    }
  }


  int out_var_act_len = fnLabels.length();
  if (out_var_act_len != numFns) {
    Cerr << "Error: Mismatch in the number of responses in mc_api_run." << endl;
    abort_handler(-1);
  }
 
  for (i=0; i<out_var_act_len; i++) {
    if (mc_ptr_int) {
      // **** f:
      if (directFnASV[i] & 1) {
	const char* outStr = fnLabels[i].c_str();
	mc_get_value(ireturn,iprint,mc_ptr_int,fnVals[i],outStr);  
	if(ireturn == -1) abort_handler(-1);
      }
      // **** df/dx:
      if (directFnASV[i] & 2) {
	Cerr << "Error: Analytic gradients not supported in mc_api_run." <<endl;
	abort_handler(-1);
      }
      // **** d^2f/dx^2:
      if (directFnASV[i] & 4) {
	Cerr << "Error: Analytic Hessians not supported in mc_api_run." << endl;
	abort_handler(-1);
      }
    }
    else {
      Cerr << "Error: null ModelCenter pointer for mc_get_value" << endl;
      abort_handler(-1);
    }
  }

  return 0; // no failure
}
#endif // DAKOTA_MODELCENTER


#ifdef DAKOTA_MATLAB
int DirectApplicInterface::matlab_engine_run()
{
  // direct interface to Matlab through Mathworks external API, BMA 11/28/2005
  // mfile executed is specified through analysisComponents
  // Special thanks to Lee Peterson for substantial enhancements 12/15/2007:
  // - Added output buffer for the MATLAB command response and error messages
  // - Made the Dakota variable persistent in the MATLAB engine workspace
  // - Added robustness to the user deleting required Dakota fields

  const int MATLAB_FAIL = 1;
  const int RESPONSE_TYPES[] = {1, 2, 4};
  int i, j, k, iF;

  /* variables for Matlab data type objects */
  mxArray *dakota_matlab = NULL;
  mxArray *mx_tmp[NUMBER_OF_FIELDS];
  mxArray *mx_string;

  /* temporary variables */
  mwSize dims[3];
  const mwSize *dim_ptr;
  double *ptr;
  String analysis_command;
  const int BUFSIZE = 32768;
  char buffer_char[BUFSIZE+1] = {'\0'};
  String buffer_string;
  
  int fail_code;
  bool fn_flag;
  
  // make sure the Matlab user did not corrupt the workspace variable Dakota
  if ( (dakota_matlab = engGetVariable(matlabEngine, "Dakota")) == NULL ) {
    Cerr << "Error (Direct:Matlab): Failed to get variable Dakota from "
         << "Matlab." << endl;
    return(MATLAB_FAIL);
  }
  if ( mxIsEmpty(dakota_matlab) ) {
    Cerr << "Error (Direct:Matlab): Dakota variable from Matlab "
         << "is empty. " << endl;
    return(MATLAB_FAIL);
  }
  if ( !mxIsStruct(dakota_matlab) ) {
    Cerr << "Error (Direct:Matlab): Dakota variable from Matlab "
         << "is not a structure array. " << endl;
    return(MATLAB_FAIL);
  }
  if ( (mxGetM(dakota_matlab) != 1) | (mxGetN(dakota_matlab) != 1) ) {
    Cerr << "Error (Direct:Matlab): Dakota variable from Matlab "
         << "is not a scalar. " << endl;
    return(MATLAB_FAIL);
  }

  /* Set scalar valued data */
  if ( (iF = matlab_field_prep(dakota_matlab, FIELD_NAMES[0])) >= 0)
    mxSetFieldByNumber(dakota_matlab,0,iF,
		       mxCreateDoubleScalar((double) numFns));
  else
    return(MATLAB_FAIL);
  
  if ( (iF = matlab_field_prep(dakota_matlab, FIELD_NAMES[1])) >= 0)
    mxSetFieldByNumber(dakota_matlab,0,iF,
		       mxCreateDoubleScalar((double) numVars));
  else
    return(MATLAB_FAIL);
  
  if ( (iF = matlab_field_prep(dakota_matlab, FIELD_NAMES[2])) >= 0)
    mxSetFieldByNumber(dakota_matlab,0,iF,
		       mxCreateDoubleScalar((double) numACV));
  else
    return(MATLAB_FAIL);
    
  if ( (iF = matlab_field_prep(dakota_matlab, FIELD_NAMES[3])) >= 0)
    mxSetFieldByNumber(dakota_matlab,0,iF,
		       mxCreateDoubleScalar((double) numADV));
  else
    return(MATLAB_FAIL);
  
  if ( (iF = matlab_field_prep(dakota_matlab, FIELD_NAMES[4])) >= 0)
    mxSetFieldByNumber(dakota_matlab,0,iF, 
		       mxCreateDoubleScalar((double) numDerivVars));
  else
    return(MATLAB_FAIL);
  
  /* Create Matlab matrix, get pointer to it, populate, put in structure 
     NOTE that mxSetFieldByNumber needs to retain allocated memory -- it
     just points to the created mxArray mx_tmp[k], so don't deallocate mx_tmp. 
     MAY be able to use mxSetPr, depending on what types 
     RealVector, IntVector, StringArray are
  */

  /* continuous variables */
  if ( (iF = matlab_field_prep(dakota_matlab, FIELD_NAMES[5])) >= 0) {
    mx_tmp[5] = mxCreateDoubleMatrix(1,numACV,mxREAL);
    ptr = mxGetPr(mx_tmp[5]);
    for( i=0; i<numACV; i++)  
      *(ptr+i) = xC[i];  
    mxSetFieldByNumber(dakota_matlab, 0, iF, mx_tmp[5]);
  }
  else
    return(MATLAB_FAIL);

  /* discrete variables */
  if ( (iF = matlab_field_prep(dakota_matlab, FIELD_NAMES[6])) >= 0) {
    mx_tmp[6] = mxCreateDoubleMatrix(1,numADV,mxREAL);
    ptr = mxGetPr(mx_tmp[6]);
    for( i=0; i<numADV; i++)  
      *(ptr+i) = (double) xD[i];
    mxSetFieldByNumber(dakota_matlab, 0, iF, mx_tmp[6]);
  }
  else
    return(MATLAB_FAIL);

  /* continuous labels */
  if ( (iF = matlab_field_prep(dakota_matlab, FIELD_NAMES[7])) >= 0) {
    dims[0] = numACV;
    dims[1] = 1;
    mx_tmp[7] = mxCreateCellArray(2,dims);
    for( i=0; i<numACV; i++)  
      mxSetCell(mx_tmp[7], i, mxCreateString( xCLabels[i] ));
    mxSetFieldByNumber(dakota_matlab, 0, iF, mx_tmp[7]);
  }
  else
    return(MATLAB_FAIL);

  /* discrete labels */
  if ( (iF = matlab_field_prep(dakota_matlab, FIELD_NAMES[8])) >= 0) {
    dims[0] = numADV;
    dims[1] = 1;
    mx_tmp[8] = mxCreateCellArray(2,dims);
    for( i=0; i<numADV; i++)
      mxSetCell(mx_tmp[8], i, mxCreateString( xDLabels[i] ));
    mxSetFieldByNumber(dakota_matlab, 0, iF, mx_tmp[8]);
  }
  else
    return(MATLAB_FAIL);

  /* active set vector (ASV) / matrix */
  if ( (iF = matlab_field_prep(dakota_matlab, FIELD_NAMES[9])) >= 0) {
    mx_tmp[9] = mxCreateDoubleMatrix(1,numFns,mxREAL);
    for( i=0; i<numFns; i++)
      *(mxGetPr(mx_tmp[9])+i) = directFnASV[i];  
    mxSetFieldByNumber(dakota_matlab, 0, iF, mx_tmp[9]);
  }
  else
    return(MATLAB_FAIL);
  
  /* Create boolean version of ASV.  Rows are fnval, grad, hess; col per fn
     CAREFUL -- Matlab stores by column */
  if ( (iF = matlab_field_prep(dakota_matlab, FIELD_NAMES[10])) >= 0) {
    mx_tmp[10] = mxCreateDoubleMatrix(3,numFns,mxREAL);
    for( i=0; i<3; i++)
      for ( j=0; j<numFns; j++)
	// FIX THIS MESS!
	(directFnASV[j] & RESPONSE_TYPES[i]) ? (*(mxGetPr(mx_tmp[10])+3*j+i) = 1)
	  : (*(mxGetPr(mx_tmp[10])+3*j+i) = 0);
    mxSetFieldByNumber(dakota_matlab, 0, iF, mx_tmp[10]);  
  }
  else
    return(MATLAB_FAIL);

  /* derivative variables vector (DVV) / DVV_bool */
  if ( (iF = matlab_field_prep(dakota_matlab, FIELD_NAMES[11])) >= 0) {
    mx_tmp[11] = mxCreateDoubleMatrix(1,numDerivVars,mxREAL);
    for( i=0; i<numDerivVars; i++)
      *(mxGetPr(mx_tmp[11])+i) = directFnDVV[i];  
    mxSetFieldByNumber(dakota_matlab, 0, iF, mx_tmp[11]);
  }
  else
    return(MATLAB_FAIL);

  if ( (iF = matlab_field_prep(dakota_matlab, FIELD_NAMES[12])) >= 0) {
    mx_tmp[12] = mxCreateDoubleMatrix(1,numVars,mxREAL);
    for ( j=0; j<numDerivVars; j++)
      *(mxGetPr(mx_tmp[12])+directFnDVV[j]-1) = 1;
    mxSetFieldByNumber(dakota_matlab, 0, iF, mx_tmp[12]);
  }
  else
    return(MATLAB_FAIL);
  
  /* fn, grad, hess Flags; as needed, allocate & initialize matrices to zero */
  fn_flag = false;
  for ( j=0; j<numFns; j++)
    if (directFnASV[j] & 1) {
      fn_flag = true;
      if ( (iF = matlab_field_prep(dakota_matlab, FIELD_NAMES[16])) >= 0) {
	mx_tmp[16] = mxCreateDoubleMatrix(1,numFns,mxREAL);
	mxSetFieldByNumber(dakota_matlab, 0, iF, mx_tmp[16]);
      }
      else 
	return(MATLAB_FAIL);
      break;
    }

  if (gradFlag) {
    if ( (iF = matlab_field_prep(dakota_matlab, FIELD_NAMES[17])) >= 0) {
      mx_tmp[17] = mxCreateDoubleMatrix(numFns,numDerivVars,mxREAL);
      mxSetFieldByNumber(dakota_matlab, 0, iF, mx_tmp[17]);
    }
    else	
      return(MATLAB_FAIL);
  }

  if (hessFlag) {
    if ( (iF = matlab_field_prep(dakota_matlab, FIELD_NAMES[18])) >= 0) {
      dims[0] = numFns;
      dims[1] = numDerivVars;
      dims[2] = numDerivVars;
      mx_tmp[18] = mxCreateNumericArray(3,dims,mxDOUBLE_CLASS,mxREAL);
      mxSetFieldByNumber(dakota_matlab, 0, iF, mx_tmp[18]);
    }
    else 
      return(MATLAB_FAIL);
  }
 
  if ( (iF = matlab_field_prep(dakota_matlab, FIELD_NAMES[13])) >= 0) {
    mxSetFieldByNumber(dakota_matlab,0,iF,
		       mxCreateDoubleScalar((double) fn_flag));
  }
  else	
    return(MATLAB_FAIL);
  
  if ( (iF = matlab_field_prep(dakota_matlab, FIELD_NAMES[14])) >= 0) {
    mxSetFieldByNumber(dakota_matlab,0,iF,
		       mxCreateDoubleScalar((double) gradFlag));
  }
  else	
    return(MATLAB_FAIL);

  if ( (iF = matlab_field_prep(dakota_matlab, FIELD_NAMES[15])) >= 0) {
    mxSetFieldByNumber(dakota_matlab,0,iF,
		       mxCreateDoubleScalar((double) hessFlag));
  }
  else	
    return(MATLAB_FAIL);
 
  /* put the structure into the Matlab workspace, then 
     iterate over provided analysis components, checking for error each time */
  engPutVariable(matlabEngine, "Dakota", dakota_matlab);
  buffer_char[BUFSIZE] = '\0';
  engOutputBuffer(matlabEngine, buffer_char, BUFSIZE);

  for (int aci=0; aci<analysisComponents[analysisDriverIndex].size(); aci++) {

    // strip away any .m the user might have included
    size_t pos = analysisComponents[analysisDriverIndex][aci].find(".");
    analysis_command = "Dakota = " + 
      analysisComponents[analysisDriverIndex][aci].substr(0,pos) + "(Dakota);";
    engEvalString(matlabEngine, analysis_command);
  }
  buffer_string = buffer_char;
  if (outputLevel > SILENT_OUTPUT && buffer_string.length() > 3)
    Cout << "Warning (Direct:Matlab): Matlab function output:\n" 
	 << buffer_char+3 << endl;  
  
  /* retrieve and parse the response */
  dakota_matlab = NULL;
  if ( (dakota_matlab = engGetVariable(matlabEngine, "Dakota")) == NULL ) {
    Cerr << "Error (Direct:Matlab): Failed to get variable Dakota from " 
         << "Matlab." << endl;
    fail_code = MATLAB_FAIL;
  } else if ( (mx_tmp[20] = mxGetField(dakota_matlab, 0, "failure")) != NULL &&
              mxGetClassID(mx_tmp[20]) == mxDOUBLE_CLASS && 
              *(mxGetPr(mx_tmp[20])) != 0 ) {

    /* Matlab user indicated failure, don't process */
    fail_code = (int) *(mxGetPr(mx_tmp[20]));

  } else {
    
    fail_code = 0;
    /* get fields by name in case the user somehow reordered, or decided to
       only return some of the fields to us... update all of
         fns:   1 x numFns
         grads: numFns * numDerivVars
         hess:  numFns * numDerivVars * numDerivVars
       if any of these come back wrong, set fail_code
    */    
     
    if (fn_flag)
      if ( (mx_tmp[16]=mxGetField(dakota_matlab, 0, "fnVals")) == NULL ) {
      
        Cerr << "Error (Direct:Matlab): Failed to get field fnVals from " 
             << "Dakota structure." << endl;
        fail_code = MATLAB_FAIL;
      
      } else if ( mxGetM(mx_tmp[16]) != 1 | mxGetN(mx_tmp[16]) != numFns ) {
        
        Cerr << "Error (Direct:Matlab): Dakota.fnVals must be [1 x numFns]." 
             << endl;    
        fail_code = MATLAB_FAIL;
      
      } else {
        
        ptr=mxGetPr(mx_tmp[16]);
        for (i=0; i<numFns; i++)
          fnVals[i] += *(ptr + i);
      }   
  
    if (gradFlag) 
      if ( (mx_tmp[17]=mxGetField(dakota_matlab, 0, "fnGrads")) == NULL ) {
      
        Cerr << "Error (Direct:Matlab): Failed to get field fnGrads from " 
             << "Dakota structure." << endl;
        fail_code = MATLAB_FAIL;     
      
      } else if ( mxGetM(mx_tmp[17]) != numFns |
		  mxGetN(mx_tmp[17]) != numDerivVars ) {
      
        Cerr << "Error (Direct:Matlab): Dakota.fnVals must be "
             << "[numFns x numDerivVars]." << endl;    
        fail_code = MATLAB_FAIL;
      
      } else  {
      
        ptr=mxGetPr(mx_tmp[17]);
        for (i=0; i<numFns; i++)
          for (j=0; j<numDerivVars; j++)
            fnGrads[i][j] += *(ptr + numFns*j + i);
      
      }
  
    if (hessFlag) 
      if ( (mx_tmp[18]=mxGetField(dakota_matlab, 0, "fnHessians")) == NULL ) {
        
        Cerr << "Error (Direct:Matlab): Failed to get field fnHessians from " 
             << "Dakota structure." << endl;
        fail_code = MATLAB_FAIL;
      
      } else if ( mxGetNumberOfDimensions(mx_tmp[18]) != 3 ) { 
        
        Cerr << "Error (Direct:Matlab): Dakota.fnHessians must be "
             << "3 dimensional." << endl;    
        fail_code = MATLAB_FAIL;
        
      } else {
        
        dim_ptr = mxGetDimensions(mx_tmp[18]);
        if ( dim_ptr[0] != numFns | dim_ptr[1] != numDerivVars | 
             dim_ptr[2] != numDerivVars ) {
          
          Cerr << "Error (Direct:Matlab): Dakota.fnHessians must be "
               << "[numFns x numDerivVars x numDerivVars]." << endl;     
          fail_code = MATLAB_FAIL;
          
        } else {
          
          ptr=mxGetPr(mx_tmp[18]);
          for (i=0; i<numFns; i++)
            for (j=0; j<numDerivVars; j++)
              for (k=0; k<numDerivVars; k++)
                fnHessians[i][j][k] += *(ptr + numDerivVars*numFns*k 
                                            + numFns*j + i );
        }
      
      }
    
    /* get fnLabels -- optional return value */
    if ( (mx_tmp[19]=mxGetField(dakota_matlab, 0, "fnLabels")) != NULL )
      if ( mxGetClassID(mx_tmp[19]) == mxCELL_CLASS &&
           mxGetNumberOfElements(mx_tmp[19]) == numFns )
        for (i=0; i<numFns; i++) {
        
          mx_string = mxGetCell(mx_tmp[19], i);  
          if ( mxGetClassID(mx_string) != mxCHAR_CLASS ) 
            Cout << "Warning (Direct:Matlab): Dakota.fnLabels{" << i
                 << "} is not a string.  Not returning fnLabel " << i 
                 << "." << endl;     
          else {
            
            mxGetString(mx_string, buffer_char, 256);
            fnLabels[i] = buffer_char;
        
          }
        }
   
  } /* end else parse response */

  return(fail_code);    

}

int DirectApplicInterface::matlab_field_prep(mxArray* dakota_matlab,
					       const char* field_name)
{
  int iF;
  if ( (iF=mxGetFieldNumber(dakota_matlab, field_name)) == -1 )
    if ( (iF=mxAddField(dakota_matlab, field_name)) == -1 ) {
      Cerr << "Error (Direct:Matlab): Cannot init Dakota." << field_name
	   << "."<< endl;
      return(iF);
    }
  mxFree(mxGetFieldByNumber(dakota_matlab,0,iF));
  return(iF);
}
#endif // DAKOTA_MATLAB

#ifdef DAKOTA_PYTHON
int DirectApplicInterface::python_run()
{
  // probably need to convert all of the following with SWIG or Boost!!
  // (there is minimal error checking for now)
  // need to cleanup ref counts on Python objects
  int fail_code = 0;

  // probably want to load the modules and functions at construction time, incl.
  // validation and store the objects for later, but need to resolve use of
  // analysisDriverIndex

  // for now we presume a single analysis component containing module:function
  const String& an_comp = analysisComponents[analysisDriverIndex][0].c_str();
  size_t pos = an_comp.find(":");
  String module_name = an_comp.substr(0,pos);
  String function_name = an_comp.substr(pos+1);

  // import the module and function and test for callable
  PyObject *pModule = PyImport_Import(PyString_FromString(module_name.c_str()));
  if (pModule != NULL) {

    char fn[256];
    strcpy(fn,function_name.c_str());
    PyObject *pFunc = PyObject_GetAttrString(pModule, fn);

    if (pFunc && PyCallable_Check(pFunc)) {

      // must use empty tuple here to pass to function taking only kwargs
      PyObject *pArgs = PyTuple_New(0);
      PyObject *pDict = PyDict_New();

      // convert DAKOTA data types to Python objects (lists and/or numpy arrays)
      PyObject *cdv, *cdv_labels, *ddv, *ddv_labels, *av, *av_labels, *asv,
	*dvv;
      python_convert(xC, &cdv);
      python_convert(xCLabels, &cdv_labels);
      python_convert_int(xD, &ddv);
      python_convert(xDLabels, &ddv_labels);
      python_convert(xC, xD, &av);
      python_convert(xCLabels, xDLabels, &av_labels);
      python_convert_int(directFnASV, &asv);
      python_convert_int(directFnDVV, &dvv);
      // analysis components -- TODO

      // assemble everything into a dictionary to pass to user function
      // this should eat references to the objects declared above
      PyDict_SetItem(pDict, PyString_FromString("variables"), 
		     PyInt_FromLong((long) numVars));
      PyDict_SetItem(pDict, PyString_FromString("functions"), 
		     PyInt_FromLong((long) numFns)); 
      PyDict_SetItem(pDict, PyString_FromString("cv"), cdv);
      PyDict_SetItem(pDict, PyString_FromString("cv_labels"), cdv_labels);
      PyDict_SetItem(pDict, PyString_FromString("dv"), ddv);
      PyDict_SetItem(pDict, PyString_FromString("dv_labels"), ddv_labels);
      PyDict_SetItem(pDict, PyString_FromString("av"), av);
      PyDict_SetItem(pDict, PyString_FromString("av_labels"), av_labels);
      PyDict_SetItem(pDict, PyString_FromString("asv"), asv);
      PyDict_SetItem(pDict, PyString_FromString("dvv"), dvv);

      // perform analysis
      cout << "Calling Python function " << function_name << " in module " 
	   << module_name << "." << endl;
      PyObject *retVal = PyObject_Call(pFunc, pArgs, pDict);
      Py_DECREF(pDict);
      Py_DECREF(pArgs);    
      Py_DECREF(pFunc);
      Py_DECREF(pModule);
      if (!retVal) {
	// TODO: better error reporting from Python
	Cerr << "Unknown error evaluating python function." << endl;
	abort_handler(-1);
      }
      
      bool fn_flag = false;
      for (int i=0; i<numFns; i++)
	if (directFnASV[i] & 1) {
	  fn_flag = true;
	  break;
	}

      // process return type as dictionary, else assume list of functions only
      if (PyDict_Check(retVal)) {
	// or the user may return a dictionary containing entires fns, fnGrads,
	// fnHessians, fnLabels, failure (int)
	// fnGrads, e.g. is a list of lists of doubles
	// this is where Boost or SWIG could really help
	// making a lot of assumptions on types being returned
	PyObject *obj;
	if (fn_flag) {
	  if ( !(obj = PyDict_GetItemString(retVal, "fns")) ) {
	    Cerr << "Python dictionary must contain list 'fns'" << endl;
	    Py_DECREF(retVal);
	    abort_handler(-1);
	  }
	  else
	    if (!python_convert(obj, fnVals, numFns)) {
	      Py_DECREF(retVal);
	      abort_handler(-1);
	    }
	}
	if (gradFlag) {
	  if ( !(obj = PyDict_GetItemString(retVal, "fnGrads")) ) {
	    Cerr << "Python dictionary must contain list 'fnGrads'" << endl;
	    Py_DECREF(retVal);
	    abort_handler(-1);
	  }
	  else
	    if (!python_convert(obj, fnGrads)) {
	      Py_DECREF(retVal);
	      abort_handler(-1);
	    }
	}
	if (hessFlag) {
	  if ( !(obj = PyDict_GetItemString(retVal, "fnHessians")) ) {
	    Cerr << "Python dictionary must contain list 'fnHessians'" << endl;
	    Py_DECREF(retVal);
	    abort_handler(-1);
	  }
	  else
	    if (!python_convert(obj, fnHessians)){
	      Py_DECREF(retVal);
	      abort_handler(-1);
	    }
	}
	// optional returns
	if (obj = PyDict_GetItemString(retVal, "failure"))
	  fail_code = PyInt_AsLong(obj);

	if (obj = PyDict_GetItemString(retVal, "fnLabels")) {
	  if (!PyList_Check(obj) || PyList_Size(obj) != numFns) {
	    Cerr << "'fnLabels' must be list of length numFns." << endl;
	    Py_DECREF(retVal);
	    abort_handler(-1);
	  }
	  else
	    for (int i=0; i<numFns; i++)
	      fnLabels[i] = PyString_AsString(PyList_GetItem(obj, i));
	}
      }
      else {
	// asssume list/numpy array containing only functions
	if (fn_flag)
	  python_convert(retVal, fnVals, numFns);
      }
      Py_DECREF(retVal);
    }
  }
  return(fail_code);
}


// convert all integer array types including Vector<int>, Array<int>,
// Array<short>, to Python list of ints or numpy array of ints
template<class ArrayT> 
bool DirectApplicInterface::
python_convert_int(const ArrayT & src, PyObject** dst)
{
  // don't rely on numACV, etc.
  int sz = src.size();
#ifdef DAKOTA_PYTHON_NUMPY
  if (userNumpyFlag) {
    int dims[1];
    dims[0] = sz;
    if (!(*dst = PyArray_FromDims(1, dims, PyArray_INT))) {
      Cerr << "Error creating Python numpy array." << endl;
      return(false);
    }
    PyArrayObject *pao = (PyArrayObject *) *dst;
    for (int i=0; i<sz; i++)
      *(int *)(pao->data + i*(pao->strides[0])) = (int) src[i];
  }
  else 
#endif
  {
    if (!(*dst = PyList_New(sz))) {
      Cerr << "Error creating Python list." << endl;
      return(false);
    }
    for (int i=0; i<sz; i++)
      PyList_SetItem(*dst, i, PyInt_FromLong((long) src[i]));
  }
  return(true);
}


// convert RealVector to list of floats or numpy array of doubles
bool DirectApplicInterface::
python_convert(const RealVector& src, PyObject** dst)
{
  int sz = src.size();
#ifdef DAKOTA_PYTHON_NUMPY
  if (userNumpyFlag) {
    int dims[1];
    dims[0] = sz;
    if (!(*dst = PyArray_FromDims(1, dims, PyArray_DOUBLE))) {
      Cerr << "Error creating Python numpy array." << endl;
      return(false);
    }
    PyArrayObject *pao = (PyArrayObject *) *dst;
    for (int i=0; i<sz; i++)
      *(double *)(pao->data + i*(pao->strides[0])) = src[i];
  }
  else
#endif
  {
    if (!(*dst = PyList_New(sz))) {
      Cerr << "Error creating Python list." << endl;
      return(false);
    }
    for (int i=0; i<sz; i++)
      PyList_SetItem(*dst, i, PyFloat_FromDouble(src[i]));
  }
  return(true);
}


// helper for converting xC and xD to single Python array of all variables
bool DirectApplicInterface::
python_convert(const RealVector& c_src, const IntVector& d_src, PyObject** dst)
{
  int c_sz = c_src.size();
  int d_sz = d_src.size();
#ifdef DAKOTA_PYTHON_NUMPY
  if (userNumpyFlag) {
    int dims[1];
    dims[0] = c_sz + d_sz;
    if (!(*dst = PyArray_FromDims(1, dims, PyArray_DOUBLE))) {
      Cerr << "Error creating Python numpy array." << endl;
      return(false);
    }
    PyArrayObject *pao = (PyArrayObject *) *dst;
    for (int i=0; i<c_sz; i++)
      *(double *)(pao->data + i*(pao->strides[0])) = c_src[i];
    for (int i=0; i<d_sz; i++)
      *(double *)(pao->data + (c_sz+i)*(pao->strides[0])) = (double) d_src[i];
  }
  else
#endif
  {
    if (!(*dst = PyList_New(c_sz + d_sz))) {
      Cerr << "Error creating Python list." << endl;
      return(false);
    }
    for (int i=0; i<c_sz; i++)
      PyList_SetItem(*dst, i, PyFloat_FromDouble(c_src[i]));
    for (int i=0; i<d_sz; i++)
      PyList_SetItem(*dst, c_sz + i, PyInt_FromLong((long) d_src[i]));
  }
  return(true);
}


// convert StringArray to list of strings
bool DirectApplicInterface::
python_convert(const StringArray& src, PyObject** dst)
{
  int sz = src.size();
  if (!(*dst = PyList_New(sz))) {
      Cerr << "Error creating Python list." << endl;
      return(false);
  }
  for (int i=0; i<sz; i++)
    PyList_SetItem(*dst, i, PyString_FromString(src[i]));

  return(true);
}


// convert continuous and discrete label strings to single list
bool DirectApplicInterface::
python_convert(const StringArray& c_src, const StringArray& d_src,
	       PyObject** dst)
{
  int c_sz = c_src.size();
  int d_sz = d_src.size();
  if (!(*dst = PyList_New(c_sz + d_sz))) {
    Cerr << "Error creating Python list." << endl;
    return(false);
  }
  for (int i=0; i<c_sz; i++)
    PyList_SetItem(*dst, i, PyString_FromString(c_src[i]));
  for (int i=0; i<d_sz; i++)
    PyList_SetItem(*dst, c_sz+i, PyString_FromString(d_src[i]));

  return(true);
}


// takes python list, dakota vector, expected dimension
// returns false if conversion failed
// (using RealBaseVector to support passing a row of a matrix)
bool DirectApplicInterface::
python_convert(PyObject *pyv, RealBaseVector &rv, const int& dim)
{
#ifdef DAKOTA_PYTHON_NUMPY
  // could automatically detect return type instead of throwing error
  if (userNumpyFlag) {
    if (!PyArray_Check(pyv) || PyArray_NDIM(pyv) != 1 || 
	PyArray_DIM(pyv,0) != dim) {
      Cerr << "Python numpy array not 1D of size " << dim << "." << endl;
      return(false);
    }
    else {
      PyArrayObject *pao = (PyArrayObject *) pyv;
      for (int i=0; i<dim; i++)
	rv[i] = *(double *)(pao->data + i*(pao->strides[0]));
    }
    return(true);
  }
  else
#endif
  {
    PyObject *val;
    if (!PyList_Check(pyv) || PyList_Size(pyv) != dim) {
      Cerr << "Python vector must have length " << dim << "." << endl;
      return(false);
    }
    else
      for (int i=0; i<dim; i++) {
	val = PyList_GetItem(pyv, i);
	if (PyFloat_Check(val))
	  rv[i] = PyFloat_AsDouble(val);
	else if (PyInt_Check(val))
	  rv[i] = (double) PyInt_AsLong(val);
	else {
	  Cerr << "Unsupported Python data type converting vector." << endl;
	  Py_DECREF(val);
	  return(false);
	}
      }
  }
  return(true);
}


// assume numDerivVars x numDerivVars
// returns false if conversion failed
bool DirectApplicInterface::python_convert(PyObject *pym, RealMatrix &rm)
{
#ifdef DAKOTA_PYTHON_NUMPY
  if (userNumpyFlag) {
    int M = rm.num_rows(); 
    int N = rm.num_columns(); 
    if (!PyArray_Check(pym) || PyArray_NDIM(pym) != 2 || 
	PyArray_DIM(pym,0) != M  ||  PyArray_DIM(pym,1) != N) {
      Cerr << "Python numpy array not 2D of size " << M << "x" << N << "." 
	   << endl;
      return(false);
    }
    else {
      PyArrayObject *pao = (PyArrayObject *) pym;
      for (int i=0; i<M; i++)
	for (int j=0; j<N; j++)
	  rm[i][j] = *(double *)(pao->data + i*(pao->strides[0]) + 
				 j*(pao->strides[1]));
    }
  }
  else
#endif
  {
    PyObject *val;
    if (!PyList_Check(pym) || PyList_Size(pym) != numFns) {
      Cerr << "Python matrix must have " << numFns << "rows." << endl;
      return(false);
    }
    else
      for (int i=0; i<numFns; i++) {
	val = PyList_GetItem(pym, i);
	if (PyList_Check(val)) {
	  if (!python_convert(val, rm[i], numDerivVars))
	    return(false);
	}
	else {
	  Cerr << "Each row of Python matrix must be a list." << endl;
	  Py_DECREF(val);
	  return(false);
	}
      }
  }
  return(true);
}


// assume numFns x numDerivVars x numDerivVars
// returns false if conversion failed
bool DirectApplicInterface::
python_convert(PyObject *pyma, RealMatrixArray &rma)
{
#ifdef DAKOTA_PYTHON_NUMPY
  if (userNumpyFlag) {
  // instead of recursing, perform here for clarity
    int M = rma.size();
    int N = rma[0].num_rows(); 
    int P = rma[0].num_columns(); 
    if (!PyArray_Check(pyma) || PyArray_NDIM(pyma) != 3 || 
	PyArray_DIM(pyma,0) != M || PyArray_DIM(pyma,1) != N ||
	PyArray_DIM(pyma,2) != P ) {
      Cerr << "Python numpy array not 3D of size " << M << "x" << N << "x"
	   << P << "." << endl;
      return(false);
    }
    else {
      PyArrayObject *pao = (PyArrayObject *) pyma;
      for (int i=0; i<M; i++)
	for (int j=0; j<N; j++)
	  for (int k=0; k<P; j++)
	    rma[i][j][k] = *(double *)(pao->data + i*(pao->strides[0]) + 
				       j*(pao->strides[1]) +
				       k*(pao->strides[2]));
    }
  }
  else
#endif
  {
    PyObject *val;
    if (!PyList_Check(pyma) || PyList_Size(pyma) != numFns) {
      Cerr << "Python matrix array must have " << numFns << "rows." << endl;
      return(false);
    }
    else
      for (int i=0; i<numFns; i++) {
	val = PyList_GetItem(pyma, i);
	if (PyList_Check(val)) {
	  if (!python_convert(val, rma[i]))
	    return(false);
	}
	else {
	  Cerr << "Each row of Python matrix must be a list." << endl;
	  Py_DECREF(val);
	  return(false);
	}
      }
  }
  return(true);
}
#endif // DAKOTA_PYTHON

} // namespace Dakota
