/*  _______________________________________________________________________

    DAKOTA: Design Analysis Kit for Optimization and Terascale Applications
    Copyright (c) 2006, Sandia National Laboratories.
    This software is distributed under the GNU General Public License.
    For more information, see the README file in the top Dakota directory.
    _______________________________________________________________________ */

//- Class:        Variables
//- Description:  Class implementation
//- Owner:        Mike Eldred

#include "ProblemDescDB.H"
#include "DakotaVariables.H"
#include "AllVariables.H"
#include "MergedVariables.H"
#include "DistinctVariables.H"
#include "DakotaBinStream.H"

//#define REFCOUNT_DEBUG

static const char rcsId[]="@(#) $Id";

namespace Dakota {


/** This constructor is the one which must build the base class data for all
    derived classes.  get_variables() instantiates a derived class letter
    and the derived constructor selects this base class constructor in its 
    initialization list (to avoid the recursion of the base class constructor
    calling get_variables() again).  Since the letter IS the representation, 
    its representation pointer is set to NULL (an uninitialized pointer causes
    problems in ~Variables). */
Variables::
Variables(BaseConstructor, const ProblemDescDB& problem_db,
	  const pair<short,short>& view):
  variablesView(view), idVariables(problem_db.get_string("variables.id")),
  variablesRep(NULL), referenceCount(1)
{
  variablesComponents.reshape(3);
  // design
  variablesComponents[0].reshape(3);
  variablesComponents[0][1]
    = problem_db.get_sizet("variables.continuous_design");
  variablesComponents[0][2] = problem_db.get_sizet("variables.discrete_design");
  // uncertain
  variablesComponents[1].reshape(14);
  variablesComponents[1][1]
    = problem_db.get_sizet("variables.normal_uncertain");
  variablesComponents[1][2]
    = problem_db.get_sizet("variables.lognormal_uncertain");
  variablesComponents[1][3]
    = problem_db.get_sizet("variables.uniform_uncertain");
  variablesComponents[1][4]
    = problem_db.get_sizet("variables.loguniform_uncertain");
  variablesComponents[1][5]
    = problem_db.get_sizet("variables.triangular_uncertain");
  variablesComponents[1][6]
    = problem_db.get_sizet("variables.exponential_uncertain");
  variablesComponents[1][7] = problem_db.get_sizet("variables.beta_uncertain");
  variablesComponents[1][8] = problem_db.get_sizet("variables.gamma_uncertain");
  variablesComponents[1][9]
    = problem_db.get_sizet("variables.gumbel_uncertain");
  variablesComponents[1][10]
    = problem_db.get_sizet("variables.frechet_uncertain");
  variablesComponents[1][11]
    = problem_db.get_sizet("variables.weibull_uncertain");
  variablesComponents[1][12]
    = problem_db.get_sizet("variables.histogram_uncertain");
  variablesComponents[1][13]
    = problem_db.get_sizet("variables.interval_uncertain");
  // state
  variablesComponents[2].reshape(3);
  variablesComponents[2][1]
    = problem_db.get_sizet("variables.continuous_state");
  variablesComponents[2][2] = problem_db.get_sizet("variables.discrete_state");
  // sum total number of design, uncertain, state and store as first entry
  for (size_t i=0; i<3; i++) {
    size_t sum = 0, len = variablesComponents[i].length();
    for (size_t j=1; j<len; j++)
      sum += variablesComponents[i][j];
    variablesComponents[i][0] = sum;
  }

#ifdef REFCOUNT_DEBUG
  Cout << "Variables::Variables(BaseConstructor) called to build base class "
       << "data for letter object." << endl;
#endif
}


/** The default constructor: variablesRep is NULL in this case (a populated
    problem_db is needed to build a meaningful Variables object).  This
    makes it necessary to check for NULL in the copy constructor, assignment
    operator, and destructor. */
Variables::Variables(): variablesRep(NULL), referenceCount(1)
{
#ifdef REFCOUNT_DEBUG
  Cout << "Variables::Variables() called to build empty variables object."
       << endl;
#endif
}


/** This is the primary envelope constructor which uses problem_db to
    build a fully populated variables object.  It only needs to
    extract enough data to properly execute get_variables(problem_db),
    since the constructor overloaded with BaseConstructor builds the
    actual base class data inherited by the derived classes. */
Variables::Variables(const ProblemDescDB& problem_db):
  referenceCount(1) // not used since this is the envelope, not the letter
{
#ifdef REFCOUNT_DEBUG
  Cout << "Variables::Variables(ProblemDescDB&) called to instantiate envelope."
       << endl;
#endif

  // Set the rep pointer to the appropriate derived variables class
  variablesRep = get_variables(problem_db);
  if ( !variablesRep ) // bad type or insufficient memory
    abort_handler(-1);
}


/** Initializes variablesRep to the appropriate derived type, as given
    by problem_db attributes.  The standard derived class constructors
    are invoked.  */
Variables* Variables::get_variables(const ProblemDescDB& problem_db)
{
#ifdef REFCOUNT_DEBUG
  Cout << "Envelope instantiating letter in get_variables(ProblemDescDB&)." 
       << endl;
#endif

  pair<short,short> view = get_view(problem_db);

  // This get_variables version invokes the standard constructor.
  int active_view = view.first;
  if (active_view == MERGED_DISTINCT_DESIGN ||
      active_view == MERGED_DISTINCT_UNCERTAIN ||
      active_view == MERGED_DISTINCT_STATE)
    return new MergedVariables(problem_db, view);
  else if (active_view == MIXED_ALL)
    return new AllVariables(problem_db, view);
  // The AllMerged case has been deleted since this combination is unlikely
  // (discrete variable relaxation is only envisioned for optimization).
  //else if (active_view == MERGED_ALL)
  //  return new AllMergedVariables(problem_db, view);
  else if (active_view == MIXED_DISTINCT_DESIGN ||
	   active_view == MIXED_DISTINCT_UNCERTAIN ||
	   active_view == MIXED_DISTINCT_STATE)
    return new DistinctVariables(problem_db, view);
  else {
    Cerr << "Variables active view " << active_view
	 << " not currently supported in derived Variables classes." << endl;
    return NULL;
  }
}


/** This is the alternate envelope constructor for instantiations on
    the fly.  Since it does not have access to problem_db, the letter
    class is not fully populated.  This constructor executes
    get_variables(view), which invokes the default constructor of the
    derived letter class, which in turn invokes the default
    constructor of the base class. */
Variables::Variables(const pair<short,short>& view):
  referenceCount(1) // not used since this is the envelope, not the letter
{
#ifdef REFCOUNT_DEBUG
  Cout << "Variables::Variables(pair<short,short>&) called to instantiate "
       << "envelope." << endl;
#endif

  // Set the rep pointer to the appropriate derived variables class
  variablesRep = get_variables(view);
  if ( !variablesRep ) // bad type or insufficient memory
    abort_handler(-1);
}


/** Initializes variablesRep to the appropriate derived type, as given
    by view.  The default derived class constructors are invoked. */
Variables* Variables::get_variables(const pair<short,short>& view) const
{
#ifdef REFCOUNT_DEBUG
  Cout << "Envelope instantiating letter in get_variables(pair<short,short>&)."
       << endl;
#endif

  // This get_variables version invokes the default constructor.  Virtual
  // copy_rep() function must then be called to copy derived class attributes.
  int active_view = view.first;
  Variables* vars;
  if (active_view == MERGED_DISTINCT_DESIGN ||
      active_view == MERGED_DISTINCT_UNCERTAIN ||
      active_view == MERGED_DISTINCT_STATE)
    vars = new MergedVariables();
  else if (active_view == MIXED_ALL)
    vars = new AllVariables();
  // The AllMerged case has been deleted since this combination is unlikely
  // (discrete variable relaxation is only envisioned for optimization; perhaps
  // global surrogate-based branch and bound would require it).
  //else if (active_view == MERGED_ALL)
  //  vars = new AllMergedVariables(problem_db);
  else if (active_view == MIXED_DISTINCT_DESIGN ||
	   active_view == MIXED_DISTINCT_UNCERTAIN ||
	   active_view == MIXED_DISTINCT_STATE)
    vars = new DistinctVariables();
  else {
    Cerr << "Variables view " << active_view
	 << " not currently supported in derived Variables classes." << endl;
    vars = NULL;
  }
  if (vars)
    vars->variablesView = view;
  return vars;
}


pair<short,short> Variables::get_view(const ProblemDescDB& problem_db) const
{
  // Default variables behavior uses the distinct arrays for continuous
  // and discrete vars/bnds (used by most iterators as well as mixed variable 
  // strategies with separate continuous and discrete iterators).  In some 
  // cases (e.g., branch & bound), a merged data approach is used in which 
  // continuous and discrete arrays are combined into a single continuous array
  // (integrality is relaxed; the converse of truncating or discretizing reals
  // could also be used in the future if needed).  And in other cases (e.g., 
  // parameter studies and dace), no distinction is drawn between design,
  // uncertain, and state variable types and all types are collected into 
  // allVars arrays.

  // Since this is used by the envelope, don't set variablesView
  pair<short,short> view; // view.first = active, view.second = inactive

  // Possibilities:  distinct types vs. all types combined
  //                 mixed cont./discrete vs. merged cont./discrete
  // If distinct, then active/inactive could be design, uncertain, or state.
  const String& strat_name  = problem_db.get_string("strategy.type");
  const String& method_name = problem_db.get_string("method.algorithm");
  if (strat_name == "branch_and_bound") {
    view.first  = MERGED_DISTINCT_DESIGN;    // active
    view.second = MERGED_DISTINCT_UNCERTAIN; // inactive
  }
  else if ( method_name.ends("_parameter_study") || method_name == "dace" ||
	    method_name.begins("fsu_") || method_name.begins("psuade_") || 
	    ( method_name.begins("nond_") &&
	      problem_db.get_bool("method.nond.all_variables") ) ) {
    view.first  = MIXED_ALL;                 // active
    view.second = EMPTY;                     // inactive
  }
  else if (method_name.begins("nond_")) {
    view.first  = MIXED_DISTINCT_UNCERTAIN;  // active
    view.second = MIXED_DISTINCT_DESIGN;     // inactive
  }
  else {
    view.first  = MIXED_DISTINCT_DESIGN;     // active
    view.second = MIXED_DISTINCT_UNCERTAIN;  // inactive
  }
  return view;
}


/** Copy constructor manages sharing of variablesRep and incrementing
    of referenceCount. */
Variables::Variables(const Variables& vars)
{
  // Increment new (no old to decrement)
  variablesRep = vars.variablesRep;
  if (variablesRep) // Check for an assignment of NULL
    variablesRep->referenceCount++;

#ifdef REFCOUNT_DEBUG
  Cout << "Variables::Variables(Variables&)" << endl;
  if (variablesRep)
    Cout << "variablesRep referenceCount = " << variablesRep->referenceCount
	 << endl;
#endif
}


/** Assignment operator decrements referenceCount for old variablesRep, assigns
    new variablesRep, and increments referenceCount for new variablesRep. */
Variables Variables::operator=(const Variables& vars)
{
  if (variablesRep != vars.variablesRep) { // normal case: old != new
    // Decrement old
    if (variablesRep) // Check for NULL
      if ( --variablesRep->referenceCount == 0 ) 
	delete variablesRep;
    // Assign new
    variablesRep = vars.variablesRep;
  }
  // Increment new (either case: old == new or old != new)
  if (variablesRep) // Check for NULL
    variablesRep->referenceCount++;

#ifdef REFCOUNT_DEBUG
  Cout << "Variables::operator=(Variables&)" << endl;
  if (variablesRep)
    Cout << "variablesRep referenceCount = " << variablesRep->referenceCount
	 << endl;
#endif

  return *this; // calls copy constructor since returned by value
}


/** Destructor decrements referenceCount and only deletes variablesRep
    when referenceCount reaches zero. */
Variables::~Variables()
{ 
  // Check for NULL pointer 
  if (variablesRep) {
    --variablesRep->referenceCount;
#ifdef REFCOUNT_DEBUG
    Cout << "variablesRep referenceCount decremented to " 
         << variablesRep->referenceCount << endl;
#endif
    if (variablesRep->referenceCount == 0) {
#ifdef REFCOUNT_DEBUG
      Cout << "deleting variablesRep" << endl;
#endif
      delete variablesRep;
    }
  }
}


void Variables::uncertain_var_types(size_t& acv_cntr)
{
  size_t i, num_nuv = variablesComponents[1][1],
    num_lnuv = variablesComponents[1][2],  num_uuv = variablesComponents[1][3],
    num_luuv = variablesComponents[1][4],  num_tuv = variablesComponents[1][5],
    num_euv  = variablesComponents[1][6],  num_buv = variablesComponents[1][7],
    num_gauv = variablesComponents[1][8], num_guuv = variablesComponents[1][9],
    num_fuv  = variablesComponents[1][10], num_wuv = variablesComponents[1][11],
    num_huv  = variablesComponents[1][12], num_iuv = variablesComponents[1][13];

  for (i=0; i<num_nuv; i++)
    allContinuousVarTypes[acv_cntr++] = "normal_uncertain";
  for (i=0; i<num_lnuv; i++)
    allContinuousVarTypes[acv_cntr++] = "lognormal_uncertain";
  for (i=0; i<num_uuv; i++)
    allContinuousVarTypes[acv_cntr++] = "uniform_uncertain";
  for (i=0; i<num_luuv; i++)
    allContinuousVarTypes[acv_cntr++] = "loguniform_uncertain";
  for (i=0; i<num_tuv; i++)
    allContinuousVarTypes[acv_cntr++] = "triangular_uncertain";
  for (i=0; i<num_euv; i++)
    allContinuousVarTypes[acv_cntr++] = "exponential_uncertain";
  for (i=0; i<num_buv; i++)
    allContinuousVarTypes[acv_cntr++] = "beta_uncertain";
  for (i=0; i<num_gauv; i++)
    allContinuousVarTypes[acv_cntr++] = "gamma_uncertain";
  for (i=0; i<num_guuv; i++)
    allContinuousVarTypes[acv_cntr++] = "gumbel_uncertain";
  for (i=0; i<num_fuv; i++)
    allContinuousVarTypes[acv_cntr++] = "frechet_uncertain";
  for (i=0; i<num_wuv; i++)
    allContinuousVarTypes[acv_cntr++] = "weibull_uncertain";
  for (i=0; i<num_huv; i++)
    allContinuousVarTypes[acv_cntr++] = "histogram_uncertain";
  for (i=0; i<num_iuv; i++)
    allContinuousVarTypes[acv_cntr++] = "interval_uncertain";
}


void Variables::read(istream& s)
{
  if (variablesRep)
    variablesRep->read(s); // envelope fwd to letter
  else { // letter lacking redefinition of virtual fn.!
    Cerr << "Error: Letter lacking redefinition of virtual read function.\n"
         << "No default defined at base class." << endl;
    abort_handler(-1);
  }
}


void Variables::write(ostream& s) const
{
  if (variablesRep)
    variablesRep->write(s); // envelope fwd to letter
  else { // letter lacking redefinition of virtual fn.!
    Cerr << "Error: Letter lacking redefinition of virtual write function.\n"
         << "No default defined at base class." << endl;
    abort_handler(-1);
  }
}


void Variables::write_aprepro(ostream& s) const
{
  if (variablesRep)
    variablesRep->write_aprepro(s); // envelope fwd to letter
  else { // letter lacking redefinition of virtual fn.!
    Cerr << "Error: Letter lacking redefinition of virtual write_aprepro "
         << "function.\nNo default defined at base class." << endl;
    abort_handler(-1);
  }
}


void Variables::read_annotated(istream& s)
{
  pair<short,short> view;
  s >> view.first;
  if (s.eof()) // exception handling since EOF may not be captured properly
    throw String("Empty record in Variables::read_annotated(istream&)");
  s >> view.second;

  if (variablesRep) { // should not occur in current usage
    if (variablesRep->variablesView == view)
      variablesRep->read_annotated(s); // envelope fwd to existing letter
    else { // decrement old reference count and replace with new letter
      Cerr << "Warning: variables type mismatch in Variables::read(BiStream&)."
	   << endl;
      if (--variablesRep->referenceCount == 0) 
        delete variablesRep;
      variablesRep = get_variables(view);
      variablesRep->read_annotated(s);
    }
  }
  else { // read from neutral file: variablesRep must be instantiated
    variablesRep = get_variables(view);
    variablesRep->read_annotated(s); // envelope fwd to new letter
  }
}


void Variables::write_annotated(ostream& s) const
{
  if (variablesRep) {
    s << variablesRep->variablesView.first  << ' '
      << variablesRep->variablesView.second << ' ';
    variablesRep->write_annotated(s); // envelope fwd to letter
    s << '\n';
  }
  else { // letter lacking redefinition of virtual fn.!
    Cerr << "Error: Letter lacking redefinition of virtual write_annotated "
         << "function.\nNo default defined at base class." << endl;
    abort_handler(-1);
  }
}


void Variables::write_tabular(ostream& s) const
{
  if (variablesRep)
    variablesRep->write_tabular(s); // envelope fwd to letter
  else { // letter lacking redefinition of virtual fn.!
    Cerr << "Error: Letter lacking redefinition of virtual write_tabular "
         << "function.\nNo default defined at base class." << endl;
    abort_handler(-1);
  }
}


void Variables::read(BiStream& s)
{
  pair<short,short> view;
  s >> view.first;
  if (s.eof()) // exception handling since EOF not captured properly by BiStream
    throw String("Empty record in Variables::read(BiStream&)");
  s >> view.second;

  if (variablesRep) { // should not occur in current usage
    if (variablesRep->variablesView == view)
      variablesRep->read(s); // envelope fwd to existing letter
    else { // decrement old reference count and replace with new letter
      Cerr << "Warning: variables type mismatch in Variables::read(BiStream&)."
	   << endl;
      if (--variablesRep->referenceCount == 0) 
        delete variablesRep;
      variablesRep = get_variables(view);
      variablesRep->read(s);
    }
  }
  else { // read from restart: variablesRep must be instantiated
    variablesRep = get_variables(view);
    variablesRep->read(s); // envelope fwd to new letter
  }
}


void Variables::write(BoStream& s) const
{
  if (variablesRep) {
    s << variablesRep->variablesView.first <<variablesRep->variablesView.second;
    variablesRep->write(s); // envelope fwd to letter
  }
  else { // letter lacking redefinition of virtual fn.!
    Cerr << "Error: Letter lacking redefinition of virtual write function.\n"
         << "No default defined at base class." << endl;
    abort_handler(-1);
  }
}


void Variables::read(MPIUnpackBuffer& s)
{
  bool letter;
  s >> letter;
  if (letter) {
    pair<short,short> view;
    s >> view.first >> view.second;
    if (variablesRep) { // should not occur in current usage
      if (variablesRep->variablesView == view)
	variablesRep->read(s); // envelope fwd to existing letter
      else { // decrement old reference count and replace with new letter
	Cerr << "Warning: variables type mismatch in "
	     << "Variables::read(MPIUnpackBuffer&)." << endl;
	if (--variablesRep->referenceCount == 0) 
	  delete variablesRep;
	variablesRep = get_variables(view);
	variablesRep->read(s);
      }
    }
    else { // buffer read on slaves: variablesRep must be instantiated
      variablesRep = get_variables(view);
      variablesRep->read(s); // envelope fwd to new letter
    }
  }
  else if (variablesRep) {
    if (--variablesRep->referenceCount == 0)
      delete variablesRep;
    variablesRep == NULL;
  }
}


void Variables::write(MPIPackBuffer& s) const
{
  s << !is_null();
  if (variablesRep) {
    s << variablesRep->variablesView.first
      << variablesRep->variablesView.second;
    // VarTypes/VarIds not required
    variablesRep->write(s); // envelope fwd to letter
  }
}


/** Deep copies are used for history mechanisms such as bestVariables
    and data_pairs since these must catalogue copies (and should not
    change as the representation within currentVariables changes). */
Variables Variables::copy() const
{
  // the envelope class instantiates a new envelope and a new letter and copies
  // current attributes into the new objects.

#ifdef REFCOUNT_DEBUG
  Cout << "Variables::copy() called to generate a deep copy with no "
       << "representation sharing." << endl;
#endif

  Variables vars; // new envelope: referenceCount=1, variablesRep=NULL

  if (variablesRep) {
    // new letter: allocate a variablesRep and copy derived attributes
    vars.variablesRep = get_variables(variablesRep->variablesView);
    vars.copy_rep(variablesRep); // or vice-versa as convenient

    // Copy attributes used by all letters (so code not replicated in copy_rep)
    vars.variablesRep->variablesView       = variablesRep->variablesView;
    vars.variablesRep->variablesComponents = variablesRep->variablesComponents;
    vars.variablesRep->allContinuousVarTypes
      =  variablesRep->allContinuousVarTypes;
    vars.variablesRep->allContinuousVarIds = variablesRep->allContinuousVarIds;
  }

  return vars;
}


void Variables::copy_rep(const Variables* vars_rep)
{
  if (variablesRep)
    variablesRep->copy_rep(vars_rep);
  else { // letter lacking redefinition of virtual fn.!
    Cerr << "Error: Letter lacking redefinition of virtual copy_rep function.\n"
         << "       No default defined at base class." << endl;
    abort_handler(-1);
  }
}


void Variables::reshape(const Sizet2DArray& vars_comps)
{
  if (variablesRep) { // envelope
    variablesRep->variablesComponents = vars_comps;
    variablesRep->reshape_rep(vars_comps);
  }
  else { // letter
    variablesComponents = vars_comps;
    reshape_rep(vars_comps);
  }
}


void Variables::reshape_rep(const Sizet2DArray& vars_comps)
{
  if (variablesRep)
    variablesRep->reshape_rep(vars_comps);
  else { // letter lacking redefinition of virtual fn.!
    Cerr << "Error: Letter lacking redefinition of virtual reshape_rep "
	 << "function.\n       No default defined at base class." << endl;
    abort_handler(-1);
  }
}


size_t Variables::tv() const
{
  if (!variablesRep) { // letter lacking redefinition of virtual fn.!
    Cerr << "Error: Letter lacking redefinition of virtual tv() function.\n"
         << "       No default defined at base class." << endl;
    abort_handler(-1);
  }
 
  // envelope fwd to letter
  return variablesRep->tv();
}


const UIntArray& Variables::merged_discrete_ids() const
{
  if (!variablesRep) { // letter lacking redefinition of virtual fn.!
    Cerr << "Error: Letter lacking redefinition of virtual merged_discrete_ids"
	 << "() function.\n       No default defined at base class." << endl;
    abort_handler(-1);
  }
 
  // envelope fwd to letter
  return variablesRep->merged_discrete_ids();
}


const RealVector& Variables::continuous_variables() const
{
  if (!variablesRep) { // letter lacking redefinition of virtual fn.!
    Cerr << "Error: Letter lacking redefinition of virtual continuous_variables"
	 << "() function.\n       No default defined at base class." << endl;
    abort_handler(-1);
  }

  return variablesRep->continuous_variables(); // envelope fwd to letter
}


void Variables::continuous_variable(const Real& c_var, const size_t& i)
{
  if (variablesRep)
    variablesRep->continuous_variable(c_var, i); // envelope fwd to letter
  else { // letter lacking redefinition of virtual fn.
    Cerr << "Error: Letter lacking redefinition of virtual continuous_variable("
	 << "Real) function.\n       No default defined at base class." << endl;
    abort_handler(-1);
  }
}


void Variables::continuous_variables(const RealVector& c_vars)
{
  if (variablesRep)
    variablesRep->continuous_variables(c_vars); // envelope fwd to letter
  else { // letter lacking redefinition of virtual fn.
    Cerr << "Error: Letter lacking redefinition of virtual continuous_variables"
	 << "(RealVector) function.\n       No default defined at base class."
	 << endl;
    abort_handler(-1);
  }
}


const IntVector& Variables::discrete_variables() const
{
  if (variablesRep)
    return variablesRep->discrete_variables(); // envelope fwd to letter
  else // default definition for letters lacking redefinition (merged types)
    return emptyIntVector;
}


void Variables::discrete_variable(const int& d_var, const size_t& i)
{
  if (variablesRep)
    variablesRep->discrete_variable(d_var, i); // envelope fwd to letter
  else { // letter lacking redefinition of virtual fn.
    Cerr << "Error: Letter lacking redefinition of virtual discrete_variable("
	 << "int) function.\n       No default defined at base class." << endl;
    abort_handler(-1);
  }
}


void Variables::discrete_variables(const IntVector& d_vars)
{
  if (variablesRep)
    variablesRep->discrete_variables(d_vars); // envelope fwd to letter
  else if (!d_vars.empty()) {
    // default definition for letters lacking redefinition (merged types)
    Cerr << "Error: There are no active discrete variables in this variables "
	 << "view.\n       Setting this array is erroneous." << endl;
    abort_handler(-1);
  }
}


const StringArray& Variables::continuous_variable_labels() const
{
  if (!variablesRep) { // letter lacking redefinition of virtual fn.!
    Cerr << "Error: Letter lacking redefinition of virtual continuous_variable_"
	 << "labels() function.\n       No default defined at base class."
	 << endl;
    abort_handler(-1);
  }

  return variablesRep->continuous_variable_labels(); // envelope fwd to letter
}


void Variables::continuous_variable_labels(const StringArray& cv_labels)
{
  if (variablesRep)
    variablesRep->continuous_variable_labels(cv_labels); //envelope fwd->letter
  else { // letter lacking redefinition of virtual fn.!
    Cerr << "Error: Letter lacking redefinition of virtual continuous_variable_"
         << "labels(StringArray) function.\n       "
	 << "No default defined at base class." << endl;
    abort_handler(-1);
  }
}


const StringArray& Variables::discrete_variable_labels() const
{
  if (variablesRep)
    return variablesRep->discrete_variable_labels(); // envelope fwd to letter
  else // default definition for letters lacking redefinition (merged types)
    return emptyStringArray;
}


void Variables::discrete_variable_labels(const StringArray& dv_labels)
{
  if (variablesRep)
    variablesRep->discrete_variable_labels(dv_labels); //envelope fwd to letter
  else if (!dv_labels.empty()) {
    // default definition for letters lacking redefinition (merged types)
    Cerr << "Error: There are no active discrete variable labels in this "
	 << "variables view.\n       Setting this array is erroneous." << endl;
    abort_handler(-1);
  }
}


const StringArray& Variables::continuous_variable_types() const
{
  if (!variablesRep) { // letter lacking redefinition of virtual fn.!
    Cerr << "Error: Letter lacking redefinition of virtual continuous_variable_"
	 << "types() function.\n       No default defined at base class."<<endl;
    abort_handler(-1);
  }

  return variablesRep->continuous_variable_types(); // envelope fwd to letter
}


const StringArray& Variables::discrete_variable_types() const
{
  if (variablesRep)
    return variablesRep->discrete_variable_types(); // envelope fwd to letter
  else // default definition for letters lacking redefinition (merged types)
    return emptyStringArray;
}


const UIntArray& Variables::continuous_variable_ids() const
{
  if (!variablesRep) { // letter lacking redefinition of virtual fn.!
    Cerr << "Error: Letter lacking redefinition of virtual continuous_variable_"
	 << "ids() function.\n       No default defined at base class." << endl;
    abort_handler(-1);
  }

  return variablesRep->continuous_variable_ids(); // envelope fwd to letter
}


const RealVector& Variables::inactive_continuous_variables() const
{
  if (variablesRep)
    return variablesRep->inactive_continuous_variables(); // envelope->letter
  else // default definition for letters lacking redefinition of virtual fn.
    return emptyRealVector;
}


void Variables::inactive_continuous_variables(const RealVector& i_c_vars)
{
  if (variablesRep)
    variablesRep->inactive_continuous_variables(i_c_vars); // envelope->letter
  else if (!i_c_vars.empty()) {
    // default definition for letters lacking redefinition of virtual fn.
    Cerr << "Error: There are no inactive continuous variables in this "
	 << "variables view.\n       Setting this array is erroneous." << endl;
    abort_handler(-1);
  }
}


const IntVector& Variables::inactive_discrete_variables() const
{
  if (variablesRep)
    return variablesRep->inactive_discrete_variables(); // envelope->letter
  else // default definition for letters lacking redefinition of virtual fn.
    return emptyIntVector;
}


void Variables::inactive_discrete_variables(const IntVector& i_d_vars)
{
  if (variablesRep)
    variablesRep->inactive_discrete_variables(i_d_vars); // envelope->letter
  else if (!i_d_vars.empty()) {
    // default definition for letters lacking redefinition of virtual fn.
    Cerr << "Error: There are no inactive discrete variables in this "
	 << "variables view.\n       Setting this array is erroneous." << endl;
    abort_handler(-1);
  }
}


const StringArray& Variables::inactive_continuous_variable_labels() const
{
  if (variablesRep) // envelope->letter
    return variablesRep->inactive_continuous_variable_labels();
  else // default definition for letters lacking redefinition of virtual fn.
    return emptyStringArray;
}


void Variables::
inactive_continuous_variable_labels(const StringArray& i_c_v_labels)
{
  if (variablesRep) // envelope->letter
    variablesRep->inactive_continuous_variable_labels(i_c_v_labels);
  else if (!i_c_v_labels.empty()) {
    // default definition for letters lacking redefinition of virtual fn.
    Cerr << "Error: There are no inactive continuous variable labels in this "
	 << "variables view.\n       Setting this array is erroneous." << endl;
    abort_handler(-1);
  }
}


const StringArray& Variables::inactive_discrete_variable_labels() const
{
  if (variablesRep) // envelope->letter
    return variablesRep->inactive_discrete_variable_labels();
  else // default definition for letters lacking redefinition of virtual fn.
    return emptyStringArray;
}


void Variables::
inactive_discrete_variable_labels(const StringArray& i_d_v_labels)
{
  if (variablesRep) // envelope->letter 
    variablesRep->inactive_discrete_variable_labels(i_d_v_labels);
  else if (!i_d_v_labels.empty()) {
    // default definition for letters lacking redefinition of virtual fn.
    Cerr << "Error: There are no inactive discrete variable labels in this "
	 << "variables view.\n       Setting this array is erroneous." << endl;
    abort_handler(-1);
  }
}


const UIntArray& Variables::inactive_continuous_variable_ids() const
{
  if (variablesRep)
    return variablesRep->inactive_continuous_variable_ids(); // envelope->letter
  else // default definition for letters lacking redefinition of virtual fn.
    return emptyUIntArray;
}


size_t Variables::acv() const
{
  if (!variablesRep) { // letter lacking redefinition of virtual fn.!
    Cerr << "Error: Letter lacking redefinition of virtual acv() function.\n"
         << "       No default defined at base class." << endl;
    abort_handler(-1);
  }

  return variablesRep->acv(); // envelope fwd to letter
}


size_t Variables::adv() const
{
  if (variablesRep)
    return variablesRep->adv(); // envelope fwd to letter
  else // default definition for letters lacking redefinition (merged types)
    return 0;
}


RealVector Variables::all_continuous_variables() const
{
  if (!variablesRep) { // letter lacking redefinition of virtual fn.!
    Cerr << "Error: Letter lacking redefinition of virtual all_continuous_"
         << "variables() function.\n       No default defined at base class."
	 << endl;
    abort_handler(-1);
  }

  return variablesRep->all_continuous_variables(); // envelope->letter
}


void Variables::all_continuous_variables(const RealVector& a_c_vars)
{
  if (variablesRep)
    variablesRep->all_continuous_variables(a_c_vars); // envelope->letter
  else { // letter lacking redefinition of virtual fn.
    Cerr << "Error: Letter lacking redefinition of virtual all_continuous_"
	 << "variables(RealVector) function.\n       No default defined at "
	 << "base class." << endl;
    abort_handler(-1);
  }
}


IntVector Variables::all_discrete_variables() const
{
  if (variablesRep)
    return variablesRep->all_discrete_variables(); // envelope->letter
  else // default definition for letters lacking redefinition (merged types)
    return emptyIntVector;
}


void Variables::all_discrete_variables(const IntVector& a_d_vars)
{
  if (variablesRep)
    variablesRep->all_discrete_variables(a_d_vars); // envelope->letter
  else if (!a_d_vars.empty()) {
    // default definition for letters lacking redefinition of virtual fn.
    Cerr << "Error: all discrete variables is not supported in this "
	 << "variables view.\n       Setting this array is erroneous." << endl;
    abort_handler(-1);
  }
}


StringArray Variables::all_continuous_variable_labels() const
{
  if (!variablesRep) { // letter lacking redefinition of virtual fn.!
    Cerr << "Error: Letter lacking redefinition of virtual all_continuous_"
         << "variable_labels() function.\n       "
	 << "No default defined at base class." << endl;
    abort_handler(-1);
  }

  return variablesRep->all_continuous_variable_labels(); // envelope->letter
}


void Variables::all_continuous_variable_labels(const StringArray& a_c_v_labels)
{
  if (variablesRep) // envelope->letter
    variablesRep->all_continuous_variable_labels(a_c_v_labels);
  else if (!a_c_v_labels.empty()) {
    // default definition for letters lacking redefinition of virtual fn.
    Cerr << "Error: all continuous variable labels not available in this "
	 << "variables view.\n       Setting this array is erroneous." << endl;
    abort_handler(-1);
  }
}


StringArray Variables::all_discrete_variable_labels() const
{
  if (variablesRep)
    return variablesRep->all_discrete_variable_labels(); // envelope->letter
  else // default definition for letters lacking redefinition (merged types)
    return emptyStringArray;
}


void Variables::all_discrete_variable_labels(const StringArray& a_d_v_labels)
{
  if (variablesRep) // envelope->letter 
    variablesRep->all_discrete_variable_labels(a_d_v_labels);
  else if (!a_d_v_labels.empty()) {
    // default definition for letters lacking redefinition of virtual fn.
    Cerr << "Error: all discrete variable labels not available in this "
	 << "variables view.\n       Setting this array is erroneous." << endl;
    abort_handler(-1);
  }
}


StringArray Variables::all_variable_labels() const
{
  if (!variablesRep) { // letter lacking redefinition of virtual fn.!
    Cerr << "Error: Letter lacking redefinition of virtual all_variable_labels "
         << "function.\nNo default defined at base class." << endl;
    abort_handler(-1);
  }
   
  return variablesRep->all_variable_labels(); // envelope->letter
}


const StringArray& Variables::all_discrete_variable_types() const
{
  if (variablesRep)
    return variablesRep->all_discrete_variable_types();// envelope fwd to letter
  else // default definition for letters lacking redefinition (merged types)
    return emptyStringArray;
}


/// equality operator for Variables
bool operator==(const Variables& vars1, const Variables& vars2)
{
  // this operator is a friend of Variables

  // Check derived type since operator== below is only defined for identical 
  // derived types.  If it becomes needed in the future, could add translations
  // between types or use all_merged_variables (nonconst and expensive).
  if (vars1.variablesRep->variablesView != vars2.variablesRep->variablesView)
    return false;

  // Check content

  // No good since variablesRep is a Variables* (base class pointer)
  //if ( !(*vars1.variablesRep == *vars2.variablesRep) )

  int active_view  = vars1.variablesRep->variablesView.first;
  if (active_view == MIXED_ALL)
    return ( *((AllVariables*)vars1.variablesRep) == 
	     *((AllVariables*)vars2.variablesRep) );
  else if (active_view == MERGED_DISTINCT_DESIGN ||
	   active_view == MERGED_DISTINCT_UNCERTAIN ||
	   active_view == MERGED_DISTINCT_STATE)
    return ( *((MergedVariables*)vars1.variablesRep) == 
	     *((MergedVariables*)vars2.variablesRep) );
  else if (active_view == MIXED_DISTINCT_DESIGN ||
	   active_view == MIXED_DISTINCT_UNCERTAIN ||
	   active_view == MIXED_DISTINCT_STATE)
    return ( *((DistinctVariables*)vars1.variablesRep) == 
	     *((DistinctVariables*)vars2.variablesRep) );
  else {
    Cerr << "Error: bad variables view in operator==(Variables&,Variables&)."
	 << endl;
    abort_handler(-1);
    return false;
  }
}


#ifdef DAKOTA_BOOST
/// binary_equal_to (since 'operator==' is not suitable for boost/hashed lookup)
bool binary_equal_to(const Variables& vars1, const Variables& vars2)
{
  // this function is a friend of Variables

  // Check derived type since 'binary_equal_to'  is only defined for identical
  // derived types.  If it becomes needed in the future, could add translations
  // between types or use all_merged_variables (nonconst and expensive).
  if (vars1.variablesRep->variablesView != vars2.variablesRep->variablesView)
    return false;

  // Check content

  // No good since variablesRep is a Variables* (base class pointer)
  //if ( !binary_equal_to(*vars1.variablesRep, *vars2.variablesRep) )

  int active_view  = vars1.variablesRep->variablesView.first;
  if (active_view == MIXED_ALL) {
    return binary_equal_to( *static_cast<AllVariables*>(vars1.variablesRep),
                            *static_cast<AllVariables*>(vars2.variablesRep) );
  }
  else if (active_view == MERGED_DISTINCT_DESIGN ||
           active_view == MERGED_DISTINCT_UNCERTAIN ||
           active_view == MERGED_DISTINCT_STATE) {
    return binary_equal_to( *static_cast<MergedVariables*>(vars1.variablesRep),
                            *static_cast<MergedVariables*>(vars2.variablesRep));  }
  else if (active_view == MIXED_DISTINCT_DESIGN ||
           active_view == MIXED_DISTINCT_UNCERTAIN ||
           active_view == MIXED_DISTINCT_STATE) {
    return binary_equal_to(*static_cast<DistinctVariables*>(vars1.variablesRep),             *static_cast<DistinctVariables*>(vars2.variablesRep));
  }
  else {
    Cerr << "Error: bad vars view in binary_equal_to(Variables&,Variables&)."
         << endl;
    abort_handler(-1);
    return false;
  }
}

/// hash_value for Variables - required by the new BMI hash_set of PRPairs
std::size_t hash_value(const Variables& vars)
{
  // this function is a friend of Variables

  std::size_t seed = 0;

  // Check derived type since hash_value is only defined for identical
  // derived types.  If it becomes needed in the future, could add translations
  // between types or use all_merged_variables (nonconst and expensive).

  boost::hash_combine(seed, vars.variablesRep->variablesView);

  // Combine hashes of content

  int active_view  = vars.variablesRep->variablesView.first;

  if (active_view == MIXED_ALL)
    boost::hash_combine(seed, *((AllVariables*)vars.variablesRep) );
  else if (active_view == MERGED_DISTINCT_DESIGN ||
           active_view == MERGED_DISTINCT_UNCERTAIN ||
           active_view == MERGED_DISTINCT_STATE)
    boost::hash_combine(seed, *((MergedVariables*)vars.variablesRep) );
  else if (active_view == MIXED_DISTINCT_DESIGN ||
           active_view == MIXED_DISTINCT_UNCERTAIN ||
           active_view == MIXED_DISTINCT_STATE)
    boost::hash_combine(seed, *((DistinctVariables*)vars.variablesRep) );
  else {
    Cerr << "Error: bad variables view in hash_value(Variables&)."
         << endl;
    abort_handler(-1);
    return 0;
  }

  return seed;
}
#endif  // DAKOTA_BOOST

} // namespace Dakota
