/*  _______________________________________________________________________

    DAKOTA: Design Analysis Kit for Optimization and Terascale Applications
    Copyright (c) 2006, Sandia National Laboratories.
    This software is distributed under the GNU General Public License.
    For more information, see the README file in the top Dakota directory.
    _______________________________________________________________________ */

//- Class:       Dakota::Constraints
//- Description: Container class of variables and bounds, each with fundamental
//-              arrays (from specification) and derived arrays (for special 
//-              cases).
//- Owner:       Mike Eldred
//- Version: $Id: DakotaConstraints.H 4799 2008-01-12 02:49:52Z mseldre $

#ifndef DAKOTA_CONSTRAINTS_H
#define DAKOTA_CONSTRAINTS_H

#include "data_types.h"

namespace Dakota {

class ProblemDescDB;


/// Base class for the variable constraints class hierarchy.

/** The Constraints class is the base class for the class hierarchy
    managing bound, linear, and nonlinear constraints.  Using the
    variable lower and upper bounds arrays from the input
    specification, different derived classes define different views of
    this data.  The linear and nonlinear constraint data is consistent
    in all views and is managed at the base class level.  For memory
    efficiency and enhanced polymorphism, the variable constraints
    hierarchy employs the "letter/envelope idiom" (see Coplien
    "Advanced C++", p. 133), for which the base class (Constraints)
    serves as the envelope and one of the derived classes (selected in
    Constraints::get_constraints()) serves as the letter. */

class Constraints
{
public:

  //
  //- Heading: Constructors, destructor, assignment operator
  //

  /// default constructor
  Constraints();
  /// standard constructor
  Constraints(const ProblemDescDB& problem_db, const pair<short,short>& view);
  /// alternate constructor for instantiations on the fly
  Constraints(const pair<short,short>& view);
  /// copy constructor
  Constraints(const Constraints& con);

  /// destructor
  virtual ~Constraints();

  /// assignment operator
  Constraints operator=(const Constraints& con);

  //
  //- Heading: Virtual functions
  //

  // ACTIVE VARIABLES

  /// return the active continuous variable lower bounds
  virtual const RealVector& continuous_lower_bounds() const;
  /// set the active continuous variable lower bounds
  virtual void continuous_lower_bounds(const RealVector& c_l_bnds);
  /// return the active continuous variable upper bounds
  virtual const RealVector& continuous_upper_bounds() const;
  /// set the active continuous variable upper bounds
  virtual void continuous_upper_bounds(const RealVector& c_u_bnds);

  /// return the active discrete variable lower bounds
  virtual const IntVector& discrete_lower_bounds() const;
  /// set the active discrete variable lower bounds
  virtual void discrete_lower_bounds(const IntVector& d_l_bnds);
  /// return the active discrete variable upper bounds
  virtual const IntVector& discrete_upper_bounds() const;
  /// set the active discrete variable upper bounds
  virtual void discrete_upper_bounds(const IntVector& d_u_bnds);

  // INACTIVE VARIABLES

  /// return the inactive continuous lower bounds
  virtual const RealVector& inactive_continuous_lower_bounds() const;
  /// set the inactive continuous lower bounds
  virtual void inactive_continuous_lower_bounds(const RealVector& i_c_l_bnds);
  /// return the inactive continuous upper bounds
  virtual const RealVector& inactive_continuous_upper_bounds() const;
  /// set the inactive continuous upper bounds
  virtual void inactive_continuous_upper_bounds(const RealVector& i_c_u_bnds);

  /// return the inactive discrete lower bounds
  virtual const IntVector& inactive_discrete_lower_bounds() const;
  /// set the inactive discrete lower bounds
  virtual void inactive_discrete_lower_bounds(const IntVector& i_d_l_bnds);
  /// return the inactive discrete upper bounds
  virtual const IntVector& inactive_discrete_upper_bounds() const;
  /// set the inactive discrete upper bounds
  virtual void inactive_discrete_upper_bounds(const IntVector& i_d_u_bnds);

  // ALL VARIABLES

  /// returns a single array with all continuous lower bounds
  virtual RealVector all_continuous_lower_bounds() const;
  /// sets all continuous lower bounds using a single array
  virtual void all_continuous_lower_bounds(const RealVector& a_c_l_bnds);
  /// returns a single array with all continuous upper bounds
  virtual RealVector all_continuous_upper_bounds() const;
  /// sets all continuous upper bounds using a single array
  virtual void all_continuous_upper_bounds(const RealVector& a_c_u_bnds);

  /// returns a single array with all discrete lower bounds
  virtual IntVector all_discrete_lower_bounds() const;
  /// sets all discrete lower bounds using a single array
  virtual void all_discrete_lower_bounds(const IntVector& a_d_l_bnds);
  /// returns a single array with all discrete upper bounds
  virtual IntVector all_discrete_upper_bounds() const;
  /// sets all discrete upper bounds using a single array
  virtual void all_discrete_upper_bounds(const IntVector& a_d_u_bnds);

  // INPUT/OUTPUT

  /// write a variable constraints object to an ostream
  virtual void write(ostream& s)    const;
  /// read a variable constraints object from an istream
  virtual void read(istream& s);

  //
  //- Heading: Member functions
  //

  // LINEAR CONSTRAINTS

  /// return the number of linear inequality constraints
  size_t num_linear_ineq_constraints() const;
  /// return the number of linear equality constraints
  size_t num_linear_eq_constraints() const;

  /// return the linear inequality constraint coefficients
  const RealMatrix& linear_ineq_constraint_coeffs() const;
  /// set the linear inequality constraint coefficients
  void linear_ineq_constraint_coeffs(const RealMatrix& lin_ineq_coeffs);
  /// return the linear inequality constraint lower bounds
  const RealVector& linear_ineq_constraint_lower_bounds() const;
  /// set the linear inequality constraint lower bounds
  void linear_ineq_constraint_lower_bounds(const RealVector& lin_ineq_l_bnds);
  /// return the linear inequality constraint upper bounds
  const RealVector& linear_ineq_constraint_upper_bounds() const;
  /// set the linear inequality constraint upper bounds
  void linear_ineq_constraint_upper_bounds(const RealVector& lin_ineq_u_bnds);
  /// return the linear equality constraint coefficients
  const RealMatrix& linear_eq_constraint_coeffs() const;
  /// set the linear equality constraint coefficients
  void linear_eq_constraint_coeffs(const RealMatrix& lin_eq_coeffs);
  /// return the linear equality constraint targets
  const RealVector& linear_eq_constraint_targets() const;
  /// set the linear equality constraint targets
  void linear_eq_constraint_targets(const RealVector& lin_eq_targets);

  // NONLINEAR CONSTRAINTS

  /// return the number of nonlinear inequality constraints
  size_t num_nonlinear_ineq_constraints() const;
  /// return the number of nonlinear equality constraints
  size_t num_nonlinear_eq_constraints() const;

  /// return the nonlinear inequality constraint lower bounds
  const RealVector& nonlinear_ineq_constraint_lower_bounds() const;
  /// set the nonlinear inequality constraint lower bounds
  void nonlinear_ineq_constraint_lower_bounds(
    const RealVector& nln_ineq_l_bnds);
  /// return the nonlinear inequality constraint upper bounds
  const RealVector& nonlinear_ineq_constraint_upper_bounds() const;
  /// set the nonlinear inequality constraint upper bounds
  void nonlinear_ineq_constraint_upper_bounds(
    const RealVector& nln_ineq_u_bnds);
  /// return the nonlinear equality constraint targets
  const RealVector& nonlinear_eq_constraint_targets() const;
  /// set the nonlinear equality constraint targets
  void nonlinear_eq_constraint_targets(const RealVector& nln_eq_targets);

  /// for use when a deep copy is needed (the representation is _not_ shared)
  Constraints copy() const;

  /// reshape the linear/nonlinear constraints arrays within the
  /// Constraints hierarchy
  void reshape(const size_t& num_nln_ineq_cons, const size_t& num_nln_eq_cons,
	       const size_t& num_lin_ineq_cons, const size_t& num_lin_eq_cons);

  /// reshape the bounds arrays within the Constraints hierarchy
  void reshape(const Sizet2DArray& vars_comps);

  /// function to check constraintsRep (does this envelope contain a letter)
  bool is_null() const;

protected:

  //
  //- Heading: Constructors
  //

  /// constructor initializes the base class part of letter classes
  /// (BaseConstructor overloading avoids infinite recursion in the
  /// derived class constructors - Coplien, p. 139)
  Constraints(BaseConstructor, const ProblemDescDB& problem_db,
	      const pair<short,short>& view);

  //
  //- Heading: Virtual functions
  //

  /// Used by copy() to copy the contents of a letter class
  virtual void copy_rep(const Constraints* con_rep);

  /// Used by reshape(Sizet2DArray&) to rehape the contents of a letter class
  virtual void reshape_rep(const Sizet2DArray& vars_comps);

  //
  //- Heading: Member functions
  //

  /// perform checks on user input, convert linear constraint
  /// coefficient input to matrices, and assign defaults
  void manage_linear_constraints(const ProblemDescDB& problem_db);

  //
  //- Heading: Data
  //

  /// the variables view pair containing active (first) and inactive (second)
  /// view enumerations
  pair<short,short> variablesView;

  /// number of nonlinear inequality constraints
  size_t numNonlinearIneqCons;
  /// number of nonlinear equality constraints
  size_t numNonlinearEqCons;
  /// nonlinear inequality constraint lower bounds
  RealVector nonlinearIneqConLowerBnds;
  /// nonlinear inequality constraint upper bounds
  RealVector nonlinearIneqConUpperBnds;
  /// nonlinear equality constraint targets
  RealVector nonlinearEqConTargets;

  /// number of linear inequality constraints
  size_t numLinearIneqCons;
  /// number of linear equality constraints
  size_t numLinearEqCons;
  /// linear inequality constraint coefficients
  RealMatrix linearIneqConCoeffs; 
  /// linear equality constraint coefficients
  RealMatrix linearEqConCoeffs;
  /// linear inequality constraint lower bounds
  RealVector linearIneqConLowerBnds;
  /// linear inequality constraint upper bounds
  RealVector linearIneqConUpperBnds;
  /// linear equality constraint targets
  RealVector linearEqConTargets;

  /// an empty real vector returned in get functions when there are
  /// no variable constraints corresponding to the request
  RealVector emptyRealVector;
  /// an empty int vector returned in get functions when there are
  /// no variable constraints corresponding to the request
  IntVector emptyIntVector;

private:

  //
  //- Heading: Member functions
  //

  /// Used only by the constructor to initialize constraintsRep to the 
  /// appropriate derived type.
  Constraints* get_constraints(const ProblemDescDB& problem_db,
			       const pair<short,short>& view);
  /// Used by copy() to initialize constraintsRep to the appropriate
  /// derived type.
  Constraints* get_constraints(const pair<short,short>& view) const;

  //
  //- Heading: Data
  //

  /// pointer to the letter (initialized only for the envelope)
  Constraints* constraintsRep;
  /// number of objects sharing constraintsRep
  int referenceCount;
};


// nonvirtual functions can access letter attributes directly (only need to fwd
// member function call when the function could be redefined).
inline size_t Constraints::num_linear_ineq_constraints() const
{
  return (constraintsRep) ? constraintsRep->numLinearIneqCons
                          : numLinearIneqCons;
}


inline size_t Constraints::num_linear_eq_constraints() const
{ return (constraintsRep) ? constraintsRep->numLinearEqCons : numLinearEqCons; }


inline const RealMatrix& Constraints::linear_ineq_constraint_coeffs() const
{
  return (constraintsRep) ? constraintsRep->linearIneqConCoeffs
                          : linearIneqConCoeffs;
}


inline void Constraints::
linear_ineq_constraint_coeffs(const RealMatrix& lin_ineq_coeffs)
{
  if (constraintsRep)
    constraintsRep->linearIneqConCoeffs = lin_ineq_coeffs;
  else
    linearIneqConCoeffs = lin_ineq_coeffs;
}


inline const RealVector& Constraints::
linear_ineq_constraint_lower_bounds() const
{
  return (constraintsRep) ? constraintsRep->linearIneqConLowerBnds
                          : linearIneqConLowerBnds;
}


inline void Constraints::
linear_ineq_constraint_lower_bounds(const RealVector& lin_ineq_l_bnds)
{
  if (constraintsRep)
    constraintsRep->linearIneqConLowerBnds = lin_ineq_l_bnds;
  else
    linearIneqConLowerBnds = lin_ineq_l_bnds;
}


inline const RealVector& Constraints::
linear_ineq_constraint_upper_bounds() const
{
  return (constraintsRep) ? constraintsRep->linearIneqConUpperBnds
                          : linearIneqConUpperBnds;
}


inline void Constraints::
linear_ineq_constraint_upper_bounds(const RealVector& lin_ineq_u_bnds)
{
  if (constraintsRep)
    constraintsRep->linearIneqConUpperBnds = lin_ineq_u_bnds;
  else
    linearIneqConUpperBnds = lin_ineq_u_bnds;
}


inline const RealMatrix& Constraints::linear_eq_constraint_coeffs() const
{
  return (constraintsRep) ? constraintsRep->linearEqConCoeffs
                          : linearEqConCoeffs;
}


inline void Constraints::
linear_eq_constraint_coeffs(const RealMatrix& lin_eq_coeffs)
{
  if (constraintsRep)
    constraintsRep->linearEqConCoeffs = lin_eq_coeffs;
  else
    linearEqConCoeffs = lin_eq_coeffs;
}


inline const RealVector& Constraints::linear_eq_constraint_targets() const
{
  return (constraintsRep) ? constraintsRep->linearEqConTargets
                          : linearEqConTargets;
}


inline void Constraints::
linear_eq_constraint_targets(const RealVector& lin_eq_targets)
{
  if (constraintsRep)
    constraintsRep->linearEqConTargets = lin_eq_targets;
  else
    linearEqConTargets = lin_eq_targets;
}


inline size_t Constraints::num_nonlinear_ineq_constraints() const
{
  return (constraintsRep) ? constraintsRep->numNonlinearIneqCons
                          : numNonlinearIneqCons;
}


inline size_t Constraints::num_nonlinear_eq_constraints() const
{
  return (constraintsRep) ? constraintsRep->numNonlinearEqCons
                          : numNonlinearEqCons;
}


inline const RealVector& Constraints::
nonlinear_ineq_constraint_lower_bounds() const
{
  return (constraintsRep) ? constraintsRep->nonlinearIneqConLowerBnds
                          : nonlinearIneqConLowerBnds;
}


inline void Constraints::
nonlinear_ineq_constraint_lower_bounds(const RealVector& nln_ineq_l_bnds)
{
  if (constraintsRep)
    constraintsRep->nonlinearIneqConLowerBnds = nln_ineq_l_bnds;
  else
    nonlinearIneqConLowerBnds = nln_ineq_l_bnds;
}


inline const RealVector& Constraints::
nonlinear_ineq_constraint_upper_bounds() const
{
  return (constraintsRep) ? constraintsRep->nonlinearIneqConUpperBnds
                          : nonlinearIneqConUpperBnds;
}


inline void Constraints::
nonlinear_ineq_constraint_upper_bounds(const RealVector& nln_ineq_u_bnds)
{
  if (constraintsRep)
    constraintsRep->nonlinearIneqConUpperBnds = nln_ineq_u_bnds;
  else
    nonlinearIneqConUpperBnds = nln_ineq_u_bnds;
}


inline const RealVector& Constraints::nonlinear_eq_constraint_targets() const
{
  return (constraintsRep) ? constraintsRep->nonlinearEqConTargets
                          : nonlinearEqConTargets;
}


inline void Constraints::
nonlinear_eq_constraint_targets(const RealVector& nln_eq_targets)
{
  if (constraintsRep)
    constraintsRep->nonlinearEqConTargets = nln_eq_targets;
  else
    nonlinearEqConTargets = nln_eq_targets;
}


inline bool Constraints::is_null() const
{ return (constraintsRep) ? false : true; }


// Having overloaded operators call read/write means that the operators need 
// not be a friend to private data because read/write functions are public.

/// istream extraction operator for Constraints
inline istream& operator>>(istream& s, Constraints& con)
{ con.read(s); return s; }


/// ostream insertion operator for Constraints
inline ostream& operator<<(ostream& s, const Constraints& con)
{ con.write(s); return s; }

} // namespace Dakota

#endif
