/*  _______________________________________________________________________

    DAKOTA: Design Analysis Kit for Optimization and Terascale Applications
    Copyright (c) 2006, Sandia National Laboratories.
    This software is distributed under the GNU General Public License.
    For more information, see the README file in the top Dakota directory.
    _______________________________________________________________________ */

//- Class:       AllConstraints
//- Description: Container class of linear constraints and bounds employing
//-              the "all" approach to variables usage.
//- Owner:       Mike Eldred
//- Version: $Id: AllConstraints.H 4799 2008-01-12 02:49:52Z mseldre $

#ifndef ALL_CONSTRAINTS_H
#define ALL_CONSTRAINTS_H

#include "DakotaConstraints.H"


namespace Dakota {

/// Derived class within the Constraints hierarchy which
/// employs the all data view.

/** Derived variable constraints classes take different views of the
    design, uncertain, and state variable types and the continuous and
    discrete domain types.  The AllConstraints derived class
    combines design, uncertain, and state variable types but separates
    continuous and discrete domain types.  The result is combined
    continuous bounds arrays (allContinuousLowerBnds,
    allContinuousUpperBnds) and combined discrete bounds arrays
    (allDiscreteLowerBnds, allDiscreteUpperBnds).  Parameter and DACE
    studies currently use this approach (see
    Variables::get_variables(problem_db) for variables view selection;
    variables view is passed to the Constraints constructor in Model). */

class AllConstraints: public Constraints
{
public:

  //
  //- Heading: Constructor and destructor
  //

  /// default constructor
  AllConstraints();
  /// standard constructor
  AllConstraints(const ProblemDescDB& problem_db,
		 const pair<short,short>& view);
  /// destructor
  ~AllConstraints();

  //
  //- Heading: Virtual function redefinitions
  //

  const RealVector& continuous_lower_bounds() const;
  void continuous_lower_bounds(const RealVector& c_l_bnds);
  const RealVector& continuous_upper_bounds() const;
  void continuous_upper_bounds(const RealVector& c_u_bnds);

  const IntVector& discrete_lower_bounds() const;
  void discrete_lower_bounds(const IntVector& d_l_bnds);
  const IntVector& discrete_upper_bounds() const;
  void discrete_upper_bounds(const IntVector& d_u_bnds);

  //const RealVector& inactive_continuous_lower_bounds() const;
  //void inactive_continuous_lower_bounds(const RealVector& i_c_l_bnds);
  //const RealVector& inactive_continuous_upper_bounds() const;
  //void inactive_continuous_upper_bounds(const RealVector& i_c_u_bnds);

  //const IntVector& inactive_discrete_lower_bounds() const;
  //void inactive_discrete_lower_bounds(const IntVector& i_d_l_bnds);
  //const IntVector& inactive_discrete_upper_bounds() const;
  //void inactive_discrete_upper_bounds(const IntVector& i_d_u_bnds);

  RealVector all_continuous_lower_bounds() const;
  void all_continuous_lower_bounds(const RealVector& a_c_l_bnds);
  RealVector all_continuous_upper_bounds() const;
  void all_continuous_upper_bounds(const RealVector& a_c_u_bnds);

  IntVector all_discrete_lower_bounds() const;
  void all_discrete_lower_bounds(const IntVector& a_d_l_bnds);
  IntVector all_discrete_upper_bounds() const;
  void all_discrete_upper_bounds(const IntVector& a_d_u_bnds);

  void write(ostream& s) const;
  void read(istream& s);

protected:

  void copy_rep(const Constraints* con_rep);

  void reshape_rep(const Sizet2DArray& vars_comps);

private:

  //
  //- Heading: Private data members
  //

  /// a continuous lower bounds array combining continuous design,
  /// uncertain, and continuous state variable types (all view).
  RealVector allContinuousLowerBnds;
  /// a continuous upper bounds array combining continuous design,
  /// uncertain, and continuous state variable types (all view).
  RealVector allContinuousUpperBnds;
  /// a discrete lower bounds array combining discrete design and
  /// discrete state variable types (all view).
  IntVector  allDiscreteLowerBnds;
  /// a discrete upper bounds array combining discrete design and
  /// discrete state variable types (all view).
  IntVector  allDiscreteUpperBnds;

  // needed to perform I/O in cdv/ddv/uv/csv/dsv order
  size_t numCDV; ///< number of continuous design variables
  size_t numDDV; ///< number of discrete design variables
};


inline AllConstraints::AllConstraints()
{ }


inline AllConstraints::~AllConstraints()
{ }


inline const RealVector& AllConstraints::continuous_lower_bounds() const
{ return allContinuousLowerBnds; }


inline void AllConstraints::
continuous_lower_bounds(const RealVector& c_l_bnds)
{
  // in the AllConstraints case, we cannot in general allow assignments to
  // change the length of allContinuousLowerBnds since this invalidates numCDV
  // and can result in I/O errors.  However, allow assignment in the case of
  // default construction.
  size_t num_aclb = allContinuousLowerBnds.length();
  if (num_aclb && c_l_bnds.length() != num_aclb) {
    Cerr << "Error: length of assigned vector (" << c_l_bnds.length()
	 << ") does not equal number of continuous variable lower bounds ("
	 << num_aclb << ") in AllConstraints::continuous_lower_bounds()."
	 << endl;
    abort_handler(-1);
  }
  allContinuousLowerBnds = c_l_bnds;
}


inline const RealVector& AllConstraints::continuous_upper_bounds() const
{ return allContinuousUpperBnds; }


inline void AllConstraints::
continuous_upper_bounds(const RealVector& c_u_bnds)
{
  // in the AllConstraints case, we cannot in general allow assignments to
  // change the length of allContinuousUpperBnds since this invalidates numCDV
  // and can result in I/O errors.  However, allow assignment in the case of
  // default construction.
  size_t num_acub = allContinuousUpperBnds.length();
  if (num_acub && c_u_bnds.length() != num_acub) {
    Cerr << "Error: length of assigned vector (" << c_u_bnds.length()
	 << ") does not equal number of continuous variable upper bounds ("
	 << num_acub << ") in AllConstraints::continuous_upper_bounds()."
	 << endl;
    abort_handler(-1);
  }
  allContinuousUpperBnds = c_u_bnds;
}


inline const IntVector& AllConstraints::discrete_lower_bounds() const
{ return allDiscreteLowerBnds; }


inline void AllConstraints::discrete_lower_bounds(const IntVector& d_l_bnds)
{
  // in the AllConstraints case, we cannot in general allow assignments to
  // change the length of allDiscreteLowerBnds since this invalidates numDDV
  // and can result in I/O errors.  However, allow assignment in the case of
  // default construction.
  size_t num_adlb = allDiscreteLowerBnds.length();
  if (num_adlb && d_l_bnds.length() != num_adlb) {
    Cerr << "Error: length of assigned vector (" << d_l_bnds.length()
	 << ") does not equal number of discrete variable lower bounds ("
	 << num_adlb << ") in AllConstraints::discrete_lower_bounds()."
	 << endl;
    abort_handler(-1);
  }
  allDiscreteLowerBnds = d_l_bnds;
}


inline const IntVector& AllConstraints::discrete_upper_bounds() const
{ return allDiscreteUpperBnds; }


inline void AllConstraints::discrete_upper_bounds(const IntVector& d_u_bnds)
{
  // in the AllConstraints case, we cannot in general allow assignments to
  // change the length of allDiscreteUpperBnds since this invalidates numDDV
  // and can result in I/O errors.  However, allow assignment in the case of
  // default construction.
  size_t num_adub = allDiscreteUpperBnds.length();
  if (num_adub && d_u_bnds.length() != num_adub) {
    Cerr << "Error: length of assigned vector (" << d_u_bnds.length()
	 << ") does not equal number of discrete variable upper bounds ("
	 << num_adub << ") in AllConstraints::discrete_upper_bounds()."
	 << endl;
    abort_handler(-1);
  }
  allDiscreteUpperBnds = d_u_bnds;
}


inline RealVector AllConstraints::all_continuous_lower_bounds() const
{ return allContinuousLowerBnds; }


inline void AllConstraints::
all_continuous_lower_bounds(const RealVector& a_c_l_bnds)
{ allContinuousLowerBnds = a_c_l_bnds; }


inline RealVector AllConstraints::all_continuous_upper_bounds() const
{ return allContinuousUpperBnds; }


inline void AllConstraints::
all_continuous_upper_bounds(const RealVector& a_c_u_bnds)
{ allContinuousUpperBnds = a_c_u_bnds; }


inline IntVector AllConstraints::all_discrete_lower_bounds() const
{ return allDiscreteLowerBnds; }


inline void AllConstraints::
all_discrete_lower_bounds(const IntVector& a_d_l_bnds)
{ allDiscreteLowerBnds = a_d_l_bnds; }


inline IntVector AllConstraints::all_discrete_upper_bounds() const
{ return allDiscreteUpperBnds; }


inline void AllConstraints::
all_discrete_upper_bounds(const IntVector& a_d_u_bnds)
{ allDiscreteUpperBnds = a_d_u_bnds; }

} // namespace Dakota

#endif
