/*  _______________________________________________________________________

    DAKOTA: Design Analysis Kit for Optimization and Terascale Applications
    Copyright (c) 2006, Sandia National Laboratories.
    This software is distributed under the GNU General Public License.
    For more information, see the README file in the top Dakota directory.
    _______________________________________________________________________ */

//- Class:        AllConstraints
//- Description:  Class implementation
//- Owner:        Mike Eldred

#include "AllConstraints.H"
#include "ProblemDescDB.H"

static const char rcsId[]="@(#) $Id";


namespace Dakota {

/** In this class, the all data approach (design, uncertain, and state
    types are combined) is used.  Iterators/strategies which use this
    class include: parameter studies, dace, and nond_sampling in
    all_variables mode.  Extract fundamental lower and upper bounds
    and combine them into allContinuousLowerBnds, allContinuousUpperBnds,
    allDiscreteLowerBnds, and allDiscreteUpperBnds. */
AllConstraints::
AllConstraints(const ProblemDescDB& problem_db, const pair<short,short>& view):
  Constraints(BaseConstructor(), problem_db, view),
  numCDV(problem_db.get_sizet("variables.continuous_design")),
  numDDV(problem_db.get_sizet("variables.discrete_design"))
{
#ifdef REFCOUNT_DEBUG
  Cout << "Letter instantiated: variablesView active = " << variablesView.first
       << " inactive = " << variablesView.second << endl;
#endif

  // Get fundamental arrays from problem_db
  const RealVector& cdv_lower_bnds
    = problem_db.get_drv("variables.continuous_design.lower_bounds");
  const RealVector& cdv_upper_bnds
    = problem_db.get_drv("variables.continuous_design.upper_bounds");
  const IntVector& ddv_lower_bnds
    = problem_db.get_div("variables.discrete_design.lower_bounds");
  const IntVector& ddv_upper_bnds
    = problem_db.get_div("variables.discrete_design.upper_bounds");
  const RealVector& uv_lower_bnds
    = problem_db.get_drv("variables.uncertain.lower_bounds");
  const RealVector& uv_upper_bnds
    = problem_db.get_drv("variables.uncertain.upper_bounds");
  const RealVector& csv_lower_bnds
    = problem_db.get_drv("variables.continuous_state.lower_bounds");
  const RealVector& csv_upper_bnds
    = problem_db.get_drv("variables.continuous_state.upper_bounds");
  const IntVector& dsv_lower_bnds
    = problem_db.get_div("variables.discrete_state.lower_bounds");
  const IntVector& dsv_upper_bnds
    = problem_db.get_div("variables.discrete_state.upper_bounds");

  // Initialize derived arrays for this class.
  aggregate_data(cdv_lower_bnds, uv_lower_bnds, csv_lower_bnds,
		 allContinuousLowerBnds);
  aggregate_data(cdv_upper_bnds, uv_upper_bnds, csv_upper_bnds,
		 allContinuousUpperBnds);
  aggregate_data(ddv_lower_bnds, dsv_lower_bnds, allDiscreteLowerBnds);
  aggregate_data(ddv_upper_bnds, dsv_upper_bnds, allDiscreteUpperBnds);

  // Manage linear constraints.
  manage_linear_constraints(problem_db);
}


void AllConstraints::read(istream& s)
{
  size_t num_uv_csv = allContinuousLowerBnds.length() - numCDV,
         num_dsv    = allDiscreteLowerBnds.length()   - numDDV;
  allContinuousLowerBnds.read_partial(s, 0, numCDV);
  allDiscreteLowerBnds.read_partial(s, 0, numDDV);
  allContinuousLowerBnds.read_partial(s, numCDV, num_uv_csv);
  allDiscreteLowerBnds.read_partial(s, numDDV, num_dsv);

  allContinuousUpperBnds.read_partial(s, 0, numCDV);
  allDiscreteUpperBnds.read_partial(s, 0, numDDV);
  allContinuousUpperBnds.read_partial(s, numCDV, num_uv_csv);
  allDiscreteUpperBnds.read_partial(s, numDDV, num_dsv);
}


void AllConstraints::write(ostream& s) const
{
  size_t num_uv_csv = allContinuousLowerBnds.length() - numCDV,
         num_dsv    = allDiscreteLowerBnds.length()   - numDDV;
  allContinuousLowerBnds.write_partial(s, 0, numCDV);
  allDiscreteLowerBnds.write_partial(s, 0, numDDV);
  allContinuousLowerBnds.write_partial(s, numCDV, num_uv_csv);
  allDiscreteLowerBnds.write_partial(s, numDDV, num_dsv);

  allContinuousUpperBnds.write_partial(s, 0, numCDV);
  allDiscreteUpperBnds.write_partial(s, 0, numDDV);
  allContinuousUpperBnds.write_partial(s, numCDV, num_uv_csv);
  allDiscreteUpperBnds.write_partial(s, numDDV, num_dsv);
}


void AllConstraints::copy_rep(const Constraints* con_rep)
{
  // copy derived class attributes
  allContinuousLowerBnds = ((AllConstraints*)con_rep)->allContinuousLowerBnds;
  allContinuousUpperBnds = ((AllConstraints*)con_rep)->allContinuousUpperBnds;
  allDiscreteLowerBnds   = ((AllConstraints*)con_rep)->allDiscreteLowerBnds;
  allDiscreteUpperBnds   = ((AllConstraints*)con_rep)->allDiscreteUpperBnds;
  numCDV                 = ((AllConstraints*)con_rep)->numCDV;
  numDDV                 = ((AllConstraints*)con_rep)->numDDV;
}


void AllConstraints::reshape_rep(const Sizet2DArray& vars_comps)
{
  numCDV = vars_comps[0][1];
  numDDV = vars_comps[0][2];
  size_t num_acv = numCDV + vars_comps[1][0] + vars_comps[2][1], // cdv+uv+csv
         num_adv = numDDV + vars_comps[2][2];                    // ddv+dsv
  allContinuousLowerBnds.reshape(num_acv);
  allContinuousUpperBnds.reshape(num_acv);
  allDiscreteLowerBnds.reshape(num_adv);
  allDiscreteUpperBnds.reshape(num_adv);
}

} // namespace Dakota
