%{
/*********************************************************************
Copyright 2008 Sandia Corporation.  Under the terms of Contract
DE-AC04-94AL85000 with Sandia Corporation, the U.S. Government
retains certain rights in this software.  All rights reserved.

Redistribution and use in source and binary forms, with or without
modification, are permitted provided that the following conditions
are met:

* Redistributions of source code must retain the above copyright
notice, this list of conditions and the following disclaimer.

* Redistributions in binary form must reproduce the above copyright
notice, this list of conditions and the following disclaimer in the
documentation and/or other materials provided with the distribution.

* Neither the name of Sandia Corporation nor the names of its
contributors may be used to endorse or promote products derived from
this software without specific prior written permission.

THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
"AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
A PARTICULAR PURPOSE ARE DISCLAIMED.  IN NO EVENT SHALL THE COPYRIGHT
OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
(INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
***********************************************************************/

#include <ctype.h>
#include <stdarg.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include "nidr.h"	/* for KeyWordKind */

#ifndef MBLK_GULP
#define MBLK_GULP 8191
#endif
#ifndef KWSTACKLEN
#define KWSTACKLEN 100	/*should be overkill*/
#endif

 typedef union
YYSTYPE { char *s; } YYSTYPE;

 enum LastSeen {
	Saw_kwname = 1,
	Saw_type = 2,
	Saw_func = 3
	};

 enum ParState {
	/* parstate values... */
	PPar_emtpy = 0,
	PPar_starting = 1,
	PPar_started = 2,
	PPar_leftpar = 3,
	Dupchecking = 4,
	Dupchecked = 5,
	Printing = 6,
	KwStacked = 7,
	Sorted = 8,
	Leftput = 9,
	Parchecked = 10
	};

 /* values for lastseen also include single characters ( [ { } ] ) ; */

 /* enums to make constants visible to debuggers */
 enum {
	Mblk_gulp = MBLK_GULP,
	KW_stacklen = KWSTACKLEN,
	Br_stacklen = 2*KW_stacklen,
	HTlen = 127,
	KWPlen = 32,
	KWOUTlen0 = 128 };

 typedef struct HashHead HashHead;
 typedef struct KWctx KWctx;
 typedef struct Mblk Mblk;
 typedef struct StrList StrList;

 struct
Mblk {
	Mblk *next;
	void *stuff[Mblk_gulp];
	};

 struct
HashHead { HashHead *hnext, *hprev; };

 struct
StrList {
	HashHead h;
	StrList *next;
	char *val;
	int Lineno;
	};

 struct
KWctx {
	KWctx *next;
	StrList *names;	/* list of name and aliases */
	KWctx *kw;	/* list of contained keywords */
	KWctx *master;	/* non-null ==> this is an alias; use master's kwkno */
	KWctx *pparent;	/* paren parent, for ((...)|(...) ...) */
	KWctx *rparent;	/* "real" parent, for stuff inside ((...)) */
	StrList *funcs;	/* pre- and post order functions and (void*) args */
	char *Br_top;	/* Br_top when this keyword began */
	int alt;	/* alternative group number of this keyword */
	int kind;	/* kind of values for this keyword */
	int kno;	/* number of this keyword, for generating kw_n names */
	int kwkno;	/* number of this keyword's kw array (if any) */
	int kwknop;	/* if this is a pparent and has a pparent, kwkno for this and parents */
	int nalt;	/* number of alternative groups in contained keywords */
	int nfcn;	/* number of function and arg entries in funcs */
	int nkw;	/* number of contained keywords */
	int nreq;	/* number of required elements in contained keywords */
	int req;	/* required group for this keyword (0 if optional) */
	int parstate;	/* state in handling ((...)) */
	int altoffset;	/* for exapnd */
	int reqoffset;	/* for expand */
	int level;
	int objno;	/* for kwcomp2 */
	};

 static int lastobjno;

 HashHead HTab[HTlen];
 KWctx KWbase, *lastkw, *lastkw1;
 KWctx *KWStack[KW_stacklen], **KWtop = KWStack;
 Mblk FirstMblk, *CurMblk = &FirstMblk;
 YYSTYPE yyval;
 char Br_stack[Br_stacklen], *Br_top = Br_stack;
 char *infname, *progname;
 int Lineno = 1, startwarn = 1, wantfuncs = 1;
 int brace, dupnames, expand, lastkno, lastseen, missing, needcomma, or_mode;
 size_t mbavail = Mblk_gulp;
 void **mbnext = FirstMblk.stuff;
 static void Keywd(char *);

 /* Possible values for *Br_top:
 //	 0  ==> empty (for Br_stack[0] only)
 //	'(' ==> open paren seen, awaiting ')'
 //	'[' ==> open square bracket, awaiting ']'
 */

 static void
botch(const char *fmt, ...)
{
	va_list ap;
	va_start(ap, fmt);
	fprintf(stderr, "\n%s:\n\tbotch on line %d", progname, Lineno);
	if (infname)
		fprintf(stderr, " of \"%s\"", infname);
	fprintf(stderr, ":\n\t");
	vfprintf(stderr, fmt, ap);
	fputs(".\n", stderr);
	va_end(ap);
	exit(1);
	}

 static void*
Malloc(size_t L, const char *where)
{
	void *rv = malloc(L);
	if (!rv)
		botch("malloc(%lu) failure in %s()", (unsigned long)L);
	return rv;
	}

 void
unexpected(const char *what)
{ botch("unexpected \"%s\"", what); }

 static void *
Alloc(size_t len)
{
	Mblk *mb;
	size_t L;
	void *rv;

	L = (len + sizeof(void*) - 1)/sizeof(void*);
	if (L >= Mblk_gulp)
		botch("Alloc(%lu) failure", (unsigned long)len);
	if (L > mbavail) {
		if (!(mb = CurMblk->next)) {
			mb = (Mblk*)Malloc(sizeof(Mblk), "Alloc");
			CurMblk->next = mb;
			}
		CurMblk = mb;
		mbavail = Mblk_gulp;
		mbnext = mb->stuff;
		mb->next = 0;
		memset(mbnext, 0, sizeof(mb->stuff));
		}
	rv = mbnext;
	mbnext += L;
	mbavail -= L;
	return rv;
	}

#if 0
 static void
Mblk_reset(void)
{
	CurMblk = FirstMblk;
	mbavail = Mblk_gulp;
	mbnext = FirstMblk.stuff;
	memset(mbnext, 0, sizeof(FirstMblk.stuff));
	}
#endif

 StrList *
new_StrList(const char *s, StrList *nxt)
{
	size_t L = strlen(s) + 1;
	StrList *rv = (StrList*)Alloc(sizeof(StrList) + L);
	strcpy(rv->val = (char*)(rv + 1), s);
	rv->next = nxt;
	rv->Lineno = Lineno;
	return rv;
	}

 static void
Alias(char *s)
{
	KWctx *kw;
	if (lastseen != Saw_kwname)
		botch("ALIAS out of place");
	kw = *KWtop;
	if (kw->kw)
		kw = kw->kw;
	for(s += 6; *s <= ' '; s++);
	kw->names = new_StrList(s, kw->names);
	lastseen = Saw_kwname;
	}

 static void
reqchk(const char *a, const char *s)
{
	if (a)
		while(*a)
			if (*a++ == lastseen)
				return;
	botch("\"%s\" out of place", s);
	}

 static void
reqchk1(const char *a, const char *s)
{
	int c;

	if (*Br_top != *s) {
		if ((c = *Br_top)) {
			switch(c) {
			 case '[': c = ']'; break;
			 case '(': c = ')'; break;
			 default:  c = '?';
			 }
			botch("Missing '%c'", c);
			}
		else
			a = 0;
		}
	reqchk(a, s+1);
	}

 static void
Bar(void)
{
	static char allowed[] = { /*({*/ '}', ')', Saw_kwname, Saw_type, 0 };
	reqchk(allowed, "|");
	lastseen = '|';
	lastkw = lastkw1 = 0;
	or_mode = 1;
	}

 static void
Lpar(void)
{
	KWctx *kw;
	int olastseen = lastseen;
	static char allowed[] = { '(', '[', '|', ']', /*{*/ '}', ')', Saw_kwname, Saw_type, 0 };

	reqchk(allowed, "("/*)*/);
	lastseen = '('; /*)*/
	if (KWtop > KWStack) {
		kw = KWtop[0];
		if (or_mode && kw->parstate == PPar_leftpar && !kw->kw) {
			--KWtop;
			Keywd(0);
			kw->pparent = KWtop[0];
			}
		else if (lastseen == olastseen)
			Keywd(0);
		lastseen = '(' /*)*/;
		}
	if (++Br_top - Br_stack >= Br_stacklen)
		botch("Br_stack overflow");
	*Br_top = '('; /*)*/
	lastkw = lastkw1 = 0;
	}

 static void
Lsqb(void)
{
	static char allowed[] = { /*({[*/ ']', '}', ')', Saw_kwname, Saw_type, 0 };
	reqchk(allowed, "["/*]*/);
	if (++Br_top - Br_stack >= Br_stacklen)
		botch("Br_stack overflow");
	lastseen = *Br_top = '['; /*]*/
	lastkw = lastkw1 = 0;
	}

 static void
Rpar(void)
{
	static char allowed[] =  {/*[({*/ '}', ')', ']', Saw_kwname, Saw_type, 0 };
	reqchk1(allowed, "()");
	if (KWtop <= KWStack)
		botch("bug: KW stack error");
	--Br_top;
	if (!KWtop[0]->kw && KWtop - KWStack > 2
	 && KWtop[-1]->parstate == PPar_starting)
		KWtop[-1]->Br_top = Br_top;
	--KWtop;
	lastseen = /*(*/ ')';
	lastkw = lastkw1 = 0;
	}

 static void
Rsqb(void)
{
	static char allowed[] = {/*{(*/'}', ')', '[', ']', Saw_kwname, Saw_type, 0 };
	reqchk1(allowed, "[]");
	if (KWtop <= KWStack)
		botch("bug: KW stack error");
	--Br_top;
	--KWtop;
	lastseen = /*[*/ ']';
	lastkw = lastkw1 = 0;
	}

 static void
Lbrace(void)
{
	if (brace++ || (lastseen != Saw_kwname && lastseen != Saw_type))
		unexpected("{");
	lastseen = '{'; /*}}*/
	}

 static void
Rbrace(void)
{
	if (--brace || (lastseen != Saw_func && lastseen != ';'))
		/*{{*/unexpected("}");
	lastseen = '}';
	lastkw = lastkw1 = 0;
	}

 static void
Type(int n, char *s)
{
	if (lastseen != Saw_kwname || !lastkw)
		botch("type %s out of place", s);
	lastkw->kind = n;
	lastkw = 0;
	lastseen = Saw_type;
	}

 static void
Keywd(char *s)
{
	KWctx *kw, *kw0, *kw1, *kw2, *kwm;
	int alt, req;
	static char allowed[] = { /*{(*/'}','(',')','[',']','|', Saw_kwname, Saw_type, 0 };

	kw0 = kw1 = *KWtop;
	kwm = 0;
	if (!kw1->names && s) {
		switch(kw1->parstate) {
		  case PPar_emtpy:
			do {
				kw1->parstate = PPar_starting;
				kw1 = kw1->pparent;
				}
				while(kw1 && kw1->parstate == PPar_emtpy);
			kw1 = kw0->rparent;
			break;
		  case PPar_starting:
			if (or_mode)
				kw1 = kw0->rparent;
			else
				kw1->parstate = PPar_started;
			break;
		  }
		}
	if (or_mode && !kw1->kw && KWtop - KWStack > 1) {
		kw2 = KWtop[-1];
		if (!kw2->names && kw2->parstate == PPar_starting)
			kw2 = kw2->rparent;
		if (s)
			kw1->master = kwm = kw1;
		kw1 = kw2;
		}
	alt = req = 0;
	if (s) {
		alt = ++kw1->nalt;
		if (*Br_top != '['/*]*/ || Br_top == kw1->Br_top)
			req = ++kw1->nreq;
		if (or_mode) {
			alt = --kw1->nalt;
			if (req)
				req = --kw1->nreq;
			}
		if (lastseen)
			reqchk(allowed, s);
		or_mode = 0;
		}
	lastkw = lastkw1 = kw = (KWctx*) Alloc(sizeof(KWctx));
	if (!kw0->names)
		kw->pparent = kw0;
	if (s) {
		kw->names = new_StrList(s,0);
		kw->next = kw1->kw;
		kw1->kw = kw;
		if (lastseen == '(' /*)*/)
			kw->parstate = PPar_leftpar;
		}
	else {
		kw->objno = ++lastobjno;
		if ((kw->rparent = kw1->rparent))
			kw1 = kw->rparent;
		else
			kw->rparent = kw1;
		alt = kw1->alt;
		req = kw1->req;
		}
	kw->alt = alt;
	kw->req = req;
	kw->Br_top = Br_top;
	kw->master = kwm;
	kw->level = KWtop - KWStack;
	if (lastseen == '(' || lastseen == '[' /*])*/ || KWtop == KWStack) {
		if (++KWtop - KWStack >= KW_stacklen)
			botch("KW stack overflow");
		*KWtop = kw;
		}
	lastseen = Saw_kwname;
	}

 static void
kw_finish(void)
{
	if (*Br_top)
		botch("unmatched '%c'", *Br_top);
	if (KWtop - KWStack > 1)
		botch("bug: KW stack botch in kw_finish");
	KWtop = KWStack;
	lastseen = 0;
	}

 static void
Fnc(char *s)
{
	KWctx *kw;

	if (!(kw = lastkw1))
		botch("misplaced {...}");
	if (++kw->nfcn > 4)
		botch("too many functions in {...}");
	kw->funcs = new_StrList(s, kw->funcs);
	lastseen = Saw_func;
	}

%}

ALias	"ALIAS"[ \t]+[a-z][a-z_0-9]*
Kword	[a-z][a-z_0-9]*
Func	[a-zA-Z_0-9()][,a-zA-Z_0-9<>():.+*/\-\[\] \t]*
ws	[ \t]+

%s FNC

%%

\n		{ ++Lineno; }
{ws}		{}
<INITIAL>{ALias}	{ Alias(yytext); }
<INITIAL>"INTEGER"	{ Type(KWKind_Int, yytext); }
<INITIAL>"INTEGERLIST"	{ Type(KWKind_Int | KWKind_List, yytext); }
<INITIAL>"KEYWORD"	{ kw_finish(); }
<INITIAL>"REAL"		{ Type(KWKind_Real, yytext); }
<INITIAL>"REALLIST"	{ Type(KWKind_Real | KWKind_List, yytext); }
<INITIAL>"STRING"	{ Type(KWKind_Str, yytext); }
<INITIAL>"STRINGLIST"	{ Type(KWKind_Str | KWKind_List, yytext); }
<INITIAL>{Kword}	{ Keywd(yytext); }
<INITIAL>";"		{ kw_finish(); }

"{"		{
		  Lbrace();
		  BEGIN FNC;
		}
"}"		{ Rbrace();
		  BEGIN INITIAL;
		}
"|"		{ Bar(); }
"("		{ Lpar(); }
")"		{ Rpar(); }
"["		{ Lsqb(); }
"]"		{ Rsqb(); }
<FNC>{Func}	{ Fnc(yytext); }
<FNC>";"	{}
"#".*\n		{++Lineno;}
.	{ botch("\"%s\" unexpected", yytext); }
%%
 int
yywrap(void)
{ return 1; }

 static int
usage(int rc)
{
	fprintf(rc ? stderr : stdout, "Usage: %s [options] specfile\noptions:\n\
	-d		{ suppress check for ambiguity from duplicate keywords }\n\
	-e		{ expand ((...)) for debugging }\n\
	-f		{ omit functions, i.e., {...}; implies -p }\n\
	-h headername	{ Specify header file(s), in addition to nidr.h;\n\
			  several -h options can be given;\n\
			  default = -h- ==> no headers }\n\
	-n namespname	{ assume namespace namespname;\n\
			  default = -n Dakota\n\
			  -n- ==> no namespace }\n\
	-p[w]		{ pretty-print rather than produce keywd header;\n\
			  add w spaces (default 2) for each nested context. }\n\
	-s[w]		{ like -p[w], but sort the keywords }\n\
	-w		{ cancel warn mode: do not report (on stderr)\n\
			  the number of implicitly named start routines;\n\
			  still flag their names  with /*!!*/ }\n\
	-o outfile	{ Write keywords to outfile rather than stdout. }\n",
		progname);
	return rc;
	}

 static StrList *
reverse(StrList **x)
{
	StrList *s, *t, *u;

	t = 0;
	for(s = *x; s; s = u) {
		u = s->next;
		s->next = t;
		t = s;
		}
	return *x = t;
	}

 static int
kwcomp(const void *a, const void *b)
{
	return strcmp((*(const KWctx**)a)->names->val, (*(const KWctx**)b)->names->val);
	}

 static KWctx *
kwsort(KWctx *kw0)
{
	/* Sort keywords after expanding kw list with aliases; compute nkw, */
	/* including aliases. */

	KWctx *kw, *kw1, *kwm, **x, **x0, **xe, *x00[512];
	StrList *minname, *nam, **pmin, **pnam;
	int nk;
	size_t L, nkw, nkwa, onkwa;

	reverse(&kw0->funcs);
	nkw = nkwa = 0;
	for(kw = kw0->kw; kw; kw = kw->next) {
		++nkw;
		pmin = &kw->names;
		minname = kw->names;
		pnam = &minname->next;
		onkwa = nkwa;
		while((nam = *pnam)) {
			++nkwa;
			if (strcmp(nam->val, minname->val) < 0) {
				pmin = pnam;
				minname = nam;
				}
			pnam = &nam->next;
			}
		if (nkwa > onkwa) {
			*pmin = minname->next;
			minname->next = kw->names;
			nam = kw->names = minname;
			nk = kw->kno;	/* ALias() made room for nk use below */
			kwm = kw;
			kw = 0;
			while((nam = nam->next)) {
				kw1 = (KWctx*)Alloc(sizeof(KWctx));
				if (!kw)
					kw = kw1;	/* so kw->next is right */
				kw1->next = kwm->next;
				kwm->next = kw1;
				kw1->names = nam;
				kw1->master = kwm;
				kw1->kind = kwm->kind;
				kw1->kno = ++nk;
				}
			}
		}
	kw0->nkw = (int)(nkw += nkwa);
	if (nkw <= 1)
		return kw0->kw;
	x0 = x00;
	if (nkw > sizeof(x00)/sizeof(x00[0]))
		x0 = (KWctx**)Malloc(L = nkw*sizeof(KWctx*), "kwsort");
	for(x = x0, kw = kw0->kw; kw; kw = kw->next)
		*x++ = kw;
	qsort(x0, nkw, sizeof(KWctx*), kwcomp);
	for(xe = x - 1, x = x0; x < xe; ++x) {
		if (!strcmp(x[0]->names->val, x[1]->names->val)) {
			fprintf(stderr, "%s: duplicate appearance of \"%s\"\n",
				progname, x[0]->names->val);
			++dupnames;
			}
		x[0]->next = x[1];
		}
	x[0]->next = 0;
	kw0->kw = kw = x0[0];
	if (x0 != x00)
		free(x0);
	return kw;
	}

 static int
alphanum(char *s)
{
	for(; *s; ++s)
		if (!isalnum(*s))
			return 0;
	return 1;
	}

 static void
kwout(KWctx *kw, KWctx **kwtop)
{
	KWctx *kw0, *kw1, *kw2, *kwm, **kws, **x, *x0[KWOUTlen0];
	StrList *S;
	char *fmt, *s;
	int aoff, i, m, m1, n, roff, sentinnel;

	if (expand && !kw->names)
		return;
	if (needcomma) {
		printf(",\n");
		needcomma = 0;
		}
	kw1 = kw->kw;
	n = 0;
	if (kw->names) {
		kw0 = kw1;
		if (kw->parstate != KwStacked)
			for(kw2 = kw->pparent; kw2; kw2 = kw2->pparent) {
				kw2->next = kw1;
				kw1 = kw2;
				++n;
				}
		if (!kw0) {
			if (!--n) {
				kw->nkw = kw1->nkw;
				if (kw1->kwkno) {
					kw->kwkno = kw1->kwkno;
					return;
					}
				kw->kwkno = lastkno + 1;
				kw = kw1;
				kw1 = kw->kw;
				}
			else {
				kw2 = kw->pparent;
				if (kw2->kwknop) {
					kw->kwkno = kw2->kwknop;
					return;
					}
				kw2->kwknop = lastkno + 1;
				kw1 = kw2;
				while((kw2 = kw2->pparent)) {
					kw2->next = kw1;
					kw1 = kw2;
					}
				}
			}
		}
	for(m = 0, kw0 = kw1; kw0; kw0 = kw0->next)
		++m;
	if (expand)
		for(kw0 = kw1; kw0 && !kw0->names; kw0 = kw0->next)
			for(kw2 = kw0->kw; kw2; kw2 = kw2->next)
				++m;
	x = x0;
	if (m > KWOUTlen0)
		x = (KWctx**)Malloc(m*sizeof(KWctx*), "kwout");
	m = 0;
	kw0 = kw1;
	if (expand) {
		aoff = kw->nalt;
		roff = kw->nreq;
		for(; kw0 && !kw0->names; kw0 = kw0->next) {
			for(kw2 = kw0->kw; kw2; kw2 = kw2->next) {
				kw2->altoffset = aoff;
				kw2->reqoffset = roff;
				x[m++] = kw2;
				}
			aoff += kw0->nalt;
			roff += kw0->nreq;
			}
		kw->nalt = aoff;
		kw->nreq = roff;
		}
	for(; kw0; kw0 = kw0->next)
		x[m++] = kw0;
	sentinnel = 0;
	if (expand) {
		kw->nkw = m;
		qsort(x, m, sizeof(KWctx*), kwcomp);
		for(i = 1; i < m; ++i)
			if (!strcmp(x[i-1]->names->val, x[i]->names->val)) {
				fprintf(stderr, "%s: duplicate appearance of \"%s\"\n",
					progname, x[i]->names->val);
				++dupnames;
				}
		}
	else if (kw->nkw == 0)
		sentinnel = 1;
	m1 = m + sentinnel;
	printf("\tkw_%d[%d] = {\n", kw->kwkno = ++lastkno, m1);
	needcomma = 1;
	for(i = 0; i < m;) {
		kw1 = x[i++];
		if (!(kwm = kw1->master))
			kwm = kw1;
		fputs("\t\t{", stdout);
		if (kw1->names)
			printf("\"%s\",%d,%d,%d,%d,", kw1->names->val,
				kw1->kind, kwm->nkw, kwm->alt + kwm->altoffset,
				kwm->req + kwm->reqoffset);
		else
			printf("0,0,%d,0,0,",kwm->nkw);
		n = kwm->kwkno;
		printf(n ? "kw_%d" : "0", n);
		if ((S = kw1->funcs)) {
			if (!kw1->kw) {
				reverse(&kw1->funcs);
				S = kw1->funcs;
				}
			goto funcput;
			}
		if ((S = kwm->funcs)) {
 funcput:
			if (S->next || strcmp(S->val,"0"))
				for(n = 0; S; S = S->next, ++n) {
					s = S->val;
					if (!(n & 1) || !strcmp(s,"0"))
						fmt = ",%s";
					else if (alphanum(s))
						fmt = ",(void*)%s";
					else
						fmt = ",(void*)(%s)";
					printf(fmt, s);
					}
			}
		else if (kwm->names) {
			++missing;
			printf(",/*!!*/");
			for(kws = KWStack; ++kws < kwtop; ) {
				if (!(kw2 = *kws)->names)
					kw2 = kw2->rparent;
				printf("%s_", kw2->names->val);
				}
			printf("%s_start", kwm->names->val);
			}
		printf("}%s\n", i < m1 ? "," : "");
		}
	if (sentinnel)
		fputs("\t\t{\"\"}\n", stdout);
	printf("\t\t}");
	if (x != x0)
		free(x);
	}

 static void
hashclear(KWctx *kw)
{
	StrList *sl;

	kw->parstate = Dupchecked;
	for(kw = kw->kw; kw; kw = kw->next)
		for(sl = kw->names; sl; sl = sl->next) {
			sl->h.hnext->hprev = sl->h.hprev;
			sl->h.hprev->hnext = sl->h.hnext;
			sl->h.hprev = sl->h.hnext = 0;
			}
	}

 static void
dupfound(HashHead *h0, StrList *sl, StrList *sl1)
{
	StrList *sl2, *sl3;

	++dupnames;
	for(sl2 = sl1; sl2->h.hnext != h0; sl2 = sl3) {
		sl3 = (StrList*)sl2->h.hnext;
		if (strcmp(sl->val, sl3->val))
			break;
		}
	fprintf(stderr, "Warning: \"%s\" on line %d also appears on line%s %d",
		sl->val, sl->Lineno, "s" + (sl2 == sl1), sl1->Lineno);
	while(sl1 != sl2) {
		sl1 = (StrList*)sl1->h.hnext;
		fprintf(stderr, ", %d", sl1->Lineno);
		}
	fputs(".\n", stderr);
	}

 static KWctx **
hashadd(KWctx *kw, KWctx **kwtop)
{
	HashHead *h, *h0;
	KWctx *kw1;
	StrList *sl, *sl1;
	char *s;
	int n;
	unsigned int c, x;

	kw->parstate = Dupchecking;
	for(kw1 = kw->kw; kw1; kw1 = kw1->next)
		for(sl = kw1->names; sl; sl = sl->next) {
			x = 0;
			s = sl->val;
			while((c = *(unsigned char*)s++))
				x = 43*x + c - ' ';
			h0 = &HTab[x % HTlen];
			for(h = h0->hnext; h != h0; h = h->hnext) {
				sl1 = (StrList*)h;
				n = strcmp(sl->val, sl1->val);
				if (n <= 0) {
					if (n == 0)
						dupfound(h0,sl,sl1);
					break;
					}
				}
			sl->h.hnext = h;
			h->hprev = (sl->h.hprev = h->hprev)->hnext = (HashHead *)sl;
			}
	if ((kw1 = kw->kw))
		*++kwtop = kw1;
	return kwtop;
	}

 static void
dup_check(void)
{
	HashHead *h, *he;
	KWctx *kw, *kw1, *kw2, **kwtop, **kwbot, *plist;
	int lev;

	for(h = HTab, he = h + HTlen; h < he; ++h)
		h->hnext = h->hprev = h;
	plist = 0;
	kwtop = kwbot = KWStack;
	*kwtop = &KWbase;
	for(;;) {
		kw = *kwtop;
		lev = kw->level;
		while(plist && plist->level >= lev) {
			hashclear(plist);
			plist = plist->next;
			}
		if (kw->parstate == Dupchecking) {
			hashclear(kw);
			if ((*kwtop = kw->next))
				continue;
			if (kwtop <= kwbot)
				break;
			--kwtop;
			continue;
			}
		if ((kw1 = kw->pparent) && kw1->parstate != Dupchecking) {
			for(kw2 = kw1; plist && plist != kw2; ) {
				hashclear(plist);
				if (!(plist = plist->next))
					break;
				while((kw2 = kw2->pparent) && kw2->level > plist->level);
				}
			kw2 = 0;
			do {
				if (kw1->kw) {
					kw1->next = kw2;
					kw2 = kw1;
					}
				}
				while((kw1 = kw1->pparent) && kw1->parstate != Dupchecking);
			while(kw2) {
				kwtop = hashadd(kw2, kwtop);
				kw1 = kw2;
				kw2 = kw2->next;
				kw1->next = plist;
				plist = kw1;
				}
			}
		kwtop = hashadd(kw, kwtop);
		}
	}

 static void
prkwname(KWctx *kw, const char *suf, int needrev)
{
	StrList *S;
	static const char *kname[] = { "BUG", "INTEGER", "REAL", "STRING" };

	if (needrev)
		reverse(&kw->names);
	S = kw->names;
	printf("%s", S->val);
	while((S = S->next))
		printf(" ALIAS %s", S->val);
	if (kw->kind) {
		printf(" %s", kname[kw->kind & 3]);
		if (kw->kind & KWKind_List)
			printf("LIST");
		}
	if (wantfuncs && kw->funcs) {
		fputs(" {", stdout);
		reverse(&kw->funcs);
		for(S = kw->funcs;;) {
			fputs(S->val, stdout);
			if (!(S = S->next))
				break;
			putchar(',');
			}
		putchar('}');
		}
	if (suf)
		fputs(suf, stdout);
	}

 static KWctx *
kwrev(KWctx **pkw)
{
	KWctx *kw, *kw0, *kw1;
	kw1 = *pkw;
	for(kw0 = 0; (kw = kw1); kw0 = kw) {
		kw1 = kw->next;
		kw->next = kw0;
		}
	return *pkw = kw0;
	}

 static void
paradj(int needrev)
{
	KWctx *kw, *kw1, **kwp, **kwtop, **kwbot;

	kwtop = kwbot = KWStack;
	*kwtop = &KWbase;
	if (needrev)
		kwrev(kwtop);
	for(;; --kwtop) {
		for(kw = *kwtop; kw; kw = kw->next) {
 top:
			if (needrev && (kw1 = kw->kw) && kw1->next)
				kwrev(&kw->kw);
			if ((kw1 = kw->pparent) && kw1->parstate != Parchecked) {
				kwp = &kw->pparent;
				for(;;) {
					if (kw1->kw) {
						*kwtop++ = kw1;
						kw1->parstate = Parchecked;
						*kwp = kw1;
						kwp = &kw1->pparent;
						}
					if (!(kw1 = *kwp = kw1->pparent)
					 || kw1->parstate == Parchecked)
						break;
					}
				}
			if (kw->kw) {
				if ((*kwtop = kw->next))
					++kwtop;
				kw = kw->kw;
				goto top;
				}
			}
		if (kwtop <= kwbot)
			break;
		}
	}

 static KWctx *
masteradj(KWctx *kw)
{
	/* to help print (nm1 | nm2 foo goo), make nm2 "master" */
	KWctx *kw1, *kw2, *kw3, *kwm, *rv;

	for(kw1 = kw, kw3 = 0; kw1; kw1 = kw1->next) {
		if ((kw2 = kw1->master))
			kw3 = kw2->master = kw2;
		}
	if (!kw3)
		return kw;
	for(rv = kw; kw; kw = kw->next) {
		kwm = kw->master;
		if ((kw1 = kw->next) && kw1->master == kwm && kwm) {
			kw3 = kw->kw ? kw : 0;
			do {
				kw2 = kw1;
				if (kw1->kw) {
					if (kw3)
						botch("Two masters in masteradj!");
					kw3 = kw1;
					}
				}
				while((kw1 = kw1->next) && kw1->master == kwm);
			if (kw3 || kw2->pparent) {
				if (kw2 != kw3) {
					kw2->pparent = kw3->pparent;
					kw2->kw = kw3->kw;
					kw3->kw = 0;
					}
				for(;;kw = kw->next) {
					kw->master = kw2;
					if (kw == kw2)
						break;
					}
				}
			else
				for(;;kw = kw->next) {
					kw->master = 0;
					if (kw == kw2)
						break;
					}
			kw = kw2;
			}
		}
	return rv;
	}

 static void
xmasteradj(KWctx **x, int n)
{
	/* variant of masteradj for print_ex1 */
	KWctx *kw, *kw1, *kw2, *kw3, *kwm;
	int i, j;

	for(i = 0, kw3 = 0; i < n; ++i) {
		kw1 = x[i];
		if ((kw2 = kw1->master))
			kw3 = kw2->master = kw2;
		}
	if (!kw3)
		return;
	for(i = 0; i < n; i = j) {
		j = i + 1;
		kw = x[i];
		kwm = kw->master;
		if (j < n && kwm && (kw1 = x[j])->master == kwm) {
			kw3 = kw->kw ? kw : 0;
			do {
				kw2 = kw1;
				if (kw1->kw) {
					if (kw3)
						botch("Two masters in xmasteradj!");
					kw3 = kw1;
					}
				}
				while(++j < n && (kw1 = x[j])->master == kwm);
			if (kw3 || kw2->pparent) {
				if (kw2 != kw3) {
					kw2->pparent = kw3->pparent;
					kw2->kw = kw3->kw;
					kw3->kw = 0;
					}
				for(;;kw = x[++i]) {
					kw->master = kw2;
					if (kw == kw2)
						break;
					}
				}
			else
				for(;; kw = x[++i]) {
					kw->master = 0;
					if (kw == kw2)
						break;
					}
			}
		}
	}

 static void
do_indent(int indent)
{
	for(indent += 8; indent >= 8; indent -= 8)
		putchar('\t');
	for(; indent > 0; --indent)
		putchar(' ');
	}

 static void
pretty_print(int iinc, int needrev)
{
	KWctx *kw, *kw1, *kw2, *kw3, **kwtop, **kwbot, *plist;
	const char *s;
	int alt, i, indent, lev, needindent, needspace;

	kwtop = kwbot = KWStack;
	indent = 0;
	plist = 0;
	needindent = 1;
	for(kw1 = KWbase.kw; kw1;) {
		fputs("KEYWORD ", stdout);
		prkwname(kw1, "\n", needrev);
		*kwtop = masteradj(kw1->kw);
		alt = 0;
		for(;;--kwtop) {
	 top:
			for(kw = *kwtop; kw; kw = kw->next) {
				lev = kw->level;
				if (needindent) {
					do_indent(indent);
					needindent = 0;
					}
				if (plist
				 && (plist->level > lev
				  || plist->level == lev && plist != kw->pparent)) {
 finishpar:
					*kwtop = kw;
					*++kwtop = plist;
					*++kwtop = masteradj(plist->kw);
					plist->alt = alt;
					alt = 0;
					plist = plist->next;
					goto top;
					}
				if (kw->parstate == Printing) {
					fputs(!kw->names || /*[(*/kw->req ? ")\n" : "]\n", stdout);
					needindent = 1;
					alt = kw->alt;
					indent -= iinc;
					if (!kw->names) {
						alt = kw->alt;
						goto bot;
						}
					continue;
					}
				if (plist)
					for (kw2 = kw->pparent;
							kw2 && kw2->parstate != Printing;
							kw2 = kw2->pparent) {
						if (plist->level >= kw2->level)
							goto finishpar;
						}
				needindent = 1;
				needspace = 0;
				if (alt == kw->alt) {
					putchar('|');
					needspace = 1;
					}
				if ((kw2 = kw->pparent) && kw2->parstate != Printing) {
					kw3 = 0;
					do {
						if (kw2->parstate == Printing)
							break;
						if (kw2->kw) {
							kw2->next = kw3;
							kw3 = kw2;
							kw2->parstate = Printing;
							if (needspace) {
								putchar('\n');
								needspace = 0;
								do_indent(indent);
								}
							indent += iinc;
							putchar('('); /*)*/
							for(i = 1; i < iinc; ++i)
								putchar(' ');
							}
						}
						while((kw2 = kw2->pparent));
					while(kw3) {
						kw2 = kw3->next;
						kw3->next = plist;
						plist = kw3;
						kw3 = kw2;
						}
					}
				if ((kw2 = kw->master) && kw == kw2)
					kw2 = 0;
				if (!kw->kw && (!kw2 || kw2->parstate == Leftput)) {
					if (needspace)
						putchar(' ');
					if (!kw->req) {
						if (alt != kw->alt) {
							fputs("[ ", stdout);
							alt = kw->alt;
							}
						s = /*[*/ " ]\n";
						if ((kw2 = kw->next) && kw2->alt == alt)
							s = "\n";
						prkwname(kw, s, needrev);
						continue;
						}
					alt = kw->alt;
					prkwname(kw, "\n", needrev);
					continue;
					}
				if (kw->parstate == Leftput) {
					if (needspace)
						goto putspace;
					}
				else {
					if (needspace) {
						putchar('\n');
						do_indent(indent);
						}
					putchar(kw->req ? '(' : '['); /*])*/
					indent += iinc;
 putspace:
					for(i = 1; i < iinc; ++i)
						putchar(' ');
					}
				kw->parstate = Printing;
				prkwname(kw, "\n", needrev);
				if (kw2) {
					alt = kw->alt;
					kw2->parstate = Leftput;
					continue;
					}
				alt = 0;
				*kwtop = kw;
				*++kwtop = masteradj(kw->kw);
				goto top;
				}
 bot:
			if (kwtop == kwbot) {
				if (plist) {
					*kwtop++ = plist;
					*kwtop++ = masteradj(plist->kw);
					plist->alt = alt;
					alt = 0;
					plist = plist->next;
					}
				else {
					if (indent) {
						if (needindent)
							do_indent(indent);
						fputs(/*(*/")\n", stdout);
						needindent = 1;
						indent -= iinc;
						}
					break;
					}
				}
			}
		if (!(kw1 = kw1->next))
			break;
		putchar('\n');
		}
	}

 static void
Output(StrList *headers, StrList *nsname, int hwant, int nwant)
{
	KWctx *kw, *kw1, *kw2, **kwtop, **kwbot;
	StrList *h;

	if (hwant && headers) {
		printf("#include \"nidr.h\"\n");
		for(h = reverse(&headers); h; h = h->next)
			printf(*h->val == '<' /*>*/
				? "#include %s\n"
				: "#include \"%s\"\n", h->val);
		}
	if (nwant) {
		if (!nsname)
			nsname = new_StrList("Dakota", nsname);
		putchar('\n');
		for(h = reverse(&nsname); h; h = h->next)
			printf("namespace %s {\n", h->val); /*}*/
		}
	printf("\nstatic KeyWord\n");
	kwtop = kwbot = KWStack;
	*kwtop = &KWbase;
	for(;;--kwtop) {
 top:
		for(kw = *kwtop; kw; kw = kw->next) {
			if ((kw->master && kw->master != kw)
			|| ((!(kw1 = kw->pparent) || kw->parstate == KwStacked) && !kw->kw)
			|| kw->kwkno)
				continue;
			if (kw1 && kw1->parstate != KwStacked) {
				if ((kw1 = kw->pparent) && kw1->parstate != KwStacked) {
					kw1->parstate = KwStacked;
					*kwtop = kw;
					*++kwtop = kw1;
					kw2 = *++kwtop = kwsort(kw1);
					do kw2->parstate = KwStacked;
						while((kw2 = kw2->next));
					while((kw1 = kw1->pparent) && kw1->parstate != KwStacked) {
						kw1->parstate = KwStacked;
						*++kwtop = kw1;
						kw2 = *++kwtop = kwsort(kw1);
						do kw2->parstate = KwStacked;
							while((kw2 = kw2->next));
						}
					goto top;
					}
				if (!kw1 && !kw->kw)
					continue;
				}
			if (!kw->nkw && kw->kw) {
				*kwtop = kw;
				*++kwtop = kwsort(kw);
				goto top;
				}
			kwout(kw, kwtop);
			}
		if (kwtop == kwbot)
			break;
		}
	printf(";\n\n");
	if (nwant) {
		for(h = reverse(&nsname); h; h = h->next)
			printf("} // namespace %s\n", h->val);
		putchar('\n');
		}
	printf("KeyWord Dakota_Keyword_Top = {\"KeywordTop\",0,%d,%d,%d",
		KWbase.nkw, KWbase.alt, KWbase.req);
	putchar(',');
	if (nwant)
		for(h = reverse(&nsname); h; h = h->next)
			printf("%s::", h->val);
	printf("kw_%d};\n", KWbase.kwkno);
	if (startwarn && missing)
		fprintf(stderr, "%s: %d start routines implicitly named.\n",
			progname, missing);
	}

 typedef int (*KWcomp)(const void*, const void*);

 static const char **zalt;

 static int
kwcomp1(const void *a, const void *b)
{
	const KWctx *ka, *kb;

	ka = *(const KWctx**)a;
	kb = *(const KWctx**)b;
	if (ka->req) {
		if (!kb->req)
			return -1;
		}
	else if (kb->req)
		return 1;
	if (ka->alt == kb->alt) {
		if (ka->master != kb->master) {
			if (!ka->master)
				return -1;
			if (!kb->master)
				return 1;
			ka = ka->master;
			kb = kb->master;
			}
		return strcmp(ka->names->val, kb->names->val);
		}
	return strcmp(zalt[ka->alt], zalt[kb->alt]);
	}

 static int
kwcomp2(const void *a, const void *b)
{
	const KWctx *ka, *kb, *pa, *pb;
	int n;

	ka = *(const KWctx**)a;
	kb = *(const KWctx**)b;
	if (ka->req) {
		if (!kb->req)
			return -1;
		}
	else if (kb->req)
		return 1;
	if (ka->alt == kb->alt) {
		pa = ka->master ? ka->master->pparent : ka->pparent;
		pb = kb->master ? kb->master->pparent : kb->pparent;
		if (pa != pb)
			return ka->objno - kb->objno;
		if (ka->master != kb->master) {
			if (!ka->master)
				return -1;
			if (!kb->master)
				return 1;
			ka = ka->master;
			kb = kb->master;
			}
		return strcmp(ka->names->val, kb->names->val);
		}
	return strcmp(zalt[ka->alt], zalt[kb->alt]);
	}

 static void
do_sort(KWctx *kw, KWcomp KWcmp)
{
	KWctx *kw1, *kwm, **x, *x0[KWPlen];
	const char *z0[KWPlen];
	int i, n, na;

	for(n = 0, kw1 = kw->kw; kw1; kw1 = kw1->next) {
		++n;
		reverse(&kw1->names);
		}
	if (n <= 1)
		return;
	x = n > KWPlen ? (KWctx**)Malloc(n*sizeof(KWctx*), "do_sort") : x0;
	na = kw->nalt + 1;
	zalt = na <= KWPlen ? z0 : (const char**)Malloc(na*sizeof(const char*), "do_sort");
	memset(zalt, 0, na*sizeof(const char*));
	for(n = 0, kw1 = kw->kw; kw1; kw1 = kw1->next) {
		x[n++] = kw1;
		if ((kwm = kw1->master))
			kwm->master = kwm;
		if (!zalt[i = kw1->alt]
		 || strcmp(kw1->names->val, zalt[i]) < 0)
			zalt[i] = kw1->names->val;
		}
	qsort(x, n--, sizeof(KWctx*), KWcmp);
	for(i = 0; i < n; ++i)
		x[i]->next = x[i+1];
	x[n]->next = 0;
	kw->kw = x[0];
	if (zalt != z0)
		free(zalt);
	if (x != x0)
		free(x);
	}

 static void
ksort(KWcomp KWcmp)
{
	KWctx *kw, *kw1, **kwtop, **kwtop0, **kwbot;

	kwtop = kwbot = KWStack;

	for(*kwtop = &KWbase;; --kwtop) {
 top:
		kw = *kwtop;
 top1:
		while(kw) {
			if ((kw1 = kw->pparent) && kw1->parstate != Sorted) {
				*kwtop = kw;
				kwtop0 = kwtop;
				do {
					kw1->parstate = Sorted;
					*++kwtop = kw1;
					}
					while((kw1 = kw1->pparent) && kw1->parstate != Sorted);
				if (kwtop > kwtop0)
					goto top;
				}
			if (kw->kw) {
				do_sort(kw, KWcmp);
				if ((*kwtop = kw->next))
					++kwtop;
				kw = *kwtop = kw->kw;
				goto top1;
				}
			kw = kw->next;
			}
		if (kwtop <= kwbot)
			break;
		}
	}

 static void
pretty_ex1(KWctx *kw, KWctx *prevkw, KWctx *nextkw, int indent, int iinc, int needrev)
{
	KWctx *kw0, *kw1, *kw2, *kwm, *kwp, *kwp0, **x, *y, *y1;
	const char *suf, *z0[KWPlen];
	int i, i0, ia, j, n, na, np;

	x = 0;
	kw1 = kw->kw;
	if ((kwp0 = kwp = kw->pparent) && (!kw->master || kw->master == kw)) {
		for(n = 0;;) {
			if (kwp->parstate != Printing) {
				kwp->parstate = Printing;
				for(kw1 = kwp->kw; kw1; kw1 = kw1->next)
					kw1->pparent = 0;	/* avoid infinite loops */
				}
			for(kw1 = kwp->kw; kw1; kw1 = kw1->next)
				++n;
			if (!(kw1 = kwp->pparent))
				break;
			kwp = kwp->pparent = kw1;
			}
		np = n;
		for(kw1 = kw->kw; kw1; kw1 = kw1->next)
			++n;
		y = y1 = (KWctx*)Malloc(np*sizeof(KWctx) + n*sizeof(KWctx*), "pretty_ex1");
		x = (KWctx**)(y + np);
		for(i = 0, kw1 = kw->kw; kw1; kw1 = kw1->next)
			x[i++] = kw1;
		i0 = i;
		for(kwp = kwp0; kwp; kwp = kwp->pparent) {
			j = i;
			na = i + kwp->nalt;
			for(kw1 = kwp->kw; kw1; kw1 = kw1->next, ++y1) {
				memcpy(y1, kw1, sizeof(KWctx));
				x[i++] = y1;
				y1->alt += j;
				}
			}
		if (!needrev) {
			for(j = i0; j < n; ++j) {
				kw1 = x[j];
				kw1->nreq = -(kw1->nreq + 1);
				}
			zalt = ++na <= KWPlen ? z0
				: (const char**)Malloc(na*sizeof(const char*), "pretty_ex1");
			memset(zalt, 0, na*sizeof(const char*));
			for(j = 0; j < n; ++j) {
				kw1 = x[j];
				if ((kwm = kw1->master))
					kwm->master = kwm;
				if (!zalt[i = kw1->alt]
				 || strcmp(kw1->names->val, zalt[i]) < 0)
					zalt[i] = kw1->names->val;
				}
			qsort(x, n, sizeof(KWctx*), kwcomp1);
			if (zalt != z0)
				free(zalt);
			for(j = 0; j < n; ++j) {
				kw1 = x[j];
				if (kw1->nreq < 0) {
					kw1->nreq = -(kw1->nreq + 1);
					kw1->pparent = 0;
					}
				}
			}
		kw1 = x[0];
		xmasteradj(x, n);
		}
	if (kw1) {
		if (!x)
			kw1 = masteradj(kw->kw);
		if (prevkw && kw->alt == prevkw->alt) {
			ia = kw->master && kw->master == prevkw->master ? iinc : 0;
			do_indent(indent + ia);
			if (!prevkw->master || prevkw->master != kw->master) {
				fputs("|\n", stdout);
				do_indent(indent + ia);
				putchar('(');
				}
			else
				putchar('|');
			}
		else {
			do_indent(indent);
			putchar(kw->req ? '(' : '[');
			}
		for(i = 1; i < iinc; ++i)
			putchar(' ');
		indent += iinc;
		prkwname(kw, "\n", needrev);
		i = 0;
		kw0 = 0;
		while(kw1) {
			if (x)
				kw2 = ++i >= n ? 0 : x[i];
			else
				kw2 = kw1->next;
			pretty_ex1(kw1, kw0, kw2, indent, iinc, needrev);
			kw0 = kw1;
			kw1 = kw2;
			}
		if (x)
			free(y);
		do_indent(indent);
		fputs(kw->req ? ")\n" : "]\n", stdout);
		}
	else {
		if (prevkw && kw->alt == prevkw->alt
		 && kw->master && kw->master == prevkw->master)
			indent += iinc;
		do_indent(indent);
		suf = "\n";
		if (prevkw && kw->alt == prevkw->alt) {
			if (kw->master && kw->master != prevkw->master) {
				fputs("|\n", stdout);
				do_indent(indent);
				fputs("( ", stdout);
				}
			else {
				fputs("| ", stdout);
				if (nextkw && !kw->req)
					goto rbcheck;
				}
			}
		else if (kw->req) {
			if (prevkw && prevkw->master != kw->master)
				fputs("( ", stdout);
			}
		else {
			fputs("[ ", stdout);
 rbcheck:
			if (!nextkw || nextkw->alt != kw->alt)
				suf = " ]\n";
			}
		prkwname(kw, suf, needrev);
		}
	}

 static void
pretty_expand(int iinc, int needrev)
{
	KWctx *kw, *kw0, *kw1, *kw2;

	if ((kw = KWbase.kw)) for(;;) {
		masteradj(kw->kw);
		fputs("KEYWORD ", stdout);
		prkwname(kw, "\n", needrev);
		kw0 = 0;
		for(kw1 = kw->kw; kw1; kw1 = kw2) {
			kw2 = kw1->next;
			pretty_ex1(kw1, kw0, kw2, 0, iinc, needrev);
			kw0 = kw1;
			}
		if (!(kw = kw->next))
			break;
		putchar('\n');
		}
	}

 static int
Str_option(char *s, char ***pav, int *want, StrList **ps)
{
	if (*++s || (s = *++*pav)) {
		if (*s == '-' && ! s[1])
			*want = 0;
		else
			*ps = new_StrList(s, *ps);
		return 0;
		}
	return 1;
	}

 static int
version(void)
{
	printf("nidrgen version 20080303\n");
	return 0;
	}

 int
main(int argc, char **argv)
{
	StrList *headers, *nsname;
	char *ofname, *s, *se;
	int dupcheck, hwant, needrev, nwant, pprint, sort, x;

	KWStack[0] = &KWbase;
	KWbase.names = new_StrList("$top",0);
	progname = argv[0];
	if (argc <= 1)
 usage1:	return usage(1);
	headers = nsname = 0;
	ofname = 0;
	dupcheck = hwant = needrev = nwant = 1;
	pprint = sort = 0;
 nextopt:
	while((s = *++argv) && *s == '-' && s[1]) {
		while(*++s)
		switch(*s) {
		 case 'd':
			dupcheck = 0;
			break;
		 case 'e':
			expand = 1;
			break;
		 case 'f':
			wantfuncs = 0;
			break;
		 case 'h':
			if (Str_option(s, &argv, &hwant, &headers))
				goto usage1;
			goto nextopt;
		 case 'n':
			if (Str_option(s, &argv, &nwant, &nsname))
				goto usage1;
			goto nextopt;
		 case 'o':
			if (!*++s && !(s = *++argv))
				goto usage1;
			ofname = s;
			goto nextopt;
		 case 's':
			sort = 1;
			needrev = 0;
			for(;;++s) {
				switch(s[1]) {
				  case 'e': expand = 1;
					    continue;
				  case 'f': wantfuncs = 0;
				  case 'p': continue;
				  }
				break;
				}
			/* no break */
		 case 'p':
			pprint = 2;
			for(;;++s) {
				switch(s[1]) {
				  case 'e':
					expand = 1;
					continue;
				  case 'f':
					wantfuncs = 0;
					continue;
				  case 's':
					sort = 1;
					needrev = 0;
					continue;
				  }
				break;
				}
			if (*++s) {
				pprint = (int)strtol(s,&se,10);
				if (pprint < 1 || se)
					goto usage1;
				}
			goto nextopt;
		 case 'v':
			return version();
		 case 'w':
			startwarn = 0;
			break;
		 case '?':
			return usage(s[1] != 0);
		 case '-':
			if (!strcmp(++s,"help"))
				return usage(0);
			if (!strcmp(s,"version"))
				return version();
			if (!*s) {
				s = *++argv;
				goto argsdone;
				}
			/* no break */
		  default:
			goto usage1;
		  }
		}
 argsdone:
	if (!s || argv[1])
		return usage(1);
	if (*s == '-' && !s[1])
		yyin = stdin;
	else if (!(yyin = fopen(infname = s, "r"))) {
		fprintf(stderr, "%s:  cannot open \"%s\"\n", progname, s);
		return 1;
		}
	if (!wantfuncs && !pprint)
		pprint = 2;
	if ((x = yylex())) {
		fprintf(stderr, "\nSurprise return %d from yylex()\n", x);
		return x;
		}
	if (Br_top > Br_stack) {
		fputs("Missing ", stderr);
		do putc(/*[*/ *Br_top-- == '(' ? ')' : ']', stderr);
			while(Br_top >Br_stack);
		fputs(" at end of file.\n", stderr);
		return 1;
		}
	if (KWbase.kw) {
		paradj(needrev);
		if (dupcheck)
			dup_check();
		}
	if (ofname && !freopen(ofname, "w", stdout)) {
		fprintf(stderr, "Cannot open output file \"%s\"\n", ofname);
		return 1;
		}
	if (KWbase.kw) {
		if (pprint) {
			if (expand) {
				if (sort)
					ksort(kwcomp1);
				pretty_expand(pprint, needrev);
				}
			else {
				if (sort)
					ksort(kwcomp2);
				pretty_print(pprint, needrev);
				}
			}
		else
			Output(headers, nsname, hwant, nwant);
		}
	return dupnames > 0;
	}
