/*  _________________________________________________________________________
 *
 *  Coliny: A Library of COLIN optimizers
 *  Copyright (c) 2003, Sandia National Laboratories.
 *  This software is distributed under the GNU Lesser General Public License.
 *  For more information, see the README.html file in the top Coliny directory.
 *  _________________________________________________________________________
 */

/**
 * \file PLGO.h
 *
 * Defines the coliny::PLGO class.
 */

#ifndef coliny_PLGO_h
#define coliny_PLGO_h

#include <acro_config.h>
#ifdef ACRO_USING_PLGO

#include <pebbl/serialLipshitzian.h>
#include <pebbl/parallelLipshitzian.h>
#include <colin/ColinUtilib.h>
#include <colin/StdOptSolver.h>
#include <colin/AppResponseAnalysis.h>
#include <colin/colin.h>

namespace coliny {

using utilib::BasicArray;
using utilib::ParameterSet;
using utilib::Ereal;
using colin::real;

class PLGO;


namespace plgo {

///
/// A class that defines an PLGO problem instance
///
class PLGOProblem
{
public:

  ///
  PLGOProblem() : problem(0), solver(0) {}

  ///
  Ereal<double> operator()(BasicArray<double>& x);

  ///
  colin::OptProblem<BasicArray<double>,colin::AppResponse_Utilib>* problem;

  ///
  colin::AppResponse_Utilib response;

  ///
  BasicArray<real> lower;

  ///
  BasicArray<real> upper; 

  ///
  Ereal<double> val;

  ///
  Ereal<double> cval;

  ///
  PLGO* solver;
};


///
/// An abstract class that defines the particular solver
///
class PLGOSolver
{
public:

  /// Destructor
  virtual ~PLGOSolver() {}

  /// Get a ParameterSet object, which can be used to set parameters
  virtual ParameterSet& parameters() = 0;

  /// Reset the solver's state
  virtual void reset() = 0;

  /// Reset the solver's state
  virtual pebbl::branching* base() = 0;

  /// Perform minimization
  virtual void minimize(std::string& termination_info, BasicArray<double>& x, colin::real& val) = 0;

  /// Returns true if this is a serial solver
  virtual bool isSerial() = 0;

  /// Set the function used in the search
  virtual void set_func(PLGOProblem* func) = 0;
};


///
/// PLGO serial solver class
///
class PLGOSerialSolver : public PLGOSolver
{
public:

  /// The serial solver
  pebbl::serialLipshitzian<PLGOProblem> solver;

  ///
  ParameterSet& parameters()
	{ return solver; }

  ///
  void reset()
	{
	int tmp1=0; char** tmp2=0;
	solver.setup(tmp1,tmp2); solver.reset(); }

  /// 
  pebbl::branching* base()
	{ return &solver; }

  ///
  void minimize(std::string& termination_info, BasicArray<double>& x, colin::real& val)
	{
	solver.solve();
        pebbl::arraySolution<double>* soln = static_cast<pebbl::arraySolution<double>*>(solver.incumbent);
	x << soln->array;
	val = soln->value;
        if (solver.abortReason)
	   termination_info = "Error";
        else
	   termination_info = "Successful";
	}

  ///
  bool isSerial()
	{ return true; }

  ///
  void set_func(PLGOProblem* func)
	{ solver.func = func; }
};


///
/// PLGO parallel solver class
///
#if defined(ACRO_HAVE_MPI)
class PLGOParallelSolver : public PLGOSolver
{
public:

  /// The serial solver
  pebbl::parallelLipshitzian<PLGOProblem> solver;

  ///
  ParameterSet& parameters()
	{ return solver; }

  ///
  void reset()
	{
	int tmp1=0; char** tmp2=0;
	solver.setup(tmp1,tmp2); solver.reset(); }

  /// 
  pebbl::branching* base()
	{ return &solver; }

  ///
  void minimize(std::string& termination_info, BasicArray<double>& x, colin::real& val)
	{
	solver.solve();
        pebbl::arraySolution<double>* soln = static_cast<pebbl::arraySolution<double>*>(solver.incumbent);
	x << soln->array;
	val = soln->value;
        if (solver.abortReason)
	   termination_info = "Error";
        else
	   termination_info = "Successful";
	}


  ///
  bool isSerial()
	{ return false; }

  ///
  void set_func(PLGOProblem* func)
	{
#if 0
	 /// WEH - this isn't portable, so I'm commenting it out for now.
	 static_cast<pebbl::serialLipshitzian<PLGOProblem> >(solver).func 
			= func;
#endif
	}
};
#endif


}


/** An interface to the Lipshitzian Global Optimizer defined by PICO
  */
class PLGO : public colin::StdOptSolver<BasicArray<double>,colin::AppResponse_Utilib>,
		public colin::AppResponseAnalysis
{
public:

  /// Constructor
  PLGO();

  ///
  virtual ~PLGO()
	{ delete solver; }

  ///
  void reset();

  ///
  void minimize();

protected:


  ///
  double step_tolerance;

  ///
  double initial_step;

  ///
  BasicArray<double> x;

  ///
  plgo::PLGOSolver* solver;

  ///
  plgo::PLGOProblem func;

};

inline Ereal<double> plgo::PLGOProblem::operator()(BasicArray<double>& x)
{
if (problem->numNonlinearConstraints() > 0)
   problem->Eval(x,response,colin::mode_f|colin::mode_cf);
else
   problem->Eval(x,response,colin::mode_f);
solver->compute_response_info(response,problem->state->constraint_lower_bounds,problem->state->constraint_upper_bounds,val,cval);
return val;
}

} // namespace coliny
#endif


#endif
