/*  _________________________________________________________________________
 *
 *  COLIN: A Common Optimization Library INterface
 *  Copyright (c) 2003, Sandia National Laboratories.
 *  This software is distributed under the GNU Lesser General Public License.
 *  For more information, see the README.html file in the top COLIN directory.
 *  _________________________________________________________________________
 */

/**
 * \file OptProblemState.h
 *
 * Defines the colin::OptProblemState class.
 */

#ifndef colin_OptProblemState_h
#define colin_OptProblemState_h

#include <acro_config.h>
#include <colin/OptProblemStateBase.h>

namespace colin {


/// The object used by OptProblem to refer to the state
/// of the problem.
template <class DomainT, class VectorT, class MatrixT>
class OptProblemState : public OptProblemStateBase
{
public:

  /// Constructor
  OptProblemState() : OptProblemStateBase() {}

  /// Set bounds with a string format.
  void set_bounds(const char* format);
 
  /// Returns true if reals are in the search domain.
  bool domain_reals()
                {return OptDomainTrait_reals<DomainT>::value;}

  /// Returns true if integers are in the search domain.
  bool domain_integers()
                {return OptDomainTrait_integers<DomainT>::value;}

  /// Returns true if booleans are in the search domain.
  bool domain_binarys()
                {return OptDomainTrait_bools<DomainT>::value;}

  /**@name Linear Constraints */
  //@{
  /// Set linear constraints (for all variables) - sparse format
  void set_linear_constraints(const MatrixT& mat, const VectorT& rhs,
				const int num_equality=0);

  /// The linear constraint matrix (in general sparse format eventually)
  MatrixT linear_constraint_mat;

  /// The RHS for linear constraints
  VectorT linear_constraint_rhs;
  //@}

};



//============================================================================
//
//
inline OptProblemStateBase::OptProblemStateBase()
: db_check(false),
  last_id(-1),
  enforcing_bound_constraints(false),
  num_real_params(0),
  num_int_params(0),
  num_binary_params(0),
  using_gradients(false),
  using_hessians(false)
{
numFunctions=1;
set_constraint_ctrs(0,0,0,0);
}



//============================================================================
//
//
inline void OptProblemStateBase::reset()
{
numConstraints=numLinearEqConstraints+numLinearIneqConstraints+
               numNonlinearEqConstraints+numNonlinearIneqConstraints;

request_mode = 0;
REQUEST_ASV.resize(numFunctions+numConstraints);
REQUEST_ASV << 0;

F_ASV.resize(numFunctions+numConstraints);
F_ASV << 0;
F_ASV[0] = static_cast<int>(mode_f);

G_ASV.resize(numFunctions+numConstraints);
G_ASV << 0;
G_ASV[0] = static_cast<int>(mode_g);

CF_ASV.resize(numFunctions+numConstraints);
CF_ASV << static_cast<int>(mode_f);
CF_ASV[0] = 0;

CG_ASV.resize(numFunctions+numConstraints);
CG_ASV << static_cast<int>(mode_g);
CG_ASV[0] = 0;
}



//============================================================================
//
//
inline void OptProblemStateBase::
	set_constraint_bounds(const std::vector<real>& lower_,
						const std::vector<real>& upper_)
{
constraint_lower_bounds << lower_;
constraint_upper_bounds << upper_;
}



//============================================================================
//
//
inline void OptProblemStateBase::
	get_constraint_bounds(std::vector<real>& lower_, std::vector<real>& upper_)
{
lower_ << constraint_lower_bounds;
upper_ << constraint_upper_bounds;
}



//============================================================================
//
//
inline void OptProblemStateBase::init_request_asv(int mode)
{
if (mode == request_mode) return;
REQUEST_ASV << 0;

if (mode & mode_f) REQUEST_ASV += F_ASV;
if (mode & mode_g) REQUEST_ASV += G_ASV;
if (mode & mode_cf) REQUEST_ASV += CF_ASV;
if (mode & mode_cg) REQUEST_ASV += CG_ASV;
}



//============================================================================
//
//
inline void OptProblemStateBase::
	set_int_bounds(const std::vector<int>& lower_, const std::vector<int>& upper_)
{
int_lower_bounds.resize(lower_.size());
int_upper_bounds.resize(upper_.size());
int_lower_bound_type.resize(lower_.size());
int_upper_bound_type.resize(lower_.size());

num_int_params = static_cast<unsigned int>(lower_.size());
enforcing_bound_constraints=true;

int_lower_bounds << lower_;
int_upper_bounds << upper_;
int_lower_bound_type << hard_bound;
int_upper_bound_type << hard_bound;
}


//============================================================================
//
//
inline void OptProblemStateBase::
	set_real_bounds(const std::vector<real>& lower_, const std::vector<real>& upper_)
{
real_lower_bounds.resize(lower_.size());
real_upper_bounds.resize(upper_.size());
real_lower_bound_type.resize(lower_.size());
real_upper_bound_type.resize(lower_.size());
 
num_real_params = static_cast<unsigned int>(lower_.size());

real_lower_bounds << lower_;
real_upper_bounds << upper_;
real_lower_bound_type << hard_bound;
real_upper_bound_type << hard_bound;
enforcing_bound_constraints=false;
for (size_type i=0; i<num_real_params; i++) {
  if (real_lower_bounds[i] == colin::real::negative_infinity)
     real_lower_bound_type[i] = no_bound;
  else 
     enforcing_bound_constraints=true;
  if (real_upper_bounds[i] == colin::real::positive_infinity)
     real_upper_bound_type[i] = no_bound;
  else 
     enforcing_bound_constraints=true;
  }
}


//============================================================================
//
//
inline void OptProblemStateBase::
	get_int_bounds(std::vector<int>& lower_, std::vector<int>& upper_) const
{
lower_.resize(num_int_params);
upper_.resize(num_int_params);
 
for (size_type i=0; i<num_int_params; i++) {
  if (int_lower_bound_type[i] != no_bound)
     lower_[i] = int_lower_bounds[i];
  else
     lower_[i] = - MAXINT;
  if (int_upper_bound_type[i] != no_bound)
     upper_[i] = int_upper_bounds[i];
  else
     upper_[i] =   MAXINT;
  }
}
 
 
//============================================================================
//
//
inline void OptProblemStateBase::
	get_real_bounds(std::vector<real>& lower_, std::vector<real>& upper_) const
{
lower_.resize(num_real_params);
upper_.resize(num_real_params);
 
for (size_type i=0; i<num_real_params; i++) {
  if (real_lower_bound_type[i] != no_bound)
     lower_[i] = real_lower_bounds[i];
  else
     lower_[i] = real :: negative_infinity;
  if (real_upper_bound_type[i] != no_bound)
     upper_[i] = real_upper_bounds[i];
  else
     upper_[i] = real :: positive_infinity;
  }
}
 
 
//============================================================================
//
//
template <class DomainT, class VectorT, class MatrixT>
void OptProblemState<DomainT,VectorT,MatrixT>::set_bounds(const char* format)
{
if (format == NULL)
   return;
 
int i, tmp, flen, ndx;
int nint=0, nreal=0, nbit=0;
 
//
// Process the 'format' string
//
flen = strlen(format);
int info_len = 0;
for (i=0; i<flen; i++)
  if (format[i] == ']')
     info_len++;
 
std::vector<std::vector<double> > Domain_Info(info_len);
ndx = 0;
 
while ((ndx < flen) && (format[ndx] != '*') && (format[ndx] != '['))
  ndx++;
 
for (i=0; i<info_len; i++) {
  while ((ndx < flen) && (format[ndx] != '['))
    ndx++;
  if (ndx == flen)  break;
 
  double tmp1,tmp2;
  if (sscanf(&(format[++ndx]),"%lf,%lf", &tmp1,&tmp2) != 2) {
     EXCEPTION_MNGR(std::runtime_error,"OptProblemStateBase::set_bounds - Domain error at position " << ndx << " in string  \"" << format << "\".");
     }
  else {
     Domain_Info[i].resize(4);
     Domain_Info[i][0] = tmp1;
     Domain_Info[i][1] = tmp2;
     }
  int flag=false;
  while ((ndx < flen) && (format[ndx] != ']')) {
    if (format[ndx] == '.') flag=true;
    ndx++;
    }
  if (ndx == flen)  break;

  ndx++;
  if (format[ndx] != '^')
     tmp = 1;
  else if (sscanf(&(format[++ndx]),"%d", &tmp) != 1)
     EXCEPTION_MNGR(std::runtime_error,"OptProblemStateBase::set_bounds - Domain error at position " << ndx << " in string \"" << format << "\".");
  Domain_Info[i][2] = tmp;

  if (flag) {
     if (! domain_reals() ) {
        EXCEPTION_MNGR(std::runtime_error,"OptProblemStateBase::set_bounds - specified a real bound for an OptProblem without real parameters.");
        }
     else {
        Domain_Info[i][3] = 1; //RealDomain;
	nreal += (int) Domain_Info[i][2];
	}
     }
  else {
     if ((Domain_Info[i][0] == 0.0) && (Domain_Info[i][1] == 1.0) && 
	 domain_binarys()) {
         Domain_Info[i][3] = 3; // BitDomain;
	 nbit += (int) Domain_Info[i][2];
	 }
     else {
	if (! domain_integers() ) {// Default to a real bound
           Domain_Info[i][3] = 1; // RealDomainT;
	   nreal += (int) Domain_Info[i][2];
	   }
	else {
           Domain_Info[i][3] = 2; // IntDomainT;
	   nint += (int) Domain_Info[i][2];
	   }
        }
     }
  }
 
//
// Process the Domain_Info
//
int_lower_bounds.resize(nint);
int_upper_bounds.resize(nint);
if (nint) {
   ndx=0;
   for (i=0; i<info_len; i++) {
     if (Domain_Info[i][3] == 2) //IntDomainT
        for (int j=0; j<Domain_Info[i][2]; j++,ndx++) {
          int_lower_bounds[ndx] = (int) Domain_Info[i][0];
          int_upper_bounds[ndx] = (int) Domain_Info[i][1];
          }
     }
   }
num_int_params = nint;

real_lower_bounds.resize(nreal);
real_upper_bounds.resize(nreal);
if (nreal) {
   ndx=0;
   for (i=0; i<info_len; i++) {
     if (Domain_Info[i][3] == 1) //RealDomainT
        for (int j=0; j<Domain_Info[i][2]; j++,ndx++) {
          real_lower_bounds[ndx] = Domain_Info[i][0];
          real_upper_bounds[ndx] = Domain_Info[i][1];
          }
     }
  }
num_real_params = nreal;

num_binary_params = nbit;
 
enforcing_bound_constraints=true;
int_lower_bound_type.resize(int_lower_bounds.size());
int_upper_bound_type.resize(int_lower_bounds.size());
real_lower_bound_type.resize(real_lower_bounds.size());
real_upper_bound_type.resize(real_lower_bounds.size());

int_lower_bound_type << hard_bound;
int_upper_bound_type << hard_bound;
real_upper_bound_type << hard_bound;
real_lower_bound_type << hard_bound;
}


//============================================================================
//
//
template <class DomainT, class VectorT, class MatrixT>
void OptProblemState<DomainT,VectorT,MatrixT>
	::set_linear_constraints(const MatrixT& mat, const VectorT& rhs, 
					const int num_eq_constr)
{
numLinearEqConstraints = num_eq_constr;
validate_linear_constraints(mat,rhs,
			num_real_params+num_int_params+num_binary_params,
			numLinearIneqConstraints,numLinearEqConstraints);
linear_constraint_mat = mat;
linear_constraint_rhs = rhs;
numConstraints=numLinearEqConstraints+numLinearIneqConstraints+
               numNonlinearEqConstraints+numNonlinearIneqConstraints;
}


} // namespace colin

#endif
