/*  _________________________________________________________________________
 *
 *  COLIN: A Common Optimization Library INterface
 *  Copyright (c) 2003, Sandia National Laboratories.
 *  This software is distributed under the GNU Lesser General Public License.
 *  For more information, see the README.html file in the top COLIN directory.
 *  _________________________________________________________________________
 */

/**
 * \file DirectFuncApplication.h
 *
 * Defines the colin::DirectFuncApplication class, along with 
 * template instances of the function colin::exec_DIrectApplication_EvalFn,
 * which are used by this class.
 */

#ifndef colin_DirectFuncApplication_h
#define colin_DirectFuncApplication_h

#include <acro_config.h>
#include <colin/DirectApplicationBase.h>

namespace colin {

/**
  * This subclass of \c DirectApplicationBase specializes the functionality in
  * that class for the case where the domain is a generic data type.  This
  * derived class calls the function \c exec_DirectApplication_EvalFn to
  * perform the transformation of the DomainT/ResponseT data into the
  * argument list of the user's objective function FuncT.
  */
template <class DomainT, class ResponseT, class FuncT>
class DirectFuncApplication : 
	public DirectApplicationBase<DomainT,ResponseT,FuncT>
{
public:

  /// Generic constructor.
  DirectFuncApplication(int app_mode_, FuncT fn) 
	: DirectApplicationBase<DomainT,ResponseT,FuncT>(app_mode_,fn) {}

  ///
  void EvalFn(DomainT& point, ResponseT& response)
	{ exec_DirectApplication_EvalFn(point,response,this->eval_fn); }
		
};


/**
  * This class provides a generic container for integrating separate
  * codes/functions for the objective and constraints.
  */
template <class DomainT, class ResponseT, class FuncT1, class FuncT2>
class FunctionContainer1
{
public:

  /// Constructor.
  FunctionContainer1(FuncT1 ofunc_, FuncT2 cfunc_)
		: ofunc(ofunc_), cfunc(cfunc_) {}

  /// Constructor.
  FunctionContainer1()
		: ofunc(0), cfunc(0) {}

  /// Evaluate a point and get the response.
  void eval(DomainT& domain, ResponseT& response);

protected:

  /// Objective function.
  FuncT1 ofunc;

  /// Constraint functino.
  FuncT2 cfunc;
};


template <class DomainT, class ResponseT, class FuncT1, class FuncT2>
void FunctionContainer1<DomainT,ResponseT,FuncT1,FuncT2>
	::eval(DomainT& domain, ResponseT& response)
{
bool f_flag, c_flag;
response.get_request_summary(f_flag,c_flag);

if (f_flag) {
   (*ofunc)(domain,response.function_value());
   response.request_vector()[0] &= mode_f;
   response.response_vector()[0] |= mode_f;
   }
if (c_flag) {
   (*cfunc)(domain,response.constraint_values());
   for (unsigned int i=0; i<response.constraint_values().size(); i++) {
     response.request_vector()[1+i] &= mode_f;
     response.response_vector()[1+i] |= mode_f;
     }
   }
}



#if !defined(DOXYGEN)
template <class DomainT, class ResponseT>
void exec_DirectApplication_EvalFn(DomainT& point, 
        ResponseT& response,
        void (*eval_fn)(DomainT&, ResponseT&))
{ (*eval_fn)(point, response); }


template <class DomainT, class ResponseT>
void exec_DirectApplication_EvalFn(DomainT& point, 
        ResponseT& response,
        void (*eval_fn)(const DomainT&, ResponseT&))
{ (*eval_fn)(point, response); }


template <class DomainT, class ResponseT>
void exec_DirectApplication_EvalFn(DomainT& point, 
        ResponseT& response,
        double (*eval_fn)(DomainT&))
{
response.function_value() = (*eval_fn)(point);
response.request_vector()[0] &= mode_f;
response.response_vector()[0] |= mode_f;
}


template <class DomainT, class ResponseT>
void exec_DirectApplication_EvalFn(DomainT& point, 
        ResponseT& response,
        double (*eval_fn)(const DomainT&))
{
response.function_value() = (*eval_fn)(point);
response.request_vector()[0] &= mode_f;
response.response_vector()[0] |= mode_f;
}


template <class DomainT, class ResponseT>
void exec_DirectApplication_EvalFn(DomainT& point, 
        ResponseT& response,
        void (*eval_fn)(DomainT&, double& ans))
{
double ans;
(*eval_fn)(point,ans);
response.function_value() = ans;
response.request_vector()[0] &= mode_f;
response.response_vector()[0] |= mode_f;
}


template <class DomainT, class ResponseT>
void exec_DirectApplication_EvalFn(DomainT& point, 
        ResponseT& response,
        void (*eval_fn)(DomainT&, real& ans))
{
(*eval_fn)(point,response.function_value());
response.request_vector()[0] &= mode_f;
response.response_vector()[0] |= mode_f;
}


template <class DomainT, class ResponseT>
void exec_DirectApplication_EvalFn(DomainT& point, 
        ResponseT& response,
        void (*eval_fn)(const DomainT&, double& ans))
{
double ans;
(*eval_fn)(point,ans);
response.function_value() = ans;
response.request_vector()[0] &= mode_f;
response.response_vector()[0] |= mode_f;
}


template <class DomainT, class ResponseT>
void exec_DirectApplication_EvalFn(DomainT& point, 
        ResponseT& response,
        void (*eval_fn)(const DomainT&, real& ans))
{
(*eval_fn)(point,response.function_value());
response.request_vector()[0] &= mode_f;
response.response_vector()[0] |= mode_f;
}


template <class DomainT, class ResponseT, class CArrayT>
void exec_DirectApplication_EvalFn(DomainT& point, 
        ResponseT& response,
        void (*eval_fn)(DomainT&, real&, CArrayT&))
{
(*eval_fn)(point,response.function_value(),response.constraint_values());
for (size_type i=0; i < response.request_vector().size(); i++) {
  response.request_vector()[i] &= mode_f;
  response.response_vector()[i] |= mode_f;
  }
}


template <class DomainT, class ResponseT, class CArrayT>
void exec_DirectApplication_EvalFn(DomainT& point, 
        ResponseT& response,
        void (*eval_fn)(const DomainT&, real&, CArrayT&))
{
(*eval_fn)(point,response.function_value(),response.constraint_values());
for (size_type i=0; i < response.request_vector().size(); i++) {
  response.request_vector()[i] &= mode_f;
  response.response_vector()[i] |= mode_f;
  }
}


template <class DomainT, class ResponseT, class FuncT1, class FuncT2>
void exec_DirectApplication_EvalFn(DomainT& point, 
        ResponseT& response,
        FunctionContainer1<DomainT,ResponseT,FuncT1,FuncT2>& container)
{ container.eval(point,response); }
#endif


}

#endif
