/*  _________________________________________________________________________
 *
 *  COLIN: A Common Optimization Library INterface
 *  Copyright (c) 2003, Sandia National Laboratories.
 *  This software is distributed under the GNU Lesser General Public License.
 *  For more information, see the README.html file in the top COLIN directory.
 *  _________________________________________________________________________
 */

/**
 * \file BatchEvaluator.h
 *
 * Defines the \c BatchEvaluator class.
 *
 * TODO: cleanup semantics of the relationship between the response object
 * and the BatchPoint data...
 **/
 
#ifndef colin_BatchEvaluator_h
#define colin_BatchEvaluator_h
 
#include <acro_config.h>
#include <utilib/BasicArray.h>
#include <utilib/LinkedList.h>
#include <utilib/ParameterSet.h>
#include <utilib/pvector.h>
#include <colin/AppResponseAnalysis.h>
#include <colin/AppResponse.h>
#include <colin/BatchPoint.h>

#if defined(USING_PROBE)
extern void probe();
#endif

namespace colin {

using utilib::LinkedList;
using utilib::BasicArray;
using utilib::Flush;

/**
  * A utility class for evaluating batches of function evaluations 
  * synchronously and asynchronously.
  */
template <class ProblemT, class DomainT>
class BatchEvaluator : public virtual utilib::ParameterSet, 
			 public virtual utilib::CommonIO,
			 public virtual colin::AppResponseAnalysis
{
public:

  #if !defined(DOXYGEN)
  typedef typename ProblemT::response_t response_t;
  #endif

  /// Constructor
  BatchEvaluator(ProblemT* problem_ptr_);

  /// Process options and prepare to be used
  void reset();

  /// Write out parameter information
  void write(std::ostream& os) const;

  /// Queue a point for an evaluation.
  /// NOTE: do not use this...
  void queue_point_for_eval(BatchPoint<DomainT, typename ProblemT::response_t> * p){
    assert(false);
    DEBUGPR(10, ucout << "In queue_point_for_eval";);
    p->which_iter = iter_no;
    evals.push_back(*p); 
    DEBUGPR(10, ucout << "evals.size() == " << evals.size() <<".\n";);
    //  p->id = -1; // we set this later.
  }

  /**
   * Queue a function evaluation for a given point.
   * The id parameter is a user-defined tag, which is returned when calling
   * execute_evaluations()
   */
  void queue_evaluation(DomainT& point, double convergence_factor, 
			real* value_ptr, real* cvalue_ptr, 
			int id=-1, int priority = -1,int iter = -1, 
			bool speculative = false)
    {
      evals.push_back();
      end = evals.end();
      end--;
      end->reset();
      end->id = id;
      end->priority = priority;
      end->which_iter = iter_no; // important for batch evaluation
      end->convergence_factor = convergence_factor;
      end->point.resize(point.size());
      end->point << point;
      end->ValuePtr = value_ptr;
      end->CValuePtr = cvalue_ptr;
      end->spec = speculative;
    }

  /**
   * Queue a function evaluation for a given point.
   * The id parameter is a user-defined tag, which is returned when calling
   * execute_evaluations()
   */
  void queue_evaluation(DomainT& point, double convergence_factor,
                        response_t& response,
                        int id=-1, int priority = -1,int iter = -1,
                        bool speculative = false)
    {
      evals.push_back();
      end = evals.end();
      end--;
      end->reset();
      end->id = id;
      end->priority = priority;
      end->which_iter = iter_no; // important for batch evaluation
      end->convergence_factor = convergence_factor;
      /// TODO - is there a way to avoid this memory allocation here?
      end->point = point;
      end->spec = speculative;
      end->set_response(&response);
    }

  /**
   * Queue a function evaluation for a given point.
   * The id parameter is a user-defined tag, which is returned when calling
   * execute_evaluations()
   */
  void queue_evaluation(DomainT& point, double convergence_factor, 
			int id=-1, int priority = -1,int iter = -1, 
			bool speculative = false)
	{ queue_evaluation(point,convergence_factor,0,0,id,priority,iter,
			speculative); }

  /// Compare two values to see if \a val1 is less than \a val2 - \a rho.
  bool compare(const real& val1, const real& val2, double rho) const
	{
	bool ans;
	if (rho > 0.0)
	   ans = (val1 < (val2 - rho));
        else
	   ans = (val1 < val2);
	DEBUGPR(1000, ucout << 
			"BatchEval::compare - val1=" << val1 << " val2=" << val2
			<< " rho=" << rho << " Result=" << ans << 
			" SufficientDecrease=" << ((val1 < val2) && !ans) << std::endl;);
	return ans;
	}

  /// TODO: Anne?
  int calc_and_update_necessary_points(){
    int howmany  = 0;
    typename LinkedList<BatchPoint<DomainT, response_t> >::iterator i;
    i = evals.begin();
    while (i != evals.end()){
      if (i->spec == false){
	if (i->which_iter == iter_no){
	  howmany++;
	}
	else{
	  i->spec = true;
	}
      }
      i++;
    }
    return howmany;
  }

  /// Reduce priorities. The lower the number, the better the priority, with 1
  ///   being the best priority.
  void reduce_priorities(){
    typename LinkedList<BatchPoint<DomainT, response_t> >::iterator i;
    i = evals.begin();
    while (i != evals.end()){
      i->spec = true;
      (i->priority) += 2;
      i++;
    }
  }
  
  /**
   * Queue a function evaluation for a given point.
   * The id parameter is a user-defined tag, which is returned when calling
   * execute_evaluations()
   */
  void queue_evaluation(DomainT& point, int id=-1)
    { queue_evaluation(point,1.0,id); }
  
  /// Perform a synchronous evaluation of a given point
  void perform_evaluation(DomainT& point, double convergence_factor,
			  typename ProblemT::response_t& response,
			  real& ans, real& cval)
    {
      problem_ptr->Eval(point,response,mode);
      compute_response_info(response,problem_ptr->state->constraint_lower_bounds,problem_ptr->state->constraint_upper_bounds,convergence_factor,ans,cval);
      DEBUGPR(1000, ucout << "BatchEvaluator::perform_evaluation - ans = " << ans << " cval = " << cval << std::endl;);
      DEBUGPR(1000, ucout << "BatchEvaluator::perform_evaluation - point = " << point << " response = " << response << std::endl;);
    }
  
    /// Perform a synchronous evaluation of a given point
    void perform_evaluation(DomainT& point,
			    typename ProblemT::response_t& response,
			    real& ans, real& cval)
      { perform_evaluation(point,1.0,response,ans,cval); }
    
    /**
     * Perform the execution, and return the best point found.  This may not
     * be the best point that was queued if a greedy execution is used
     * (e.g. async).  The data response and ans is only updated if an improving
     * point is found.  Returns true when an improving point is found.
     */
    bool execute_evaluations(typename ProblemT::response_t& response,
			     real& ans, real& cval, DomainT& point, double rho)
      {
	int tmp;
	return execute_evaluations(response,ans,cval,point,tmp,rho);
      }
    
    /**
     * Perform the execution, and return the best point found.  This may not
     * be the best point that was queued if a greedy execution is used
     * (e.g. async).  The data response and ans is only updated if an improving
     * point is found.  The value of id is the identification of the improving
     * point, and -1 if no improving point is found.  Returns true when an
     * improving point is found.
     */
    bool execute_evaluations(typename ProblemT::response_t& response, 
			     real& ans, real& cval, DomainT& point, int& id,
			     double rho);
    
    /// Execute all evaluations
    void execute_evaluations(bool clear_flag=true);

    ///
    /// Clear out all evaluations that we can, attempting to terminate
    /// evaluations where possible.  If termination fails, then
    /// we mark the remaining evaluations as 'dead'.
    ///
    /// NOTE: Validation can be screwed up in contexts where extant
    /// evaluations are still being processed (e.g. due to a former
    /// optimization run, which terminated without synchronizing all
    /// evaluations.)
    ///
    void clear_evaluations(bool validate_ids=false);
    
    ///
    bool validate_ids;

  protected:

    #if !defined(DOXYGEN)
    enum batch_enum {batch_all=0, batch_sequential=1, 
		     batch_async=2, batch_pri= 3};
    #endif
  
    /// Batch string
    std::string batch_str;

    /// Batch id
    batch_enum batch_id;
  
    /// Number of active servers.
    unsigned int active_servers;
  
    /// Maximum number of evaluations allowed on each server.
    unsigned int max_server_load;

    /// Total number of servers.
    unsigned int num_servers;

    /// The iteration that of evaluations.
    int iter_no;

    /// Linked lists used for asynch batch processing
    LinkedList<BatchPoint<DomainT, typename ProblemT::response_t> > evals;

    /// Index for async evals.
    BasicArray<int> async_ndx;

    /// Array of iterators for async evals.
    BasicArray<typename LinkedList<BatchPoint<DomainT, typename ProblemT::response_t> >
      ::iterator> async_itr;

    /// Print debugging information.
    void virt_debug_io(std::ostream& os, const bool finishing, const int io_level);
 
  private:

    /// Iterator for the current batch
    typename LinkedList<BatchPoint<DomainT, typename ProblemT::response_t> >::iterator batch_curr;
  
    /// Iterator for the end of the current evals.
    typename LinkedList<BatchPoint<DomainT, typename ProblemT::response_t> >::iterator end;
  
    /// Iterator for the best point in the current evals.
    typename LinkedList<BatchPoint<DomainT, typename ProblemT::response_t> >::iterator bestpt;
  
    /// Pointer to an OptProblem object
    ProblemT* problem_ptr;

    /// The mode of evaluation.
    int mode;

  };



  template <class ProblemT, class DomainT>
  BatchEvaluator<ProblemT,DomainT>::BatchEvaluator(ProblemT* problem_ptr_)
    :
    batch_str("sequential"),
    max_server_load(1),
    iter_no(0),
    problem_ptr(problem_ptr_),
    mode(0)
  {
  validate_ids=false;

    utilib::ParameterSet::create_parameter("batch_eval",batch_str,
	"<str>","sequential",
	"Defines how algorithms are parallelized when processing batches of\n"
	"\t  function evaluations.  A general mechanism for parallelism is\n"
	"\t  supported which all algorithms use when two or more function\n"
	"\t  evaluations could be computed simultaneously.\n"
	"\t  sequential: Function evaluations are performed sequentially in a\n"
	"\t     randomized order until an improving point is generated or all of\n"
	"\t     them have been performed.\n"
	"\t  all: All function evaluations are performed and the best is used.\n"
	"\t       (Alias: 'sync').\n"
	"\t  async: Function evaluations are performed asynchronously.\n"
	"\t     Evaluations are recorded until an improving point is generated or\n"
	"\t     all of them have completed.  If an improving point is generated,\n"
	"\t     then an attempt is made to terminate the remaining spawned\n"
	"\t     evaluations."
	);

    utilib::ParameterSet::create_parameter("max_server_load",max_server_load,
	"<unsigned int>","5",
	"Defines the maximum number of evaluations that can be simultaneously\n"
	"\t  spawned on each evaluation server.  When this is small, the\n"
	"\t  asynchronous evaluations are performed 'conservatively', which is\n"
	"\t  good when evaluations cannot be effectively terminated.  When this is\n"
	"\t  large, asynchronous evaluations are performed 'aggressively', which\n"
	"\t  is good when evaluations are quick or when they can be terminated."
	);

    evals.validate_flag=true;
  }


  template <class ProblemT,class DomainT>
  void BatchEvaluator<ProblemT,DomainT>::reset()
  {
    if (!problem_ptr) return;

    if ((batch_str == "all") || (batch_str == "sync"))
      batch_id = batch_all;
    else if (batch_str == "sequential")
      batch_id = batch_sequential;
    else if (batch_str == "async")
      batch_id = batch_async;
    else if (batch_str == "prioritized")  // %%%% added 6/07/04  pls
      batch_id = batch_pri;
    else
      EXCEPTION_MNGR(std::runtime_error,"BatchEvaluator::reset - unknown batch_eval mode:" << batch_str);

    batch_curr = evals.begin();
    end  = evals.end();
    while (batch_curr != end){
      batch_curr = evals.erase(batch_curr);
    }

    // %%%% NOTE:  this is how to find how many servers are out there.  
    //  pls 6/7/04
    //  the max_server_load field is a member of the 
    //  BatchEvaluator class, though, and
    //  is at this writing hard wired to 5.  Is this going to be a problem?
    num_servers = max_server_load;
    if (problem_ptr->num_evaluation_servers() > 1)
       num_servers *= problem_ptr->num_evaluation_servers();
    DEBUGPR(100,ucout << "maxservload=" << max_server_load << " numservers=" << std::max(problem_ptr->num_evaluation_servers(),(unsigned int)1) << std::endl;);
    active_servers=0;
    if (problem_ptr->numNonlinearConstraints() > 0)
      mode = colin::mode_f | colin::mode_cf;
    else
      mode = colin::mode_f;
  }



  template <class ProblemT,class DomainT>
  void BatchEvaluator<ProblemT,DomainT>::execute_evaluations(bool clear_flag)
  {
  if (evals.size() == 0) return;
  if (mode == 0)
     EXCEPTION_MNGR(std::runtime_error, "BatchEvaluator::execute_evaluations - bad mode value: 0");

  batch_curr = evals.begin();
  end  = evals.end();
  //
  // Launch all evals
  //
  while (batch_curr != end) {
#if defined(USING_PROBE)
    probe();
#endif
    //
    // Note: we ignore evaluations that we have already spawned,
    //   but which we haven't gotten information on yet.
    //
    if (batch_curr->spawned == false)
       problem_ptr->AsyncEval(batch_curr->point,batch_curr->priority,
			   batch_curr->response,mode);
    batch_curr++;
    }
  //
  // Synchronize evals
  //
  problem_ptr->synchronize();
  //
  // Compute response information
  //
  batch_curr = evals.begin();
  while (batch_curr != end) {
    if (batch_curr->ignore == false)
       compute_response_info(*(batch_curr->response),
		problem_ptr->state->constraint_lower_bounds,
		problem_ptr->state->constraint_upper_bounds,
		batch_curr->convergence_factor,
		batch_curr->value(), batch_curr->cvalue());
    batch_curr++;
    }
  //
  // Empty the evals list
  //
  // Note: we don't need to validate these ids, since that logic
  // uses the server data structure, which isn't needed here...
  //
  if (clear_flag)
     clear_evaluations(false);
  }


  template <class ProblemT,class DomainT>
  bool BatchEvaluator<ProblemT,DomainT>::execute_evaluations(
			typename ProblemT::response_t& response,
			real& ans, real& cval,
			DomainT& point, int& best_id, double rho)
  {
    if (evals.size() == 0) return false;
    batch_enum batch_used = batch_id;
    if (batch_used != batch_pri){
      bestpt = evals.begin();
    }
    batch_curr = evals.begin();
    end  = evals.end();
    best_id = -1;

    //
    // If you only have one evaluation, compute it 'sequentially'
    // If you only have one evaluation server, then async evaluation is
    // serial.
    if ((batch_id == batch_async) &&
	((evals.size() == 1))){
	 // || (problem_ptr->num_evaluation_servers() == 1)))
      batch_used = batch_sequential;
    }
#if defined(USING_PROBE)
      probe();
#endif
    //
    if (batch_used == batch_all) {
      DEBUGPR(100,ucout << "batch_all evaluation: " << 
			     evals.size() << std::endl);
      execute_evaluations(false);
      //
      // Find best eval
      //
      batch_curr = evals.begin();
      bestpt = evals.begin();
      compute_response_info(*(bestpt->response), 
				problem_ptr->state->constraint_lower_bounds,
				problem_ptr->state->constraint_upper_bounds,
				bestpt->convergence_factor,
				bestpt->value(),bestpt->cvalue());
      if (batch_curr != end)
	batch_curr++;
      while (batch_curr != end) {
	compute_response_info(*(batch_curr->response),
		problem_ptr->state->constraint_lower_bounds,
		problem_ptr->state->constraint_upper_bounds,
		batch_curr->convergence_factor,
		batch_curr->value(), batch_curr->cvalue());
	DEBUGPR(100,ucout << "batch_all evaluation: " << 
			     batch_curr->value() << 
			     " ::: " << batch_curr->point << std::endl;);

	if ((batch_curr->value()) < (bestpt->value())) {
	    evals.erase(bestpt);
	    bestpt = batch_curr;
	    bestpt->evaluated=true;
	    batch_curr++;
	  }
	  else {
	    batch_curr = evals.erase(batch_curr);
	  }
      }
      //
      // If (best != end) then we have found a new point.
      // If compare(best->value,ans,rho) is true, then we have a sufficient
      //    decrease in this point compared with the previous iteration.
      // Otherwise, we haven't been successful, so we erase 'best' from 
      // 'evals' and reset 'best'.
      //
      if ((bestpt != end) && (!compare(bestpt->value(),ans,rho))) {
	// %%% why >= and not > ????
	evals.erase(bestpt);
	bestpt = end;
      }
    }

    // %%%% Note---pls--- adapt this for batch_pri, add batch_pri to list.
    else if (batch_used == batch_async) {

#if defined(USING_PROBE)
      probe();
#endif
      bestpt = evals.end();
      bool finished=false;
      unsigned int total_jobs = evals.size();
      unsigned int completed_jobs = 0;
      DEBUGPR(100, ucout << "Starting async: evals.size=" << evals.size() << " queued=" << problem_ptr->num_queued_evaluations() << " active_servers=" << active_servers << " num_servers=" << num_servers << std::endl;);
	async_ndx.resize(std::max(async_ndx.size(),(size_type)total_jobs));
      async_itr.resize(async_ndx.size());
      while ((completed_jobs < total_jobs) && !finished) {
	//
	// Launch additional evals if we there are more points and idle servers
        batch_curr = evals.begin();
	while ((batch_curr != end) && (active_servers < num_servers)) {
	  if (batch_curr->spawned == false) {
#if defined(USING_PROBE)
		  probe();
#endif
	    problem_ptr->AsyncEval(batch_curr->point,batch_curr->priority,batch_curr->response, mode);
	    async_ndx[active_servers]   = problem_ptr->last_id();
	    DEBUGPR(100, ucout << "Spawned eval: " << async_ndx[active_servers] << std::endl;);
	    async_itr[active_servers++] = batch_curr;
	    batch_curr->spawned = true;
          }
	  else
	    DEBUGPR(100, ucout << "Found previously spawned eval" << std::endl; );
	  batch_curr++;
	}
	//
	// Test to see if async evals have completed
	//
	int id=-1;
	while ((id = problem_ptr->next_eval()) != -1) {
	  completed_jobs++;
	  DEBUGPR(100, ucout << "Completed job " << id << std::endl << Flush;);
	  bool valid=false;
	  unsigned int i=0;
	  for (; i<active_servers; i++)
	    if (async_ndx[i] == id) {
	      valid = true;
	      break;
	    }
	  if (validate_ids && !valid)
	    EXCEPTION_MNGR(std::runtime_error,"BatchEvaluator::execute_evaluations - bad id value returned: " << id;
		ucerr << "Active IDs:" << std::endl;
          	unsigned int i=0;
          	for (; i<active_servers; i++)
            	  ucerr << " " << i << " " << async_ndx[i] << " " << async_itr[i]->evaluated << std::endl;
		);
	  if (!valid) {
	     // If we are validating IDs, then we never get here.  Otherwise, we need to abort and not 
	     // process this ID.  Since it's not valid, one of the active servers is not working on it!
	     break;
	     }
	  //
	  // Process the id found
	  //
	  async_itr[i]->evaluated=true;
	  DEBUGPR(100,ucout << "Response Info:" << std::endl << *(async_itr[i]->response) << Flush;);
	  if (!(async_itr[i]->ignore)) {
	     compute_response_info(*(async_itr[i]->response),
			  	problem_ptr->state->constraint_lower_bounds,
			  	problem_ptr->state->constraint_upper_bounds,
				async_itr[i]->convergence_factor, 
				async_itr[i]->value(), async_itr[i]->cvalue());
	     if (compare(async_itr[i]->value(),ans,rho)) {
	       finished=true;
	       bestpt = async_itr[i];
	       break;
             }
          }
	  active_servers--;
	  async_ndx[i] = async_ndx[active_servers];
	  async_ndx[active_servers] = -1;
	  evals.erase(async_itr[i]);
	  async_itr[i] = async_itr[active_servers];
	  async_itr[active_servers] = end;
	}
      }
    //
    // At this point, we have an improving point.  However, we may not
    // be able to terminate all of the remaining jobs.  Thus, we
    // terminate as many as we can.
    //
    clear_evaluations();
    }

//===========================================================================
//  %%%% added 6/04 pls
//===========================================================================

    else if (batch_used == batch_pri) {
      bool newbest = false;
      utilib::pvector<DomainT>  points;
      utilib::pvector<int *> task_priorities;
      utilib::pvector<response_t * >  response_vec;
      utilib::pvector<int> id_vec;

      // best = evals.end();
      bool finished=false;
      unsigned int total_jobs = evals.size();
      int required_jobs = calc_and_update_necessary_points();
      unsigned int completed_jobs = 0;
      int req_jobs_done = 0;
      DEBUGPR(100, ucout << "Starting pri: evals.size=" << evals.size() 
	      << " queued=" << problem_ptr->num_queued_evaluations() 
	      << " active_servers=" << active_servers << std::endl;);
      async_ndx.resize(std::max(async_ndx.size(),(size_type)(total_jobs+1)));
      async_itr.resize(async_ndx.size());
      // count how many of the evals are new this go-round.
      // we want to send only the new points to be eval'd

      // reset the async.ndx array.
      int foo;
      for(foo = 0; foo < (int)async_ndx.size(); foo++){
	async_ndx[foo] = -1;
      } 
      foo = 0;
      batch_curr = evals.begin();
      while(batch_curr != evals.end()){
	if(batch_curr->which_iter == iter_no){  // new point
	  foo++;
	}
	else{                             // old point
	  assert(batch_curr->id >= 0 && batch_curr->id < (int)async_ndx.size());
	  async_ndx[batch_curr->id] = batch_curr->id;
	}
	batch_curr++;
      }

      points.resize(foo);
      task_priorities.resize(foo);
      response_vec.resize(foo);
      id_vec.resize(foo);

      batch_curr = evals.begin();
      while( (batch_curr != evals.end())
	     && (batch_curr->which_iter != iter_no) ){
	batch_curr++;	
      }
      // We should add a total of foo new points.  When we leave
      //  this for-loop, holder should equal foo.
      int holder = 0;
      int q = 0;
      for(q = 0; 
	  (q < (int)async_ndx.size()) && (batch_curr != evals.end()) && 
	    (holder < foo); 
	  q++){
	if(async_ndx[q] == -1){
	  while(batch_curr->which_iter != iter_no && batch_curr != evals.end())
	    { 
	      batch_curr++;
	    }
	  if(batch_curr != evals.end()){
	    batch_curr->response->info->id = q;
	    points[holder] = (batch_curr->point);
	    task_priorities[holder] =  &(batch_curr->priority);
	    response_vec[holder] = batch_curr->response;
	    // cout<< "\nadddress of batch_curr->response == " 
	    //<< &(batch_curr->response) <<std::endl;
	    async_itr[q] = batch_curr;
	    async_ndx[q] = q;
	    async_itr[q]->id = q;
	    holder++;
	    batch_curr++;
	  } // if
	} // if there's a free spot here
      } // for q
      assert(holder == foo);
      ucout << "\nq == "<< q ;

      ucout<<"\nGoing in from BatchEval, the response vec id's look like this:\n";
      for(int qw = 0; qw < foo; qw++){
	response_vec[qw]->info->id_generate=false;
	id_vec[qw] =  response_vec[qw]->info->id;
	ucout<< response_vec[qw]->info->id <<"  ";
      }
      ucout << "\nSending "<< points.size()<<" points off for evaluation.";
#if defined (USING_PROBE)
      probe();
#endif
      problem_ptr->AsyncEval(points, task_priorities, 
			    	     response_vec, mode, id_vec);
      
      batch_curr = evals.begin();
      finished = false;
      while ((completed_jobs < total_jobs) && !finished) {
	//cerr<<"  blah  ";
	//
	// Test to see if async evals have completed
	//
	int id=-1;
	//cerr<<"  blah2  ";
	while ((id = problem_ptr->next_eval()) != -1) {
	  // We need to come up with a way to figure out which jobs are
	  //  finishing, because for certain purposes we are most concerned
	  //  about the non-spec jobs.  We need to match the outgoing id to
	  //  the one that comes in from the calls to next_eval and last_id
	  //  and so on.  
	  //cerr<<"  blah3  ";
	  completed_jobs++;
	  DEBUGPR(100, ucout << "****Completed job " << id << std::endl;);
	  DEBUGPR(100, ucout << "****Completed jobs==  " 
		  << completed_jobs << std::endl;);
	  bool valid=false;
	  unsigned int i=0;
	  for (i = 0; i< async_ndx.size(); i++){
	    if (async_ndx[i] == id) {
	      valid = true;
	      ucout<< "Point returned: "<< async_itr[i]->response->function_value(); 
	      ucout<< " with i== "<<i<<std::endl;
	      break;
	    }
	  }
	  if (validate_ids && !valid){
	    ucout<<" \nasync_ndx: \n";
	    for(int w = 0; w< (int)async_ndx.size(); w++){
	      ucout<<" "<<async_ndx[w]<<"  ";
	    }
	    ucout<<std::endl;
	    if(async_ndx.size() > i){
	      ucout<< "\ni == "<< i <<", async_ndx[i] == "
		  << async_ndx[i]
		  << ",  and async_ndx.size() == "
		  << async_ndx.size()<<std::endl;
	    }
	    else{
	      ucout<< "\ni == "<< i
		  << " and async_ndx.size() == "
		  << async_ndx.size()<<std::endl;
	    }
	    EXCEPTION_MNGR(std::runtime_error,
             "PatternSearch::execute_evaluations - bad id value returned: " 
			   << id);
	  }
	  //
	  // Process the id found
	  //
	  // NOTE::: %%%%%% We have to be sure that  if
	  //   there is improvement, best is
	  //   at the beginning of the list for later. 
	  //   if no improving point, set best to end.
	  async_itr[i]->evaluated=true;
	  if(!(async_itr[i]->spec)
	     && (async_itr[i]->which_iter == iter_no)){
	    req_jobs_done++;
	  }
	  // %%% ACK!!! we need this response obj to be a pointer---
	  // otherwise the information in it will not persist.  
	  //  This is causing huge problems because we keep getting
	  //  a "value" of zero for anything that was sent off
	  compute_response_info(*(async_itr[i]->response),
			  	problem_ptr->state->constraint_lower_bounds,
			  	problem_ptr->state->constraint_upper_bounds,
				async_itr[i]->convergence_factor, 
				async_itr[i]->value(), async_itr[i]->cvalue());
	  ucout<<"\nasync_itr[i]->value == "<< async_itr[i]->value()
	      <<", id number == "<< async_itr[i]->response->info->id
	      <<", resp value == "<< async_itr[i]->response->function_value()
	      <<",\n address is "<< async_itr[i]->response
	      << ", and ans == "<<ans<<std::endl;
	  if (compare(async_itr[i]->value(),ans,rho)) {
	    ucout<<"@@@@@@@New best point!"<<std::endl;
	    finished=true;
	    bestpt = async_itr[i];
	    newbest = true;
	    break;
          }
	  else{
	    evals.erase(async_itr[i]);
	    async_itr[i] = NULL;
	    async_ndx[i] = -1;
	  }
	}
	if(req_jobs_done >= required_jobs){  
	  finished = true;
	}
      }
      if(!newbest){
	bestpt = evals.end();
      }
      // increment the iteration number.
      iter_no++;
    }

    else { // batch_d == batch_sequential
      //
      // Consider each evaluation sequentially
      //
      DEBUGPR(1000, ucout << "Batch_Eval::sequential" << std::endl;);
      bestpt = evals.end();
      while (batch_curr != end) {
        if (batch_curr->spawned == false) {
	   problem_ptr->Eval(batch_curr->point, *(batch_curr->response), mode);
	   compute_response_info(*(batch_curr->response),
			      problem_ptr->state->constraint_lower_bounds,
			      problem_ptr->state->constraint_upper_bounds,
			      batch_curr->convergence_factor, 
			      batch_curr->value(),
			      batch_curr->cvalue());
	   DEBUGPR(10, ucout << "Trial Pt Value: " << batch_curr->value() << std::endl;);
	   DEBUGPR(1000, ucout << "Trial Pt: " << batch_curr->point << std::endl;);
	
	   batch_curr->evaluated=true;
	   if (compare(batch_curr->value(),ans,rho)) {
	     bestpt = batch_curr;
	     batch_curr++;
	     break;
           }
	   else{
	     batch_curr = evals.erase(batch_curr);
           }
	}
        else batch_curr++;
      }
      //
      // Remove the rest of the evals that haven't been spawned.
      //
      while (batch_curr != end){
        if (batch_curr->spawned == false)
	   batch_curr = evals.erase(batch_curr);
        else
	   batch_curr++;
      }
    }

    if (bestpt != end) {
       ans = bestpt->value();
       cval = bestpt->cvalue();
       response << *(bestpt->response);
       point << bestpt->point;
       best_id = bestpt->id;
       DEBUGPR(100, ucout << "Best Pt: Value=" << ans << " Point: " 
			  << point << std::endl;);
 
       }
    DEBUGPR(100, ucout << "Cleanup Evals: evals.size() = " << evals.size() << 
			  std::endl;);
      
      /*        
      if(batch_used != batch_pri
	 && batch_used != batch_async ){
	if (evals.size() > 0) {
	  cerr<< "\n*)*)*)*)*)*)*)*)*)   ACK?   "<<std::endl;
	  DEBUGPR(100, ucout << "Evals: " << evals << std::endl;)
	  EXCEPTION_MNGR( std::runtime_error, 
	    "BatchEvaluator::execute_evaluations - Should have no evals left when the iteration failed.");
	}
      }
      */
      
      
      if (batch_used == batch_pri) {
	batch_curr = evals.begin();
	while (batch_curr != end){
	  if(batch_curr->priority < -1){  
	    batch_curr = evals.erase(batch_curr);
	  }
	  else batch_curr++;
	} // while	
	reduce_priorities();
      }

  //
  // We return true if a best point was found.  However, we still need
  // to remove this from the list before we go!
  //
  bool status = (bestpt != end);
  if (status && (evals.size() > 0) && (batch_used != batch_async)) {
     evals.erase(bestpt);
     bestpt = end;
     }
  return status;
  }


  template <class ProblemT, class DomainT>
  void BatchEvaluator<ProblemT,DomainT>::write(std::ostream& os) const
  {
    os << "batch_mode\t\t";
    switch (batch_id) {

    case batch_sequential:
      os << "sequential";
      os << "\t# Function evaluations are performed sequentially in a" << std::endl;
      os << "\t\t\t\t# randomized order until an improving point is generated or all of\n";
      os << "\t\t\t\t# them have been performed.\n";
      break;

    case batch_all:
      os << "all";
      os << "\t# All function evaluations are performed and the best is used.\n";
      break;

    case batch_async:
      os << "async";
      os << "\t# Function evaluations are performed asynchronously.\n";
      os << "\t\t\t\t# Up to the 'max_server_load' evaluations are spawned on each evaluation\n";
      os << "\t\t\t\t# server.  Evaluations are recorded until an improving point is generated or all of them\n";
      os << "\t\t\t\t# have completed.  If an improving point is generated, then an attempt\n";
      os << "\t\t\t\t# is made to terminate the remaining spawned evaluations.\n";
      break;

    default:
      break;
    }


    os << "max_server_load\t" << max_server_load << std::endl;
  }


  template <class ProblemT, class DomainT>
  void BatchEvaluator<ProblemT,DomainT>::virt_debug_io(std::ostream& os, 
								 const bool , const int )
  {
    switch (batch_id) {
    case batch_sequential:
      os << "\tUsing \"sequential\" batch evaluation:"<<std::endl;
      os << "\t\tEvaluate one point at a time." << std::endl;
      break;
    case batch_all:
      os << "\tUsing \"all\" batch evaluation:"<<std::endl;
      os << "\t\tEvaluate all points before selecting the best." << std::endl;
      break;
    case batch_async:
      os << "\tUsing \"async\" batch evaluation:"<<std::endl;
      os << "\t\tAsynchronously spawn points and keep the first" << std::endl;
      os << "\t\timproving point." << std::endl;
      break;
    case batch_pri:
      os << "\tUsing \"pri\" batch evaluation:"<<std::endl;
    };
  }


template <class ProblemT, class DomainT>
void BatchEvaluator<ProblemT,DomainT>::clear_evaluations(bool _validate_ids)
{
_validate_ids = _validate_ids || validate_ids;
//
// Process other evaluations that may have terminated
//
int id=-1;
while ((id = problem_ptr->next_eval()) != -1) {
  DEBUGPR(100, ucout << "Completed job " << id << std::endl << Flush;);
  bool valid=false;
  unsigned int i=0;
  for (; i<active_servers; i++)
    if (async_ndx[i] == id) {
       valid = true;
       break;
       }
  if (!valid) {
     if (_validate_ids) {
        EXCEPTION_MNGR(std::runtime_error,"BatchEvaluator::clear_evaluations - bad id value returned: " << id;);
     }
     else
        continue;
     }
  //
  // Process the id found
  //
  async_itr[i]->evaluated=true;
  DEBUGPR(100,ucout << "Response Info:" << std::endl << *(async_itr[i]->response) << Flush;);
  compute_response_info(*(async_itr[i]->response),
			  	problem_ptr->state->constraint_lower_bounds,
			  	problem_ptr->state->constraint_upper_bounds,
				async_itr[i]->convergence_factor, 
				async_itr[i]->value(), async_itr[i]->cvalue());
  active_servers--;
  async_ndx[i] = async_ndx[active_servers];
  async_ndx[active_servers] = -1;
  evals.erase(async_itr[i]);
  async_itr[i] = async_itr[active_servers];
  async_itr[active_servers] = end;
  }
//
// Terminate as many of the evaluations as we can
//
unsigned int i=0;
while (i < active_servers) {
  bool status = (async_itr[i]->evaluated 
			  || problem_ptr->terminate_eval(async_ndx[i]));
  if (status) {
      active_servers--;
     async_ndx[i] = async_ndx[active_servers];
     async_ndx[active_servers] = -1;
     evals.erase(async_itr[i]);
     async_itr[i] = async_itr[active_servers];
     async_itr[active_servers] = end;
     }
  else
     i++;
  }
//
// Delete all unspawned jobs
//
batch_curr = evals.begin();
while (batch_curr != end){
  if (!batch_curr->spawned) {
     batch_curr = evals.erase(batch_curr);
     }
  else batch_curr++;
  }	
//
// Make the rest of the evaluations 'dead'
if (active_servers > 0) {
   unsigned int i=0;
   while (i < active_servers) {
     problem_ptr->ignore_eval(async_ndx[i]);
     async_itr[i]->ignore=true;
     i++;
     }
   }
active_servers=0;

batch_curr = evals.begin();
end  = evals.end();
while (batch_curr != end){
  batch_curr = evals.erase(batch_curr);
  }

}


} // namespace colin

#endif
