/*
================================================================================
    PROJECT:

        John Eddy's Genetic Algorithms (JEGA)

    CONTENTS:

        Implementation of class BitManipulator.

    NOTES:

        See notes of BitManipulator.hpp.

    PROGRAMMERS:

        John Eddy (jpeddy@sandia.gov) (JE)

    ORGANIZATION:

        Sandia National Laboratories

    COPYRIGHT:

        See the LICENSE file in the top level JEGA directory.

    VERSION:

        1.0.0

    CHANGES:

        Thu Jun 12 10:11:58 2003 - Original Version (JE)

================================================================================
*/




/*
================================================================================
Document This File
================================================================================
*/
/** \file
 * \brief Contains the implementation of the BitManipulator class.
 */


/*
================================================================================
Includes
================================================================================
*/
// JEGAConfig.hpp should be the first include in all JEGA files.
#include <../Utilities/include/JEGAConfig.hpp>

#include <ostream>
#include <BitManipulator.hpp>
#include <utilities/include/Math.hpp>
#include <../Utilities/include/Design.hpp>
#include <../Utilities/include/Logging.hpp>
#include <../Utilities/include/DesignTarget.hpp>
#include <utilities/include/EDDY_DebugScope.hpp>
#include <../Utilities/include/DesignVariableInfo.hpp>

/*
================================================================================
Namespace Using Directives
================================================================================
*/
using namespace std;
using namespace JEGA;
using namespace JEGA::Logging;
using namespace JEGA::Utilities;
using namespace eddy::utilities;







/*
================================================================================
Begin Namespace
================================================================================
*/
namespace JEGA {
    namespace Algorithms {










/*
================================================================================
Static Member Data Definitions
================================================================================
*/







/*
================================================================================
Mutators
================================================================================
*/








/*
================================================================================
Accessors
================================================================================
*/








/*
================================================================================
Public Methods
================================================================================
*/


eddy::utilities::int64_t
BitManipulator::ConvertToShiftedInt(
    const Design& des,
    std::size_t dv
    ) const
{
    EDDY_FUNC_DEBUGSCOPE
    return ConvertToShiftedInt(des.GetVariableRep(dv), dv);
}

eddy::utilities::int64_t
BitManipulator::ConvertToShiftedInt(
    double val,
    std::size_t dv
    ) const
{
    EDDY_FUNC_DEBUGSCOPE
    EDDY_ASSERT(_mins.size() > dv);
    EDDY_ASSERT(_mults.size() > dv);

    return static_cast<eddy::utilities::int64_t>(
        Math::Round((val+_mins.at(dv))*_mults.at(dv)));
}

double
BitManipulator::ConvertFromShiftedInt(
    eddy::utilities::int64_t val,
    std::size_t dv
    ) const
{
    EDDY_FUNC_DEBUGSCOPE
    EDDY_ASSERT(_mins.size() > dv);
    EDDY_ASSERT(_mults.size() > dv);

    return val/_mults.at(dv) - _mins.at(dv);
}

eddy::utilities::uint8_t
BitManipulator::CountBits(
    eddy::utilities::int64_t of,
    bool withVal
    )
{
    EDDY_FUNC_DEBUGSCOPE

    eddy::utilities::uint8_t ret = 0;
    eddy::utilities::uint8_t want = withVal ? 1 : 0;

    for(int i=0; i<64; ++i)
        ret += (((of << i) & 1) == want) ? 1 : 0;

    return ret;
}

eddy::utilities::uint32_t
BitManipulator::GetTotalNumOfBits(
    ) const
{
    EDDY_FUNC_DEBUGSCOPE

    vector<eddy::utilities::uint16_t>::const_iterator it(_nbits.begin());
    eddy::utilities::uint32_t ret = 0;
    for(; it!=_nbits.end(); ++it) ret += *it;
    return ret;
}

void
BitManipulator::PrintGenome(
    ostream& stream,
    const Design& des
    ) const
{
    EDDY_FUNC_DEBUGSCOPE

    std::size_t ndv = _target.GetNDV();

    if(ndv == 0) return;

    const DesignVariableInfoVector& dvinfos = _target.GetDesignVariableInfos();
    DesignVariableInfoVector::const_iterator it(dvinfos.begin());
    DesignVariableInfoVector::const_iterator e(dvinfos.end());

    for(std::size_t dv=0; dv<(ndv-1); ++dv)
    {
        PrintBits(stream, des, dv);
        stream << ' ';
    }

    // now do the last one without the space.
    PrintBits(stream, des, ndv-1);
}

void
BitManipulator::PrintBits(
    ostream& stream,
    const JEGA::Utilities::Design& des,
    std::size_t dv
    ) const
{
    EDDY_FUNC_DEBUGSCOPE

    BitManipulator::PrintBits(
        stream, ConvertToShiftedInt(des, dv), 0, GetNumberOfBits(dv)-1
        );
}


void
BitManipulator::PrintBits(
    ostream& stream,
    eddy::utilities::int64_t val,
    eddy::utilities::uint16_t lobit,
    eddy::utilities::uint16_t hibit
    )
{
    EDDY_FUNC_DEBUGSCOPE
    EDDY_ASSERT(hibit < 64);
    EDDY_ASSERT(lobit <= hibit);

    for(eddy::utilities::int32_t i=hibit;
        i>=static_cast<eddy::utilities::int32_t>(lobit); --i)
            stream << ((val>>i)&1);
}

bool
BitManipulator::AreContentsCurrent(
    ) const
{
    EDDY_FUNC_DEBUGSCOPE

    // get the needed info items
    const DesignVariableInfoVector& dvinfos =
        _target.GetDesignVariableInfos();

    // now loop through the design variable infos and extract
    // the needed information.
    DesignVariableInfoVector::const_iterator it(dvinfos.begin());

    for(DoubleVector::size_type i=0; it!=dvinfos.end(); ++it, ++i)
    {
        // compute and store the multiplier / divider.
        double mult = Math::Pow(10.0, (*it)->GetPrecision());
        if(mult != _mults.at(i)) return false;

        // store the minimum representation.
        double mini = (*it)->GetMinDoubleRep();
        if(mini != _mins.at(i)) return false;

        // compute and store the number of bits for this variable.
        double temp = ((*it)->GetMaxDoubleRep()-mini) * mult;

        eddy::utilities::uint16_t nbits =
            static_cast<eddy::utilities::uint16_t>(
                Math::Floor(Math::Log(temp, 2.0)) + 1
                );

        if(nbits != _nbits.at(i)) return false;
    }
    return true;
}

void
BitManipulator::ReValidateContents(
    )
{
    EDDY_FUNC_DEBUGSCOPE

    // clear the current data.
    _nbits.clear();
    _mults.clear();
    _mins.clear();

    // get the needed info items
    const DesignVariableInfoVector& dvinfos =
        _target.GetDesignVariableInfos();

    // now loop through the design variable infos and extract
    // the needed information.
    DesignVariableInfoVector::const_iterator it(dvinfos.begin()),
                          e(dvinfos.end());

    for(; it!=e; ++it)
    {
        // compute and store the multiplier / divider.
        double mult =
            Math::Pow(10, static_cast<double>((*it)->GetPrecision()));
        _mults.push_back(mult);

        // store the minimum representation.
        double mini = (*it)->GetMinDoubleRep();
        _mins.push_back(mini);

        // compute and store the number of bits for this variable.
        double temp = ((*it)->GetMaxDoubleRep()-mini) * mult;

        eddy::utilities::uint16_t nbits =
            static_cast<eddy::utilities::uint16_t>(
                Math::Floor(Math::Log(temp, 2.0)) + 1.0
                );

        EDDY_DEBUG(
            nbits > 64, "Variable span too large for 64 bit representation."
            );

        JEGAIFLOG_II_G_F(nbits > 64, this,
                text_entry(lfatal(), "BitManipulator: Variable \"" +
                           (*it)->GetLabel() + "\"'s range is too large to "
                           "be represented by 64 bits.")
                )

        _nbits.push_back(nbits);
    }
}

/*
================================================================================
Subclass Visible Methods
================================================================================
*/








/*
================================================================================
Subclass Overridable Methods
================================================================================
*/








/*
================================================================================
Private Methods
================================================================================
*/
BitManipulator::BinaryString
BitManipulator::ToBinaryString(
    double v
    )
{
    EDDY_FUNC_DEBUGSCOPE

    v = Math::Truncate(v);

    // figure out how many bits we are going to need.
    size_t nbits = static_cast<size_t>(Math::Floor(Math::Log(v, 2.0))) + 1;

    // now prepare our return object.
    BinaryString ret(nbits, 0);

    // now load up the string.
    for(size_t b=nbits-1; b>0; --b)
    {
        eddy::utilities::uintmax_t bv = 1 << b;
        if(v >= bv)
        {
            ret.set_bit(b);
            v -= bv;
        }
        if(v == 0.0) return ret;
    }

    // now do the last bit separately.
    if(v >= 1) ret.set_bit(0);

    return ret;
}


double
BitManipulator::ToDouble(
    const BinaryString& bstr
    )
{
    EDDY_FUNC_DEBUGSCOPE

    double ret = 0.0;

    BinaryString::const_iterator e(bstr.end());
    BinaryString::size_type loc = 0;
    for(BinaryString::const_iterator it(bstr.begin()); it!=e; ++it)
        if(*it) ret += Math::Pow(2.0, loc++);

    return ret;
}






/*
================================================================================
Structors
================================================================================
*/


BitManipulator::BitManipulator(
    const DesignTarget& target
    ) :
        _nbits(),
        _mults(),
        _mins(),
        _target(target)
{
    EDDY_FUNC_DEBUGSCOPE
    ReValidateContents();
}





/*
================================================================================
End Namespace
================================================================================
*/
    } // namespace Algorithms
} // namespace JEGA
