/*
================================================================================
    PROJECT:

        John Eddy's Genetic Algorithms (JEGA)

    CONTENTS:

        Implementation of class DesignTargetImpl.

    NOTES:

        See notes of DesignTargetImpl.hpp.

    PROGRAMMERS:

        John Eddy (jpeddy@sandia.gov) (JE)

    ORGANIZATION:

        Sandia National Laboratories

    COPYRIGHT:

        See the LICENSE file in the top level JEGA directory.

    VERSION:

        2.0.0

    CHANGES:

        Tue Dec 20 08:11:48 2005 - Original Version (JE)

================================================================================
*/




/*
================================================================================
Document This File
================================================================================
*/
/** \file
 * \brief Contains the implementation of the DesignTargetImpl class.
 */




/*
================================================================================
Includes
================================================================================
*/
// JEGAConfig.hpp should be the first include in all JEGA files.
#include <../Utilities/include/JEGAConfig.hpp>

#ifdef JEGA_THREADSAFE
#include <threads/include/mutex.hpp>
#include <threads/include/mutex_lock.hpp>
#endif

#include <../Utilities/include/Design.hpp>
#include <../Utilities/include/Logging.hpp>
#include <utilities/include/EDDY_DebugScope.hpp>
#include <../Utilities/include/RegionOfSpace.hpp>
#include <../Utilities/include/ConstraintInfo.hpp>
#include <../Utilities/include/DesignTargetImpl.hpp>
#include <../Utilities/include/DesignVariableInfo.hpp>
#include <../Utilities/include/ObjectiveFunctionInfo.hpp>







/*
================================================================================
Namespace Using Directives
================================================================================
*/
JEGA_IF_THREADSAFE(using namespace eddy::threads;)
using namespace JEGA::Logging;






/*
================================================================================
Begin Namespace
================================================================================
*/
namespace JEGA {
    namespace Utilities {



/*
================================================================================
Nested Utility Class Implementations
================================================================================
*/
#ifdef JEGA_THREADSAFE

/// A class housing all mutexes used by the DesignTargetImpl.
/**
 * This gets removed if JEGA_THREADSAFE is not defined.
 */
class DesignTargetImpl::Mutexes
{
    /*
    ===========================================================================
    Member Data Declarations
    ===========================================================================
    */
    public:

        /// A mutext to protect the collection of discards.
        mutable mutex _discardMutex;

        /// A mutex to protect the collection of design variable infos.
        mutable mutex _dvInfoMutex;

        /// A mutex to protect the collection of objective function infos.
        mutable mutex _ofInfoMutex;

        /// A mutex to protect the collection of constraint infos.
        mutable mutex _cnInfoMutex;

    /*
    ===========================================================================
    Structors
    ===========================================================================
    */
    public:

        /**
         * \brief Default constructs a Mutexes object which default constructs
         *        all mutexes.
         */
        Mutexes(
            ) :
                _discardMutex(PTHREAD_MUTEX_RECURSIVE),
                _dvInfoMutex(PTHREAD_MUTEX_RECURSIVE),
                _ofInfoMutex(PTHREAD_MUTEX_RECURSIVE),
                _cnInfoMutex(PTHREAD_MUTEX_RECURSIVE)
        {
            EDDY_FUNC_DEBUGSCOPE
        }

}; // class DesignTargetImpl::Mutexes

#endif // JEGA_THREADSAFE

/*
================================================================================
Static Member Data Definitions
================================================================================
*/








/*
================================================================================
Mutators
================================================================================
*/

void
DesignTargetImpl::SetTrackDiscards(
    bool use
    )
{
    this->_trackDiscards = use;
}










/*
================================================================================
Accessors
================================================================================
*/


bool
DesignTargetImpl::GetTrackDiscards(
    ) const
{
    return this->_trackDiscards;
}






/*
================================================================================
Public Methods
================================================================================
*/








/*
================================================================================
Subclass Visible Methods
================================================================================
*/








/*
================================================================================
Subclass Overridable Methods
================================================================================
*/
const DesignDVSortSet&
DesignTargetImpl::CheckoutDiscards(
    ) const
{
    EDDY_FUNC_DEBUGSCOPE
    JEGA_IF_THREADSAFE(_mutexes->_discardMutex.lock();)
    return _discards;
}

void
DesignTargetImpl::CheckinDiscards(
    ) const
{
    EDDY_FUNC_DEBUGSCOPE
    JEGA_IF_THREADSAFE(_mutexes->_discardMutex.unlock();)
}

const DesignVariableInfoVector&
DesignTargetImpl::GetDesignVariableInfos(
    ) const
{
    EDDY_FUNC_DEBUGSCOPE
    JEGA_IF_THREADSAFE(mutex_lock l(_mutexes->_dvInfoMutex);)
    return _dvInfos;
}

const ConstraintInfoVector&
DesignTargetImpl::GetConstraintInfos(
    ) const
{
    EDDY_FUNC_DEBUGSCOPE
    JEGA_IF_THREADSAFE(mutex_lock l(_mutexes->_cnInfoMutex);)
    return _cnInfos;
}

const ObjectiveFunctionInfoVector&
DesignTargetImpl::GetObjectiveFunctionInfos(
    ) const
{
    EDDY_FUNC_DEBUGSCOPE
    JEGA_IF_THREADSAFE(mutex_lock l(_mutexes->_ofInfoMutex);)
    return _ofInfos;
}

std::size_t
DesignTargetImpl::GetNDV(
    ) const
{
    EDDY_FUNC_DEBUGSCOPE
    JEGA_IF_THREADSAFE(mutex_lock l(_mutexes->_dvInfoMutex);)
    return _dvInfos.size();
}

std::size_t
DesignTargetImpl::GetNOF(
    ) const
{
    EDDY_FUNC_DEBUGSCOPE
    JEGA_IF_THREADSAFE(mutex_lock l(_mutexes->_ofInfoMutex);)
    return _ofInfos.size();
}

std::size_t
DesignTargetImpl::GetNCN(
    ) const
{
    EDDY_FUNC_DEBUGSCOPE
    JEGA_IF_THREADSAFE(mutex_lock l(_mutexes->_cnInfoMutex);)
    return _cnInfos.size();
}

Design*
DesignTargetImpl::GetNewDesign(
    ) const
{
    EDDY_FUNC_DEBUGSCOPE
    return new Design(const_cast<DesignTargetImpl&>(*this));
}

Design*
DesignTargetImpl::GetNewDesign(
    const Design& copy
    ) const
{
    EDDY_FUNC_DEBUGSCOPE
    return new Design(copy);
}

bool
DesignTargetImpl::CheckFeasibility(
    Design& des
    ) const
{
    EDDY_FUNC_DEBUGSCOPE

    CheckSideConstraints(des);
    CheckNonSideConstraints(des);
    return des.IsFeasible();
}

bool
DesignTargetImpl::CheckSideConstraints(
    Design& des
    ) const
{
    EDDY_FUNC_DEBUGSCOPE

    // Iterating through the design variables
    // and look for out-of-bounds values
    JEGA_IF_THREADSAFE(mutex_lock l(_mutexes->_dvInfoMutex);)
    DesignVariableInfoVector::const_iterator dit(_dvInfos.begin());

    for(; dit!=_dvInfos.end(); ++dit)
    {
        if(!(*dit)->IsRepInBounds((*dit)->WhichDoubleRep(des)))
        {
            des.SetSatisfiesBounds(false);
            return false;
        }
    }

    // if we make it here, we satisfied them all.
    des.SetSatisfiesBounds(true);
    return true;
}

bool
DesignTargetImpl::CheckNonSideConstraints(
    Design& des
    ) const
{
    EDDY_FUNC_DEBUGSCOPE

    // iterate through the constraint info list and
    // check for violations
    JEGA_IF_THREADSAFE(mutex_lock l(this->_mutexes->_cnInfoMutex);)
    ConstraintInfoVector::const_iterator cit(this->_cnInfos.begin());

    for(; cit!=this->_cnInfos.end(); ++cit)
    {
        if((*cit)->GetViolationAmount(des) != 0.0)
        {
            des.SetSatisfiesConstraints(false);
            return false;
        }
    }

    // if we make it here, we satisfied them all.
    des.SetSatisfiesConstraints(true);
    return true;
}

void
DesignTargetImpl::TakeDesign(
    Design* des
    )
{
    EDDY_FUNC_DEBUGSCOPE
    EDDY_ASSERT(des != 0x0);

    if(this->_trackDiscards && des->IsEvaluated())
    {
        JEGA_IF_THREADSAFE(mutex_lock l(this->_mutexes->_discardMutex);)
        this->_discards.insert(des);
    }
    else delete des;
}

bool
DesignTargetImpl::ReclaimDesign(
    const Design& des
    )
{
    EDDY_FUNC_DEBUGSCOPE

    JEGA_IF_THREADSAFE(mutex_lock l(_mutexes->_discardMutex);)
    DesignDVSortSet::iterator it(
        _discards.find_exact(const_cast<Design*>(&des))
        );

    if(it == _discards.end()) return false;
    _discards.erase(it);
    return true;
}

void
DesignTargetImpl::RecordAllConstraintViolations(
    const Design& des
    ) const
{
    EDDY_FUNC_DEBUGSCOPE

    JEGA_IF_THREADSAFE(mutex_lock l(_mutexes->_cnInfoMutex);)
    for(ConstraintInfoVector::const_iterator it(_cnInfos.begin());
         it!=_cnInfos.end(); ++it)
            (*it)->RecordViolation(des);
}

bool
DesignTargetImpl::AddDesignVariableInfo(
    DesignVariableInfo& info
    )
{
    EDDY_FUNC_DEBUGSCOPE
    EDDY_ASSERT(&info.GetDesignTarget() == this);
    EDDY_ASSERT(_discards.empty());

    if(&info.GetDesignTarget() != this) return false;

    info.SetNumber(GetNDV());

    JEGA_IF_THREADSAFE(mutex_lock l1(_mutexes->_dvInfoMutex);)
    _dvInfos.push_back(&info);

    JEGALOG_II_G(lverbose(), this,
        ostream_entry(lverbose(), "Design Target: Design variable " +
            info.GetLabel() + " added.  ") << GetNDV() << " design variables "
            "now in target."
            )

    JEGA_IF_THREADSAFE(mutex_lock l2(_mutexes->_discardMutex);)
    _discards.flush();
    return true;
}


bool
DesignTargetImpl::AddConstraintInfo(
    ConstraintInfo& info
    )
{
    EDDY_FUNC_DEBUGSCOPE
    EDDY_ASSERT(&info.GetDesignTarget() == this);
    EDDY_ASSERT(_discards.empty());

    if(&info.GetDesignTarget() != this) return false;

    info.SetNumber(GetNCN());

    JEGA_IF_THREADSAFE(mutex_lock l1(_mutexes->_cnInfoMutex);)
    _cnInfos.push_back(&info);

    JEGALOG_II_G(lverbose(), this,
        ostream_entry(lverbose(), "Design Target: Constraint " +
            info.GetLabel() + " added.  ") << GetNCN() << " constraints "
            "now in target."
            )

    JEGA_IF_THREADSAFE(mutex_lock l2(_mutexes->_discardMutex);)
    _discards.flush();
    return true;
}

bool
DesignTargetImpl::AddObjectiveFunctionInfo(
    ObjectiveFunctionInfo& info
    )
{
    EDDY_FUNC_DEBUGSCOPE
    EDDY_ASSERT(&info.GetDesignTarget() == this);
    EDDY_ASSERT(_discards.empty());

    if(&info.GetDesignTarget() != this) return false;

    info.SetNumber(GetNOF());

    JEGA_IF_THREADSAFE(mutex_lock l1(_mutexes->_ofInfoMutex);)
    _ofInfos.push_back(&info);

    JEGALOG_II_G(lverbose(), this,
        ostream_entry(lverbose(), "Design Target: Objective " +
            info.GetLabel() + " added.  ") << GetNOF() << " objectives "
            "now in target."
            )

    JEGA_IF_THREADSAFE(mutex_lock l2(_mutexes->_discardMutex);)
    _discards.flush();
    return true;
}

RegionOfSpace
DesignTargetImpl::GetDesignSpace(
    ) const
{
    EDDY_FUNC_DEBUGSCOPE
    RegionOfSpace ret(_dvInfos.size());
    for(DesignVariableInfoVector::const_iterator it(_dvInfos.begin());
        it!=_dvInfos.end(); ++it)
            ret.SetLimits(
                (*it)->GetNumber(),
                (*it)->GetMinDoubleRep(),
                (*it)->GetMaxDoubleRep()
                );

    return ret;
}


/*
================================================================================
Private Methods
================================================================================
*/








/*
================================================================================
Structors
================================================================================
*/
DesignTargetImpl::DesignTargetImpl(
    ) :
        DesignTarget(),
        _trackDiscards(true),
        _discards(),
        _dvInfos(),
        _ofInfos(),
        _cnInfos() JEGA_COMMA_IF_THREADSAFE
        JEGA_IF_THREADSAFE(_mutexes(new Mutexes()))
{
    EDDY_FUNC_DEBUGSCOPE
}

DesignTargetImpl::~DesignTargetImpl(
    )
{
    EDDY_FUNC_DEBUGSCOPE

    _discards.flush();

    DesignVariableInfoVector::iterator dit(_dvInfos.begin());
    for(; dit!=_dvInfos.end(); ++dit) delete *dit;

    ConstraintInfoVector::iterator cit(_cnInfos.begin());
    for(; cit!=_cnInfos.end(); ++cit) delete *cit;

    ObjectiveFunctionInfoVector::iterator oit(_ofInfos.begin());
    for(; oit!=_ofInfos.end(); ++oit) delete *oit;

    JEGA_IF_THREADSAFE(delete _mutexes;)
}






/*
================================================================================
End Namespace
================================================================================
*/
    } // namespace Utilities
} // namespace JEGA

