/*  _______________________________________________________________________

    DAKOTA: Design Analysis Kit for Optimization and Terascale Applications
    Copyright (c) 2006, Sandia National Laboratories.
    This software is distributed under the GNU General Public License.
    For more information, see the README file in the top Dakota directory.
    _______________________________________________________________________ */

//- Class:        OrthogPolyApproximation
//- Description:  Implementation code for OrthogPolyApproximation class
//-               
//- Owner:        Mike Eldred

#include "OrthogPolyApproximation.H"
#include "OrthogonalPolynomial.H"
#include "NonDQuadrature.H"
#include "ProblemDescDB.H"

//#define DEBUG

// Prototype for the LAPACK subroutine used to perform linear least squares
// for obtaining the PCE coefficients by the point collocation method.
#define DGELSS_F77 F77_FUNC(dgelss,DGELSS)
#define DGGLSE_F77 F77_FUNC(dgglse,DGGLSE)

extern "C" {

void DGELSS_F77(int&, int&, int&, double*, int&, double*, int&, double*,
		double&, int&, double*, int&, int&);

void DGGLSE_F77(int&, int&, int&, double*, int&, double*, int&, double*,
		double*, double*, double*, int&, int&);

}


namespace Dakota {

OrthogPolyApproximation::
OrthogPolyApproximation(const ProblemDescDB& problem_db, const size_t& num_acv):
  BasisPolyApproximation(problem_db, num_acv),
  numExpansionTerms(problem_db.get_int("method.nond.expansion_terms"))
{
  approxOrder = problem_db.get_short("method.nond.expansion_order");
  resolve_inputs();
}


int OrthogPolyApproximation::min_coefficients() const
{
  // return the minimum number of data instances required to build the 
  // surface of multiple dimensions
  if (expansionCoeffFlag || expansionGradFlag)
    switch (expCoeffsSolnApproach) {
    case QUADRATURE: case SPARSE_GRID: case SAMPLING:
      return 1; // quadrature: (int)pow((Real)MIN_GAUSS_PTS, numVars);
      break;
    case REGRESSION:
      return numExpansionTerms;
      break;
    case -1: default: // coefficient import 
      return 0;
      break;
    }
  else
    return 0;
}


void OrthogPolyApproximation::find_coefficients()
{
  if (!expansionCoeffFlag && !expansionGradFlag) {
    Cerr << "Warning: neither expansion coefficients nor expansion coefficient "
	 << "gradients\n         are active in OrthogPolyApproximation::find_"
	 << "coefficients().\n         Bypassing approximation construction."
	 << endl;
    return;
  }

  //if (approxOrder < 0 || !numExpansionTerms)
  //  resolve_inputs();
  allocate_arrays();
#ifdef DEBUG
  gradient_check();
#endif // DEBUG

  // ----------------------------------------------------
  // Array sizing is divided into two parts:
  // > data used in all cases (allocate_arrays())
  // > data not used in the expansion import case
  //   (chaosSamples and expansionCoeffGrads sized below) 
  // ----------------------------------------------------

  // For testing of anchorPoint logic:
  //anchorPoint = currentPoints.front();
  //currentPoints.pop_front();

  size_t i, j, offset = 0, num_pts = currentPoints.entries();
  // anchorPoint, if present, is the first chaosSample.  It is handled
  // differently for different expCoeffsSolnApproach:
  //   SAMPLING:          treat it as another currentPoint
  //   QUADRATURE:        error
  //   REGRESSION: use equality-constrained least squares
  if (!anchorPoint.is_null()) {
    offset   = 1;
    num_pts += offset;
  }
  if (!num_pts) {
    Cerr << "Error: nonzero number of sample points required in "
	 << "OrthogPolyApproximation::find_coefficients()." << endl;
    abort_handler(-1);
  }
  if (expansionGradFlag && expansionCoeffGrads.empty()) {
    const SurrogateDataPoint& sdp = (anchorPoint.is_null()) ?
      *currentPoints.begin() : anchorPoint;
    size_t num_deriv_vars = sdp.response_gradient().length();
    expansionCoeffGrads.reshape_2d(numExpansionTerms, num_deriv_vars);
  }

  // calculate multivariate polynomial sets for all current points,
  // including the anchorPoint if present.
  chaosSamples.reshape(num_pts);
  if (!anchorPoint.is_null())
    chaosSamples[0]
      = get_multivariate_polynomials(anchorPoint.continuous_variables());
  List<SurrogateDataPoint>::iterator it = currentPoints.begin();
  for (i=offset; i<num_pts; i++, it++)
    chaosSamples[i] = get_multivariate_polynomials(it->continuous_variables());
  
  // calculate polynomial chaos coefficients

  switch (expCoeffsSolnApproach) {
  case QUADRATURE: {
    if (!integrationRep) {
      Cerr << "Error: pointer to NonDIntegration iterator required in "
	   << "OrthogPolyApproximation::find_coefficients()." << endl;
      abort_handler(-1);
    }
    // verify quad_order stencil matches num_pts
    const UShortArray& quad_order
      = ((NonDQuadrature*)integrationRep)->quadrature_order();
    if (quad_order.length() != numVars) {
      Cerr << "Error: quadrature order array is not consistent with number of "
	   << "variables (" << numVars << ")\n       in "
	   << "OrthogPolyApproximation::find_coefficients()." << endl;
      abort_handler(-1);
    }
    size_t num_gauss_pts = 1;
    for (i=0; i<numVars; i++)
      num_gauss_pts *= quad_order[i];
    if (num_pts != num_gauss_pts ||
	num_pts != integrationRep->weight_products().length()) {
      Cerr << "Error: number of current points (" << num_pts << ") is not "
	   << "consistent with\n       quadrature data in "
	   << "OrthogPolyApproximation::find_coefficients()." << endl;
      abort_handler(-1);
    }

#ifdef DEBUG
    for (i=0; i<numVars; i++) {
      OrthogonalPolynomial* poly_rep
	= (OrthogonalPolynomial*)polynomialBasis[i].polynomial_rep();
      for (j=1; j<=quad_order[i]; j++)
	poly_rep->gauss_check(j);
    }
#endif // DEBUG

    integration();
    break;
  }
  case SPARSE_GRID:
    if (!integrationRep) {
      Cerr << "Error: pointer to NonDIntegration iterator required in "
	   << "OrthogPolyApproximation::find_coefficients()." << endl;
      abort_handler(-1);
    }
    if (num_pts != integrationRep->weight_products().length()) {
      Cerr << "Error: number of current points (" << num_pts << ") is not "
	   << "consistent with\n       sparse grid data in "
	   << "OrthogPolyApproximation::find_coefficients()." << endl;
      abort_handler(-1);
    }

    integration();
    break;
  case REGRESSION:
    regression();
    break;
  case SAMPLING:
    expectation();
    break;
  }

  // deallocate memory that's no longer needed for evaluation of this
  // approximation (since multiple approximations may be built in sequence).
  wienerAskeyChaos.clear();
  chaosSamples.clear();
}


void OrthogPolyApproximation::resolve_inputs()
{
  // see DAKOTA reference manual for P = fn(s,p,n,r) expression
  if (approxOrder >= 0) { // default is -1

    // numExpansionTerms may be computed from (n+p)!/(n!p!)
    numExpansionTerms = 1;
    for (size_t i=1; i<=approxOrder; i++) {
      // do in 2 steps rather than 1 to avoid truncation from integer division
      numExpansionTerms *= numVars+i;
      numExpansionTerms /= i;
    }
    Cout << "Orthogonal polynomial approximation order = " << approxOrder
	 << " using full expansion of " << numExpansionTerms << " terms\n";
  }
  else if (numExpansionTerms > 0) { // default is 0
    approxOrder = 0;
    size_t full_order_expansion = 1;
    while (numExpansionTerms > full_order_expansion) {
      approxOrder++;
      // do in 2 steps rather than 1 to avoid truncation from integer division
      full_order_expansion *= numVars+approxOrder;
      full_order_expansion /= approxOrder;
    }
    Cout << "Orthogonal polynomial approximation order = " << approxOrder;
    if (numExpansionTerms == full_order_expansion)
      Cout << " using full expansion of " << numExpansionTerms << " terms\n";
    else
      Cout << " using partial expansion of " << numExpansionTerms << " terms\n";
  }
  else {
    Cerr << "Error: bad expansion specification in OrthogPolyApproximation::"
	 << "resolve_inputs()." << endl;
    abort_handler(-1);
  }
}


void OrthogPolyApproximation::allocate_arrays()
{
  // now that terms & order are known, shape some arrays.  This is done here,
  // rather than in find coefficients(), in order to support array sizing for
  // the data import case.
  if (polynomialBasis.empty()) {
    polynomialBasis.reshape(numVars);
    for (size_t i=0; i<numVars; i++)
      polynomialBasis[i] = BasisPolynomial(basisTypes[i]);
    /*
    // Could reuse objects as in InterpPolyApproximation, but this would
    // require caching {jacobiAlphas,jacobiBetas,genLagAlphas} and moving code
    // from {basis_types(),jacobi_alphas(),jacobi_betas(),
    // generalized_laguerre_alphas()} to here, which isn't yet well motivated.
    size_t i, j;
    for (i=0; i<numVars; i++) {
      // reuse prev instance via shared rep or instantiate new unique instance
      short basis_type_i = basisTypes[i];
      bool found = false;
      for (j=0; j<i; j++)
	if ( basis_type_i == basisTypes[j] && ( basis_type_i <= LAGUERRE ||
	     ( basis_type_i == JACOBI && jacobiAlphas[] == jacobiAlphas[] &&
	       jacobiBetas[] == jacobiBetas[] ) ||
	     ( basis_type_i == GENERALIZED_LAGUERRE &&
	       genLagAlphas[] == genLagAlphas[] ) ) )
	  { found = true; break; }
      polynomialBasis[i]
	= (found) ? polynomialBasis[j] : BasisPolynomial(basis_type_i);
    }
    */
  }
  if (wienerAskeyChaos.empty())
    wienerAskeyChaos.reshape(numExpansionTerms);
  if (expansionCoeffFlag && expansionCoeffs.empty())
    expansionCoeffs.reshape(numExpansionTerms);
  if (multiIndex.empty()) {
    multiIndex.reshape(numExpansionTerms);
    for (size_t i=0; i<numExpansionTerms; i++)
      multiIndex[i].reshape(numVars);

    // populate multiIndex for use in get_multivariate_polynomials() and
    // norm_squared().  Implementation follows ordering of Eq. 4.1 in
    // [Xiu and Karniadakis, 2002].
    size_t cntr = 0;
    multiIndex[cntr++] = 0;
    if (approxOrder >= 1) {
      for (size_t i=0; i<numVars && cntr<numExpansionTerms; i++, cntr++) {
	multiIndex[cntr] = 0;
	multiIndex[cntr][i] = 1;
      }
    }
    if (approxOrder >= 2) {
      for (unsigned short order_nd=2; order_nd<=approxOrder; order_nd++) {
	UShortArray terms(order_nd, 1); // # of indices = current order
	bool order_complete = false;
	while (!order_complete) {
	  // this is the inner-most loop w/i the nested looping managed by terms
	  size_t last_index = order_nd - 1, prev_index = order_nd - 2;
	  for (terms[last_index]=1;
	       terms[last_index]<=terms[prev_index] && cntr<numExpansionTerms; 
	       terms[last_index]++, cntr++) {
	    // store the orders of the univariate polynomials to be used for
	    // constructing the current multivariate basis function
	    for (size_t k=0; k<numVars; k++)
	      multiIndex[cntr][k] = terms.count(k+1);
#ifdef DEBUG
	    Cout << "terms:\n" << terms << endl;
#endif // DEBUG
	  }
	  if (cntr == numExpansionTerms)
	    order_complete = true;
	  else { // increment term hierarchy
	    bool increment_complete = false;
	    while (!increment_complete) {
	      terms[last_index] = 1;
	      terms[prev_index]++;
	      if (prev_index == 0) {
		increment_complete = true;
		if (terms[prev_index] > numVars)
		  order_complete = true;
	      }
	      else {
		last_index = prev_index;
		prev_index--;
		if (terms[last_index]<=terms[prev_index])
		  increment_complete = true;
	      }
	    }
	  }
	}
      }
    }
#ifdef DEBUG
    Cout << "multiIndex:\n" << multiIndex << endl;
#endif // DEBUG
  }
}


/** The coefficients of the PCE for the response are calculated using a
    Galerkin projection of the response against each multivariate orthogonal
    polynomial basis fn using the inner product ratio <f,Psi>/<Psi^2>, where
    inner product <a,b> is the n-dimensional integral of a*b*weighting over
    the support range of the n-dimensional (composite) weighting function.
    1-D quadrature rules are defined for specific 1-D weighting functions
    and support ranges and approximate the integral of f*weighting as the
    Sum_i of w_i f_i.  To extend this to n-dimensions, a tensor product
    quadrature rule or Smolyak sparse grid rule is applied using the product
    of 1-D weightings applied to the n-dimensional stencil of points.  It is
    not necessary to approximate the integral for the denominator numerically,
    since this is available analytically. */
void OrthogPolyApproximation::integration()
{
  if (!anchorPoint.is_null()) { // TO DO: verify this creates a problem
    Cerr << "Error: anchor point not supported for numerical integration in "
	 << "OrthogPolyApproximation::integration()." << endl;
    abort_handler(-1);
  }

  // Perform tensor-product quadrature/Smolyak sparse grid using
  // point & weight sets computed in NonDQuadrature/NonDSparseGrid
  size_t i, j, k, num_deriv_vars = expansionCoeffGrads.num_columns();
  const RealVector& wt_products = integrationRep->weight_products();
  List<SurrogateDataPoint>::iterator it;
  Real empty_r;
  RealBaseVector empty_rbv;
  for (i=0; i<numExpansionTerms; i++) {

    Real& chaos_coeff_i = (expansionCoeffFlag) ? expansionCoeffs[i] : empty_r;
    RealBaseVector& chaos_grad_i
      = (expansionGradFlag) ? expansionCoeffGrads[i] : empty_rbv;
    if (expansionCoeffFlag) chaos_coeff_i = 0.;
    if (expansionGradFlag)  chaos_grad_i  = 0.;
    for (j=0, it=currentPoints.begin(); it!=currentPoints.end(); j++, it++) {

      // sum contribution to inner product <f, Psi_j>
      //num += wt_products[j] * chaosSamples[j][i] * it->response_function();
      // sum contribution to inner product <Psi_j, Psi_j>
      //den += wt_products[j] * pow(chaosSamples[j][i], 2);

      Real wt_samp_prod = wt_products[j] * chaosSamples[j][i];
      if (expansionCoeffFlag)
	chaos_coeff_i += wt_samp_prod * it->response_function();
      if (expansionGradFlag) {
	const RealBaseVector& curr_pt_grad = it->response_gradient();
	for (k=0; k<num_deriv_vars; k++)
	  chaos_grad_i[k] += wt_samp_prod * curr_pt_grad[k];
      }
    }

    //expansionCoeffs[i] = num / den;
    const Real& norm_sq = norm_squared(i);
    if (expansionCoeffFlag)
      chaos_coeff_i /= norm_sq;
    if (expansionGradFlag)
      for (k=0; k<num_deriv_vars; k++)
	chaos_grad_i[k] /= norm_sq;
#ifdef DEBUG
    Cout << "coeff[" << i <<"] = " << chaos_coeff_i
       //<< "coeff_grad[" << i <<"] = " << chaos_grad_i
	 << " norm squared[" << i <<"] = " << norm_sq << "\n\n";
#endif // DEBUG
  }
}


/** In this case, regression is used in place of Galerkin projection.  That
    is, instead of calculating the PCE coefficients using inner product
    ratios, linear least squares is used to estimate the PCE coefficients
    which best match a set of response samples.  This approach is also known
    as stochastic response surfaces.  The least squares estimation is
    performed using DGELSS (SVD) or DGGLSE (equality-constrained) from
    LAPACK, based on the presence of an anchorPoint. */
void OrthogPolyApproximation::regression()
{
  size_t i, j, num_pts = currentPoints.entries();

  int info         = 0; // output flag from DGELSS/DGGLSE subroutines
  int num_rows_A   = num_pts;  // Number of rows in matrix A
  int num_cols_A   = numExpansionTerms; // Number of columns in matrix A
  // Matrix of polynomial terms unrolled into a vector.
  double* A_matrix = new double [num_rows_A*num_cols_A]; // "A" in A*x = b
  List<SurrogateDataPoint>::iterator it;
  bool err_flag = false;

  if (anchorPoint.is_null()) {
    // Use DGELSS for LLS soln using SVD.  Solves min ||b - Ax||_2
    // where {b} may have multiple RHS -> multiple {x} solutions.

    size_t num_deriv_vars = (expansionGradFlag)  ?
      expansionCoeffGrads.num_columns() : 0;
    size_t num_coeff_rhs  = (expansionCoeffFlag) ? 1 : 0;
    int num_rhs  = num_coeff_rhs + num_deriv_vars; // input: # of RHS vectors
    double rcond = -1.; // input:  use macheps to rank singular vals of A
    int rank     =  0;  // output: effective rank of matrix A
    // input: vectors of response data that correspond to samples in matrix A.
    double* b_vectors = new double [num_rows_A*num_rhs]; // "b" in A*x = b
    // output: vector of singular values, dimensioned for overdetermined system
    double* s_vector  = new double [num_cols_A];

    // Get the optimal work array size
    int lwork    = -1; // special code for workspace query
    double* work = new double [1]; // temporary work array
    DGELSS_F77(num_rows_A, num_cols_A, num_rhs, A_matrix, num_rows_A, b_vectors,
	       num_rows_A, s_vector, rcond, rank, work, lwork, info);
    lwork = (int)work[0]; // optimal work array size returned by query
    delete [] work;
    work = new double [lwork]; // Optimal work array

    // chaosSamples define the rectangular coefficient matrix:
    // > chaosSamples is a num_pts (rows) x numExpansionTerms (cols)
    // > The "A" matrix is a contiguous block of memory packed in column-major
    //   ordering as required by F77 for the DGELSS subroutine from LAPACK.
    //   For example, the 6 elements of A(2,3) are stored in the order:
    //   A(1,1), A(2,1), A(1,2), A(2,2), A(1,3), A(2,3).
    copy_data(chaosSamples, A_matrix, num_rows_A*num_cols_A, "fortran");

    // response data (values/gradients) define the multiple RHS which are
    // matched in the LS soln.  b_vectors is num_pts (rows) x num_rhs (cols),
    // arranged in column-major order.
    for (i=0, it=currentPoints.begin(); i<num_rows_A; i++, it++) {
     if (expansionCoeffFlag)
       b_vectors[i] = it->response_function(); // i-th point: response value
      if (expansionGradFlag) {
	const RealBaseVector& curr_pt_grad = it->response_gradient();
	for (j=0; j<num_deriv_vars; j++) // i-th point, j-th gradient component
	  b_vectors[(j+num_coeff_rhs)*num_pts+i] = curr_pt_grad[j];
      }
    }

    // Least squares computation using LAPACK's DGELSS subroutine which uses a
    // SVD method for solving the least squares problem
    info = 0;
    DGELSS_F77(num_rows_A, num_cols_A, num_rhs, A_matrix, num_rows_A, b_vectors,
	       num_rows_A, s_vector, rcond, rank, work, lwork, info);
    if (info)
      err_flag = true;

    // DGELSS returns the x solution in the numExpansionTerms x num_rhs
    // submatrix of b_vectors
    if (expansionCoeffFlag)
      copy_data(b_vectors, numExpansionTerms, expansionCoeffs);
    if (expansionGradFlag)
      for (i=0; i<numExpansionTerms; i++)
	for (j=0; j<num_deriv_vars; j++)
	  expansionCoeffGrads[i][j] = b_vectors[(j+num_coeff_rhs)*num_pts+i];

    delete [] b_vectors;
    delete [] s_vector;
    delete [] work;
  }
  else {
    // Use DGGLSE for equality-constrained LLS soln using GRQ factorization.
    // Solves min ||b - Ax||_2 s.t. Cx = d (Note: b,C switched from LAPACK docs)
    // where {b,d} are single vectors (multiple RHS not supported).

    size_t offset = 1;
    num_pts += offset;

    int num_cons = 1; // only 1 anchor point constraint
    // Vector of response values that correspond to the samples in matrix A.
    double* b_vector = new double [num_rows_A]; // "b" in A*x = b
    // Matrix of constraints unrolled into a vector
    double* C_matrix = new double [num_cols_A]; // "C" in C*x = d
    // RHS of constraints
    double* d_vector = new double [num_cons];   // "d" in C*x = d
    // Solution vector
    double* x_vector = new double [num_cols_A]; // "x" in b - A*x, C*x = d

    // Get the optimal work array size
    int lwork    = -1; // special code for workspace query
    double* work = new double [1]; // temporary work array
    DGGLSE_F77(num_rows_A, num_cols_A, num_cons, A_matrix, num_rows_A, C_matrix,
	       num_cons, b_vector, d_vector, x_vector, work, lwork, info);
    lwork = (int)work[0]; // optimal work array size returned by query
    delete [] work;
    work = new double [lwork]; // Optimal work array

    if (expansionCoeffFlag) {
      // chaosSamples define the rectangular coefficient matrix:
      // > chaosSamples is a num_pts (rows) x numExpansionTerms (cols)
      // > The "A" matrix is a contiguous block of memory packed in column-major
      //   ordering as required by F77 for the DGELSS subroutine from LAPACK.
      //   For example, the 6 elements of A(2,3) are stored in the order:
      //   A(1,1), A(2,1), A(1,2), A(2,2), A(1,3), A(2,3).
      size_t cntr = 0;
      for (j=0; j<num_cols_A; j++) // loop over columns
	for (i=offset; i<num_pts; i++) // loop over rows
	  A_matrix[cntr++] = chaosSamples[i][j];// currentPoints data, no anchor
      copy_data(chaosSamples[0], C_matrix, num_cols_A); // anchor data
      d_vector[0] = anchorPoint.response_function();    // anchor data

      // Least squares computation using LAPACK's DGGLSE subroutine which uses a
      // GRQ factorization method for solving the least squares problem
      info = 0;
      DGGLSE_F77(num_rows_A, num_cols_A, num_cons, A_matrix, num_rows_A,
		 C_matrix, num_cons, b_vector, d_vector, x_vector, work,
		 lwork, info);
      if (info)
	err_flag = true;
      copy_data(x_vector, numExpansionTerms, expansionCoeffs);
    }

    if (expansionGradFlag) {
      size_t num_deriv_vars = expansionCoeffGrads.num_columns();
      for (i=0; i<num_deriv_vars; i++) {
	// must be recopied each time since DGGLSE solves in place
	size_t cntr = 0;
	for (j=0; j<num_cols_A; j++) // loop over columns
	  for (i=offset; i<num_pts; i++) // loop over rows
	    A_matrix[cntr++] = chaosSamples[i][j];
	copy_data(chaosSamples[0], C_matrix, num_cols_A);
	// the Ax=b RHS is the currentPoints realizations of the i-th
	// gradient component
	for (j=0, it=currentPoints.begin(); j<num_rows_A; j++, it++)
	  b_vector[j] = it->response_gradient()[i];
	// the Cx=d RHS is the anchorPoint value for the i-th gradient component
	d_vector[0] = anchorPoint.response_gradient()[i];
	// solve for the PCE coefficients for the i-th gradient component
	info = 0;
	DGGLSE_F77(num_rows_A, num_cols_A, num_cons, A_matrix, num_rows_A,
		   C_matrix, num_cons, b_vector, d_vector, x_vector, work,
		   lwork, info);
	if (info)
	  err_flag = true;
	for (j=0; j<numExpansionTerms; j++)
	  expansionCoeffGrads[j][i] = x_vector[j];
      }
    }

    delete [] b_vector;
    delete [] C_matrix;
    delete [] d_vector;
    delete [] x_vector;
    delete [] work;
  }

  delete [] A_matrix;

  if (err_flag) { // if numerical problems in LLS, abort with error message
    Cerr << "Error: nonzero return code from LAPACK linear least squares in "
	 << "OrthogPolyApproximation::find_coefficients()" << endl;
    abort_handler(-1);
  }
}


/** The coefficients of the PCE for the response are calculated using a
    Galerkin projection of the response against each multivariate orthogonal
    polynomial basis fn using the inner product ratio <f,Psi>/<Psi^2>,
    where inner product <a,b> is the n-dimensional integral of a*b*weighting
    over the support range of the n-dimensional (composite) weighting
    function.  When interpreting the weighting function as a probability
    density function, <a,b> = expected value of a*b, which can be evaluated
    by sampling from the probability density function and computing the mean
    statistic.  It is not necessary to compute the mean statistic for the
    denominator, since this is available analytically. */
void OrthogPolyApproximation::expectation()
{
  // "lhs" or "random", no weights needed
  List<SurrogateDataPoint>::iterator it;
  size_t i, j, k, offset = 0, num_pts = currentPoints.entries(),
    num_deriv_vars = expansionCoeffGrads.num_columns();
  if (!anchorPoint.is_null()) {
    offset   = 1;
    num_pts += offset;
  }

  Real empty_r;
  RealBaseVector empty_rbv;
  /*
  // The following implementation evaluates all PCE coefficients
  // using a consistent expectation formulation
  for (i=0; i<numExpansionTerms; i++) {
    Real& chaos_coeff_i = expansionCoeffs[i];
    chaos_coeff_i = (anchorPoint.is_null()) ? 0.0 :
      anchorPoint.response_function() * chaosSamples[0][i];
    for (j=offset, it=currentPoints.begin(); j<num_pts; j++, it++)
      chaos_coeff_i += it->response_function() * chaosSamples[j][i];
    chaos_coeff_i /= num_pts * norm_squared(i);
#ifdef DEBUG
    Cout << "coeff[" << i << "] = " << chaos_coeff_i
	 << " norm squared[" << i <<"] = " << norm_squared(i) << '\n';
#endif // DEBUG
  }
  */

  // This alternate implementation evaluates the first PCE coefficient (the
  // response mean) as an expectation and then removes the mean from the
  // expectation evaluation of all subsequent coefficients.  This approach
  // has been observed to result in better results for small sample sizes.
  Real& mean = (expansionCoeffFlag) ? expansionCoeffs[0] : empty_r;
  RealBaseVector& mean_grad
    = (expansionGradFlag) ? expansionCoeffGrads[0] : empty_rbv;
  if (anchorPoint.is_null()) {
    if (expansionCoeffFlag) mean      = 0.0;
    if (expansionGradFlag)  mean_grad = 0.0;
  }
  else {
    if (expansionCoeffFlag) mean      = anchorPoint.response_function();
    if (expansionGradFlag)  mean_grad = anchorPoint.response_gradient();
  }
  for (it=currentPoints.begin(); it!=currentPoints.end(); it++) {
    if (expansionCoeffFlag)
      mean += it->response_function();
    if (expansionGradFlag) {
      const RealBaseVector& curr_pt_grad = it->response_gradient();
      for (j=0; j<num_deriv_vars; j++)
	mean_grad[j] += curr_pt_grad[j];
    }
  }
  if (expansionCoeffFlag)
    mean /= num_pts;
  if (expansionGradFlag)
    for (j=0; j<num_deriv_vars; j++)
      mean_grad[j] /= num_pts;
  for (i=1; i<numExpansionTerms; i++) {
    Real& chaos_coeff_i = (expansionCoeffFlag) ? expansionCoeffs[i] : empty_r;
    RealBaseVector& chaos_grad_i = (expansionGradFlag) ?
      expansionCoeffGrads[i] : empty_rbv;
    if (anchorPoint.is_null()) {
      if (expansionCoeffFlag) chaos_coeff_i = 0.0;
      if (expansionGradFlag)  chaos_grad_i  = 0.0;
    }
    else {
      const Real& chaos_sample = chaosSamples[0][i];
      if (expansionCoeffFlag)
	chaos_coeff_i = (anchorPoint.response_function() - mean) * chaos_sample;
      if (expansionGradFlag) {
	const RealBaseVector& anchor_grad = anchorPoint.response_gradient();
	for (j=0; j<num_deriv_vars; j++)
	  chaos_grad_i[j] = (anchor_grad[j] - mean_grad[j]) * chaos_sample;
      }
    }
    for (j=offset, it=currentPoints.begin(); j<num_pts; j++, it++) {
      const Real& chaos_sample = chaosSamples[j][i];
      if (expansionCoeffFlag)
	chaos_coeff_i += (it->response_function() - mean) * chaos_sample;
      if (expansionGradFlag) {
	const RealBaseVector& curr_pt_grad = it->response_gradient();
	for (k=0; k<num_deriv_vars; k++)
	  chaos_grad_i[k] += (curr_pt_grad[k] - mean_grad[k]) * chaos_sample;
      }
    }
    Real term = num_pts * norm_squared(i);
    if (expansionCoeffFlag)
      chaos_coeff_i /= term;
    if (expansionGradFlag)
      for (j=0; j<num_deriv_vars; j++)
	chaos_grad_i[j] /= term;
#ifdef DEBUG
    Cout << "coeff[" << i << "] = " << chaos_coeff_i
       //<< "coeff_grad[" << i <<"] = " << chaos_grad_i
	 << " norm squared[" << i <<"] = " << norm_squared(i) << '\n';
#endif // DEBUG
  }
}


const RealVector& OrthogPolyApproximation::
get_multivariate_polynomials(const RealVector& xi)
{
  size_t i, j;
  unsigned short order_1d;
  for (i=0; i<numExpansionTerms; i++) {
    wienerAskeyChaos[i] = 1.0;
    for (j=0; j<numVars; j++) {
      order_1d = multiIndex[i][j];
      if (order_1d)
	wienerAskeyChaos[i] *= polynomialBasis[j].get_value(xi[j], order_1d);
    }
  }
  return wienerAskeyChaos;
}


const Real& OrthogPolyApproximation::get_value(const RealVector& x)
{
  // calculate values of all multivariate orthogonal polynomials at x
  //const RealVector& Psi = get_multivariate_polynomials(x);

  // Error check for required data
  if (!expansionCoeffFlag) {
    Cerr << "Error: expansion coefficients not defined in "
	 << "OrthogPolyApproximation::get_value()" << endl;
    abort_handler(-1);
  }

  // sum expansion to get response prediction
  approxValue = 0.0;
  Real Psi;
  size_t i, j;
  unsigned short order_1d;
  for (i=0; i<numExpansionTerms; i++) {
    Psi = 1.0;
    for (j=0; j<numVars; j++) {
      order_1d = multiIndex[i][j];
      if (order_1d)
	Psi *= polynomialBasis[j].get_value(x[j], order_1d);
    }
    approxValue += Psi*expansionCoeffs[i];
  }
  return approxValue;
}


const RealBaseVector& OrthogPolyApproximation::get_gradient(const RealVector& x)
{
  // this could define a default_dvv and call get_gradient(x, dvv),
  // but we want this fn to be as fast as possible

  // Error check for required data
  if (!expansionCoeffFlag) {
    Cerr << "Error: expansion coefficients not defined in "
	 << "OrthogPolyApproximation::get_gradient()" << endl;
    abort_handler(-1);
  }

  if (approxGradient.length() != numVars)
    approxGradient.reshape(numVars);
  approxGradient = 0.0;

  // sum expansion to get response gradient prediction
  size_t i, j, k;
  for (i=0; i<numExpansionTerms; i++) {
    for (j=0; j<numVars; j++) {
      Real term_i_grad_j = 1.0;
      for (k=0; k<numVars; k++)
	term_i_grad_j *= (k == j) ?
	  polynomialBasis[k].get_gradient(x[k], multiIndex[i][k]) :
	  polynomialBasis[k].get_value(x[k],    multiIndex[i][k]);
      approxGradient[j] += expansionCoeffs[i]*term_i_grad_j;
    }
  }
  return approxGradient;
}


const RealBaseVector& OrthogPolyApproximation::
get_gradient(const RealVector& x, const UIntArray& dvv)
{
  // Error check for required data
  if (!expansionCoeffFlag) {
    Cerr << "Error: expansion coefficients not defined in "
	 << "OrthogPolyApproximation::get_gradient()" << endl;
    abort_handler(-1);
  }

  size_t num_deriv_vars = dvv.length();
  if (approxGradient.length() != num_deriv_vars)
    approxGradient.reshape(num_deriv_vars);
  approxGradient = 0.0;

  // sum expansion to get response gradient prediction
  size_t i, j, k, deriv_index;
  for (i=0; i<numExpansionTerms; i++) {
    for (j=0; j<num_deriv_vars; j++) {
      deriv_index = dvv[j] - 1; // *** requires an "All" view
      Real term_i_grad_j = 1.0;
      for (k=0; k<numVars; k++)
	term_i_grad_j *= (k == deriv_index) ?
	  polynomialBasis[k].get_gradient(x[k], multiIndex[i][k]) :
	  polynomialBasis[k].get_value(x[k],    multiIndex[i][k]);
      approxGradient[j] += expansionCoeffs[i]*term_i_grad_j;
    }
  }
  return approxGradient;
}


/** In this case, all expansion variables are random variables and the
    mean of the expansion is simply the first chaos coefficient. */
const Real& OrthogPolyApproximation::get_mean()
{
  // Error check for required data
  if (!expansionCoeffFlag) {
    Cerr << "Error: expansion coefficients not defined in "
	 << "OrthogPolyApproximation::get_mean()" << endl;
    abort_handler(-1);
  }

  //expansionMean = expansionCoeffs[0];
  //return expansionMean;
  return expansionCoeffs[0];
}


/** In this case, a subset of the expansion variables are random
    variables and the mean of the expansion involves evaluating the
    expectation over this subset. */
const Real& OrthogPolyApproximation::get_mean(const RealVector& x)
{
  // Error check for required data
  if (!expansionCoeffFlag) {
    Cerr << "Error: expansion coefficients not defined in "
	 << "OrthogPolyApproximation::get_mean()" << endl;
    abort_handler(-1);
  }

  // calculate values of all multivariate orthogonal polynomials at x
  //const RealVector& Psi = get_multivariate_polynomials(x);

  // sum expansion to get response prediction
  expansionMean = expansionCoeffs[0];

  size_t i;
  SizetList::iterator it;
  for (i=1; i<numExpansionTerms; i++) {
    bool include = true;
    for (it=randomIndices.begin(); it!=randomIndices.end(); it++)
      if (multiIndex[i][*it]) // expectation of this expansion term is zero
	{ include = false; break; }
    if (include) {
      Real Psi = 1.0;
      for (it=nonRandomIndices.begin(); it!=nonRandomIndices.end(); it++) {
	size_t index = *it;
	unsigned short order_1d = multiIndex[i][index];
	if (order_1d)
	  Psi *= polynomialBasis[index].get_value(x[index], order_1d);
      }
      expansionMean += Psi*expansionCoeffs[i];
#ifdef DEBUG
      Cout << "Mean estimate inclusion: term index = " << i << " Psi = " << Psi
	   << " PCE coeff = " << expansionCoeffs[i] << " total = "
	   << expansionMean << '\n';
#endif // DEBUG
    }
  }

  return expansionMean;
}


/** In this function, all expansion variables are random variables and
    any design/state variables are omitted from the expansion.  In
    this case, the derivative of the expectation is the expectation of
    the derivative.  The mixed derivative case (some design variables
    are inserted and some are augmented) requires no special treatment. */
const RealBaseVector& OrthogPolyApproximation::get_mean_gradient()
{
  // d/ds \mu_R = d/ds \alpha_0 = <dR/ds>

  // Error check for required data
  if (!expansionGradFlag) {
    Cerr << "Error: expansion coefficient gradients not defined in "
	 << "OrthogPolyApproximation::get_mean_gradient()." << endl;
    abort_handler(-1);
  }

  //expansionMeanGrad = expansionCoeffGrads[0];
  //return expansionMeanGrad;
  return expansionCoeffGrads[0];
}


/** In this function, a subset of the expansion variables are random
    variables and any augmented design/state variables (i.e., not
    inserted as random variable distribution parameters) are included
    in the expansion.  In this case, the mean of the expansion is the
    expectation over the random subset and the derivative of the mean
    is the derivative of the remaining expansion over the non-random
    subset.  This function must handle the mixed case, where some
    design/state variables are augmented (and are part of the
    expansion: derivatives are evaluated as described above) and some
    are inserted (derivatives are obtained from expansionCoeffGrads). */
const RealBaseVector& OrthogPolyApproximation::
get_mean_gradient(const RealVector& x, const UIntArray& dvv)
{
  size_t i, j, deriv_index, num_deriv_vars = dvv.length();
  if (expansionMeanGrad.length() != num_deriv_vars)
    expansionMeanGrad.reshape(num_deriv_vars);
  expansionMeanGrad = 0.0;

  SizetList::iterator it;
  size_t cntr = 0; // insertions carried in order within expansionCoeffGrads
  for (i=0; i<num_deriv_vars; i++) {
    deriv_index = dvv[i] - 1; // OK since we are in an "All" view
    if (randomVarsKey[deriv_index]) { // deriv w.r.t. design var insertion
      // Error check for required data
      if (!expansionGradFlag) {
	Cerr << "Error: expansion coefficient gradients not defined in "
	     << "OrthogPolyApproximation::get_mean_gradient()." << endl;
	abort_handler(-1);
      }
      expansionMeanGrad[i] = expansionCoeffGrads[0][cntr];
    }
    else if (!expansionCoeffFlag) { // Error check for required data
      Cerr << "Error: expansion coefficients not defined in "
	   << "OrthogPolyApproximation::get_mean_gradient()" << endl;
      abort_handler(-1);
    }
    for (j=1; j<numExpansionTerms; j++) {
      bool include = true;
      for (it=randomIndices.begin(); it!=randomIndices.end(); it++)
	if (multiIndex[j][*it]) // expectation of this expansion term is zero
	  { include = false; break; }
      if (include) {
	// In both cases below, the term to differentiate is alpha_j(s) Psi_j(s)
	// since <Psi_j>_xi = 1 for included terms.  The difference occurs based
	// on whether a particular s_i dependence appears in alpha (for
	// inserted) or Psi (for augmented).
	if (randomVarsKey[deriv_index]) {
	  // -------------------------------------------
	  // derivative w.r.t. design variable insertion
	  // -------------------------------------------
	  Real Psi = 1.0;
	  for (it=nonRandomIndices.begin(); it!=nonRandomIndices.end(); it++) {
	    size_t index = *it;
	    unsigned short order_1d = multiIndex[j][index];
	    if (order_1d)
	      Psi *= polynomialBasis[index].get_value(x[index], order_1d);
	  }
	  expansionMeanGrad[i] += Psi*expansionCoeffGrads[j][cntr];
	}
	else {
	  // ----------------------------------------------
	  // derivative w.r.t. design variable augmentation
	  // ----------------------------------------------
	  Real term_j_grad_i = 1.0;
	  for (it=nonRandomIndices.begin(); it!=nonRandomIndices.end(); it++) {
	    size_t index = *it;
	    term_j_grad_i *= (index == deriv_index) ?
	      polynomialBasis[index].get_gradient(x[index],multiIndex[j][index])
	    : polynomialBasis[index].get_value(x[index], multiIndex[j][index]);
	  }
	  expansionMeanGrad[i] += expansionCoeffs[j]*term_j_grad_i;
	}
      }
    }
    if (randomVarsKey[deriv_index]) // derivative w.r.t. design var insertion
      cntr++;
  }

  return expansionMeanGrad;
}


/** In this case, all expansion variables are random variables and the
    variance of the expansion is the sum over all but the first term
    of the coefficients squared times the polynomial norms squared. */
const Real& OrthogPolyApproximation::get_variance()
{
  // Error check for required data
  if (!expansionCoeffFlag) {
    Cerr << "Error: expansion coefficients not defined in "
	 << "OrthogPolyApproximation::get_variance()" << endl;
    abort_handler(-1);
  }

  expansionVariance = 0.0;
  for (size_t i=1; i<numExpansionTerms; i++)
    expansionVariance += pow(expansionCoeffs[i], 2) * norm_squared(i);
  return expansionVariance;
}


/** In this case, a subset of the expansion variables are random
    variables and the variance of the expansion involves summations
    over this subset. */
const Real& OrthogPolyApproximation::get_variance(const RealVector& x)
{
  // Error check for required data
  if (!expansionCoeffFlag) {
    Cerr << "Error: expansion coefficients not defined in "
	 << "OrthogPolyApproximation::get_variance()" << endl;
    abort_handler(-1);
  }

  expansionVariance = 0.0;
  size_t i, j;
  SizetList::iterator it;
  for (i=1; i<numExpansionTerms; i++) {
    // For r = random_vars and nr = non_random_vars,
    // sigma^2_R(nr) = < (R(r,nr) - \mu_R(nr))^2 >_r
    // -> include only those terms from R(r,nr) which do not appear in \mu_R(nr)
    bool include_i = false;
    for (it=randomIndices.begin(); it!=randomIndices.end(); it++)
      if (multiIndex[i][*it]) // term does not appear in mean(nr)
	{ include_i = true; break; }
    if (include_i) {
      Real norm_sq_i = norm_squared_random(i);
      for (j=1; j<numExpansionTerms; j++) {

	// random part of polynomial must be identical to contribute to variance
	// (else orthogonality drops term)
	bool include_j = true;
	for (it=randomIndices.begin(); it!=randomIndices.end(); it++)
	  if (multiIndex[i][*it] != multiIndex[j][*it])
	    { include_j = false; break; }
	if (include_j) {
	  Real var_ij = expansionCoeffs[i] * expansionCoeffs[j] * norm_sq_i;
	  for (it=nonRandomIndices.begin(); it!=nonRandomIndices.end(); it++) {
	    size_t index = *it;
	    unsigned short order_1d_i = multiIndex[i][index],
	      order_1d_j = multiIndex[j][index];
	    BasisPolynomial& poly_1d = polynomialBasis[index];
	    if (order_1d_i)
	      var_ij *= poly_1d.get_value(x[index], order_1d_i);
	    if (order_1d_j)
	      var_ij *= poly_1d.get_value(x[index], order_1d_j);
	  }
	  expansionVariance += var_ij;
#ifdef DEBUG
	  Cout << "Variance estimate inclusion: term index = " << i
	       << " variance = " << var_ij << " total = " << expansionVariance
	       <<'\n';
#endif // DEBUG
	}
      }
    }
  }

  return expansionVariance;
}


/** In this function, all expansion variables are random variables and
    any design/state variables are omitted from the expansion.  The
    mixed derivative case (some design variables are inserted and some
    are augmented) requires no special treatment. */
const RealBaseVector& OrthogPolyApproximation::get_variance_gradient()
{
  // d/ds \sigma^2_R = Sum_{j=1}^P <Psi^2_j> d/ds \alpha^2_j
  //                 = 2 Sum_{j=1}^P \alpha_j <dR/ds, Psi_j>

  // Error check for required data
  if (!expansionCoeffFlag) {
    Cerr << "Error: expansion coefficients not defined in "
	 << "OrthogPolyApproximation::get_variance_gradient()" << endl;
    abort_handler(-1);
  }
  if (!expansionGradFlag) {
    Cerr << "Error: expansion coefficient gradients not defined in "
	 << "OrthogPolyApproximation::get_variance_gradient()." << endl;
    abort_handler(-1);
  }

  size_t i, j, num_deriv_vars = expansionCoeffGrads.num_columns();
  if (expansionVarianceGrad.length() != num_deriv_vars)
    expansionVarianceGrad.reshape(num_deriv_vars);
  expansionVarianceGrad = 0.0;
  for (i=1; i<numExpansionTerms; i++) {
    Real term_i = 2. * expansionCoeffs[i] * norm_squared(i);
    for (j=0; j<num_deriv_vars; j++)
      expansionVarianceGrad[j] += term_i * expansionCoeffGrads[i][j];
  }
  return expansionVarianceGrad;
}


/** In this function, a subset of the expansion variables are random
    variables and any augmented design/state variables (i.e., not
    inserted as random variable distribution parameters) are included
    in the expansion.  This function must handle the mixed case, where
    some design/state variables are augmented (and are part of the
    expansion) and some are inserted (derivatives are obtained from
    expansionCoeffGrads). */
const RealBaseVector& OrthogPolyApproximation::
get_variance_gradient(const RealVector& x, const UIntArray& dvv)
{
  // Error check for required data
  if (!expansionCoeffFlag) {
    Cerr << "Error: expansion coefficients not defined in "
	 << "OrthogPolyApproximation::get_variance_gradient()" << endl;
    abort_handler(-1);
  }

  size_t i, j, k, deriv_index, num_deriv_vars = dvv.length();
  if (expansionVarianceGrad.length() != num_deriv_vars)
    expansionVarianceGrad.reshape(num_deriv_vars);
  expansionVarianceGrad = 0.0;

  SizetList::iterator it;
  size_t cntr = 0; // insertions carried in order within expansionCoeffGrads
  for (i=0; i<num_deriv_vars; i++) {
    deriv_index = dvv[i] - 1; // OK since we are in an "All" view
    if (randomVarsKey[deriv_index] && !expansionGradFlag) { // Error check 
      Cerr << "Error: expansion coefficient gradients not defined in "
	   << "OrthogPolyApproximation::get_variance_gradient()." << endl;
      abort_handler(-1);
    }
    for (j=1; j<numExpansionTerms; j++) {
      bool include_j = false;
      for (it=randomIndices.begin(); it!=randomIndices.end(); it++)
	if (multiIndex[j][*it]) // term does not appear in mean(nr)
	  { include_j = true; break; }
      if (include_j) {
	Real norm_sq_j = norm_squared_random(j);
	for (k=1; k<numExpansionTerms; k++) {
	  // random part of polynomial must be identical to contribute to
	  // variance (else orthogonality drops term)
	  bool include_k = true;
	  for (it=randomIndices.begin(); it!=randomIndices.end(); it++)
	    if (multiIndex[j][*it] != multiIndex[k][*it])
	      { include_k = false; break; }
	  if (include_k) {
	    // In both cases below, the term to differentiate is
	    // alpha_j(s) alpha_k(s) <Psi_j^2>_xi Psi_j(s) Psi_k(s) and the
	    // difference occurs based on whether a particular s_i dependence
	    // appears in alpha (for inserted) or Psi (for augmented).
	    if (randomVarsKey[deriv_index]) {
	      // -------------------------------------------
	      // derivative w.r.t. design variable insertion
	      // -------------------------------------------
	      Real var_jk = norm_sq_j *
		(expansionCoeffs[j] * expansionCoeffGrads[k][cntr] +
		 expansionCoeffs[k] * expansionCoeffGrads[j][cntr]);
	      for (it=nonRandomIndices.begin(); it!=nonRandomIndices.end();
		   it++) {
		size_t index = *it;
		unsigned short order_1d_j = multiIndex[j][index],
		  order_1d_k = multiIndex[k][index];
		BasisPolynomial& poly_1d = polynomialBasis[index];
		if (order_1d_j)
		  var_jk *= poly_1d.get_value(x[index], order_1d_j);
		if (order_1d_k)
		  var_jk *= poly_1d.get_value(x[index], order_1d_k);
	      }
	      expansionVarianceGrad[i] += var_jk;
	    }
	    else {
	      // ----------------------------------------------
	      // derivative w.r.t. design variable augmentation
	      // ----------------------------------------------
	      Real var_jk = expansionCoeffs[j] * expansionCoeffs[k] * norm_sq_j;
	      Real Psi_j = 1., Psi_k = 1., dPsi_j_ds_i = 1., dPsi_k_ds_i = 1.;
	      for (it=nonRandomIndices.begin(); it!=nonRandomIndices.end();
		   it++) {
		size_t index = *it;
		unsigned short order_1d_j = multiIndex[j][index],
		  order_1d_k = multiIndex[k][index];
		BasisPolynomial& poly_1d = polynomialBasis[index];
		if (order_1d_j)
		  Psi_j *= poly_1d.get_value(x[index], order_1d_j);
		if (order_1d_k)
		  Psi_k *= poly_1d.get_value(x[index], order_1d_k);
		dPsi_j_ds_i *= (index == deriv_index) ?
		  poly_1d.get_gradient(x[index], order_1d_j) :
		  poly_1d.get_value(   x[index], order_1d_j);
		dPsi_k_ds_i *= (index == deriv_index) ?
		  poly_1d.get_gradient(x[index], order_1d_k) :
		  poly_1d.get_value(   x[index], order_1d_k);
	      }
	      expansionVarianceGrad[i] += var_jk * 
		(Psi_j*dPsi_k_ds_i + dPsi_j_ds_i*Psi_k);
	    }
	  }
	}
      }
    }
    if (randomVarsKey[deriv_index]) // derivative w.r.t. design var insertion
      cntr++;
  }

  return expansionVarianceGrad;
}


/** This test works in combination with DEBUG settings in
    (Legendre,Laguerre,Jacobi,GenLaguerre)OrthogPolynomial::get_gradient(). */
void OrthogPolyApproximation::gradient_check()
{
  BasisPolynomial legendre_poly(LEGENDRE), laguerre_poly(LAGUERRE),
    jacobi_poly(JACOBI), gen_laguerre_poly(GENERALIZED_LAGUERRE);
  // alpha/beta selections mirror dakota_uq_rosenbrock_pce.in
  jacobi_poly.alpha_stat(1.5);
  jacobi_poly.beta_stat(2.);
  gen_laguerre_poly.alpha_stat(2.5);

  Real x = 0.5; // valid for all support ranges: [-1,1], [0,Inf], [-Inf, Inf]
  Cout << "-------------------------------------------------\n";
  for (size_t n=0; n<=10; n++) {
    Cout << "Gradients at " << x << " for order " << n << '\n';
    legendre_poly.get_gradient(x, n);
    laguerre_poly.get_gradient(x, n);
    jacobi_poly.get_gradient(x, n);
    gen_laguerre_poly.get_gradient(x, n);
    Cout << "-------------------------------------------------\n";
  }
}


const Real& OrthogPolyApproximation::norm_squared(size_t expansion_index)
{
  // the norm squared of a particular multivariate polynomial is the product of
  // the norms squared of the numVars univariate polynomials that comprise it.
  multiPolyNormSq = 1.0;
  unsigned short order;
  for (size_t i=0; i<numVars; i++) {
    order = multiIndex[expansion_index][i];
    if (order)
      multiPolyNormSq *= polynomialBasis[i].norm_squared(order);
  }
  return multiPolyNormSq;
}


const Real& OrthogPolyApproximation::norm_squared_random(size_t expansion_index)
{
  // the norm squared of a particular multivariate polynomial is the product of
  // the norms squared of the numVars univariate polynomials that comprise it.
  multiPolyNormSq = 1.0;
  unsigned short order;
  for (size_t i=0; i<numVars; i++) {
    if (randomVarsKey[i]) {
      order = multiIndex[expansion_index][i];
      if (order)
	multiPolyNormSq *= polynomialBasis[i].norm_squared(order);
    }
  }
  return multiPolyNormSq;
}


void OrthogPolyApproximation::print_coefficients(ostream& s) const
{
  size_t i, j;
  char tag[10];

  // terms and term identifiers
  for (i=0; i<numExpansionTerms; i++) {
    s << "\n  " << setw(17) << expansionCoeffs[i];
    for (j=0; j<numVars; j++) {
      switch (basisTypes[j]) {
      case HERMITE:
	sprintf(tag,  "He%i", multiIndex[i][j]);
	break;
      case LEGENDRE:
	sprintf(tag,   "P%i", multiIndex[i][j]);
	break;
      case LAGUERRE:
	sprintf(tag,   "L%i", multiIndex[i][j]);
	break;
      case JACOBI:
	sprintf(tag, "Pab%i", multiIndex[i][j]);
	break;
      case GENERALIZED_LAGUERRE:
	sprintf(tag,  "La%i", multiIndex[i][j]);
	break;
      default:
	Cerr << "Error: bad polynomial type = " << basisTypes[j]
	     << " in OrthogPolyApproximation::print_coefficients()." << endl;
	abort_handler(-1);
	break;
      }
      s << setw(5) << tag;
    }
  }
  s << '\n';
}

} // namespace Dakota
