/*  _______________________________________________________________________

    DAKOTA: Design Analysis Kit for Optimization and Terascale Applications
    Copyright (c) 2006, Sandia National Laboratories.
    This software is distributed under the GNU General Public License.
    For more information, see the README file in the top Dakota directory.
    _______________________________________________________________________ */

//- Class:        JacobiOrthogPolynomial
//- Description:  Implementation code for JacobiOrthogPolynomial class
//-               
//- Owner:        Mike Eldred, Sandia National Laboratories

#include "JacobiOrthogPolynomial.H"
#ifdef DAKOTA_QUADRATURE
#include "sandia_rules.H"
#ifdef HAVE_GSL
#include "gsl/gsl_sf_gamma.h"
#endif
#endif

//#define DEBUG


namespace Dakota {

const Real& JacobiOrthogPolynomial::
get_value(const Real& x, unsigned short order)
{
  switch (order) {
  case 0:
    basisPolyValue = 1.;
    break;
  case 1:
    basisPolyValue = (alphaPoly + betaPoly + 2.)*(x-1.)/2. + alphaPoly + 1.;
    break;
  case 2: {
    Real xm1 = x - 1.;
    basisPolyValue
      = ( (alphaPoly + betaPoly + 3.)*(alphaPoly + betaPoly + 4.)*xm1*xm1 +
	  4.*(alphaPoly + betaPoly + 3.)*(alphaPoly + 2.)*xm1 +
	  4.*(alphaPoly + 1.)*(alphaPoly + 2.) ) / 8.;
    break;
  }
  default: {
    // Support higher order polynomials using the 3 point recursion formula:
    Real xm1 = x - 1.,
      Pab_n = ( (alphaPoly + betaPoly + 3.)*(alphaPoly + betaPoly + 4.)*xm1*xm1
		+ 4*(alphaPoly + betaPoly + 3.)*(alphaPoly + 2.)*xm1
		+ 4.*(alphaPoly + 1.)*(alphaPoly + 2.) ) / 8.,       // Pab_2
      Pab_nm1 = (alphaPoly + betaPoly + 2.)*xm1/2. + alphaPoly + 1.; // Pab_1
    for (size_t i=2; i<order; i++) {
      Real ab2i = alphaPoly + betaPoly + 2.*i;
      basisPolyValue // Pab_np1
	= ( ( (ab2i+1.)*(alphaPoly*alphaPoly-betaPoly*betaPoly)
	      + x*pochhammer(ab2i,3) )*Pab_n
	    - 2.*(i+alphaPoly)*(i+betaPoly)*(ab2i+2.)*Pab_nm1 )
	/ ( 2.*(i+1.)*(i+alphaPoly+betaPoly+1.)*ab2i );
      if (i != order-1) {
	Pab_nm1 = Pab_n;
	Pab_n   = basisPolyValue;
      }
    }
    break;
  }
  }

  return basisPolyValue;
}


const Real& JacobiOrthogPolynomial::
get_gradient(const Real& x, unsigned short order)
{
#ifdef DEBUG
  // See Abramowitz & Stegun, Section 22.8, p.783
  //basisPolyGradient = (order) ?
  //  (g1*get_value(x,order) + g0*get_value(x,order-1))/g2 :0.;
  if (order) { // be careful with reference to changing basisPolyValue
    Real ab2n = 2.*order+alphaPoly+betaPoly,
      g0 = 2.*(order+alphaPoly)*(order+betaPoly),
      g1 = order*(alphaPoly-betaPoly-ab2n*x), g2 = ab2n*(1.-x*x),
      Pab_n = get_value(x, order), Pab_nminus1 = get_value(x, order-1);
    basisPolyGradient = (g1*Pab_n + g0*Pab_nminus1)/g2;
  }
  else
    basisPolyGradient = 0.;
  Cout << "Jacobi gradient approach 1: " << basisPolyGradient << '\n';
#endif // DEBUG

  // The previous approach, while very compact, produces 0/0 = NaN at x = +/-1.
  // To avoid NaN issues at bounds, differentiate the 3 pt value recursion
  // to get a 3 point gradient recursion
  switch (order) {
  case 0:
    basisPolyGradient = 0.;
    break;
  case 1:
    basisPolyGradient = (alphaPoly + betaPoly + 2.)/2.;
    break;
  case 2: {
    Real xm1 = x - 1.;
    basisPolyGradient
      = ( (alphaPoly + betaPoly + 3.)*(alphaPoly + betaPoly + 4.)*xm1 +
	  2.*(alphaPoly + betaPoly + 3.)*(alphaPoly + 2.) ) / 4.;
    break;
  }
  default: {
    // Support higher order polynomials using the 3 point recursion formula:
    Real xm1 = x - 1.,
      dPabdx_n = ( (alphaPoly + betaPoly + 3.)*(alphaPoly + betaPoly + 4.)*xm1 +
		   2.*(alphaPoly + betaPoly + 3.)*(alphaPoly + 2.) ) / 4.,//P'_2
      dPabdx_nm1 = (alphaPoly + betaPoly + 2.)/2.;                       // P'_1
    for (size_t i=2; i<order; i++) {
      Real ab2i = alphaPoly + betaPoly + 2.*i, pab2i3 = pochhammer(ab2i, 3);
      basisPolyGradient // dPabdx_np1
	= ( ( (ab2i+1.)*(alphaPoly*alphaPoly-betaPoly*betaPoly) + x*pab2i3 )
	    * dPabdx_n + pab2i3*get_value(x,i)
	    - 2.*(i+alphaPoly)*(i+betaPoly)*(ab2i+2.)*dPabdx_nm1 )
	/ ( 2.*(i+1.)*(i+alphaPoly+betaPoly+1.)*ab2i );
      if (i != order-1) {
	dPabdx_nm1 = dPabdx_n;
	dPabdx_n   = basisPolyGradient;
      }
    }
    break;
  }
  }
#ifdef DEBUG
  Cout << "Jacobi gradient approach 2: " << basisPolyGradient << '\n';
#endif // DEBUG

  return basisPolyGradient;
}


const Real& JacobiOrthogPolynomial::norm_squared(unsigned short order)
{
  orthogPolyNormSq = (alphaPoly+betaPoly+1.) / (2.*order+alphaPoly+betaPoly+1.)
    * pochhammer(alphaPoly+1.,order) * pochhammer(betaPoly+1.,order)
    / pochhammer(alphaPoly+betaPoly+1.,order) / factorial(order);
  return orthogPolyNormSq;
}


const RealVector& JacobiOrthogPolynomial::gauss_points(unsigned short order)
{
  // pull this out from default below since order=0 is initial gauss pts length
  if (order < 1) {
    Cerr << "Error: underflow in minimum quadrature order (1) in "
	 << "JacobiOrthogPolynomial::gauss_points()." << endl;
    abort_handler(-1);
  }

  if (gaussPoints.length() != order) { // if not already computed
    gaussPoints.reshape(order);
    switch (order) {
    case 1: // zeros of Pab_1(x) for one Gauss-Jacobi point:
      gaussPoints[0] = (betaPoly - alphaPoly) / (alphaPoly + betaPoly + 2.);
      break;
    case 2: { // zeros of Pab_2(x) for two Gauss-Jacobi points:
      Real a = (alphaPoly+betaPoly+3.)*(alphaPoly+betaPoly+4.),
	   b = 4.*(alphaPoly+betaPoly+3.)*(alphaPoly+2.),
	   c = 4.*(alphaPoly+1.)*(alphaPoly+2.),
	   srdiscrim = sqrt(b*b-4.*a*c), a2 = 2.*a;
      gaussPoints[0] = 1. - (b+srdiscrim)/a2;
      gaussPoints[1] = 1. - (b-srdiscrim)/a2;
      break;
    }
    default:
#if defined(DAKOTA_QUADRATURE) && defined(HAVE_GSL)
      if (gaussWeights.length() != order)
	gaussWeights.reshape(order);
      webbur2::jacobi_compute(order, alphaPoly, betaPoly, gaussPoints,
			      gaussWeights);
      Real factor = pow(2., alphaPoly + betaPoly + 1.)
	          * gsl_sf_beta(alphaPoly + 1., betaPoly + 1.);
      for (size_t i=0; i<order; i++)
	gaussWeights[i] /= factor; // polynomial weight fn -> PDF
#else
      Cerr << "Error: overflow in maximum quadrature order limit (2) in "
	   << "JacobiOrthogPolynomial::gauss_points()." << endl;
      abort_handler(-1);
#endif
      break;
    }
  }

  return gaussPoints;
}


const RealVector& JacobiOrthogPolynomial::gauss_weights(unsigned short order)
{
  // Derived from (A_n gamma_{n-1})/(A_{n-1} Phi_n'(x_i) Phi_{n-1}(x_i))

  // The sums of the weights = 1, which is the integral of the density
  // function (1-x)^alpha (1+x)^beta / (2^(alpha+beta+1) B(alpha+1,beta+1))
  // over the support range of [-1,+1].

  if (gaussWeights.length() != order) { // if not already computed
    gaussWeights.reshape(order);
    switch (order) {
    case 1: // weights for one Gauss-Jacobi point:
      gaussWeights[0] = 1.0;
      break;
    default:
#if defined(DAKOTA_QUADRATURE) && defined(HAVE_GSL)
      if (gaussPoints.length() != order)
	gaussPoints.reshape(order);
      webbur2::jacobi_compute(order, alphaPoly, betaPoly, gaussPoints,
			      gaussWeights);
      Real factor = pow(2., alphaPoly + betaPoly + 1.)
	          * gsl_sf_beta(alphaPoly + 1., betaPoly + 1.);
      for (size_t i=0; i<order; i++)
	gaussWeights[i] /= factor; // polynomial weight fn -> PDF
#else
      // define Gauss wts from Gauss pts using formula above
      const RealVector& gauss_pts = gauss_points(order);
      for (size_t i=0; i<order; i++) {
	const Real& x_i = gauss_pts[i];
	Real AnoAnm1
	  = (2.*order+alphaPoly+betaPoly) * (2.*order+alphaPoly+betaPoly-1.)
	  / (2.*order) / (order+alphaPoly+betaPoly);
	gaussWeights[i]
	  = AnoAnm1 * norm_squared(order-1) / get_value(x_i, order-1)
	  / get_gradient(x_i, order);
      }
#endif
      break;
    }
  }

  return gaussWeights;
}

} // namespace Dakota
