/*
================================================================================
    PROJECT:

        John Eddy's Genetic Algorithms (JEGA) Managed Front End

    CONTENTS:

        Implementation of class MSolution.

    NOTES:

        See notes of MSolution.hpp.

    PROGRAMMERS:

        John Eddy (jpeddy@sandia.gov) (JE)

    ORGANIZATION:

        Sandia National Laboratories

    COPYRIGHT:

        See the LICENSE file in the top level JEGA directory.

    VERSION:

        1.0.0

    CHANGES:

        Mon Feb 13 13:14:41 2006 - Original Version (JE)

================================================================================
*/




/*
================================================================================
Document This File
================================================================================
*/
/** \file
 * \brief Contains the implementation of the MSolution class.
 */




/*
================================================================================
Includes
================================================================================
*/
#include <stdafx.h>
#include <MSolution.hpp>

#pragma unmanaged
#include <../Utilities/include/Design.hpp>
#include <utilities/include/EDDY_DebugScope.hpp>
#pragma managed







/*
================================================================================
Namespace Using Directives
================================================================================
*/
using namespace JEGA::Utilities;







/*
================================================================================
Begin Namespace
================================================================================
*/
namespace JEGA {
    namespace FrontEnd {
        namespace Managed {





/*
================================================================================
Static Member Data Definitions
================================================================================
*/








/*
================================================================================
Mutators
================================================================================
*/








/*
================================================================================
Accessors
================================================================================
*/

double
MSolution::GetVariable(
    std::size_t num
    )
{
    EDDY_FUNC_DEBUGSCOPE
    return System::Convert::ToDouble(MANAGED_LIST_ITEM(_X, num));
}

double
MSolution::GetObjective(
    std::size_t num
    )
{
    EDDY_FUNC_DEBUGSCOPE
    return System::Convert::ToDouble(MANAGED_LIST_ITEM(_F, num));
}

double
MSolution::GetConstraint(
    std::size_t num
    )
{
    EDDY_FUNC_DEBUGSCOPE
    return System::Convert::ToDouble(MANAGED_LIST_ITEM(_G, num));
}






/*
================================================================================
Public Methods
================================================================================
*/


std::size_t
MSolution::GetNDV(
    )
{
    EDDY_FUNC_DEBUGSCOPE
    return _X->Count;

} // MSolution::GetNDV

std::size_t
MSolution::GetNOF(
    )
{
    EDDY_FUNC_DEBUGSCOPE
    return _F->Count;

} // MSolution::GetNOF

std::size_t
MSolution::GetNCN(
    )
{
    EDDY_FUNC_DEBUGSCOPE
    return _G->Count;

} // MSolution::GetNCN

bool
MSolution::IsEvaluated(
    )
{
    EDDY_FUNC_DEBUGSCOPE
    return HasAttribute(Design::Evaluated);
}

bool
MSolution::IsFeasible(
    )
{
    EDDY_FUNC_DEBUGSCOPE
    return SatisfiesConstraints() && SatisfiesBounds();
}

bool
MSolution::SatisfiesConstraints(
    )
{
    EDDY_FUNC_DEBUGSCOPE
    return HasAttribute(Design::FeasibleConstraints);
}

bool
MSolution::SatisfiesBounds(
    )
{
    EDDY_FUNC_DEBUGSCOPE
    return HasAttribute(Design::FeasibleBounds);
}

bool
MSolution::IsIllconditioned(
    )
{
    EDDY_FUNC_DEBUGSCOPE
    return HasAttribute(Design::Illconditioned);
}





/*
================================================================================
Subclass Visible Methods
================================================================================
*/


bool
MSolution::HasAttribute(
    size_t attrIndex
    )
{
    EDDY_FUNC_DEBUGSCOPE
    return (_attributes & (1 << attrIndex)) != 0;
}







/*
================================================================================
Subclass Overridable Methods
================================================================================
*/








/*
================================================================================
Private Methods
================================================================================
*/








/*
================================================================================
Structors
================================================================================
*/


MSolution::MSolution(
    const JEGA::Utilities::Design& des
    ) :
        _X(MANAGED_GCNEW DoubleVector(static_cast<int>(des.GetNDV()))),
        _F(MANAGED_GCNEW DoubleVector(static_cast<int>(des.GetNOF()))),
        _G(MANAGED_GCNEW DoubleVector(static_cast<int>(des.GetNCN()))),
        _attributes(static_cast<eddy::utilities::bitmask8_t>(
            des.GetRawAttributes().to_ulong()
            ))
{
    EDDY_FUNC_DEBUGSCOPE

    // record the design variables first.
    const std::size_t ndv = des.GetNDV();
    for(std::size_t i=0; i<ndv; ++i)
        _X->Add(MANAGED_BOX(des.GetVariableValue(i)));

    // now the objective function values.
    const std::size_t nof = des.GetNOF();
    for(std::size_t i=0; i<nof; ++i)
        _F->Add(MANAGED_BOX(des.GetObjective(i)));

    // and now the constraint values.
    const std::size_t ncn = des.GetNCN();
    for(std::size_t i=0; i<ncn; ++i)
        _G->Add(MANAGED_BOX(des.GetConstraint(i)));
}





/*
================================================================================
End Namespace
================================================================================
*/
        } // namespace Managed
    } // namespace FrontEnd
} // namespace JEGA

