/*  _______________________________________________________________________

    DAKOTA: Design Analysis Kit for Optimization and Terascale Applications
    Copyright (c) 2006, Sandia National Laboratories.
    This software is distributed under the GNU General Public License.
    For more information, see the README file in the top Dakota directory.
    _______________________________________________________________________ */

#include <cstdlib>
#include <iostream>
#include <fstream>
#include <vector>
#include <string>
using namespace std;


int main(int argc, char** argv)
{
  // The Rosenbrock function may be solved as either a general minimization
  // problem with Objective function = 100.*(x1-x0^2)^2 + (1-x0)^2
  // or a least squares problem with Term1 = 10.*(x1-x0^2) and Term2 = (1-x0).
  // See p. 95 in Practical Optimization by Gill, Murray, and Wright. 

  // This application program reads and writes parameter and response data 
  // directly so no input/output filters are needed.
  ifstream fin(argv[1]);
  if (!fin) {
    cerr << "\nError: failure opening " << argv[1] << endl;
    exit(-1);
  }
  size_t i, num_vars, num_fns, num_deriv_vars;
  string vars_text, fns_text, dvv_text;

  // Get the parameter vector and ignore the labels
  fin >> num_vars >> vars_text;
  vector<double> x(num_vars);
  for (i=0; i<num_vars; i++) {
    fin >> x[i];
    fin.ignore(256, '\n');
  }

  // Get the ASV vector and ignore the labels
  fin >> num_fns >> fns_text;
  vector<int> ASV(num_fns);
  for (i=0; i<num_fns; i++) {
    fin >> ASV[i];
    fin.ignore(256, '\n');
  }

  // Get the DVV vector and ignore the labels
  fin >> num_deriv_vars >> dvv_text;
  vector<int> DVV(num_deriv_vars);
  for (i=0; i<num_deriv_vars; i++) {
    fin >> DVV[i];
    fin.ignore(256, '\n');
  }

  if (num_vars != 2) {
    cerr << "Wrong number of variables for the rosenbrock problem\n";
    exit(-1);
  }
  if (num_fns < 1 || num_fns > 2) { // 1 fn -> opt, 2 fns -> least sq
    cerr << "Wrong number of functions in rosenbrock problem\n";
    exit(-1);
  }

  // Compute and output responses
  bool least_sq_flag = (num_fns > 1) ? true : false;
  double f0 = x[1]-x[0]*x[0];
  double f1 = 1.-x[0];

  ofstream fout(argv[2]);
  if (!fout) {
    cerr << "\nError: failure creating " << argv[2] << endl;
    exit(-1);
  }
  fout.precision(15); // 16 total digits
  fout.setf(ios::scientific);
  fout.setf(ios::right);

  if (least_sq_flag) {
    if (ASV[0] & 1) // **** Residual R1:
      fout << "                     " << 10.*f0  << " f1\n";
    if (ASV[1] & 1) // **** Residual R2:
      fout << "                     " << f1  << " f2\n";

    if (ASV[0] & 2) { // **** dR1/dx:
      fout << "[ ";
      for (size_t i=0; i<num_deriv_vars; i++) {
	int var_index = DVV[i] - 1;
	switch (var_index) {
	case 0: fout << -20.*x[0] << ' '; break;
	case 1: fout <<  10.      << ' '; break;
	}
      }
      fout << "]\n ";
    }
    if (ASV[1] & 2) { // **** dR2/dx:
      fout << "[ ";
      for (size_t i=0; i<num_deriv_vars; i++) {
	int var_index = DVV[i] - 1;
	switch (var_index) {
	case 0: fout << -1. << ' '; break;
	case 1: fout <<  0. << ' '; break;
	}
      }
      fout << "]\n";
    }

    if (ASV[0] & 4) { // **** d^2R1/dx^2:
      fout << "[[ ";
      for (size_t i=0; i<num_deriv_vars; i++) {
	int var_index_i = DVV[i] - 1;
	for (size_t j=0; j<num_deriv_vars; j++) {
	  int var_index_j = DVV[j] - 1;
	  if (var_index_i == 0 && var_index_j == 0)
	    fout <<  -20. << ' ';
	  else
	    fout <<    0. << ' ';
	}
      }
      fout << "]]\n";
    }
    if (ASV[1] & 4) { // **** d^2R2/dx^2:
      fout << "[[ ";
      for (size_t i=0; i<num_deriv_vars; i++)
	for (size_t j=0; j<num_deriv_vars; j++)
	  fout << 0. << ' ';
      fout << "]]\n";
    }
  }
  else {
    if (ASV[0] & 1) // **** f:
      fout << "                     " << 100.*f0*f0+f1*f1  << " f\n";

    if (ASV[0] & 2) { // **** df/dx:
      fout << "[ ";
      for (size_t i=0; i<num_deriv_vars; i++) {
	int var_index = DVV[i] - 1;
	switch (var_index) {
	case 0: fout << -400.*f0*x[0] - 2.*f1 << ' '; break;
	case 1: fout <<  200.*f0              << ' '; break;
	}
      }
      fout << "]\n";
    }

    if (ASV[0] & 4) { // **** d^2f/dx^2:
      fout << "[[ ";
      for (size_t i=0; i<num_deriv_vars; i++) {
	int var_index_i = DVV[i] - 1;
	for (size_t j=0; j<num_deriv_vars; j++) {
	  int var_index_j = DVV[j] - 1;
	  if (var_index_i == 0 && var_index_j == 0)
	    fout << -400.*(x[1] - 3.*x[0]*x[0]) + 2. << ' ';
	  else if ( (var_index_i == 0 && var_index_j == 1) ||
		    (var_index_i == 1 && var_index_j == 0) )
	    fout << -400.*x[0] << ' ';
	  else if (var_index_i == 1 && var_index_j == 1)
	    fout <<  200. << ' ';
	}
      }
      fout << "]]\n";
    }
  }

  fout.flush();
  fout.close();
  return 0;
}
