/*  _______________________________________________________________________

    DAKOTA: Design Analysis Kit for Optimization and Terascale Applications
    Copyright (c) 2006, Sandia National Laboratories.
    This software is distributed under the GNU General Public License.
    For more information, see the README file in the top Dakota directory.
    _______________________________________________________________________ */

//- Class:       Variables
//- Description: Base class for variables.  Using the foundational
//-              cdv/ddv/uv/csv/dsv arrays from an input specification,
//-              derived classes use different views (All, Distinct, Merged)
//-              to define active and inactive variable sets.
//- Owner:       Mike Eldred
//- Version: $Id: DakotaVariables.H 5337 2008-10-09 18:12:05Z mseldre $

#ifndef DAKOTA_VARIABLES_H
#define DAKOTA_VARIABLES_H

#include "data_types.h"
#ifdef DAKOTA_BOOST
#include "boost/functional/hash/hash.hpp"
#endif


namespace Dakota {

class ProblemDescDB;


/// Base class for the variables class hierarchy.

/** The Variables class is the base class for the class hierarchy
    providing design, uncertain, and state variables for continuous
    and discrete domains within a Model.  Using the fundamental
    arrays from the input specification, different derived classes
    define different views of the data.  For memory efficiency and
    enhanced polymorphism, the variables hierarchy employs the
    "letter/envelope idiom" (see Coplien "Advanced C++", p. 133), for
    which the base class (Variables) serves as the envelope and one of
    the derived classes (selected in Variables::get_variables())
    serves as the letter. */

class Variables
{
  //
  //- Heading: Friends
  //

  /// equality operator
  friend bool operator==(const Variables& vars1, const Variables& vars2);
  /// inequality operator
  friend bool operator!=(const Variables& vars1, const Variables& vars2);

#ifdef DAKOTA_BOOST
  /// hash_value
  friend std::size_t hash_value(const Variables& vars);

  /// binary_equal_to (since 'operator==' is not suitable for boost/hash_set)
  friend bool binary_equal_to(const Variables& vars1, const Variables& vars2);
#endif

public:

  //
  //- Heading: Constructors, destructor, assignment operator
  //

  /// default constructor
  Variables();
  /// standard constructor
  Variables(const ProblemDescDB& problem_db);
  /// alternate constructor for instantiations on the fly
  Variables(const pair<short,short>& view);
  /// copy constructor
  Variables(const Variables& vars);

  /// destructor
  virtual ~Variables();

  /// assignment operator
  Variables operator=(const Variables& vars);

  //
  //- Heading: Virtual functions
  //

  virtual size_t tv() const; ///< Returns total number of vars

  /// returns the list of discrete variables merged into a continuous array
  virtual const UIntArray& merged_discrete_ids() const;

  // ACTIVE VARIABLES

  /// return the active continuous variables
  virtual const RealVector& continuous_variables() const;
  /// set an active continuous variable
  virtual void continuous_variable(const Real& c_var, const size_t& i);
  /// set the active continuous variables
  virtual void continuous_variables(const RealVector& c_vars);
  /// return the active discrete variables
  virtual const IntVector& discrete_variables() const;
  /// set an active discrete variable
  virtual void discrete_variable(const int& d_var, const size_t& i);
  /// set the active discrete variables
  virtual void discrete_variables(const IntVector& d_vars);

  /// return the active continuous variable labels
  virtual const StringArray& continuous_variable_labels() const;
  /// set the active continuous variable labels
  virtual void continuous_variable_labels(const StringArray& cv_labels);
  /// return the active discrete variable labels
  virtual const StringArray& discrete_variable_labels()   const;
  /// set the active discrete variable labels
  virtual void discrete_variable_labels(const StringArray& dv_labels);

  /// return the active continuous variable types
  virtual const StringArray& continuous_variable_types() const;
  /// return the active discrete variable types
  virtual const StringArray& discrete_variable_types() const;

  /// return the active continuous variable position identifiers
  virtual const UIntArray& continuous_variable_ids() const;

  // INACTIVE VARIABLES

  // This facility would be one way to update the inactive variables for doing
  // mixed iteration (such as OUU).  The set operations would use an
  // insert(sub_array, total_array, total_start, total_end) function to update
  // entries in concatenated arrays.  The disadvantage of this approach is that
  // it is not polymorphic (e.g., does not flexibly accommodate both OUU & UOO).
  //virtual void continuous_design_variables(const RealVector& cdv);
  //virtual void discrete_design_variables(const IntVector& ddv);
  //virtual void uncertain_variables(const RealVector& uv);
  //virtual void continuous_state_variables(const RealVector& csv);
  //virtual void discrete_state_variables(const IntVector& dsv);

  // This facility is polymorphic and it flexibly accomodates both OUU and UOO
  // in nested models.  However, it requires the omission of state variables
  // to be non-ambiguous.
  /// return the inactive continuous variables
  virtual const RealVector& inactive_continuous_variables() const;
  /// set the inactive continuous variables
  virtual void inactive_continuous_variables(const RealVector& i_c_vars);
  /// return the inactive discrete variables
  virtual const IntVector& inactive_discrete_variables() const;
  /// set the inactive discrete variables
  virtual void inactive_discrete_variables(const IntVector& i_d_vars);

  /// return the inactive continuous variable labels
  virtual const StringArray& inactive_continuous_variable_labels() const;
  /// set the inactive continuous variable labels
  virtual void inactive_continuous_variable_labels(const StringArray& i_c_vars);
  /// return the inactive discrete variable labels
  virtual const StringArray& inactive_discrete_variable_labels() const;
  /// set the inactive discrete variable labels
  virtual void inactive_discrete_variable_labels(const StringArray& i_d_vars);

  /// return the inactive continuous variable position identifiers
  virtual const UIntArray& inactive_continuous_variable_ids() const;

  // ALL VARIABLES

  // This facility is used when performing approximate maps to mimic the use
  // of all variables by ApplicationInterface::map file I/O
  virtual size_t acv() const; ///< returns total number of continuous vars
  virtual size_t adv() const; ///< returns total number of discrete vars

  /// returns a single array with all continuous variables
  virtual RealVector all_continuous_variables() const;
  /// sets all continuous variables using a single array
  virtual void all_continuous_variables(const RealVector& a_c_vars);
  /// returns a single array with all discrete variables
  virtual IntVector all_discrete_variables() const;
  /// sets all discrete variables using a single array
  virtual void all_discrete_variables(const IntVector& a_d_vars);

  /// returns a single array with all continuous variable labels
  virtual StringArray all_continuous_variable_labels() const;
  /// sets all continuous variable labels using a single array
  virtual void all_continuous_variable_labels(const StringArray& a_c_v_labels);
  /// returns a single array with all discrete variable labels
  virtual StringArray all_discrete_variable_labels() const;
  /// sets all discrete variable labels using a single array
  virtual void all_discrete_variable_labels(const StringArray& a_d_v_labels);
  /// returns a single array with all variable labels
  virtual StringArray all_variable_labels() const;

  /// return the all discrete variable types
  virtual const StringArray& all_discrete_variable_types() const;

  // INPUT/OUTPUT

  /// read a variables object from an istream
  virtual void read(istream& s);
  /// write a variables object to an ostream
  virtual void write(ostream& s) const;
  /// write a variables object to an ostream in aprepro format
  virtual void write_aprepro(ostream& s) const;

  // For neutral file I/O:
  /// read a variables object in annotated format from an istream
  virtual void read_annotated(istream& s);
  /// write a variables object in annotated format to an ostream
  virtual void write_annotated(ostream& s) const;

  /// write a variables object in tabular format to an ostream
  virtual void write_tabular(ostream& s) const;

  /// read a variables object from the binary restart stream
  virtual void read(BiStream& s);
  /// write a variables object to the binary restart stream
  virtual void write(BoStream& s) const;

  /// read a variables object from a packed MPI buffer
  virtual void read(MPIUnpackBuffer& s);
  /// write a variables object to a packed MPI buffer
  virtual void write(MPIPackBuffer& s) const;

  //
  //- Heading: Member functions
  //

  size_t cv()  const; ///< Returns number of active continuous vars
  size_t dv()  const; ///< Returns number of active discrete vars
  size_t icv() const; ///< returns number of inactive continuous vars
  size_t idv() const; ///< returns number of inactive discrete vars

  /// return an active continuous variable
  const Real& continuous_variable(const size_t& i) const;
  /// return an active discrete variable
  const int&  discrete_variable(const size_t& i)   const;

  /// for use when a deep copy is needed (the representation is _not_ shared)
  Variables copy() const;

  /// reshapes an existing Variables object based on the incoming
  /// variablesComponents
  void reshape(const Sizet2DArray& vars_comps);

  /// returns variablesView
  const pair<short,short>& view() const;
  /// defines variablesView from problem_db attributes
  pair<short,short> get_view(const ProblemDescDB& problem_db) const;

  /// returns the variables identifier string
  const String& variables_id() const;

  /// returns the number of variables for each of the constitutive components
  const Sizet2DArray& variables_components() const;

  /// return all continuous variable types
  const StringArray& all_continuous_variable_types() const;

  /// return all continuous variable position identifiers
  const UIntArray& all_continuous_variable_ids() const;

  /// function to check variablesRep (does this envelope contain a letter)
  bool is_null() const;

protected:

  //
  //- Heading: Constructors
  //

  /// constructor initializes the base class part of letter classes
  /// (BaseConstructor overloading avoids infinite recursion in the
  /// derived class constructors - Coplien, p. 139)
  Variables(BaseConstructor, const ProblemDescDB& problem_db,
	    const pair<short,short>& view);

  //
  //- Heading: Virtual functions
  //

  /// Used by copy() to copy the contents of a letter class
  virtual void copy_rep(const Variables* vars_rep);

  /// Used by reshape() to reshape the contents of a letter class
  virtual void reshape_rep(const Sizet2DArray& vars_comps);

  //
  //- Heading: Member functions
  //

  /// Convenience function for shared code within build_types_ids()
  void uncertain_var_types(size_t& acv_cntr);

  //
  //- Heading: Data
  //

  /// the variables view pair containing active (first) and inactive (second)
  /// view enumerations
  pair<short,short> variablesView;

  /// two-dimensional array containing variable type counts for
  /// design/uncertain/state (first index) by sub-type (second index)
  Sizet2DArray variablesComponents;

  /// array of variable types for the all continuous variables array
  StringArray allContinuousVarTypes;
  /// array of position identifiers for the all continuous variables array
  /** These identifiers define positions of the all continuous variables array
      within the total variable sequence. */
  UIntArray allContinuousVarIds;

  /// an empty real vector returned in get functions when there are
  /// no variables corresponding to the request
  RealVector emptyRealVector;
  /// an empty int vector returned in get functions when there are
  /// no variables corresponding to the request
  IntVector emptyIntVector;
  /// an empty label array returned in get functions when there are
  /// no variables corresponding to the request
  StringArray emptyStringArray;
  /// an empty unsigned int array returned in get functions when there
  /// are no variables corresponding to the request
  UIntArray emptyUIntArray;

private:

  //
  //- Heading: Member functions
  //

  /// Used by the standard envelope constructor to instantiate the
  /// correct letter class
  Variables* get_variables(const ProblemDescDB& problem_db);

  /// Used by the alternate envelope constructor, by read functions,
  /// and by copy() to instantiate a new letter class
  Variables* get_variables(const pair<short,short>& view) const;

  //
  //- Heading: Data
  //

  /// variables identifier string from the input file
  String idVariables;
  /// pointer to the letter (initialized only for the envelope)
  Variables* variablesRep;
  /// number of objects sharing variablesRep
  int referenceCount;
};


// nonvirtual functions can access letter attributes directly (only need to fwd
// member function call when the function could be redefined).
inline size_t Variables::cv() const
{ return continuous_variables().length(); }


inline size_t Variables::dv() const
{ return discrete_variables().length(); }


inline size_t Variables::icv() const
{ return inactive_continuous_variables().length(); }


inline size_t Variables::idv() const
{ return inactive_discrete_variables().length(); }


inline const Real& Variables::continuous_variable(const size_t& i) const
{ return continuous_variables()[i]; }


inline const int& Variables::discrete_variable(const size_t& i) const
{ return discrete_variables()[i]; }


inline const pair<short,short>& Variables::view() const
{ return (variablesRep) ? variablesRep->variablesView : variablesView; }


inline const String& Variables::variables_id() const
{ return (variablesRep) ? variablesRep->idVariables : idVariables; }


inline const Sizet2DArray& Variables::variables_components() const
{
  return (variablesRep) ? variablesRep->variablesComponents
                        : variablesComponents;
}


inline const StringArray& Variables::all_continuous_variable_types() const
{
  return (variablesRep) ? variablesRep->allContinuousVarTypes
                        : allContinuousVarTypes;
}


inline const UIntArray& Variables::all_continuous_variable_ids() const
{
  return (variablesRep) ? variablesRep->allContinuousVarIds
                        : allContinuousVarIds;
}


inline bool Variables::is_null() const
{ return (variablesRep) ? false : true; }


/// global comparison function for Variables
inline bool variables_id_compare(const Variables& vars, const void* id)
{ return ( *(const String*)id == vars.variables_id() ); }


// Having overloaded operators call read/write means that the operators need 
// not be a friend to private data because read/write functions are public.

/// istream extraction operator for Variables.
inline istream& operator>>(istream& s, Variables& vars)
{ vars.read(s); return s; }


/// ostream insertion operator for Variables.
inline ostream& operator<<(ostream& s, const Variables& vars)
{ vars.write(s); return s; }


/// BiStream extraction operator for Variables.
inline BiStream& operator>>(BiStream& s, Variables& vars)
{ vars.read(s); return s; }


/// BoStream insertion operator for Variables.
inline BoStream& operator<<(BoStream& s, const Variables& vars)
{ vars.write(s); return s; }


/// MPIUnpackBuffer extraction operator for Variables.
inline MPIUnpackBuffer& operator>>(MPIUnpackBuffer& s, Variables& vars)
{ vars.read(s); return s; }


/// MPIPackBuffer insertion operator for Variables.
inline MPIPackBuffer& operator<<(MPIPackBuffer& s, const Variables& vars)
{ vars.write(s); return s; }

/// inequality operator for Variables
inline bool operator!=(const Variables& vars1, const Variables& vars2)
{ return !(vars1 == vars2); }

} // namespace Dakota

#endif
