/*  _______________________________________________________________________

    DAKOTA: Design Analysis Kit for Optimization and Terascale Applications
    Copyright (c) 2006, Sandia National Laboratories.
    This software is distributed under the GNU General Public License.
    For more information, see the README file in the top Dakota directory.
    _______________________________________________________________________ */

/*
 *  not_executable(const char *dname) checks whether dname is an executable
 *  file appearing somewhere in $PATH and returns 0 if so, 1 if not found,
 *  2 if found but not executable.
 */

#include <stdlib.h>
#include <string.h>
#include <sys/types.h>
#include <sys/stat.h>
#ifndef _WIN32
#include <unistd.h>
#endif
#ifdef TEST
#include <stdio.h>
#endif

#undef  WIN32
#ifdef __CYGWIN__
#define WIN32
#undef _WIN32
#elif defined(_WIN32)
#define WIN32
#endif

/* The following routines assume ASCII or UTF-encoded Unicode. */

 static void
find_token(const char *s0, const char **s1, const char **s2, const char **s3)
{
	char ebuf[256];
	const char *t;
	const unsigned char *s;
	int q;
	size_t i, n;

 top:
	for(s = (const unsigned char*)s0; *s <= ' '; ++s)
		if (!*s) {
			*s1 = *s2 = *s3 = (const char*)s;
			return;
			}
	*s1 = t = (const char*)s;
	if ((q = *s) == '"' || q == '\'') {
		for(*s1 = (const char*)++s; *s != q; ++s) {
			if (!*s) {
				*s2 = *s3 = (const char*)s;
				return;
				}
			}
		*s2 = (const char*)s;
		*s3 = (const char*)s + 1;
		return;
		}
	while((q = *++s) > ' ' && q != '"' && q != '\'' && q != '$');
	*s2 = *s3 = (const char*)s;
	if (*t == '$' && (n = *s2 - t) <= sizeof(ebuf)) {
		--n;
		for(i = 0; i < n; ++i)
			ebuf[i] = *++t;
		ebuf[n] = 0;
		if ((t = getenv(ebuf))) {
			*s1 = t;
			while(*t)
				++t;
			*s2 = t;
			}
		else {
			s0 = *s3;
			goto top;
			}
		}
	}

/* adjust for quoted strings in arg_list[0] */
 const char**
arg_list_adjust(const char **arg_list, void **a0)
{
	const char **al, *s, *s1, *s2, *s3;
	char *t;
	int n, n0;
	size_t L;

	s = arg_list[0];
	find_token(s = arg_list[0], &s1, &s2, &s3);
	if (!*s3 && s == s1) {
		if (a0)
			*a0 = 0;
		return arg_list;
		}
	n = 0;
	while(arg_list[n++]);
	n0 = n;
	L = s2 - s1 + 1;
	while(*s3) {
		find_token(s3, &s1, &s2, &s3);
		if (s1 == s3)
			break;
		L += s2 - s1 + 1;
		if (*(unsigned char*)s3 <= ' ')
			++n;
		}
	al = (const char**)malloc(n*sizeof(const char*) + L); /* freed implicitly by execvp */
	if (a0)
		*a0 = (void*)al;
	if (!al)
		return arg_list; /* should not happen */
	t = (char*)(al + n);
	find_token(s = arg_list[0], &s1, &s2, &s3);
	for(n = 0;;) {
		al[n++] = t;
		while(s1 < s2)
			*t++ = *s1++;
		while(*(unsigned char*)s3 > ' ') {
			find_token(s3, &s1, &s2, &s3);
			while(s1 < s2)
				*t++ = *s1++;
			}
		*t++ = 0;
		if (!*s3)
			break;
		find_token(s3, &s1, &s2, &s3);
		if (s1 == s3)
			break;
		}
	n0 = 0;
	while((al[n] = arg_list[++n0]))
		++n;
	for(n0 = 0; n0 < n && strchr(al[n0], '='); ++n0);
	if (n0 && n0 < n) {
		if (!a0) {
			for(n = 0; n < n0; ++n)
				putenv((char*)al[n]);
			}
		al += n0;
		}
	return al;
	}

 int
not_executable(const char *dname)
{
	static char *p0;
	struct stat sb;
	char *b, buf[2048], *p, *p1;
	const char *a2[2], **al;
	int rc, sv;
	size_t dlen, plen;
	void *a0;
#ifdef WIN32
	char dbuf[128];
#else
	static uid_t myuid;
	static gid_t mygid;
#endif

	/* allow shell assignments and quotes around executable names */
	/* that may involve blanks */
	a2[0] = dname;
	a2[1] = 0;
	al = arg_list_adjust(a2,&a0);
	dname = al[0];

	rc = 0;
	if (!p0) {
		p0 = getenv("PATH");
#ifdef WIN32
		if (!p0)
			p0 = getenv("Path");
#else
		myuid = geteuid();
		mygid = getegid();
#endif
		if (p0)
			while(*p0 <= ' ' && *p0)
				++p0;
		else
			p0 = "";
		}
#ifdef WIN32
	/* make sure we have a suitable suffix */
	if ((p = strrchr(dname, '.'))) {
		if (strlen(++p) != 3)
			p = 0;
		else {
			for(b = dbuf; *p; ++b, ++p)
				*b = tolower(*p);
			*b = 0;
			if (strcmp(dbuf,"exe") && strcmp(dbuf,"bat") && strcmp(dbuf, "cmd"))
				p = 0;
			}
		}
	if (!p) {
		dlen = strlen(dname);
		if (dlen + 5 > sizeof(dbuf)) {
			rc = 1;
			goto ret;
			}
		strcpy(dbuf, dname);
		strcpy(dbuf+dlen, ".exe");
		dname = dbuf;
		}
	/* . is always implicitly in $Path under MS Windows; check it now */
	if (!stat(dname, &sb))
		goto ret;
#endif
	dlen = strlen(dname);
	p = p0;
	if (strchr(dname, '/')
#ifdef _WIN32
		|| strchr(dname, '\\') || (dlen > 2 && dname[1] == ':')
#endif
		)
		p = "";
	rc = 1;
	for(;;) {
		for(p1 = p;; ++p1) {
			switch(*p1) {
			  case 0:
#ifdef _WIN32
			  case ';':
#else
			  case ':':
#endif
				goto break2;
			  }
			}
 break2:
		if (p1 == p || (p1 == p + 1 && *p == '.'))
			sv = stat(dname, &sb);
		else {
			plen = p1 - p;
			while(plen && p[plen-1] <= ' ')
				--plen;
			if (plen + dlen + 2 > sizeof(buf))
				sv = 1;
			else {
				strncpy(buf,p,plen);
				b = buf + plen;
				*b++ = '/';
				strcpy(b, dname);
				sv = stat(buf, &sb);
				}
			}
		if (!sv) {
#ifdef __CYGWIN__
			rc = 2;
			if (sb.st_mode & (S_IXUSR|S_IXGRP|S_IXOTH)) {
				rc = 0;
				goto ret;
				}
#elif defined(_WIN32)
			rc = 0;
			goto ret;
#else
			rc = 2;
			if (sb.st_uid == myuid) {
				if (sb.st_mode & S_IXUSR)
					goto ret0;
				}
			else if (sb.st_gid == mygid) {
				if (sb.st_mode & S_IXGRP)
					goto ret0;
				}
			else if (sb.st_mode & S_IXOTH) {
 ret0:				rc = 0;
				goto ret;
				}
#endif
			}
		if (!*p1)
			break;
		for(p = p1 + 1; *p <= ' '; ++p)
		while(*p <= ' ' && *p)
			if (!*p)
				goto ret;
		}
 ret:
	if (a0)
		free(a0);
	return rc;
	}

#ifdef TEST
 static char *progname;
 static int
usage(int rc)
{
	fprintf(rc ? stderr : stdout, "Usage: %s file [file...]\n\
	to report whether the files exist and are executable.\n", progname);
	return rc;
	}

 int
main(int argc, char **argv)
{
	char *s;
	static const char *what[3] = {
		"executable", "not found", "found but not executable" };

	progname = *argv;
	if (argc < 2)
		return usage(1);
	s = argv[1];
	if (*s == '-') {
		if (((s[1] == '?' || s[1] == 'h') && !s[2])
			  || (s[1] == '-' && !strcmp(s+2,"help")))
			return usage(0);
		if (s[1] == '-' && s[2] == 0) {
			--argc;
			++argv;
			if (argc < 2)
				return usage(1);
			}
		else
			return usage(1);
		}
	while((s = *++argv))
		printf("%s is %s.\n", s, what[not_executable(s)]);
	return 0;
	}
#endif
