/*  _________________________________________________________________________
 *
 *  UTILIB: A utility library for developing portable C++ codes.
 *  Copyright (c) 2001, Sandia National Laboratories.
 *  This software is distributed under the GNU Lesser General Public License.
 *  For more information, see the README file in the top UTILIB directory.
 *  _________________________________________________________________________
 */

#include <utilib/Any-shallow.h>
#include <iostream>

using std::cout;
using std::endl;
using std::setw;
using std::string;
using utilib::Any;
using utilib::AnyRef;
using utilib::AnyFixedRef;

namespace {

class FOO
  {
  public:
    FOO()
      { cout << "Constructing FOO" << endl; }
    FOO(const FOO &rhs)
      { cout << "Copy Constructing FOO" << endl; }

    // equality test required by Any API
    bool operator==(const FOO &rhs) const
      { return true; }
    // inequality test required by Any API
    bool operator<(const FOO &rhs) const
      { return false; }
  };

}

/// Macro for incorporating macros into string constants
/** Macro pair to "stringify" any macro (Part 1).  This works by first
 *  quoting the macro name and then expanding the macro into it's
 *  actual value. [see _QUOT(x)]
 */
#define QUOTE(x) _QUOTE(x)
/** Macro pair to "stringify" any macro (Part 2).  This works by first
 *  quoting the macro name and then expanding the macro into it's
 *  actual value. [see QUOT(x)]
 */
#define _QUOTE(x) #x

/** Macro to "demangle" the type stored in the Any to a known (fixed)
 *  string.  That is, if the Any currently stores an instance of TYPE,
 *  then "TYPE" is returned, otherwise the mangled typename of the
 *  stored type is returned.  This prevents false failures caused by
 *  different platforms using different mangling rules.
 */
#define DEMANGLE_TYPE(ANY, TYPE) \
  ( strcmp(ANY.type().name(), typeid(TYPE).name()) == 0   \
           ? QUOTE(TYPE) : ANY.type().name() )

#define PRINT_ANY(ANY, TYPE)      \
do {                              \
  int ans;                        \
  double d_val = 0;               \
  long   l_val = 0;               \
  int    i_val = 0;               \
  short  s_val = 0;               \
  cout << QUOTE(ANY) ": type="  << setw(6) << DEMANGLE_TYPE(ANY,TYPE);   \
  ans = ANY.extract(d_val);       \
  cout << ", d = " << setw(3) << d_val << " (" << setw(4) << ans << ")"; \
  ans = ANY.extract(i_val);       \
  cout << ", i = " << setw(3) << i_val << " (" << setw(4) << ans << ")"; \
  ans = ANY.extract(s_val);       \
  cout << ", s = " << setw(3) << s_val << " (" << setw(4) << ans << ")"; \
  ans = ANY.extract(l_val);       \
  cout << ", l = " << setw(3) << l_val << " (" << setw(4) << ans << ")"; \
  cout << endl;                   \
} while (false)

/** Macro to strip off the file name & line number from the exception
 *  message (so we don't have to edit the reference output
 *  (anyShallow.qa) every time wo do *any* edit to the source code.
 *  Also strips off anything after and including the first "'" character
 *  to prevent comparison failures based on differing mangling rules
 */
#define CATCH(STR, EXCEPTION)   \
  catch (EXCEPTION err)         \
    {                           \
    string msg(err.what());     \
    size_t pos = msg.find_last_of("0123456789");  \
    if ( pos == string::npos )  \
      { pos = 0; }              \
    else                        \
      { ++pos; }                \
    size_t end = msg.substr(pos).find("'");  \
    cout << STR << " failed" << msg.substr(pos, end) << endl;  \
    }                                        \
  catch (...)                                \
    { cout << STR << " failed: (unexpected exception)" << endl; }  \
  do {} while (false)


void AnyRefTest(const AnyRef& myAny)
  {
  double &tmp = const_cast<double&>(myAny.expose<double>());
  cout << tmp << " -> ";
  tmp++;
  cout << tmp;
  }

void AnyFixedRefTest(AnyFixedRef myAny)
  {
  double &tmp = const_cast<double&>(myAny.expose<double>());
  cout << tmp << " -> ";
  myAny = tmp + 2;
  cout << tmp;
  }


int test_anyShallow(int,char**)
  {
  Any::throwCastExceptions() = false;

  short s;
  int i;
  long l;
  double d;
  utilib::Any foo(5);
  utilib::Any bar = foo;
  const utilib::Any const_bar = 6.5;

  bar = 5.5;

  PRINT_ANY(foo, int);

  //foo.set(7.7);
  foo = 7.7;
  PRINT_ANY(foo, double);

  s = 7;
  //foo.set(s);
  foo = s;
  PRINT_ANY(foo, short);

  l = 8;
  //foo.set(l);
  foo = l;
  PRINT_ANY(foo, long);

  /* no longer valid as Any now stores CONST objects
  foo.expose<long>() = 9;
  PRINT_ANY(foo, long);
  */

  foo = bar;
  PRINT_ANY(foo, double);

  foo = const_bar;
  PRINT_ANY(foo, double);

  cout << "typeof<int> = " << foo.is_type(typeid(int))
       << "; typeof<double> = " << foo.is_type(typeid(double)) << endl;

  // The following will throw an exception
  //foo.expose<int>() = 1;

  i = 5;
  bar &= i;
  Any bar1(bar, true);
  Any bar2(i, false);
  cout << "reference tests: ";
  cout << DEMANGLE_TYPE(bar, int) << "(" << bar.is_reference() << "), ";
  cout << DEMANGLE_TYPE(bar1, Any) << "(" << bar1.is_reference() << "), ";
  cout << DEMANGLE_TYPE(bar2, int) << "(" << bar2.is_reference() << ")" 
       << endl;

  cout << "i = " << i << ", ";
  cout << "Any(&i) = " << bar.expose<int>() << ", ";
  cout << "Any(i) = " << bar2.expose<int>() << endl;

  ++i;
  cout << "i = " << i << ", ";
  cout << "Any(&i) = " << bar.expose<int>() << ", ";
  cout << "Any(i) = " << bar2.expose<int>() << endl;

  ++utilib::anyref_cast<int>(bar);
  ++utilib::anyref_cast<int>(bar2);
  cout << "i = " << i << ", ";
  cout << "Any(&i) = " << bar.expose<int>() << ", ";
  cout << "Any(i) = " << bar2.expose<int>() << endl;

  i = utilib::anyval_cast<int>(bar2);
  cout << "i = " << i << ", ";
  cout << "Any(&i) = " << bar.expose<int>() << ", ";
  cout << "Any(i) = " << bar2.expose<int>() << endl;

  // This will call both constructor & copy constructor
  FOO myFOO;
  Any anyFoo1(myFOO);
  // This will only call the constructor
  Any anyFoo2;
  anyFoo2.set<FOO>();

  // Test immutable Anys
  int j = 1;
  foo.set(j,true,true);
  cout << "j = " << j;
  foo = bar;
  cout << "; j = " << j;
  foo.set(11);
  cout << "; j = " << j << endl;
  try 
    { 
    bar = 10.5; 
    foo = bar;
    }
  CATCH("Assignment", utilib::bad_any_typeid);

  try 
    { foo = 10.5; }
  CATCH("Assignemnt", utilib::bad_any_typeid);
  
  try 
    { foo.set<long>(); }
  CATCH("Assignemnt", utilib::bad_any_typeid);

  Any::throwCastExceptions() = true;
  try
    { foo.extract(d); }
  CATCH("Extraction", utilib::bad_any_cast);


  // test implicit casting to reference anys
  {
  d = 1;
  AnyFixedRef tmp(d);
  cout << "refcount = " << tmp.anyCount() << endl;
  AnyRef tmp4(tmp);
  cout << "refcount = " << tmp.anyCount() << endl;
  AnyRef tmp3(tmp4);
  cout << "refcount = " << tmp.anyCount() << endl;
  Any tmp5;
  tmp5 = tmp4;
  cout << "refcount = " << tmp.anyCount() << endl;
  cout << "Test AnyRef: (d = " << d << ") ";
  AnyRefTest(tmp3);
  cout << " (d = " << d << ")" << endl;

  d = 5;
  cout << "Test cast double -> AnyRef: (d = " << d << ") ";
  AnyRefTest(d);
  cout << " (d = " << d << ")" << endl;

  d = 10;
  Any tmp2(d, true);
  cout << "Test cast Any(is_ref = true) -> AnyRef: (d = " << d << ") ";
  AnyRefTest(tmp2);
  cout << " (d = " << d << ", Any = " << tmp2.expose<double>() << ")" << endl;

  d = 20;
  tmp2 = d;
  cout << "Test cast Any(is_ref = false) -> AnyRef: (d = " << d << ") ";
  AnyRefTest(tmp2);
  cout << " (d = " << d << ", Any = " << tmp2.expose<double>() << ")" << endl;
  }

  // test implicit casting to immutable anys
  {
  d = 1;
  AnyRef tmp(d);
  cout << "refcount = " << tmp.anyCount() << endl;
  AnyFixedRef tmp4(tmp);
  cout << "refcount = " << tmp4.anyCount() << endl;
  AnyFixedRef tmp3(tmp4);
  cout << "refcount = " << tmp4.anyCount() << endl;
  Any tmp5 = tmp4;
  cout << "refcount = " << tmp4.anyCount() << endl;
  cout << "Test AnyFixedRef: (d = " << d << ") ";
  AnyFixedRefTest(tmp3);
  cout << " (d = " << d << ")" << endl;

  d = 5;
  cout << "Test cast double -> AnyFixedRef: (d = " << d << ") ";
  AnyFixedRefTest(d);
  cout << " (d = " << d << ")" << endl;

  d = 10;
  Any tmp2(d, true);
  cout << "Test cast Any(is_ref = true) -> AnyFixedRef: (d = " << d << ") ";
  AnyFixedRefTest(tmp2);
  cout << " (d = " << d << ", Any = " << tmp2.expose<double>() << ")" << endl;

  d = 20;
  tmp2 = d;
  cout << "Test cast Any(is_ref = false) -> AnyFixedRef: (d = " << d << ") ";
  AnyFixedRefTest(tmp2);
  cout << " (d = " << d << ", Any = " << tmp2.expose<double>() << ")" << endl;
  }

  // test equality tests
  {
     Any tmp1 = i;
     Any tmp1a = tmp1;
     Any tmp2(i, true);
     Any tmp3(j, true);
     Any tmp4(d, true);
     cout << "test i ref == i:  " << tmp1.reference_same_data(tmp1a) << endl;
     cout << "test i ref == &i: " << tmp1.reference_same_data(tmp2) << endl;
     cout << "test i == i:  " << (tmp1 == tmp1a) << endl;
     cout << "test i == &i: " << (tmp1 == tmp2) << endl;
     cout << "test i == j:  " << (tmp1 == tmp3) << endl;
     cout << "test i == d:  " << (tmp1 == tmp4) << endl;
     j = i;
     cout << "test i == j:  " << (tmp1 == tmp3) << endl;
  }

  return 0;
  }
