/*  _______________________________________________________________________

    DAKOTA: Design Analysis Kit for Optimization and Terascale Applications
    Copyright (c) 2006, Sandia National Laboratories.
    This software is distributed under the GNU General Public License.
    For more information, see the README file in the top Dakota directory.
    _______________________________________________________________________ */

//- Class:       RecastModel
//- Description: A model mapping variables into responses using
//-              primary/secondary function pointers.
//- Owner:       Mike Eldred
//- Checked by:
//- Version: $Id: RecastModel.H 5091 2008-06-12 00:20:53Z mseldre $

#ifndef RECAST_MODEL_H
#define RECAST_MODEL_H

#include "DakotaModel.H"
#include "DakotaInterface.H"
#include "ParallelLibrary.H"


namespace Dakota {

/// Derived model class which provides a thin wrapper around a sub-model
/// in order to recast the form of its inputs and/or outputs.

/** The RecastModel class uses function pointers to allow recasting of
    the subModel input/output into new problem forms.  This is
    currently used to recast SBO approximate subproblems, but can be
    used for multiobjective, input/output scaling, and other problem
    modifications in the future. */

class RecastModel: public Model
{
public:
  
  //
  //- Heading: Constructor and destructor
  //

  /// standard constructor
  RecastModel(Model& sub_model, const Sizet2DArray& vars_map_indices,
	      bool nonlinear_vars_mapping,
	      void (*variables_map)      (const Variables& recast_vars,
					  Variables& sub_model_vars),
	      void (*set_map)            (const ActiveSet& recast_set,
					  ActiveSet& sub_model_set),
	      const Sizet2DArray& primary_resp_map_indices,
	      const Sizet2DArray& secondary_resp_map_indices,
	      size_t recast_secondary_offset,
	      const BoolDequeArray& nonlinear_resp_mapping,
	      void (*primary_resp_map)   (const Variables& sub_model_vars,
					  const Variables& recast_vars,
					  const Response& sub_model_response,
					  Response& recast_response),
	      void (*secondary_resp_map) (const Variables& sub_model_vars,
					  const Variables& recast_vars,
					  const Response& sub_model_response,
					  Response& recast_response));
  /// alternate constructor
  RecastModel(Model& sub_model, //size_t num_deriv_vars,
	      size_t num_recast_primary_fns, size_t num_recast_secondary_fns,
	      size_t recast_secondary_offset);

  /// destructor
  ~RecastModel();
    
  //
  //- Heading: Member functions
  //

  /// completes initialization of the RecastModel after alternate construction
  void initialize(const Sizet2DArray& vars_map_indices,
		  bool nonlinear_vars_mapping,
		  void (*variables_map)     (const Variables& recast_vars,
					     Variables& sub_model_vars),
		  void (*set_map)           (const ActiveSet& recast_set,
					     ActiveSet& sub_model_set),
		  const Sizet2DArray& primary_resp_map_indices,
		  const Sizet2DArray& secondary_resp_map_indices,
		  const BoolDequeArray& nonlinear_resp_mapping,
		  void (*primary_resp_map)  (const Variables& sub_model_vars,
					     const Variables& recast_vars,
					     const Response& sub_model_response,
					     Response& recast_response),
		  void (*secondary_resp_map)(const Variables& sub_model_vars,
					     const Variables& recast_vars,
					     const Response& sub_model_response,
					     Response& recast_response));

  /// override the submodel's derivative estimation behavior
  void submodel_supports_estimated_derivatives(bool ssed_flag);

protected:

  //
  //- Heading: Virtual function redefinitions
  //

  /// portion of compute_response() specific to RecastModel
  /// (forward to subModel.compute_response())
  void derived_compute_response(const ActiveSet& set);
  /// portion of asynch_compute_response() specific to RecastModel
  /// (forward to subModel.asynch_compute_response())
  void derived_asynch_compute_response(const ActiveSet& set);
  /// portion of synchronize() specific to RecastModel
  /// (forward to subModel.synchronize())
  const ResponseArray& derived_synchronize();
  /// portion of synchronize_nowait() specific to RecastModel
  /// (forward to subModel.synchronize_nowait())
  const IntResponseMap& derived_synchronize_nowait();

  /// return sub-iterator, if present, within subModel
  Iterator& subordinate_iterator();
  /// return surrogate model, if present, within subModel
  Model& surrogate_model();
  /// return truth model, if present, within subModel
  Model& truth_model();
  /// add subModel to list and recurse into subModel
  void derived_subordinate_models(ModelList& ml, bool recurse_flag);
  /// pass request to subModel if recursing and then update from it
  void update_from_subordinate_model(bool recurse_flag = true);
  /// return subModel interface
  Interface& interface();

  /// forward to subModel
  void surrogate_function_indices(const IntSet& surr_fn_indices);

  // forward to subModel
  void surrogate_bypass(bool bypass_flag);

  /// builds the subModel approximation
  void build_approximation();
  /// builds the subModel approximation
  bool build_approximation(const Variables& vars, const Response& response);
  /// updates the subModel approximation
  void update_approximation(const Variables& vars, const Response& response,
			    bool rebuild_flag);
  /// updates the subModel approximation
  void update_approximation(const VariablesArray& vars_array,
			    const ResponseArray& resp_array, bool rebuild_flag);
  /// appends the subModel approximation
  void append_approximation(const Variables& vars, const Response& response,
			    bool rebuild_flag);
  /// appends the subModel approximation
  void append_approximation(const VariablesArray& vars_array,
			    const ResponseArray& resp_array, bool rebuild_flag);

  /// retrieve the set of Approximations from the subModel
  Array<Approximation>& approximations();
  /// retrieve the approximation coefficients from the subModel
  const RealVectorArray& approximation_coefficients();
  /// set the approximation coefficients within the subModel
  void approximation_coefficients(const RealVectorArray& approx_coeffs);
  /// print a particular set of approximation coefficients within the subModel
  void print_coefficients(ostream& s, size_t index) const;
  /// retrieve the approximation variances from the subModel
  const RealVector& approximation_variances(const RealVector& c_vars);
  /// retrieve the approximation data from the subModel
  const List<SurrogateDataPoint>& approximation_data(size_t index);

  /// RecastModel only supports parallelism in subModel, so this
  /// virtual function redefinition is simply a sanity check.
  void component_parallel_mode(short mode);

  /// return subModel local synchronization setting
  String local_eval_synchronization();
  /// return subModel local evaluation concurrency
  int local_eval_concurrency();
  /// flag which prevents overloading the master with a multiprocessor
  /// evaluation (request forwarded to subModel)
  bool derived_master_overload() const;

  /// set up RecastModel for parallel operations (request forwarded to subModel)
  void derived_init_communicators(const int& max_iterator_concurrency,
				  bool recurse_flag = true);
  /// set up RecastModel for serial operations (request forwarded to subModel).
  void derived_init_serial();
  /// set active parallel configuration within subModel
  void derived_set_communicators(const int& max_iterator_concurrency,
				 bool recurse_flag = true);
  /// deallocate communicator partitions for the RecastModel (request forwarded
  /// to subModel)
  void derived_free_communicators(const int& max_iterator_concurrency,
				  bool recurse_flag = true);

  /// Service subModel job requests received from the master.
  /// Completes when a termination message is received from stop_servers().
  void serve();
  /// executed by the master to terminate subModel server operations
  /// when RecastModel iteration is complete.
  void stop_servers();

  /// return the subModel interface identifier
  const String& interface_id() const;
  /// return the current evaluation id for the RecastModel (request
  /// forwarded to subModel)
  int evaluation_id() const;
  /// set the evaluation counter reference points for the RecastModel
  /// (request forwarded to subModel)
  void set_evaluation_reference();
  /// request fine-grained evaluation reporting within subModel
  void fine_grained_evaluation_counters();
  /// print the evaluation summary for the RecastModel (request
  /// forwarded to subModel)
  void print_evaluation_summary(ostream& s, bool minimal_header = false,
				bool relative_count = true) const;

private:

  //
  //- Heading: Convenience member functions
  //

  /// Uses recastFnMap to convert incoming recast_set from iterator
  /// into sub_model_set for use with subModel.
  void set_mapping(const ActiveSet& recast_set, ActiveSet& sub_model_set);

  /// update current variables/labels/bounds/targets from subModel
  void update_from_sub_model();

  //
  //- Heading: Data members
  //

  /// the sub-model underlying the function pointers
  Model subModel;

  /// For each subModel variable, identifies the indices of the recast
  /// variables used to define it (maps RecastModel variables to
  /// subModel variables)
  Sizet2DArray varsMapIndices;
  /// boolean set to true if the variables mapping involves a nonlinear
  /// transformation.  Used in set_mapping() to manage the requirement for
  /// gradients within the Hessian transformations.  This does not require
  /// a BoolDeque for each individual variable, since response gradients and
  /// Hessians are managed per function, not per variable.
  bool nonlinearVarsMapping;

  /// set to true if non-NULL primaryRespMapping or secondaryRespMapping
  /// are supplied
  bool respMapping;
  /// For each recast primary function, identifies the indices of the
  /// subModel functions used to define it (maps subModel response
  /// to RecastModel Response).
  Sizet2DArray primaryRespMapIndices;
  /// For each recast secondary function, identifies the indices of
  /// the subModel functions used to define it (maps subModel response
  /// to RecastModel response).
  Sizet2DArray secondaryRespMapIndices;
  /// array of BoolDeques, one for each recast response function.  Each
  /// BoolDeque defines which subModel response functions contribute to the
  /// recast function using a nonlinear mapping.  Used in set_mapping() to
  /// augment the subModel function value/gradient requirements.
  BoolDequeArray nonlinearRespMapping;

  /// map of recast active set passed to derived_asynch_compute_response().
  /// Needed for currentResponse update in synchronization routines.
  IntActiveSetMap recastSetMap;
  /// map of recast variables used by derived_asynch_compute_response().
  /// Needed for primaryRespMapping() and secondaryRespMapping() in
  /// synchronization routines.
  IntVariablesMap recastVarsMap;
  /// map of subModel variables used by derived_asynch_compute_response().
  /// Needed for primaryRespMapping() and secondaryRespMapping() in
  /// synchronization routines.
  IntVariablesMap subModelVarsMap;
  /// array of recast responses used by RecastModel::derived_synchronize()
  ResponseArray recastResponseArray;
  /// map of recast responses used by RecastModel::derived_synchronize_nowait()
  IntResponseMap recastResponseMap;

  /// holds pointer for variables mapping function passed in ctor/initialize
  void (*variablesMapping)     (const Variables& recast_vars,
			        Variables& sub_model_vars);
  /// holds pointer for set mapping function passed in ctor/initialize
  void (*setMapping)           (const ActiveSet& recast_set,
			        ActiveSet& sub_model_set);
  /// holds pointer for primary response mapping function passed in
  /// ctor/initialize
  void (*primaryRespMapping)   (const Variables& sub_model_vars,
				const Variables& recast_vars,
				const Response& sub_model_response,
				Response& recast_response);
  /// holds pointer for secondary response mapping function passed in
  /// ctor/initialize
  void (*secondaryRespMapping) (const Variables& sub_model_vars,
				const Variables& recast_vars,
				const Response& sub_model_response,
				Response& recast_response);
};


inline RecastModel::~RecastModel()
{ } // Virtual destructor handles referenceCount at Strategy level.


inline void RecastModel::submodel_supports_estimated_derivatives(bool ssed_flag)
{ subModel.supports_estimated_derivatives(ssed_flag); }


inline Iterator& RecastModel::subordinate_iterator()
{ return subModel.subordinate_iterator(); }


inline Model& RecastModel::surrogate_model()
{ return subModel.surrogate_model(); }


inline Model& RecastModel::truth_model()
{ return subModel.truth_model(); }


inline void RecastModel::
derived_subordinate_models(ModelList& ml, bool recurse_flag)
{
  ml.insert(subModel);
  if (recurse_flag)
    subModel.derived_subordinate_models(ml, true);
}


inline void RecastModel::update_from_subordinate_model(bool recurse_flag)
{
  // data flows from the bottom-up, so recurse first
  if (recurse_flag)
    subModel.update_from_subordinate_model(recurse_flag);
  // now pull the latest updates from subModel
  update_from_sub_model();
}


inline Interface& RecastModel::interface()
{ return subModel.interface(); }


inline void RecastModel::surrogate_bypass(bool bypass_flag)
{ subModel.surrogate_bypass(bypass_flag); }


inline void RecastModel::
surrogate_function_indices(const IntSet& surr_fn_indices)
{ subModel.surrogate_function_indices(surr_fn_indices); }


inline void RecastModel::build_approximation()
{ subModel.build_approximation(); }


inline bool RecastModel::
build_approximation(const Variables& vars, const Response& response)
{ return subModel.build_approximation(vars, response); }


inline void RecastModel::
update_approximation(const Variables& vars, const Response& response,
		     bool rebuild_flag)
{ subModel.update_approximation(vars, response, rebuild_flag); }


inline void RecastModel::
update_approximation(const VariablesArray& vars_array,
		     const ResponseArray& resp_array, bool rebuild_flag)
{ subModel.update_approximation(vars_array, resp_array, rebuild_flag); }


inline void RecastModel::
append_approximation(const Variables& vars, const Response& response,
		     bool rebuild_flag)
{ subModel.append_approximation(vars, response, rebuild_flag); }


inline void RecastModel::
append_approximation(const VariablesArray& vars_array,
		     const ResponseArray& resp_array, bool rebuild_flag)
{ subModel.append_approximation(vars_array, resp_array, rebuild_flag); }


inline Array<Approximation>& RecastModel::approximations()
{ return subModel.approximations(); }


inline const RealVectorArray& RecastModel::approximation_coefficients()
{ return subModel.approximation_coefficients(); }


inline void RecastModel::
approximation_coefficients(const RealVectorArray& approx_coeffs)
{ subModel.approximation_coefficients(approx_coeffs); }


inline void RecastModel::print_coefficients(ostream& s, size_t index) const
{ subModel.print_coefficients(s, index); }


inline const RealVector& RecastModel::
approximation_variances(const RealVector& c_vars)
{ return subModel.approximation_variances(c_vars); }


inline const List<SurrogateDataPoint>& RecastModel::
approximation_data(size_t index)
{ return subModel.approximation_data(index); }


inline void RecastModel::component_parallel_mode(short mode)
{
  //if (mode != SUB_MODEL) {
  //  Cerr << "Error: RecastModel only supports the SUB_MODEL component "
  //       << "parallel mode." << endl;
  //  abort_handler(-1);
  //}

  // Since we don't want the thin recast wrapper interfering with subModel
  // parallelism, we forward the parallel mode to the subModel.  This differs
  // from all other derived Model implementations (which utilize the mode
  // locally and do not forward it).
  subModel.component_parallel_mode(mode);
}


inline String RecastModel::local_eval_synchronization()
{ return subModel.local_eval_synchronization(); }


inline int RecastModel::local_eval_concurrency()
{ return subModel.local_eval_concurrency(); }


inline bool RecastModel::derived_master_overload() const
{ return subModel.derived_master_overload(); }


inline void RecastModel::
derived_init_communicators(const int& max_iterator_concurrency,
			   bool recurse_flag)
{ if (recurse_flag) subModel.init_communicators(max_iterator_concurrency); }


inline void RecastModel::derived_init_serial()
{ subModel.init_serial(); }


inline void RecastModel::
derived_set_communicators(const int& max_iterator_concurrency,
			  bool recurse_flag)
{
  if (recurse_flag)
    subModel.set_communicators(max_iterator_concurrency);

  // the following assignment is not overridden in Model::set_communicators()
  // since RecastModels do not define the ie_parallel_level
  asynchEvalFlag = subModel.asynch_flag();
}


inline void RecastModel::
derived_free_communicators(const int& max_iterator_concurrency,
			   bool recurse_flag)
{ if (recurse_flag) subModel.free_communicators(max_iterator_concurrency); }


inline void RecastModel::serve()
{ subModel.serve(); }


inline void RecastModel::stop_servers()
{ subModel.stop_servers(); }


inline const String& RecastModel::interface_id() const
{ return subModel.interface_id(); }


inline int RecastModel::evaluation_id() const
{ return subModel.evaluation_id(); }


inline void RecastModel::set_evaluation_reference()
{ subModel.set_evaluation_reference(); }


inline void RecastModel::fine_grained_evaluation_counters()
{ subModel.fine_grained_evaluation_counters(); }


inline void RecastModel::
print_evaluation_summary(ostream& s, bool minimal_header,
			 bool relative_count) const
{ subModel.print_evaluation_summary(s, minimal_header, relative_count); }

} // namespace Dakota

#endif
