/*
================================================================================
    PROJECT:

        John Eddy's Genetic Algorithms (JEGA)

    CONTENTS:

        Implementation of class BasicFitnessRecord.

    NOTES:

        See notes of BasicFitnessRecord.hpp.

    PROGRAMMERS:

        John Eddy (jpeddy@sandia.gov) (JE)

    ORGANIZATION:

        Sandia National Laboratories

    COPYRIGHT:

        See the LICENSE file in the top level JEGA directory.

    VERSION:

        2.0.0

    CHANGES:

        Wed Dec 21 15:54:57 2005 - Original Version (JE)

================================================================================
*/




/*
================================================================================
Document This File
================================================================================
*/
/** \file
 * \brief Contains the implementation of the BasicFitnessRecord class.
 */




/*
================================================================================
Includes
================================================================================
*/
// JEGAConfig.hpp should be the first include in all JEGA files.
#include <../Utilities/include/JEGAConfig.hpp>

#include <BasicFitnessRecord.hpp>
#include <utilities/include/Math.hpp>
#include <../Utilities/include/Design.hpp>
#include <utilities/include/EDDY_DebugScope.hpp>







/*
================================================================================
Namespace Using Directives
================================================================================
*/
using namespace JEGA::Utilities;







/*
================================================================================
Begin Namespace
================================================================================
*/
namespace JEGA {
    namespace Algorithms {





/*
================================================================================
Static Member Data Definitions
================================================================================
*/








/*
================================================================================
Mutators
================================================================================
*/








/*
================================================================================
Accessors
================================================================================
*/








/*
================================================================================
Public Methods
================================================================================
*/


bool
BasicFitnessRecord::AddFitness(
    const Design* des,
    double fitness
    )
{
    EDDY_FUNC_DEBUGSCOPE
    return _data.AddValue(des, fitness);
}

double
BasicFitnessRecord::GetAverageFitness(
    ) const
{
    EDDY_FUNC_DEBUGSCOPE
    double avg = this->_data.GetAverageValue();
    return avg == DesignDoubleMap::MIN_POSSIBLE ?
                     DesignDoubleMap::MAX_POSSIBLE : avg;
}





/*
================================================================================
Subclass Visible Methods
================================================================================
*/








/*
================================================================================
Subclass Overridable Methods
================================================================================
*/


double
BasicFitnessRecord::GetFitness(
    const Design& des
    ) const
{
    EDDY_FUNC_DEBUGSCOPE

    // if the fitness comes back unknown, make it large negative so
    // that it keeps with the idea of lower fitness is worse.
    double ftns = this->_data.GetValue(des);
    return ftns == DesignDoubleMap::MAX_POSSIBLE ?
                     DesignDoubleMap::MIN_POSSIBLE : ftns;
}

double
BasicFitnessRecord::GetMaxFitness(
    ) const
{
    EDDY_FUNC_DEBUGSCOPE
    return this->_data.GetMaxValue();
}


double
BasicFitnessRecord::GetMinFitness(
    ) const
{
    EDDY_FUNC_DEBUGSCOPE
    return this->_data.GetMinValue();
}

double
BasicFitnessRecord::GetTotalFitness(
    ) const
{
    EDDY_FUNC_DEBUGSCOPE
    return this->_data.GetTotalValue();
}

std::size_t
BasicFitnessRecord::GetSize(
    ) const
{
    EDDY_FUNC_DEBUGSCOPE
    return _data.size();
}




/*
================================================================================
Private Methods
================================================================================
*/








/*
================================================================================
Structors
================================================================================
*/


BasicFitnessRecord::BasicFitnessRecord(
    ) :
        FitnessRecord(),
        _data()
{
}


BasicFitnessRecord::~BasicFitnessRecord(
    )
{
}






/*
================================================================================
End Namespace
================================================================================
*/
    } // namespace Algorithms
} // namespace JEGA
