/*  _______________________________________________________________________

    DAKOTA: Design Analysis Kit for Optimization and Terascale Applications
    Copyright (c) 2006, Sandia National Laboratories.
    This software is distributed under the GNU General Public License.
    For more information, see the README file in the top Dakota directory.
    _______________________________________________________________________ */

//- Class:       DistinctConstraints
//- Description: Container class of linear constraints and bounds employing 
//-              the "distinct" approach to variables usage.
//- Owner:       Mike Eldred
//- Version: $Id: DistinctConstraints.H 5003 2008-05-01 22:43:46Z mseldre $

#ifndef DISTINCT_CONSTRAINTS_H
#define DISTINCT_CONSTRAINTS_H

#include "DakotaConstraints.H"
#include "DataVariables.H"


namespace Dakota {

/// Derived class within the Constraints hierarchy which employs
/// the default data view (no variable or domain type array merging).

/** Derived variable constraints classes take different views of the
    design, uncertain, and state variable types and the continuous and
    discrete domain types.  The DistinctConstraints derived
    class separates the design, uncertain, and state variable types as
    well as the continuous and discrete domain types.  The result is
    separate lower and upper bounds arrays for continuous design,
    discrete design, uncertain, continuous state, and discrete state
    variables.  This is the default approach, so all iterators and
    strategies not specifically utilizing the All or Merged views use
    this approach (see Variables::get_variables(problem_db) for
    variables type selection; variables type is passed to the
    Constraints constructor in Model). */

class DistinctConstraints: public Constraints
{
public:

  //
  //- Heading: Constructor and destructor
  //

  /// default constructor
  DistinctConstraints();
  /// standard constructor
  DistinctConstraints(const ProblemDescDB& problem_db,
		      const pair<short,short>& view);
  /// destructor
  ~DistinctConstraints();

  //
  //- Heading: Virtual function redefinitions
  //

  const RealVector& continuous_lower_bounds() const;
  void continuous_lower_bounds(const RealVector& c_l_bnds);
  const RealVector& continuous_upper_bounds() const;
  void continuous_upper_bounds(const RealVector& c_u_bnds);

  const IntVector& discrete_lower_bounds() const;
  void discrete_lower_bounds(const IntVector& d_l_bnds);
  const IntVector& discrete_upper_bounds() const;
  void discrete_upper_bounds(const IntVector& d_u_bnds);

  const RealVector& inactive_continuous_lower_bounds() const;
  void inactive_continuous_lower_bounds(const RealVector& i_c_l_bnds);
  const RealVector& inactive_continuous_upper_bounds() const;
  void inactive_continuous_upper_bounds(const RealVector& i_c_u_bnds);

  const IntVector& inactive_discrete_lower_bounds() const;
  void inactive_discrete_lower_bounds(const IntVector& i_d_l_bnds);
  const IntVector& inactive_discrete_upper_bounds() const;
  void inactive_discrete_upper_bounds(const IntVector& i_d_u_bnds);

  RealVector all_continuous_lower_bounds() const;
  void all_continuous_lower_bounds(const RealVector& a_c_l_bnds);
  RealVector all_continuous_upper_bounds() const;
  void all_continuous_upper_bounds(const RealVector& a_c_u_bnds);

  IntVector all_discrete_lower_bounds() const;
  void all_discrete_lower_bounds(const IntVector& a_d_l_bnds);
  IntVector all_discrete_upper_bounds() const;
  void all_discrete_upper_bounds(const IntVector& a_d_u_bnds);

  void write(ostream& s) const;
  void read(istream& s);

protected:

  void copy_rep(const Constraints* con_rep);

  void reshape_rep(const Sizet2DArray& vars_comps);

private:

  //
  //- Heading: Private data members
  //

  /// the continuous design lower bounds array
  RealVector continuousDesignLowerBnds;
  /// the continuous design upper bounds array
  RealVector continuousDesignUpperBnds;
  /// the discrete design lower bounds array
  IntVector discreteDesignLowerBnds;
  /// the discrete design upper bounds array
  IntVector discreteDesignUpperBnds;
  /// the uncertain distribution lower bounds array
  RealVector uncertainLowerBnds;
  /// the uncertain distribution upper bounds array
  RealVector uncertainUpperBnds;
  /// the continuous state lower bounds array
  RealVector continuousStateLowerBnds;
  /// the continuous state upper bounds array
  RealVector continuousStateUpperBnds;
  /// the discrete state lower bounds array
  IntVector discreteStateLowerBnds;
  /// the discrete state upper bounds array
  IntVector discreteStateUpperBnds;
};


inline DistinctConstraints::DistinctConstraints()
{ }


inline DistinctConstraints::~DistinctConstraints()
{ }


inline const RealVector& DistinctConstraints::
continuous_lower_bounds() const
{
  switch (variablesView.first) {
  case MIXED_DISTINCT_DESIGN:
    return continuousDesignLowerBnds; break;
  case MIXED_DISTINCT_UNCERTAIN:
    return uncertainLowerBnds;        break;
  case MIXED_DISTINCT_STATE:
    return continuousStateLowerBnds;  break;
  }
  return emptyRealVector;
}


inline void DistinctConstraints::
continuous_lower_bounds(const RealVector& c_l_bnds)
{
  switch (variablesView.first) {
  case MIXED_DISTINCT_DESIGN:
    continuousDesignLowerBnds = c_l_bnds; break;
  case MIXED_DISTINCT_UNCERTAIN:
    uncertainLowerBnds        = c_l_bnds; break;
  case MIXED_DISTINCT_STATE:
    continuousStateLowerBnds  = c_l_bnds; break;
  }
}


inline const RealVector& DistinctConstraints::
continuous_upper_bounds() const
{
  switch (variablesView.first) {
  case MIXED_DISTINCT_DESIGN:
    return continuousDesignUpperBnds; break;
  case MIXED_DISTINCT_UNCERTAIN:
    return uncertainUpperBnds;        break;
  case MIXED_DISTINCT_STATE:
    return continuousStateUpperBnds;  break;
  }
  return emptyRealVector;
}


inline void DistinctConstraints::
continuous_upper_bounds(const RealVector& c_u_bnds)
{
  switch (variablesView.first) {
  case MIXED_DISTINCT_DESIGN:
    continuousDesignUpperBnds = c_u_bnds; break;
  case MIXED_DISTINCT_UNCERTAIN:
    uncertainUpperBnds        = c_u_bnds; break;
  case MIXED_DISTINCT_STATE:
    continuousStateUpperBnds  = c_u_bnds; break;
  }
}


inline const IntVector& DistinctConstraints::discrete_lower_bounds() const
{
  switch (variablesView.first) {
  case MIXED_DISTINCT_DESIGN:
    return discreteDesignLowerBnds; break;
  case MIXED_DISTINCT_UNCERTAIN:
    return emptyIntVector;          break;
  case MIXED_DISTINCT_STATE:
    return discreteStateLowerBnds;  break;
  }
  return emptyIntVector;
}


inline void DistinctConstraints::
discrete_lower_bounds(const IntVector& d_l_bnds)
{
  switch (variablesView.first) {
  case MIXED_DISTINCT_DESIGN:
    discreteDesignLowerBnds = d_l_bnds; break;
  case MIXED_DISTINCT_UNCERTAIN:
    if (d_l_bnds.length()) {
      Cerr << "Error: There are no active discrete lower bounds in "
	   << "DistinctConstraints in nondeterministic mode.\n       "
	   << "Setting this array is erroneous." << endl;
      abort_handler(-1);
    }
    break;
  case MIXED_DISTINCT_STATE:
    discreteStateLowerBnds  = d_l_bnds; break;
  }
}


inline const IntVector& DistinctConstraints::discrete_upper_bounds() const
{
  switch (variablesView.first) {
  case MIXED_DISTINCT_DESIGN:
    return discreteDesignUpperBnds; break;
  case MIXED_DISTINCT_UNCERTAIN:
    return emptyIntVector;          break;
  case MIXED_DISTINCT_STATE:
    return discreteStateUpperBnds;  break;
  }
  return emptyIntVector;
}


inline void DistinctConstraints::
discrete_upper_bounds(const IntVector& d_u_bnds)
{
  switch (variablesView.first) {
  case MIXED_DISTINCT_DESIGN:
    discreteDesignUpperBnds = d_u_bnds; break;
  case MIXED_DISTINCT_UNCERTAIN:
    if (d_u_bnds.length()) {
      Cerr << "Error: There are no active discrete upper bounds in "
	   << "DistinctConstraints in nondeterministic mode.\n       "
	   << "Setting this array is erroneous." << endl;
      abort_handler(-1);
    }
    break;
  case MIXED_DISTINCT_STATE:
    discreteStateUpperBnds  = d_u_bnds; break;
  }
}


inline const RealVector& DistinctConstraints::
inactive_continuous_lower_bounds() const
{
  switch (variablesView.second) {
  case MIXED_DISTINCT_DESIGN:
    return continuousDesignLowerBnds; break;
  case MIXED_DISTINCT_UNCERTAIN:
    return uncertainLowerBnds;        break;
  case MIXED_DISTINCT_STATE:
    return continuousStateLowerBnds;  break;
  }
  return emptyRealVector;
}


inline void DistinctConstraints::
inactive_continuous_lower_bounds(const RealVector& i_c_l_bnds)
{
  switch (variablesView.second) {
  case MIXED_DISTINCT_DESIGN:
    continuousDesignLowerBnds = i_c_l_bnds; break;
  case MIXED_DISTINCT_UNCERTAIN:
    uncertainLowerBnds        = i_c_l_bnds; break;
  case MIXED_DISTINCT_STATE:
    continuousStateLowerBnds  = i_c_l_bnds; break;
  }
}


inline const RealVector& DistinctConstraints::
inactive_continuous_upper_bounds() const
{
  switch (variablesView.second) {
  case MIXED_DISTINCT_DESIGN:
    return continuousDesignUpperBnds; break;
  case MIXED_DISTINCT_UNCERTAIN:
    return uncertainUpperBnds;        break;
  case MIXED_DISTINCT_STATE:
    return continuousStateUpperBnds;  break;
  }
  return emptyRealVector;
}


inline void DistinctConstraints::
inactive_continuous_upper_bounds(const RealVector& i_c_u_bnds)
{
  switch (variablesView.second) {
  case MIXED_DISTINCT_DESIGN:
    continuousDesignUpperBnds = i_c_u_bnds; break;
  case MIXED_DISTINCT_UNCERTAIN:
    uncertainUpperBnds        = i_c_u_bnds; break;
  case MIXED_DISTINCT_STATE:
    continuousStateUpperBnds  = i_c_u_bnds; break;
  }
}


inline const IntVector& DistinctConstraints::
inactive_discrete_lower_bounds() const
{
  switch (variablesView.second) {
  case MIXED_DISTINCT_DESIGN:
    return discreteDesignLowerBnds; break;
  case MIXED_DISTINCT_UNCERTAIN:
    return emptyIntVector;          break;
  case MIXED_DISTINCT_STATE:
    return discreteStateLowerBnds;  break;
  }
  return emptyIntVector;
}


inline void DistinctConstraints::
inactive_discrete_lower_bounds(const IntVector& i_d_l_bnds)
{
  switch (variablesView.second) {
  case MIXED_DISTINCT_DESIGN:
    discreteDesignLowerBnds = i_d_l_bnds; break;
  case MIXED_DISTINCT_UNCERTAIN:
    if (i_d_l_bnds.length()) {
      Cerr << "Error: There are no inactive discrete lower bounds in "
	   << "DistinctConstraints in design mode.\n       "
	   << "Setting this array is erroneous." << endl;
      abort_handler(-1);
    }
    break;
  case MIXED_DISTINCT_STATE:
    discreteStateLowerBnds  = i_d_l_bnds; break;
  }
}


inline const IntVector& DistinctConstraints::
inactive_discrete_upper_bounds() const
{
  switch (variablesView.second) {
  case MIXED_DISTINCT_DESIGN:
    return discreteDesignUpperBnds; break;
  case MIXED_DISTINCT_UNCERTAIN:
    return emptyIntVector;          break;
  case MIXED_DISTINCT_STATE:
    return discreteStateUpperBnds;  break;
  }
  return emptyIntVector;
}


inline void DistinctConstraints::
inactive_discrete_upper_bounds(const IntVector& i_d_u_bnds)
{
  switch (variablesView.second) {
  case MIXED_DISTINCT_DESIGN:
    discreteDesignUpperBnds = i_d_u_bnds; break;
  case MIXED_DISTINCT_UNCERTAIN:
    if (i_d_u_bnds.length()) {
      Cerr << "Error: There are no inactive discrete upper bounds in "
	   << "DistinctConstraints in design mode.\n       "
	   << "Setting this array is erroneous." << endl;
      abort_handler(-1);
    }
    break;
  case MIXED_DISTINCT_STATE:
    discreteStateUpperBnds  = i_d_u_bnds; break;
  }
}


inline RealVector DistinctConstraints::all_continuous_lower_bounds() const
{
  RealVector all_c_l_bnds;
  aggregate_data(continuousDesignLowerBnds, uncertainLowerBnds, 
		 continuousStateLowerBnds, all_c_l_bnds);
  return all_c_l_bnds;
}


inline void DistinctConstraints::
all_continuous_lower_bounds(const RealVector& a_c_l_bnds)
{
  separate_data(a_c_l_bnds, continuousDesignLowerBnds, uncertainLowerBnds,
		continuousStateLowerBnds);
}


inline RealVector DistinctConstraints::all_continuous_upper_bounds() const
{
  RealVector all_c_u_bnds;
  aggregate_data(continuousDesignUpperBnds, uncertainUpperBnds, 
		 continuousStateUpperBnds, all_c_u_bnds);
  return all_c_u_bnds;
}


inline void DistinctConstraints::
all_continuous_upper_bounds(const RealVector& a_c_u_bnds)
{
  separate_data(a_c_u_bnds, continuousDesignUpperBnds, uncertainUpperBnds,
		continuousStateUpperBnds);
}


inline IntVector DistinctConstraints::all_discrete_lower_bounds() const
{
  IntVector all_d_l_bnds;
  aggregate_data(discreteDesignLowerBnds, discreteStateLowerBnds, all_d_l_bnds);
  return all_d_l_bnds;
}


inline void DistinctConstraints::
all_discrete_lower_bounds(const IntVector& a_d_l_bnds)
{ separate_data(a_d_l_bnds, discreteDesignLowerBnds, discreteStateLowerBnds); }


inline IntVector DistinctConstraints::all_discrete_upper_bounds() const
{
  IntVector all_d_u_bnds;
  aggregate_data(discreteDesignUpperBnds, discreteStateUpperBnds, all_d_u_bnds);
  return all_d_u_bnds;
}


inline void DistinctConstraints::
all_discrete_upper_bounds(const IntVector& a_d_u_bnds)
{ separate_data(a_d_u_bnds, discreteDesignUpperBnds, discreteStateUpperBnds); }

} // namespace Dakota

#endif
