/*  _________________________________________________________________________
 *
 *  Coliny: A Library of COLIN optimizers
 *  Copyright (c) 2003, Sandia National Laboratories.
 *  This software is distributed under the GNU Lesser General Public License.
 *  For more information, see the README.html file in the top Coliny directory.
 *  _________________________________________________________________________
 */

//
// Coliny_SolisWets.cpp
//
// Local search without gradients.
//
// TODO: cleanup the code
//   Create a 'response' value for that contains the function/constraint
//   values, feasibility flags, etc.
//

#include <acro_config.h>
#include <utilib/_math.h>
#include <utilib/stl_auxillary.h>
#include <coliny/Factory.h>
#include <coliny/SolisWets.h>

#define SWAP(x,y)	{NumArray<double>* tmp; tmp = x; x = y; y = tmp;}

#define SPHERE_DEV  0
#define NORMAL_DEV  1
#define UNIFORM_DEV 2

#define SOLISWETS_DEFAULT       0
#define SINGLE_EXPAND_UPDATE    1

using namespace std;

namespace coliny {

SolisWets::SolisWets()
 : batch_evaluator_t(&problem),
   max_success(5),
   max_failure(0),
   ex_factor(2.0),
   ct_factor(0.5),
   Delta_thresh(1e-6),
   Delta_init(1.0),
   update_type("default"),
   bias_flag(false),
   neighborhood_type("normal")
{
constraint_penalty=1.0;
ParameterSet::set_parameter_default("constraint_penalty","1.0");

opt_name = "SolisWets";

auto_rescale_flag=true;
ParameterSet::create_categorized_parameter("auto_rescale",auto_rescale_flag,
        "<bool>","true",
        "If true, then automatically rescale the search for bound-constrained\n"
"\t  problems. The initial scale is 10% of the range in each dimension.",
        "Step Length Control");

ParameterSet::create_categorized_parameter("initial_step",Delta_init,
        "<double>","1.0",
        "Initial step length",
	"Step Length Control");
ParameterSet::alias_parameter("initial_step","initial_stepsize");
ParameterSet::alias_parameter("initial_step","initial_steplength");

ParameterSet::create_categorized_parameter("max_success",max_success,
        "<int>","5",
        "Number of successful iterations before step length is expanded",
	"Step Length Control");

ParameterSet::create_categorized_parameter("max_failure",max_failure,
        "<int>","4*n",
        "Number of unsuccessful iterations before step length is contracted",
	"Step Length Control");

ParameterSet::create_categorized_parameter("expansion_factor",ex_factor,
        "<double>","2.0",
        "Expansion factor",
	"Step Length Control");

ParameterSet::create_categorized_parameter("contraction_factor",ct_factor,
        "<double>","0.5",
        "Contraction factor",
	"Step Length Control");

ParameterSet::create_categorized_parameter("step_tolerance",Delta_thresh,
        "<double>","1e-6",
        "Convergence tolerance step length",
	"Termination");

ParameterSet::create_categorized_parameter("update_type",update_type,
        "<string>","default",
        "Control for step length update:\n\t  default, single_expand",
	"Step Length Control");

ParameterSet::create_parameter("bias_flag",bias_flag,
        "<boolean>","false",
        "Use an adaptive step bias");

ParameterSet::create_parameter("neighborhood_type",neighborhood_type,
        "<string>","normal",
        "Type of neighorhood used:\n\t  normal, uniform, sphere");

ParameterSet::create_categorized_parameter("step_scales",Sigma,
        "<Array<double>>","All 1.0",
        "The scale factors for each dimension",
	"Step Length Control");
}


void SolisWets::reset()
{
if (!problem) return;

colin::StdOptSolver<BasicArray<double>,response_t>::reset();

if (!rng)
   EXCEPTION_MNGR(runtime_error,"SolisWets::reset - undefined random number generator");

unif_dev.generator(&rng);
normal_dev.generator(&rng);

unsigned int n = problem.num_real_params();
if ((Sigma.size() != 0) && (Sigma.size() != n))
   EXCEPTION_MNGR(runtime_error,"SolisWets::reset - Scale vector length " << Sigma.size() << " is \n\t  not equal to num_real_params: " << n);
Sigma.resize(n);
bool bc_flag = problem.enforcing_bounds();
utilib::pvector<colin::real> lower_bc, upper_bc;
if (bc_flag)
   problem.get_real_bounds(lower_bc, upper_bc);

if (auto_rescale_flag && bc_flag) {
  for (unsigned int i=0; i<n; i++)
    if ((upper_bc[i] == real :: positive_infinity) ||
        (lower_bc[i] == real :: negative_infinity))
      Sigma[i] = 1.0;
    else
      Sigma[i] = max((static_cast<double>(upper_bc[i])-static_cast<double>(lower_bc[i]))/10.0,1e-5);
  }
else
  Sigma << 1.0;




if (max_failure < 1)
   max_failure = 4*n;

if (update_type == "default")
   update_id = SOLISWETS_DEFAULT;
else if (update_type == "single_expand")
   update_id = SINGLE_EXPAND_UPDATE;
else
   EXCEPTION_MNGR(runtime_error,"SolisWets::reset - bad update type: " << update_type);

if (neighborhood_type == "normal")
   neighborhood_id = NORMAL_DEV;
else if (neighborhood_type == "uniform")
   neighborhood_id = UNIFORM_DEV;
else if (neighborhood_type == "sphere")
   neighborhood_id = SPHERE_DEV;
else
   EXCEPTION_MNGR(runtime_error,"SolisWets::reset - bad neighborhood type: " << neighborhood_type);

bias.resize(n);
vec1.resize(n);
vec2.resize(n);
vec3.resize(n);

//
// Setup algorithmic parameters
//
bias << 0.0;
expand_flag=true;
n_success=0;
n_failure=0;
Delta_min = Delta = Delta_init;

batch_evaluator_t::reset();
colin::AppResponseAnalysis::initialize(problem.numNonlinearIneqConstraints(),constraint_tolerance);
}


void SolisWets::minimize()
{
//
// Misc initialization of the optimizer
//
opt_init();
if (!(this->initial_point_flag))
   EXCEPTION_MNGR(runtime_error,"SolisWets::minimize - no initial point specified.");
if (problem.num_real_params() != best().point.size())
   EXCEPTION_MNGR(runtime_error,"SolisWets::minimize - problem has " <<
	problem.num_real_params() << " real params, but initial point has " << best().point.size() );
if (best().point.size() == 0) {
   best().termination_info = "No-Real-Params";
   return;
   }
//
//
//
unsigned int num_iters;
if (max_iters <= 0)
   num_iters = MAXINT;
else
   num_iters = curr_iter + max_iters;
//
// Setup swap vectors
//
NumArray<double>* best_pt = &vec1;
NumArray<double>* new_point  = &vec2;
NumArray<double>* temp_point = &vec3;

real fp;
*best_pt << best().point;

bool bound_feasible = problem.test_feasibility(*best_pt);
real constraint_value=0.0;

best().value() = real :: positive_infinity;
perform_evaluation(*best_pt, Delta_init/Delta_min, best().response, best().value(), best().constraint_violation);
if (!bound_feasible)
   EXCEPTION_MNGR(runtime_error,"SolisWets::minimize - initial point is not bound-feasible");
iteration_status = -1;
debug_io(ucout);

for (curr_iter++; curr_iter <= num_iters;  curr_iter++) {

  if (Delta <= Delta_thresh) {
     stringstream tmp;
     tmp << "Step-Length Delta=" << Delta << "<=" 
				<< Delta_thresh << "=Delta_thresh";
     best().termination_info = tmp.str();
     break;
     }

  if (check_convergence(best().value()))
     break;

  DEBUGPR(2, ucout << "OBJECTIVE " << best().value() << "\n";);
  DEBUGPR(2,ucout << "DELTA:       " << Delta << "\n";);
  DEBUGPR(2,ucout << "BIAS VECTOR: " << bias << "\n";);

  *temp_point << *best_pt;		// The new mean
  if (bias_flag)
     *temp_point += bias;
  gen_new_point(*new_point,*temp_point,Delta,bound_feasible);
  if (bound_feasible) {
     //tmp_response.reset();
     perform_evaluation(*new_point, Delta_init/Delta_min, tmp_response, fp, constraint_value);
     }
  DEBUGPR(2,
  {
  ucout << "Response     " << tmp_response << endl;
  ucout << "OLD PT:      " << *best_pt << "\n";
  ucout << "OLD+BIAS:    " << *temp_point << "\n";
  ucout << "NEW PT:      " << *new_point << "\n";
  double val=0.0;
  for (unsigned int i=0; i<new_point->size(); i++)
    val += ((*new_point)[i]-(*temp_point)[i]) * ((*new_point)[i]-(*temp_point)[i]);
  val = sqrt(val);
  ucout << "STEPLEN: " << val << endl;
  }
  );
  DEBUGPR(2,ucout << "FEval: value=" << fp << " cvalue=" << constraint_value << " feasible=" << (bound_feasible) << "\n");

  if (bound_feasible && (fp < best().value())) {
     DEBUGPR(2, ucout << "BETTER POINT: " << best().value() << " (" << fp << ")\n";);
     iteration_status = 0;
     best().value() = fp;
     best().constraint_violation = constraint_value;
     best().response << tmp_response;

     UpdateDelta(true);
     if (bias_flag) {
        bias *= 0.2;
        for (unsigned int i=0; i<bias.size(); i++)
          bias[i] += 0.4*((*new_point)[i] - (*best_pt)[i]);
        }

     /**
     if (Debug_path_len) {
        *best_pt -= *new_point;
        path_length += length(*best_pt);
        }
     **/
     SWAP(best_pt,new_point);
     }

  else {
	// Go through some shenanigans to avoid de/allocating memory
     *temp_point << *best_pt;
     *temp_point *= 2.0;
     *temp_point -= *new_point;

     DEBUGPR(2, ucout << "Reflected Point:    " << *temp_point << endl;);
     DEBUGPR(2, ucout << "Response     " << tmp_response << endl;);
     bound_feasible = problem.test_feasibility(*temp_point);
     if (bound_feasible) {
	//tmp_response.reset();
        perform_evaluation(*temp_point, Delta_init/Delta_min, tmp_response, fp, constraint_value);
	}
     DEBUGPR(2,ucout << "FEval: value=" << fp << " cvalue=" << constraint_value << " feasible=" << (bound_feasible) << "\n");
     if (bound_feasible && (fp < best().value())) {
        DEBUGPR(2, ucout << "BETTER POINT: " << best().value() << " (" << fp << ")\n";);
        iteration_status = 1;
        best().value() = fp;
        best().constraint_violation = constraint_value;
	best().response << tmp_response;

	UpdateDelta(true);
	if (bias_flag) {
           for (unsigned int i=0; i<bias.size(); i++)
             bias[i] -= 0.4*((*new_point)[i] - (*best_pt)[i]);
	   }

        SWAP(best_pt,temp_point);
        }
     else {
        iteration_status = 2;
	UpdateDelta(false);
	if (bias_flag)
           bias *= 0.5;
	DEBUGPR(2, ucout << "GENERATED WORSE POINTS!\n";);
        }
     }

   if (Delta < Delta_min) {
      Delta_min = Delta;
	//best_val = compute_penalty_function(best_response.function_value(0),best_cval,Delta_init/Delta_min);
      }

  best().point << static_cast<BasicArray<double> >(*best_pt);
  debug_io(ucout);
  }

best().point << static_cast<BasicArray<double> >(*best_pt);
debug_io(ucout,true);
this->clear_evaluations();
}


void SolisWets::UpdateDelta(bool flag)
{
switch (update_id) {
  case SINGLE_EXPAND_UPDATE:
        if (flag == true) {
           n_success++;
	   n_failure=0;
           if (expand_flag == true) {
              if (n_success >= max_success) {
                 DEBUGPR(10,ucout << "SINGLE_EXPAND: Expanding step length: " << Delta << " by " << ex_factor << "\n";);
                 Delta *= ex_factor;			// Expand
                 n_success=0;
                 }
	      }
           }
        else {
           n_failure++;
	   n_success=0;
           if (n_failure >= max_failure) {
              DEBUGPR(10,ucout << "SINGLE_EXPAND: Contracting step length: " << Delta << " by " << ct_factor << "\n";);
              Delta *= ct_factor;
              n_failure=0;
              expand_flag = false;
              }
           }
        break;

  case SOLISWETS_DEFAULT:
        if (flag == true) {                      // Expand
           n_success++;
	   n_failure=0;
           if (n_success >= max_success) {
              DEBUGPR(10,ucout << "SW_DEFAULT: Expanding step length: " << Delta << " by " << ex_factor << "\n";);
              Delta *= ex_factor;
              n_success=0;
              }
           }
        else {
           n_failure++;
	   n_success=0;
           if (n_failure >= max_failure) {
              DEBUGPR(10,ucout << "SW_DEFAULT: Contracting step length: " << Delta << " by " << ct_factor << "\n";);
              Delta *= ct_factor;
              n_failure=0;
              }
           }
	break;

  }
}


void SolisWets::write(ostream& os) const
{
colin::StdOptSolver<BasicArray<double>,response_t>::write(os);
 
os << "##\n## Solis-Wets Information\n##\n";
switch (neighborhood_id) {
  case NORMAL_DEV:	os << "neighborhood\tnormal" << endl; break;
  case SPHERE_DEV:	os << "neighborhood\tsphere" << endl; break;
  case UNIFORM_DEV:	os << "neighborhood\tuniform" << endl; break;
  };
os << "update_id\t" << update_id ;
if (update_id == SOLISWETS_DEFAULT)
  os << "\t\t# No restrictions on expansion and contractions" << endl;
else
  os << "\t\t# Expansions not allowed after first contraction" << endl;
os << "max_success\t" << max_success << endl;
os << "max_failure\t" << max_failure << endl;
os << "ex_factor\t" << ex_factor << endl;
os << "ct_factor\t" << ct_factor << endl;
os << "Delta_init\t" << Delta_init << endl;
os << "Delta_thresh\t" << Delta_thresh << endl;
os << "bias_flag\t" << bias_flag;
if (bias_flag)
   os << "\t\t# Using a dynamic bias in search (default)" << endl;
else
   os << "\t\t# NOT using a dynamic bias in search" << endl;
DEBUGPR(2,os << "Sigma\t" << Sigma << "\n";);
}


void SolisWets::gen_new_point(NumArray<double>& new_pt, 
		NumArray<double>& mean_vec, double _Delta, bool& bound_feasible)
{
switch (neighborhood_id) {
  case SPHERE_DEV:
     {
     for (unsigned int i=0; i<new_pt.size(); i++)
       new_pt[i] = normal_dev();
     new_pt /= length(new_pt);
     for (unsigned int i=0; i<new_pt.size(); i++)
       new_pt[i] = mean_vec[i] + new_pt[i]*_Delta*Sigma[i];
     }
     break;

  case NORMAL_DEV:
     {
     for (unsigned int i=0; i<new_pt.size(); i++)
       new_pt[i] = mean_vec[i] + normal_dev()*_Delta*Sigma[i];
     }
     break;

  case UNIFORM_DEV:
     {
     for (unsigned int i=0; i<new_pt.size(); i++)
       new_pt[i] = mean_vec[i] + (2.0*unif_dev()-1.0) * _Delta * Sigma[i];
     }
     break;
  };

bound_feasible = problem.test_feasibility(new_pt);
}


void SolisWets::virt_debug_io(ostream& os, const bool finishing,
                        const int olevel) 
{
//
// This only supports verbose output
//
if (olevel < 3)
   return;

os << endl;
if (iteration_status >= 0) {
   if (iteration_status == 0) {
      if (bias_flag)
         os << "\tCurrent Point = Prev Point + (Bias Vector + Random Deviates)" << endl << endl;
      else
         os << "\tCurrent Point = Prev Point + Random Deviates" << endl << endl;
      }
   else if (iteration_status == 1) {
      if (bias_flag)
         os << "\tCurrent Point = Prev Point - (Bias Vector + Random Deviates)" << endl << endl;
      else
         os << "\tCurrent Point = Prev Point - Random Deviates" << endl << endl;
      }
   else
      os << "\tCurrent Point = Prev Point" << endl << endl;
   }

os << "\tStep Scales: ";
for (unsigned int i=0; i<Sigma.size(); i++)
  os << Delta * Sigma[i] << " ";
os << endl;

if (curr_iter == 0) {
   os << "\n\tUsing ";
   switch (neighborhood_id) {
     case SPHERE_DEV:	os << "sphere" ; break;
     case NORMAL_DEV:	os << "normal" ; break;
     case UNIFORM_DEV:	os << "uniform" ; break;
     };
   os << " deviates to generate trial points." << endl;
   if (update_id == SOLISWETS_DEFAULT)
     os << "\tNo restrictions on expansion and contractions" << endl;
   else
     os << "\tExpansions not allowed after first contraction" << endl;
   if (bias_flag)
      os << "\tUsing a dynamic bias in search (default)" << endl;
   else
      os << "\tNOT using a dynamic bias in search" << endl;
   }
}

#if 0
void SolisWets::perform_evaluation(BasicArray<double>& point, 
			SolisWets_response_t& response,
			real& val,
			real& cval, BasicArray<real >& cvals,
			bool& constraint_feasible, bool& bound_feasible)
{				
bound_feasible = problem.test_feasibility(point);
if (!bound_feasible) return;

if (problem.numNonlinearConstraints() > 0)
   problem.Eval(point,response,colin::mode_f|colin::mode_cf);
else
   problem.Eval(point,response,colin::mode_f);

compute_response_info(response,problem.state->constraint_lower_bounds,problem.state->constraint_upper_bounds,Delta_init/Delta_min,val,cval);

constraint_feasible=(cval == 0.0);
}
#endif

} // namespace coliny

typedef colin::OptSolver<utilib::BasicArray<double>,colin::                     AppResponse_Utilib> base_t;
FACTORY_REGISTER(sw, base_t*, return new coliny::SolisWets();, "An alias to SolisWets")
FACTORY_REGISTER(SolisWets, base_t*, return new coliny::SolisWets();, "The derivative-free optimizer of Solis-Wets")

