/*  _________________________________________________________________________
 *
 *  COLIN: A Common Optimization Library INterface
 *  Copyright (c) 2007, Sandia National Laboratories.
 *  This software is distributed under the GNU Lesser General Public License.
 *  For more information, see the README.html file in the top COLIN directory.
 *  _________________________________________________________________________
 */

/**
 * \file AsyncEvaluator.h
 *
 * Defines the \c AsyncEvaluator class.
 *
 **/
 
#ifndef colin_AsyncEvaluator_h
#define colin_AsyncEvaluator_h

#include <iostream>
 
//#include <acro_config.h>
//#include <utilib/BasicArray.h>
//#include <utilib/LinkedList.h>
#include <utilib/ParameterSet.h>
#include <utilib/Ereal.h>
//#include <utilib/pvector.h>
#include <colin/AppResponseAnalysis.h>
#include <colin/AppResponse.h>
#include <colin/ResponseSet.h>

#define DEBUG_ASYNC_EVALUATOR  0

#if defined(USING_PROBE)
extern void probe();
#endif

namespace colin {

namespace ERR {
const int AsyncEvaluator_InvlaidPoint         = -1101;
const int AsyncEvaluator_NoPendingEvaluations = -1102;
const int AsyncEvaluator_NoAvailabelServers   = -1103;
};

using std::cerr;
using std::endl;

/**
 * A utility class for asynchronously scheduling and evaluating function
 * evaluations from one or more optimization algorithms.
 */
template <class ProblemT, class DomainT>
class AsyncEvaluator : public virtual utilib::ParameterSet, 
                       public virtual utilib::CommonIO,
                       public virtual colin::AppResponseAnalysis
  {
  public:
    //
    //- Heading: Class-specific interface typedefs
    //
    typedef long    evalID_t;
    typedef utilib::Ereal<double>  priority_t;
    typedef int     solverID_t;

#if !defined(DOXYGEN)
    typedef typename ProblemT::response_t    response_t;
    typedef ResponseSet<DomainT, response_t> responseSet_t;
#endif
    
  public:
    //
    //- Heading: Constructors, destructors, and whatnot
    //

    /// Constructor
    AsyncEvaluator(ProblemT* problem_ptr_);
    
    //
    //- Heading: Member functions
    //

    /// Process options and prepare to be used
    void reset();
    
    /// Write out parameter information
    void write(std::ostream& os) const;
    
    /// Queue a function evaluation for a given point.
    evalID_t queue_evaluation( responseSet_t  responseSet, 
                               priority_t     priority, 
                               solverID_t     solverID,
                               bool           duplicateEval = false );
    /// Wait for and return the next completed evaluation
    int return_evaluation( responseSet_t &responseSet, 
                           std::map<solverID_t, evalID_t> &resultMap );
    /// Immediately perform a function evaluation and wait for result.
    int perform_evaluation(responseSet_t  responseSet);

    /// Set the target (fractional) resource allocation for each solver
    void set_target_allocation(std::map<solverID_t, double> &target);
    
  protected:
    /// Print debugging information.
    void virt_debug_io( std::ostream& os, const bool finishing, 
                        const int io_level );

  private:  
    typedef std::map< solverID_t, 
                      std::map< priority_t, 
                                std::list<responseSet_t > > > requestMap_t;


    /// Fire off the next evaluation (based on priority and solver allotment)
    int spawn_evaluation();


    struct ResponseInfo {
      ResponseInfo() : evalsInProgress(0) {}

      /// Sorted map of pending eval ID's by solver and priority
      std::map<solverID_t, std::multimap<priority_t, evalID_t> > solvers;
      /// The number of evaluations currently running
      unsigned int evalsInProgress;
      };

    /* parameters */
    unsigned int m_max_server_load; ///< Simultaneous evaluations per server.
    unsigned int m_poll_interval;   ///< Polling interval for new evaluations

    /* member variables */
    unsigned int m_lastEvalID;      ///< The ID of the last queued evaluation
    unsigned int m_active_servers;  ///< Number of active servers.
    unsigned int m_num_servers;     ///< Total number of servers.
    
    ProblemT* m_problem_ptr; ///< Pointer to an OptProblem object
    int m_mode;              ///< The mode of evaluation.

    /// The list of queued (not active) requests sorted by solver & priority
    requestMap_t m_solverRequests;
    /// The info for all requests (queued and active) sorted by ResponseSet
    std::map<responseSet_t, ResponseInfo>  m_responseSetInfo;
    /// The executing requests sorted by problem eval id
    std::map<int, responseSet_t>  m_activeEvals;
    /// The response buffers for all executing requests (by problem eval id)
    std::map<int, response_t*>  m_activeResponseBuf;

    /// The charge of currently running evaluations against each solver
    std::map<solverID_t, int>      m_chargedTime;
    /// The target computational allocation
    std::map<solverID_t, double>   m_targetAlloc;
    /// A starvation factor to assure that all solvers get *some* time
    std::map<solverID_t, double>   m_starvation;
  };
  
  
  
template <class ProblemT, class DomainT>
AsyncEvaluator<ProblemT,DomainT>::
AsyncEvaluator(ProblemT* problem_ptr_)
  : m_max_server_load(1),
    m_poll_interval(100000),
    m_problem_ptr(problem_ptr_),
    m_mode(0)
  {
  utilib::ParameterSet::create_parameter
      ("max_server_load", m_max_server_load, "<unsigned int>", "1",
       "Define the maximum number of evaluations that can simultaneously\n"
       "\t  run on each evaluation server.  When this is small, the\n"
       "\t  asynchronous evaluations are performed 'conservatively', which\n"
       "\t  is good when evaluations cannot be effectively terminated.\n"
       "\t  When this is large, asynchronous evaluations are performed \n"
       "\t  'aggressively', which is good when evaluations are quick or\n"
       "\t  when they can be terminated."
       );
  utilib::ParameterSet::create_parameter
      ("poll_interval", m_poll_interval, "<unsigned int>", "100000",
       "Define the polling interval (in microseconds) for checking for \n"
       "\t  newly-completed function evaluations."
       );
  }


template <class ProblemT,class DomainT>
void
AsyncEvaluator<ProblemT,DomainT>::
reset()
  {
  if ( m_problem_ptr == NULL ) 
    { return; }

  m_lastEvalID = 0;
  m_solverRequests.clear();
  m_responseSetInfo.clear();
  m_activeEvals.clear();
  m_activeResponseBuf.clear();

  m_chargedTime.clear();
  m_targetAlloc.clear();
  m_starvation.clear();
    
  // This calculates the number of "servers" (i.e. processor slots on
  //   which we can perform a function evaluation).  
  // (Copied from the BatchEvaluator.)
  m_active_servers=0;
  m_num_servers = m_max_server_load;
  if ( m_problem_ptr->num_evaluation_servers() > 1 )
    { m_num_servers *= m_problem_ptr->num_evaluation_servers(); }

  DEBUGPR( 100, ucout << "maxservload=" << m_max_server_load << " numservers=" 
           << std::max(m_problem_ptr->num_evaluation_servers(),(unsigned int)1)
           << std::endl; );

  if ( m_problem_ptr->numNonlinearConstraints() > 0 )
    { m_mode = colin::mode_f | colin::mode_cf; }
  else
    { m_mode = colin::mode_f; }
  }


template <class ProblemT, class DomainT>
void 
AsyncEvaluator<ProblemT,DomainT>::
write(std::ostream& os) const
  {
  os << "priority queue evaluation:" << std::endl
     << "\t# Function evaluations are performed asynchronously using a " 
     << std::endl
     << "\t# dynamically-managed prioritized queue and subsolver-based "
     << std::endl
     << "\t# resource sharing policy.  Up to max_server_load evaluations "
     << std::endl
     << "\t# are spawned on each evaluation server.  Results are returned "
     << std::endl
     << "\t# one at a time as they complete."
     << std::endl;
  os << "max_server_load\t" << m_max_server_load << std::endl;
  os << "poll_interval\t" << m_poll_interval << std::endl;
  }


/** Queue a function evaluation for a given point using the specified
 *  priority and 'charged' against the specified solverID's computational
 *  allocation.
 *
 *  Note: this will NOT create a duplicate function evaluation UNLESS
 *  this solver has already requested an evaluation of this ResponseSet.
 *  If another Solver has requested a calculation by this response set,
 *  then this request will piggyback on the existing queued request.
 *
 *  Returns the new EvalID on success, <0 on failure.
 */
template <class ProblemT, class DomainT>
typename AsyncEvaluator<ProblemT,DomainT>::evalID_t
AsyncEvaluator<ProblemT,DomainT>::
queue_evaluation( responseSet_t  responseSet,
                  priority_t     priority, 
                  solverID_t     solverID,
                  bool           duplicateEval )
  {
  if ( DEBUG_ASYNC_EVALUATOR > 1 )
    { cerr << "entering queue_evaluation()" << endl; }

  if ( responseSet.invalid() )
    { return ERR::AsyncEvaluator_InvlaidPoint; }

  ResponseInfo& rsi = m_responseSetInfo[responseSet];
  std::multimap<priority_t, evalID_t> &solverPri = rsi.solvers[solverID];


  // Special case: if duplicateEval == false, then this request may
  // piggyback on a previous request by this model (the single eval is
  // OK for both requests).  In that case, we will return the EvalID of
  // the previous request, and if this priority is better than the
  // other, reset the priority.
  if (( ! duplicateEval ) && ( ! solverPri.empty() ))
    {
    evalID_t evalID = solverPri.begin()->second;
    if (( rsi.evalsInProgress > 0 ) && ( solverPri.begin()->first > priority ))
      {
      // If the highest priority eval is already running and we are an
      // even higher priority, "boost" the priority of the existing eval
      // that we are piggybacking on (this will help prevent a later
      // duplicate eval with a priority higher than the original
      // request, but lower than this one from preempting this request).
      solverPri.erase(solverPri.begin());
      solverPri.insert(std::pair<priority_t, evalID_t>(priority, evalID));
      }
    else if ( solverPri.begin()->first > priority )
      {
      // If the highest priority eval is NOT running and we are an even
      // higher priority, "boost" the priority of the existing eval that
      // we are piggybacking on.  This is more complex because we also
      // need to reprioritize the original responseSet in the master
      // request list.
      typename std::map< priority_t, std::list<responseSet_t > >::iterator 
          dupPri = m_solverRequests[solverID].find(solverPri.begin()->first);
      if ( dupPri == m_solverRequests[solverID].end() )
        {
        EXCEPTION_MNGR
            (std::runtime_error, 
             "AsyncEvaluator::queue_evaluation() cannot find priority for "
             "reprioritizing evalID " << evalID;
             );
        }

      std::list<responseSet_t > &responseList = dupPri->second;
      typename std::list<responseSet_t >::iterator oldResponse
          = find(responseList.begin(), responseList.end(), responseSet);
      if ( oldResponse == responseList.end() )
        {
        EXCEPTION_MNGR
            (std::runtime_error, 
             "AsyncEvaluator::queue_evaluation() cannot find responseset for "
             "reprioritizing evalID " << evalID;
             );
        }

      responseList.erase(oldResponse);
      if ( responseList.empty() )
        { m_solverRequests[solverID].erase(dupPri); }
      solverPri.erase(solverPri.begin());
      solverPri.insert(std::pair<priority_t, evalID_t>(priority, evalID));
      m_solverRequests[solverID][priority].push_back(responseSet);
      }
    return evalID;
    }


  // OK - we need to generate a new evalID.
  evalID_t evalID = ++m_lastEvalID;

  // insert this priority into the set of priorities for this solver
  solverPri.insert(std::pair<priority_t, evalID_t>(priority, evalID));

  if ( rsi.evalsInProgress == 0 )
    {
    // no pending requests, so we just add this request to the master list
    m_solverRequests[solverID][priority].push_back(responseSet); 
    if ( DEBUG_ASYNC_EVALUATOR )
      {
      cerr << "[queued response " << responseSet.id() << " at " 
           << solverID << ":" << priority << "]" << endl;
      }
    }
  else
    {
    // this is more complicated...  if this priority places it in the
    // front evalsInProgress of the solver's prioritized list, then this
    // request will supercede a request currently on the evals list.  We
    // need to put the priority at evalsInProgress [i.e. lowest priority
    // already running] back onto the master list... Otherwise we place
    // this priority on the master list [and allow it to be scheduled
    // normally].
    std::multimap<priority_t, evalID_t>::iterator pri_it = solverPri.begin();
    std::multimap<priority_t, evalID_t>::iterator pri_itEnd = solverPri.end();
    unsigned int i = rsi.evalsInProgress;
    while (( i > 0 ) && ( pri_it != pri_itEnd ))
      { 
      ++pri_it; 
      --i;
      }

    if ( pri_it == pri_itEnd )
      { 
      // do nothing: an "available" evaluation is already under way
      // (just charge the time)
      ++(m_chargedTime[solverID]);
      }
    else if ( pri_it->first <= priority )
      { 
      // all pending evaluations "out rank" this one... add this
      // priority to the master list
      m_solverRequests[solverID][priority].push_back(responseSet); 
      }
    else
      { 
      // this request "out ranks" one already under way... allow this
      // one to replace the request under way and put it (the in-process
      // request priority) back onto the master list.
      m_solverRequests[solverID][pri_it->first].push_back(responseSet); 
      }
    }

  // If we are just starting, this will get all the servers doing
  // *something*.  Once we are in full swing, this should never
  // evaluate to true.
  if ( m_active_servers < m_num_servers )
    {
    // yes, this will end up spawning *this* point, but I am lazy and it
    // is easier to use the same spawn_election() function than to write
    // a special-case function for this instance.
    spawn_evaluation();
    }
  return evalID;
  }



/** Wait for and return the next completed function evaluation.  This
 *  returns the newly augmented ResponseSet in responseSet, and a map of
 *  SolverIDs -> EvalIDs in resultMap
 *
 *  Returns 0 on success, <0 on failure.
 */
template <class ProblemT,class DomainT>
int
AsyncEvaluator<ProblemT,DomainT>::
return_evaluation( responseSet_t &responseSet, 
                   std::map<solverID_t, evalID_t> &resultMap )
  {
  if ( m_activeEvals.empty() )
    { return ERR::AsyncEvaluator_NoPendingEvaluations; }

  int id = -1;
  while ((id = m_problem_ptr->next_eval()) == -1) 
    {
    #if defined(MSC_VER) || defined(__MINGW32__)
    _sleep(m_poll_interval);
    #else
    usleep(m_poll_interval);
    #endif
    }

  typename std::map<int, responseSet_t>::iterator eval_it
      = m_activeEvals.find(id);
  if ( eval_it == m_activeEvals.end() )
    {
    EXCEPTION_MNGR
        (std::runtime_error, 
         "AsyncEvaluator::return_evaluation - bad id returned: " << id;
         );
    }
  responseSet = eval_it->second;

  typename std::map<int, response_t*>::iterator responseBuf_it
      = m_activeResponseBuf.find(id);
  if ( responseBuf_it == m_activeResponseBuf.end() )
    {
    EXCEPTION_MNGR
        (std::runtime_error, 
         "AsyncEvaluator::return_evaluation - bad id returned: " << id;
         );
    }
  response_t *response = responseBuf_it->second;

  // process the returned evaluation
  compute_response_info(*response,
                        m_problem_ptr->state->constraint_lower_bounds,
                        m_problem_ptr->state->constraint_upper_bounds,
                        responseSet.convergenceFactor());
  responseSet.new_response(*response);
  delete response;

  // prepare the result information
  typename std::map<responseSet_t, ResponseInfo>::iterator rsi
      = m_responseSetInfo.find(responseSet);
  if ( rsi == m_responseSetInfo.end() )
    {
    EXCEPTION_MNGR
        (std::runtime_error, 
         "AsyncEvaluator::return_evaluation - response not in RSI";
         );
    }
  std::map<solverID_t, std::multimap<priority_t, evalID_t> >::iterator 
      sol_it = rsi->second.solvers.begin();
  std::map<solverID_t, std::multimap<priority_t, evalID_t> >::iterator 
      sol_itEnd = rsi->second.solvers.end();
  resultMap.clear();
  while ( sol_it != sol_itEnd )
    {
    resultMap[sol_it->first] = sol_it->second.begin()->second;
    sol_it->second.erase(sol_it->second.begin());
    --(m_chargedTime[sol_it->first]);

    if ( sol_it->second.empty() )
      {
      std::map<solverID_t, std::multimap<priority_t, evalID_t> >::iterator 
          tmp = sol_it;
      ++sol_it;
      rsi->second.solvers.erase(tmp);
      }
    else
      { ++sol_it; }
    }
  --(rsi->second.evalsInProgress);

  // delete the request from my internal structures 
  if ( rsi->second.solvers.empty() )
    { m_responseSetInfo.erase(rsi); }
  m_activeEvals.erase(eval_it);
  m_activeResponseBuf.erase(responseBuf_it);

  // Launch the next evaluation
  m_active_servers--;
  spawn_evaluation();
  return 0;
  }



/** Immediately execute an evaluation of the point at the specified
 *  response set and wait (block) for the result.  As ResponseSets are
 *  really references, this will cause the passed response set to
 *  contain the newly-calculated value.
 *
 *  Returns 0 on success, <0 on failure.
 */
template <class ProblemT, class DomainT>
int
AsyncEvaluator<ProblemT,DomainT>::
perform_evaluation( responseSet_t responseSet )
  {
#if defined(USING_PROBE)
  probe();
#endif

  response_t response;
  m_problem_ptr->Eval(responseSet.point(), response, m_mode);
  compute_response_info(response,
                        m_problem_ptr->state->constraint_lower_bounds,
                        m_problem_ptr->state->constraint_upper_bounds,
                        responseSet.convergenceFactor());
  responseSet.new_response(response);

  DEBUGPR(1000, ucout << "AsyncEvaluator::perform_evaluation - ans = " 
          << response.augmented_function_value() << " cval = " 
          << response.l2_constraint_violation() << std::endl;);
  return 0;
  }



/** Set the target (fractional) resource allocation for each solver.
 *  This will normalize the target allocations.
 */
template <class ProblemT, class DomainT>
void
AsyncEvaluator<ProblemT,DomainT>::
set_target_allocation(std::map<solverID_t, double> &target)
  {
  m_targetAlloc.clear();
  m_targetAlloc = target;

  // normalize
  double total = 0.0;
  std::map<solverID_t, double>::iterator it = m_targetAlloc.begin();
  std::map<solverID_t, double>::iterator itEnd = m_targetAlloc.end();
  for ( ; it != itEnd; ++it )
    { 
    if ( it->second < 0.0 )
      { it->second = 0.0; }
    total += it->second; 
    }

  if ( total > 0 )
    {
    for (it = m_targetAlloc.begin() ; it != itEnd; ++it )
      { it->second /= total; }
    }
  }
    



template <class ProblemT, class DomainT>
void 
AsyncEvaluator<ProblemT,DomainT>::
virt_debug_io(std::ostream& os, const bool , const int )
  {
  os << "\tUsing max_server_load == " << m_max_server_load << std::endl; 
  os << "\tUsing poll_interval == " << m_poll_interval << std::endl; 
  }


/** Start the next function evaluation based on the current and target
 *  computational allotments.  This launches the "winning" evaluation
 *  asynchronously and charges the computer time against all solvers
 *  requesting the evaluation.
 */
template <class ProblemT, class DomainT>
int
AsyncEvaluator<ProblemT,DomainT>::
spawn_evaluation()
  {
  if ( DEBUG_ASYNC_EVALUATOR > 1 )
    { std::cerr << "entering spawn_evaluation()" << std::endl; }

  if ( m_active_servers >= m_num_servers )
    { return ERR::AsyncEvaluator_NoAvailabelServers; }
  if ( m_solverRequests.empty() )
    { return ERR::AsyncEvaluator_NoPendingEvaluations; }

  std::map<solverID_t, double> priority;

  typename requestMap_t::iterator db_it = m_solverRequests.begin();
  typename requestMap_t::iterator db_itEnd = m_solverRequests.end();

  if ( DEBUG_ASYNC_EVALUATOR > 2 )
    { 
    std::cerr << "spawn(): solvers: " << m_solverRequests.size() << endl;
    for(db_it = m_solverRequests.begin(); db_it != db_itEnd; ++db_it)
      {
      std::cerr << "         queue " << db_it->first << ": ";
      
      typename std::map< priority_t, std::list<responseSet_t > >::iterator 
          db2_it = db_it->second.begin();
      typename std::map< priority_t, std::list<responseSet_t > >::iterator 
          db2_itEnd = db_it->second.end();
      for( ; db2_it != db2_itEnd; ++db2_it)
        { std::cerr << db2_it->first << ":" << db2_it->second.size() << ", "; }
      std::cerr << endl;
      }
    }

  // Make sure that all solvers that have requested time are included in
  // the priority map
  typename requestMap_t::iterator rm_it = m_solverRequests.begin();
  typename requestMap_t::iterator rm_itEnd = m_solverRequests.end();
  for( ; rm_it != rm_itEnd; ++rm_it )
    { priority[rm_it->first]; }

  // the target allocation
  std::map<solverID_t, double>::iterator d_it = m_targetAlloc.begin();
  std::map<solverID_t, double>::iterator d_itEnd = m_targetAlloc.end();
  for ( ; d_it != d_itEnd; ++d_it )
    { priority[d_it->first] = -1.0 * d_it->second; }

  // a correction for starvation
  d_itEnd = m_starvation.end();
  for ( d_it = m_starvation.begin() ; d_it != d_itEnd; ++d_it )
    { priority[d_it->first] += d_it->second; }

  // and the current allocations
  std::map<solverID_t, int>::iterator i_it = m_chargedTime.begin();
  std::map<solverID_t, int>::iterator i_itEnd = m_chargedTime.end();
  double total = 0;
  for( ; i_it != i_itEnd; ++i_it )
    { total += i_it->second; }
  if ( total < m_num_servers )
    { total = m_num_servers; }
  for( i_it = m_chargedTime.begin() ; i_it != i_itEnd; ++i_it )
    { priority[i_it->first] += static_cast<double>(i_it->second) / total; }


  // ... the solver with the lowest number wins
  typename requestMap_t::iterator req_it;
  std::map<solverID_t, double>::iterator best;
  bool notOK = true;
  while (( notOK ) && ( ! priority.empty() ))
    {
    d_itEnd = priority.end();
    for ( best = d_it = priority.begin() ; d_it != d_itEnd; ++d_it )
      { 
      // remember anyone with an allocation < their target
      total = d_it->second;
      m_starvation[d_it->first] = ( total < 0.0 ? total : 0.0 );

      if ( d_it->second < best->second )  
        { best = d_it; }
      }
  
    // find the evaluation that won (the lowest priority for the queued
    // solvers that won).
    req_it = m_solverRequests.find(best->first);
    if (( req_it == m_solverRequests.end() ) || ( req_it->second.empty() ))
      { 
      if ( DEBUG_ASYNC_EVALUATOR )
        {
        std::cerr << "spawn(): selected solver " << best->first
                  << " has no requests" << endl;
        }
      // we get here when the winning solver has no queued requests... so
      // we have to find another one.
      m_starvation[best->first] = 0;
      priority.erase(best); 
      }
    else
      { notOK = false; }
    }
  if ( priority.empty() )
    {
    EXCEPTION_MNGR
        (std::runtime_error, 
         "AsyncEvaluator::spawn_evaluation - empty priority list");
    }

  
  // get the winning response set
  responseSet_t responseSet = req_it->second.begin()->second.front();


  // for each solver that gets to "piggyback" on the request, update the
  // charged time and the master request map.
  ResponseInfo &rsi = m_responseSetInfo[responseSet];
  std::map<solverID_t, std::multimap<priority_t, evalID_t> >::iterator 
      sol_it = rsi.solvers.begin();
  std::map<solverID_t, std::multimap<priority_t, evalID_t> >::iterator 
      sol_itEnd = rsi.solvers.end();
  for ( ; sol_it != sol_itEnd; ++sol_it)
    { 
    // Does this solver have a pending request for this ResponseSet?
    if ( sol_it->second.size() <= rsi.evalsInProgress )
      { continue; }

    // Find the priority that this solver assigned the request
    std::multimap<priority_t, evalID_t>::iterator rsiPri_it
        = sol_it->second.begin();
    unsigned int ui = rsi.evalsInProgress;
    while ( ui > 0 )
      { 
      cerr << "Multiple evals in progress" << endl;
      ++rsiPri_it;
      --ui;
      }
    
    // Now find this ResponseSet in the master queued priority list
    typename std::map< priority_t, std::list<responseSet_t > >::iterator
        pri_it = m_solverRequests[sol_it->first].find(rsiPri_it->first);
    if ( pri_it == m_solverRequests[sol_it->first].end() )
      {
      EXCEPTION_MNGR
          (std::runtime_error, 
           "AsyncEvaluator::spawn_evaluation - unable to find " 
           "priority for Solver " << sol_it->first << ", priority "
           << rsiPri_it->first << ", response id " << responseSet.id();
           );
      }
    typename std::list<responseSet_t >::iterator rs_it 
        = pri_it->second.begin();
    typename std::list<responseSet_t >::iterator rs_itEnd
        = pri_it->second.end();
    while (( rs_it != rs_itEnd ) && ( *rs_it != responseSet ))
      { ++rs_it; }
    if ( rs_it == rs_itEnd )
      {
      EXCEPTION_MNGR
          (std::runtime_error, 
           "AsyncEvaluator::spawn_evaluation - unable to find " 
           "response for Solver " << sol_it->first << ", priority "
           << rsiPri_it->first << ", response id " << responseSet.id();
           );
      }

    // delete this ResponseSet from the master queued list
    pri_it->second.erase(rs_it);
    if ( pri_it->second.empty() )
      { m_solverRequests[sol_it->first].erase(pri_it); }

    // charge the solver's computational time
    ++(m_chargedTime[sol_it->first]);

    // As the solver is getting a request run, clear the starvation flag.
    m_starvation[best->first] = 0;
    }
  
  // Mark that this response set has another evaluation in progress
  ++(rsi.evalsInProgress);

  if ( DEBUG_ASYNC_EVALUATOR )
    { cerr << "spawn(): winning solver " << best->first 
           << ", response = " << responseSet.id() << endl; }


  // Finally, fire off the evaluation
#if defined(USING_PROBE)
  probe();
#endif
  response_t *response = new response_t;
  int pri = 0;
  m_problem_ptr->AsyncEval( responseSet.point(),
                            pri, // we manage the schedule... so set this to 0
                            response, 
                            m_mode );
  int id = m_problem_ptr->last_id();
  ++m_active_servers;
  m_activeEvals[id] = responseSet;
  m_activeResponseBuf[id] = response;
  DEBUGPR(100, ucout << "Spawned AsyncEval " << id << " for ResponseSet "
          << responseSet.id());

  return 0;
  }


}; // namespace colin

#endif // colin_AsyncEvaluator_h
