/*  _______________________________________________________________________

    DAKOTA: Design Analysis Kit for Optimization and Terascale Applications
    Copyright (c) 2006, Sandia National Laboratories.
    This software is distributed under the GNU General Public License.
    For more information, see the README file in the top Dakota directory.
    _______________________________________________________________________ */

//- Class:       AllVariables
//- Description: Container class of variables, employing the "all" approach to
//-              variables usage.
//- Owner:       Mike Eldred
//- Version: $Id: AllVariables.H 5337 2008-10-09 18:12:05Z mseldre $

#ifndef ALL_VARIABLES_H
#define ALL_VARIABLES_H

#include "DakotaVariables.H"


namespace Dakota {

/// Derived class within the Variables hierarchy which employs
/// the all data view.

/** Derived variables classes take different views of the design,
    uncertain, and state variable types and the continuous and
    discrete domain types.  The AllVariables derived class combines
    design, uncertain, and state variable types but separates
    continuous and discrete domain types.  The result is a single
    array of continuous variables (allContinuousVars) and a single
    array of discrete variables (allDiscreteVars).  Parameter and DACE
    studies currently use this approach (see
    Variables::get_variables(problem_db)). */

class AllVariables: public Variables
{
  //
  //- Heading: Friends
  //

  /// equality operator
  friend bool operator==(const AllVariables& vars1, const AllVariables& vars2);

#ifdef DAKOTA_BOOST
  /// hash_value
  friend std::size_t hash_value(const AllVariables& vars);

  /// binary_equal_to (since 'operator==' is not suitable for boost/hash_set)
  friend bool binary_equal_to(const AllVariables& vars1,
                              const AllVariables& vars2);
#endif

public:

  //
  //- Heading: Constructors and destructor
  //

  /// default constructor
  AllVariables();
  /// standard constructor
  AllVariables(const ProblemDescDB& problem_db, const pair<short,short>& view);
  /// destructor
  ~AllVariables();

  //
  //- Heading: Virtual function redefinitions
  //

  size_t tv() const;
  //size_t cv() const;
  //size_t dv() const;

  const RealVector& continuous_variables() const;
  void continuous_variable(const Real& c_var, const size_t& i);
  void continuous_variables(const RealVector& c_vars);
  const IntVector& discrete_variables() const;
  void discrete_variable(const int& d_var, const size_t& i);
  void discrete_variables(const IntVector& d_vars);

  const StringArray& continuous_variable_labels() const;
  void continuous_variable_labels(const StringArray& cv_labels);
  const StringArray& discrete_variable_labels()   const;
  void discrete_variable_labels(const StringArray& dv_labels);

  const StringArray& continuous_variable_types() const;
  const StringArray& discrete_variable_types() const;

  const UIntArray& continuous_variable_ids() const;

  //size_t icv() const;
  //size_t idv() const;

  //const RealVector& inactive_continuous_variables() const;
  //void inactive_continuous_variables(const RealVector& i_c_vars);
  //const IntVector& inactive_discrete_variables() const;
  //void inactive_discrete_variables(const IntVector& i_d_vars);

  //const StringArray& inactive_continuous_variable_labels() const;
  //void inactive_continuous_variable_labels(const StringArray& i_c_v_labels);
  //const StringArray& inactive_discrete_variable_labels() const;
  //void inactive_discrete_variable_labels(const StringArray& i_d_v_labels);

  //const UIntArray& inactive_continuous_variable_ids() const;

  size_t acv() const;
  size_t adv() const;

  RealVector all_continuous_variables() const;
  void all_continuous_variables(const RealVector& a_c_vars);
  IntVector  all_discrete_variables()   const;
  void all_discrete_variables(const IntVector& a_d_vars);

  StringArray all_continuous_variable_labels() const;
  void all_continuous_variable_labels(const StringArray& a_c_v_labels);
  StringArray all_discrete_variable_labels()   const;
  void all_discrete_variable_labels(const StringArray& a_d_v_labels);
  StringArray all_variable_labels()            const;

  const StringArray& all_discrete_variable_types() const;

  void read(istream& s);
  void write(ostream& s) const;
  void write_aprepro(ostream& s) const;

  void read_annotated(istream& s);
  void write_annotated(ostream& s) const;

  void write_tabular(ostream& s) const;

  void read(BiStream& s);
  void write(BoStream& s) const;

  void read(MPIUnpackBuffer& s);
  void write(MPIPackBuffer& s) const;

protected:

  void copy_rep(const Variables* vars_rep);

  void reshape_rep(const Sizet2DArray& vars_comps);

private:

  //
  //- Heading: Private member functions
  //

  /// construct VarTypes and VarIds arrays using variablesComponents
  void build_types_ids();

  //
  //- Heading: Private data members
  //

  /// a continuous array combining all of the continuous variables
  /// (design, uncertain, and state).
  RealVector allContinuousVars;
  /// a discrete array combining all of the discrete variables
  /// (design and state).
  IntVector allDiscreteVars;

  /// a label array combining all of the continuous variable labels
  /// (design, uncertain, and state).
  StringArray allContinuousLabels;
  /// a label array combining all of the discrete variable labels
  /// (design and state).
  StringArray allDiscreteLabels;

  /// array of variable types for all of the discrete variables
  StringArray allDiscreteVarTypes;
};


inline AllVariables::AllVariables()
{ }


inline AllVariables::~AllVariables()
{ }


inline size_t AllVariables::tv() const
{ return allContinuousVars.length() + allDiscreteVars.length(); }


inline const RealVector& AllVariables::continuous_variables() const
{ return allContinuousVars; }


inline void AllVariables::
continuous_variable(const Real& c_var, const size_t& i)
{ allContinuousVars[i] = c_var; }


inline void AllVariables::continuous_variables(const RealVector& c_vars)
{
  // in the AllVariables case, we cannot in general allow assignments to change
  // the length of allContinuousVars since this invalidates variableComponents
  // and can result in I/O errors.  However, allow assignment in the case of
  // instantiate-on-the-fly which uses default construction.
  size_t num_acv = allContinuousVars.length();
  if (num_acv && c_vars.length() != num_acv) {
    Cerr << "Error: length of assigned vector (" << c_vars.length()
	 << ") does not equal number of continuous variables (" << num_acv
	 << ") in AllVariables::continuous_variables()." << endl;
    abort_handler(-1);
  }
  allContinuousVars = c_vars;
}


inline const IntVector& AllVariables::discrete_variables() const
{ return allDiscreteVars; }


inline void AllVariables::discrete_variable(const int& d_var, const size_t& i)
{ allDiscreteVars[i] = d_var; }


inline void AllVariables::discrete_variables(const IntVector& d_vars)
{
  // in the AllVariables case, we cannot in general allow assignments to change
  // the length of allContinuousVars since this invalidates variableComponents
  // and can result in I/O errors.  However, allow assignment in the case of
  // instantiate-on-the-fly which uses default construction.
  size_t num_adv = allDiscreteVars.length();
  if (num_adv && d_vars.length() != num_adv) {
    Cerr << "Error: length of assigned vector (" << d_vars.length()
	 << ") does not equal number of discrete variables (" << num_adv
	 << ") in AllVariables::discrete_variables()." << endl;
    abort_handler(-1);
  }
  allDiscreteVars = d_vars;
}


inline const StringArray& AllVariables::continuous_variable_labels() const
{ return allContinuousLabels; }


inline void AllVariables::
continuous_variable_labels(const StringArray& cv_labels)
{
  size_t num_acl = allContinuousLabels.length();
  if (num_acl && cv_labels.length() != num_acl) {
    Cerr << "Error: length of assigned string array (" << cv_labels.length()
	 << ") does not equal number of continuous variable labels (" << num_acl
	 << ") in AllVariables::continuous_variable_labels()." << endl;
    abort_handler(-1);
  }
  allContinuousLabels = cv_labels;
}


inline const StringArray& AllVariables::discrete_variable_labels() const
{ return allDiscreteLabels; }


inline void AllVariables::discrete_variable_labels(const StringArray& dv_labels)
{
  size_t num_adl = allDiscreteLabels.length();
  if (num_adl && dv_labels.length() != num_adl) {
    Cerr << "Error: length of assigned string array (" << dv_labels.length()
	 << ") does not equal number of discrete variable labels (" << num_adl
	 << ") in AllVariables::discrete_variable_labels()." << endl;
    abort_handler(-1);
  }
  allDiscreteLabels = dv_labels;
}


inline const StringArray& AllVariables::continuous_variable_types() const
{ return allContinuousVarTypes; }


inline const StringArray& AllVariables::discrete_variable_types() const
{ return allDiscreteVarTypes; }


inline const UIntArray& AllVariables::continuous_variable_ids() const
{ return allContinuousVarIds; }


inline size_t AllVariables::acv() const
{ return allContinuousVars.length(); }


inline size_t AllVariables::adv() const
{ return allDiscreteVars.length(); }


inline RealVector AllVariables::all_continuous_variables() const
{ return allContinuousVars; }


inline void AllVariables::all_continuous_variables(const RealVector& a_c_vars)
{ allContinuousVars = a_c_vars; }


inline IntVector AllVariables::all_discrete_variables() const
{ return allDiscreteVars; }


inline void AllVariables::all_discrete_variables(const IntVector& a_d_vars)
{ allDiscreteVars = a_d_vars; }


inline StringArray AllVariables::all_continuous_variable_labels() const
{ return allContinuousLabels; }


inline void AllVariables::
all_continuous_variable_labels(const StringArray& a_c_v_labels)
{ allContinuousLabels = a_c_v_labels; }


inline StringArray AllVariables::all_discrete_variable_labels() const
{ return allDiscreteLabels; }


inline void AllVariables::
all_discrete_variable_labels(const StringArray& a_d_v_labels)
{ allDiscreteLabels = a_d_v_labels; }


inline StringArray AllVariables::all_variable_labels() const
{
  size_t num_cdv    = variablesComponents[0][1],
         num_ddv    = variablesComponents[0][2],
         num_uv_csv = allContinuousVars.length() - num_cdv,
         num_dsv    = allDiscreteVars.length()   - num_ddv,
         num_dv     = num_cdv + num_ddv;
  StringArray a_v_labels(num_dv+num_uv_csv+num_dsv);
  copy_data_partial(allContinuousLabels, 0, num_cdv, a_v_labels, 0);
  copy_data_partial(allDiscreteLabels,   0, num_ddv, a_v_labels, num_cdv);
  copy_data_partial(allContinuousLabels, num_cdv, num_uv_csv, a_v_labels,
		    num_dv);
  copy_data_partial(allDiscreteLabels,   num_ddv, num_dsv,    a_v_labels,
		    num_dv+num_uv_csv);
  return a_v_labels;
}


inline const StringArray& AllVariables::all_discrete_variable_types() const
{ return allDiscreteVarTypes; }


#ifdef DAKOTA_BOOST
/// binary_equal_to for AllVariables -- NOTE: no tolerance is used!
inline bool binary_equal_to(const AllVariables& vars1,const AllVariables& vars2){
  // Check for equality in array lengths
  size_t cv_size = vars1.allContinuousVars.size();
  size_t dv_size = vars1.allDiscreteVars.size();
  if (vars2.allContinuousVars.size() != cv_size ||
      vars2.allDiscreteVars.size()   != dv_size)
    return false;

  // Check each value (labels are ignored!)
  size_t i;
  for (i=0; i<cv_size; ++i) {
    if ( vars2.allContinuousVars[i] != vars1.allContinuousVars[i] )
      return false;
  }

  for (i=0; i<dv_size; ++i) {
    if ( vars2.allDiscreteVars[i] != vars1.allDiscreteVars[i] )
      return false;
  }

  return true;
}

/// hash_value for AllVariables
inline std::size_t hash_value(const AllVariables& vars)
{
  std::size_t seed = 0;
  boost::hash_combine(seed, vars.allContinuousVars);
  boost::hash_combine(seed, vars.allDiscreteVars);

  return seed;
}
#endif  // DAKOTA_BOOST

} // namespace Dakota

#endif
