/*  _______________________________________________________________________

    DAKOTA: Design Analysis Kit for Optimization and Terascale Applications
    Copyright (c) 2001, Sandia National Laboratories.
    This software is distributed under the GNU General Public License.
    For more information, see the README file in the top Dakota directory.
    _______________________________________________________________________ */

//- Class:	 NonDQuadrature
//- Description: Implementation code for NonDQuadrature class
//- Owner:       Mike Eldred
//- Revised by:  
//- Version:

#include "data_types.h"
#include "system_defs.h"
#include "NonDQuadrature.H"
#include "DakotaModel.H"
#include "ProblemDescDB.H"
#include "BasisPolynomial.H"

static const char rcsId[]="@(#) $Id: NonDQuadrature.C,v 1.57 2004/06/21 19:57:32 mseldre Exp $";


namespace Dakota {

/** This constructor is called for a standard letter-envelope iterator
    instantiation.  In this case, set_db_list_nodes has been called
    and probDescDB can be queried for settings from the method
    specification.  It is not currently used, as there is not yet a
    separate nond_quadrature method specification. */
NonDQuadrature::NonDQuadrature(Model& model): NonDIntegration(model), 
  quadOrderSpec(probDescDB.get_dusa("method.nond.quadrature_order")),
  quadOrder(quadOrderSpec)
{
  // pure virtual fn cannot be called from NonDIntegration ctor
  check_input();
}


/** This alternate constructor is used for on-the-fly generation and
    evaluation of numerical quadrature points. */
NonDQuadrature::NonDQuadrature(Model& model, const UShortArray& order): 
  NonDIntegration(NoDBBaseConstructor(), model), quadOrderSpec(order),
  quadOrder(order)
{
  // pure virtual fn cannot be called from NonDIntegration ctor
  //check_input();
}


NonDQuadrature::~NonDQuadrature()
{ }


void NonDQuadrature::check_input()
{
  bool err_flag = false;

  const Pecos::ShortArray& x_types = natafTransform.x_types();
  numDesignVars     = std::count(x_types.begin(), x_types.end(), (short)DESIGN);
  numStateVars      = std::count(x_types.begin(), x_types.end(), (short)STATE);
  numContinuousVars = numDesignVars + numUncertainVars + numStateVars;

  // Promote quadOrder to full length, if needed.
  // (Leave quadOrderSpec as original user specification.)
  if (quadOrder.empty()) {
    Cerr << "Error: quadrature order specification required in NonDQuadrature."
	 << endl;
    err_flag = true;
  }
  else if (quadOrder.length() != numContinuousVars) {
    if (quadOrder.length() == 1) {
      unsigned short quad_order = quadOrder[0];
      quadOrder.reshape(numContinuousVars);
      quadOrder = quad_order;
    }
    else {
      Cerr << "Error: length of quadrature order specification does not equal "
	   << "number of active variables." << endl;
      err_flag = true;
    }
  }

  // Gauss-Hermite, Gauss-Legendre, Gauss-Laguerre, generalized Gauss-Laguerre,
  // and Gauss-Jacobi are currently available
  if ( numUncertainVars != numNormalVars + numUniformVars + numExponentialVars +
                           numBetaVars   + numGammaVars ) {
    Cerr << "Error: All variables in NonDQuadrature must be standard forms of "
	 << "normal, uniform,\n       exponential, beta, or gamma "
	 << "distributions since only Gauss-Hermite,\n       Gauss-Legendre, "
	 << "Gauss-Laguerre, generalized Gauss-Laguerre, and Gauss-Jacobi"
	 << "\n       numerical integration are supported." << endl;
    err_flag = true;
  }

  if (err_flag)
    abort_handler(-1);
}


void NonDQuadrature::get_parameter_sets(const Model& model)
{
  size_t i, j, num_quad_points = 1;
  Cout << "\nNumber of Gauss points per variable: { ";
  for (i=0; i<numContinuousVars; i++) {
    num_quad_points *= quadOrder[i];
    Cout << quadOrder[i] << ' ';
  }
  Cout << "}\nTotal number of integration points:  " << num_quad_points << '\n';

  BasisPolynomial hermite_poly, legendre_poly, laguerre_poly, jacobi_poly,
    gen_laguerre_poly;
  if (numNormalVars)
    hermite_poly  = BasisPolynomial(HERMITE);
  if (numDesignVars || numUniformVars || numStateVars)
    legendre_poly = BasisPolynomial(LEGENDRE);
  if (numExponentialVars)
    laguerre_poly = BasisPolynomial(LAGUERRE);
  if (numBetaVars)
    jacobi_poly   = BasisPolynomial(JACOBI);
  if (numGammaVars)
    gen_laguerre_poly = BasisPolynomial(GENERALIZED_LAGUERRE);
  if (gaussPts1D.empty() || gaussWts1D.empty()) {
    gaussPts1D.reshape(numContinuousVars);
    gaussWts1D.reshape(numContinuousVars);
  }
  const RealDenseVector& jacobi_alphas  = iteratedModel.beta_alphas();
  const RealDenseVector& jacobi_betas   = iteratedModel.beta_betas();
  const RealDenseVector& gen_lag_alphas = iteratedModel.gamma_alphas();
  const Pecos::ShortArray& u_types = natafTransform.u_types();
  size_t beta_cntr = 0, gamma_cntr = 0;
  for (i=0; i<numContinuousVars; i++) {
    gaussPts1D[i].reshape(1);
    gaussWts1D[i].reshape(1);
    switch (u_types[i]) {
    case NORMAL:
      gaussPts1D[i][0] = hermite_poly.gauss_points(quadOrder[i]);
      gaussWts1D[i][0] = hermite_poly.gauss_weights(quadOrder[i]);
      break;
    case UNIFORM:
      gaussPts1D[i][0] = legendre_poly.gauss_points(quadOrder[i]);
      gaussWts1D[i][0] = legendre_poly.gauss_weights(quadOrder[i]);
      break;
    case EXPONENTIAL:
      gaussPts1D[i][0] = laguerre_poly.gauss_points(quadOrder[i]);
      gaussWts1D[i][0] = laguerre_poly.gauss_weights(quadOrder[i]);
      break;
    case BETA:
      jacobi_poly.alpha_stat(jacobi_alphas[beta_cntr]);
      jacobi_poly.beta_stat(jacobi_betas[beta_cntr]);
      jacobi_poly.reset_gauss();
      gaussPts1D[i][0] = jacobi_poly.gauss_points(quadOrder[i]);
      gaussWts1D[i][0] = jacobi_poly.gauss_weights(quadOrder[i]);
      beta_cntr++;
      break;
    case GAMMA:
      gen_laguerre_poly.alpha_stat(gen_lag_alphas[gamma_cntr]);
      gen_laguerre_poly.reset_gauss();
      gaussPts1D[i][0] = gen_laguerre_poly.gauss_points(quadOrder[i]);
      gaussWts1D[i][0] = gen_laguerre_poly.gauss_weights(quadOrder[i]);
      gamma_cntr++;
      break;
    }
  }

  // Tensor-product quadrature: Integral of f approximated by
  // Sum_i1 Sum_i2 ... Sum_in (w_i1 w_i2 ... w_in) f(x_i1, x_i2, ..., x_in)
  // > project 1-D gauss point arrays (of potentially different type and order)
  //   into an n-dimensional stencil.
  // > compute and store products of 1-D Gauss weights at each point in stencil.
  SizetArray gauss_indices(numContinuousVars, 0);
  if (allVariables.length() != num_quad_points)
    allVariables.reshape(num_quad_points);
  if (weightProducts.length() != num_quad_points)
    weightProducts.reshape(num_quad_points);
  const pair<short,short>& view = iteratedModel.current_variables().view();
  RealVector c_vars(numContinuousVars);
  for (i=0; i<num_quad_points; i++) {
    if (allVariables[i].is_null())
      allVariables[i] = Variables(view); // minimal instantiate-on-the-fly
    Real& wt_prod_i = weightProducts[i];
    wt_prod_i = 1.0;
    for (j=0; j<numContinuousVars; j++) {
      c_vars[j]  = gaussPts1D[j][0][gauss_indices[j]];
      wt_prod_i *= gaussWts1D[j][0][gauss_indices[j]];
    }
    allVariables[i].continuous_variables(c_vars);
    // increment the n-dimensional gauss point index set
    size_t increment_index = 0;
    gauss_indices[increment_index]++;
    while (increment_index < numContinuousVars &&
	   gauss_indices[increment_index] >= quadOrder[increment_index]) {
      gauss_indices[increment_index] = 0;
      increment_index++;
      if (increment_index < numContinuousVars)
	gauss_indices[increment_index]++;
    }
  }
}

} // namespace Dakota
