/*  _______________________________________________________________________

    DAKOTA: Design Analysis Kit for Optimization and Terascale Applications
    Copyright (c) 2006, Sandia National Laboratories.
    This software is distributed under the GNU General Public License.
    For more information, see the README file in the top Dakota directory.
    _______________________________________________________________________ */

//- Class:        DataVariables
//- Description:
//-
//-
//- Owner:        Mike Eldred
//- Version: $Id: DataVariables.H 5416 2008-10-23 21:53:56Z mseldre $

#ifndef DATA_VARIABLES_H
#define DATA_VARIABLES_H

#include "data_types.h"

namespace Dakota {

// variable types for active and inactive views include EMPTY plus the 8
// combinations of All vs. Distinct (active: Design or Uncertain or State)
// [4 total] and Merged vs. Mixed [2 total].  Of these 9, 6 are currently used.
enum { EMPTY, MERGED_ALL, MIXED_ALL,
       MERGED_DISTINCT_DESIGN, MERGED_DISTINCT_UNCERTAIN, MERGED_DISTINCT_STATE,
       MIXED_DISTINCT_DESIGN, MIXED_DISTINCT_UNCERTAIN, MIXED_DISTINCT_STATE };


/// Body class for variables specification data.

/** The DataVariablesRep class is used to contain the data from a
    variables keyword specification.  Default values are managed in
    the DataVariablesRep constructor.  Data is public to avoid
    maintaining set/get functions, but is still encapsulated within
    ProblemDescDB since ProblemDescDB::dataVariablesList is private (a
    similar model is used with SurrogateDataPoint objects contained in
    Dakota::Approximation). */

class DataVariablesRep
{
  //
  //- Heading: Friends
  //

  /// the handle class can access attributes of the body class directly
  friend class DataVariables;

//private:
public:

  //
  //- Heading: Data
  //

  /// string identifier for the variables specification data set
  /// (from the \c id_variables specification in \ref VarSetId)
  String idVariables;
  /// number of continuous design variables (from the \c continuous_design
  /// specification in \ref VarDV)
  size_t numContinuousDesVars;
  /// number of discrete design variables (from the \c discrete_design
  /// specification in \ref VarDV)
  size_t numDiscreteDesVars;
  /// number of normal uncertain variables (from the \c normal_uncertain
  /// specification in \ref VarUV)
  size_t numNormalUncVars;
  /// number of lognormal uncertain variables (from the \c lognormal_uncertain
  /// specification in \ref VarUV)
  size_t numLognormalUncVars;
  /// number of uniform uncertain variables (from the \c uniform_uncertain
  /// specification in \ref VarUV)
  size_t numUniformUncVars;
  /// number of loguniform uncertain variables (from the \c
  /// loguniform_uncertain specification in \ref VarUV)
  size_t numLoguniformUncVars;
  /// number of triangular uncertain variables (from the \c
  /// triangular_uncertain specification in \ref VarUV)
  size_t numTriangularUncVars;
  /// number of exponential uncertain variables (from the \c
  /// exponential_uncertain specification in \ref VarUV)
  size_t numExponentialUncVars;
  /// number of beta uncertain variables (from the \c beta_uncertain
  /// specification in \ref VarUV)
  size_t numBetaUncVars;
  /// number of gamma uncertain variables (from the \c gamma_uncertain
  /// specification in \ref VarUV)
  size_t numGammaUncVars;
  /// number of gumbel uncertain variables (from the \c gumbel_uncertain
  /// specification in \ref VarUV)
  size_t numGumbelUncVars;
  /// number of frechet uncertain variables (from the \c frechet_uncertain
  /// specification in \ref VarUV)
  size_t numFrechetUncVars;
  /// number of weibull uncertain variables (from the \c weibull_uncertain
  /// specification in \ref VarUV)
  size_t numWeibullUncVars;
  /// number of histogram uncertain variables (from the \c histogram_uncertain
  /// specification in \ref VarUV)
  size_t numHistogramUncVars;
  /// number of interval uncertain variables (from the \c interval_uncertain
  /// specification in \ref VarUV)
  size_t numIntervalUncVars;
  /// number of continuous state variables (from the \c continuous_state
  /// specification in \ref VarSV)
  size_t numContinuousStateVars;
  /// number of discrete state variables (from the \c discrete_state
  /// specification in \ref VarSV)
  size_t numDiscreteStateVars;

  // Design arrays

  /// initial values for the continuous design variables array (from
  /// the \c cdv_initial_point specification in \ref VarDV)
  RealVector continuousDesignVars;
  /// the continuous design lower bounds array (from the \c
  /// cdv_lower_bounds specification in \ref VarDV)
  RealVector continuousDesignLowerBnds;
  /// the continuous design upper bounds array (from the \c
  /// cdv_upper_bounds specification in \ref VarDV)
  RealVector continuousDesignUpperBnds;
  /// the continuous design scale types array (from the \c
  /// cdv_scale_types specification in \ref VarDV)
  StringArray continuousDesignScaleTypes;
  /// the continuous design scales array (from the \c
  /// cdv_scales specification in \ref VarDV)
  RealVector continuousDesignScales;
  /// initial values for the discrete design variables array (from
  /// the \c ddv_initial_point specification in \ref VarDV)
  IntVector discreteDesignVars;
  /// the discrete design lower bounds array (from the \c
  /// ddv_lower_bounds specification in \ref VarDV)
  IntVector discreteDesignLowerBnds;
  /// the discrete design upper bounds array (from the \c
  /// ddv_upper_bounds specification in \ref VarDV)
  IntVector discreteDesignUpperBnds;
  /// the continuous design labels array (from the \c cdv_descriptors
  /// specification in \ref VarDV)
  StringArray continuousDesignLabels;
  /// the discrete design labels array (from the \c ddv_descriptors
  /// specification in \ref VarDV)
  StringArray discreteDesignLabels;

  // Uncertain arrays

  /// means of the normal uncertain variables (from the \c nuv_means
  /// specification in \ref VarUV)
  RealDenseVector normalUncMeans;
  /// standard deviations of the normal uncertain variables (from
  /// the \c nuv_std_deviations specification in \ref VarUV)
  RealDenseVector normalUncStdDevs;
  /// distribution lower bounds for the normal uncertain variables
  /// (from the \c nuv_lower_bounds specification in \ref VarUV)
  RealDenseVector normalUncLowerBnds;
  /// distribution upper bounds for the normal uncertain variables
  /// (from the \c nuv_upper_bounds specification in \ref VarUV)
  RealDenseVector normalUncUpperBnds;
  /// means of the lognormal uncertain variables (from the \c
  /// lnuv_means specification in \ref VarUV)
  RealDenseVector lognormalUncMeans;
  /// standard deviations of the lognormal uncertain variables (from
  /// the \c lnuv_std_deviations specification in \ref VarUV)
  RealDenseVector lognormalUncStdDevs;
  /// error factors for the lognormal uncertain variables (from
  /// the \c lnuv_error_factors specification in \ref VarUV)
  RealDenseVector lognormalUncErrFacts;
  /// distribution lower bounds for the lognormal uncertain variables
  /// (from the \c lnuv_lower_bounds specification in \ref VarUV)
  RealDenseVector lognormalUncLowerBnds;
  /// distribution upper bounds for the lognormal uncertain variables
  /// (from the \c lnuv_upper_bounds specification in \ref VarUV)
  RealDenseVector lognormalUncUpperBnds;
  /// distribution lower bounds for the uniform uncertain variables
  /// (from the \c uuv_lower_bounds specification in \ref VarUV)
  RealDenseVector uniformUncLowerBnds;
  /// distribution upper bounds for the uniform uncertain variables
  /// (from the \c uuv_upper_bounds specification in \ref VarUV)
  RealDenseVector uniformUncUpperBnds;
  /// distribution lower bounds for the loguniform uncertain variables
  /// (from the \c luuv_lower_bounds specification in \ref VarUV)
  RealDenseVector loguniformUncLowerBnds;
  /// distribution upper bounds for the loguniform uncertain variables
  /// (from the \c luuv_upper_bounds specification in \ref VarUV)
  RealDenseVector loguniformUncUpperBnds;
  /// modes of the triangular uncertain variables (from the \c tuv_modes
  /// specification in \ref VarUV)
  RealDenseVector triangularUncModes;
  /// distribution lower bounds for the triangular uncertain variables
  /// (from the \c tuv_lower_bounds specification in \ref VarUV)
  RealDenseVector triangularUncLowerBnds;
  /// distribution upper bounds for the triangular uncertain variables
  /// (from the \c tuv_upper_bounds specification in \ref VarUV)
  RealDenseVector triangularUncUpperBnds;
  /// beta factors for the exponential uncertain variables (from
  /// the \c euv_betas specification in \ref VarUV)
  RealDenseVector exponentialUncBetas;
  /// alpha factors for the beta uncertain variables (from
  /// the \c buv_means specification in \ref VarUV)
  RealDenseVector betaUncAlphas;
  /// beta factors for the beta uncertain variables (from
  /// the \c buv_std_deviations specification in \ref VarUV)
  RealDenseVector betaUncBetas;
  /// distribution lower bounds for the beta uncertain variables
  /// (from the \c buv_lower_bounds specification in \ref VarUV)
  RealDenseVector betaUncLowerBnds;
  /// distribution upper bounds for the beta uncertain variables
  /// (from the \c buv_upper_bounds specification in \ref VarUV)
  RealDenseVector betaUncUpperBnds;
  /// alpha factors for the gamma uncertain variables (from
  /// the \c gauv_alphas specification in \ref VarUV)
  RealDenseVector gammaUncAlphas;
  /// beta factors for the gamma uncertain variables (from
  /// the \c gauv_betas specification in \ref VarUV)
  RealDenseVector gammaUncBetas;
  /// alpha factors for the gumbel uncertain variables (from the \c
  /// guuv_alphas specification in \ref VarUV)
  RealDenseVector gumbelUncAlphas;
  /// beta factors for of the gumbel uncertain variables (from
  /// the \c guuv_betas specification in \ref VarUV)
  RealDenseVector gumbelUncBetas;
  /// alpha factors for the frechet uncertain variables (from
  /// the \c fuv_alphas specification in \ref VarUV)
  RealDenseVector frechetUncAlphas;
  /// beta factors for the frechet uncertain variables (from
  /// the \c fuv_betas specification in \ref VarUV)
  RealDenseVector frechetUncBetas;
  /// alpha factors for the weibull uncertain variables (from
  /// the \c wuv_alphas specification in \ref VarUV)
  RealDenseVector weibullUncAlphas;
  /// beta factors for the weibull uncertain variables (from
  /// the \c wuv_betas specification in \ref VarUV)
  RealDenseVector weibullUncBetas;
  /// an array containing a vector of (x,y) pairs for each bin-based
  /// histogram uncertain variable (see continuous linear histogram
  /// in LHS manual; from the \c huv_num_bin_pairs and \c huv_bin_pairs
  /// specifications in \ref VarUV)
  RealDenseVectorArray histogramUncBinPairs;
  /// an array containing a vector of (x,y) pairs for each point-based
  /// histogram uncertain variable (see discrete histogram in LHS manual;
  /// from the \c huv_num_point_pairs and \c huv_point_pairs
  /// specifications in \ref VarUV)
  RealDenseVectorArray histogramUncPointPairs;
  /// Probability values per interval uncertain variable (from the \c
  /// iuv_interval_probs specification in \ref VarUV)
  RealDenseVectorArray intervalUncBasicProbs;
  /// Interval Bounds per interval uncertain variable (from the \c
  /// iuv_interval_bounds specification in \ref VarUV)
  RealDenseVectorArray intervalUncBounds;

  /// correlation matrix for all uncertain variables (from the \c
  /// uncertain_correlation_matrix specification in \ref VarUV). This
  /// matrix specifies rank correlations for sampling methods (i.e.,
  /// LHS) and correlation coefficients (rho_ij = normalized covariance
  /// matrix) for analytic reliability methods.
  RealSymDenseMatrix uncertainCorrelations;
  /// array of values for all uncertain variables (built and
  /// initialized in IDRProblemDescDB::variables_kwhandler())
  RealVector uncertainVars;
  /// distribution lower bounds for all uncertain variables
  /// (collected from \c nuv_lower_bounds, \c lnuv_lower_bounds,
  /// \c uuv_lower_bounds, \c luuv_lower_bounds, \c tuv_lower_bounds,
  /// and \c buv_lower_bounds specifications in \ref VarUV, and derived
  /// for gamma, gumbel, frechet, weibull and histogram specifications)
  RealVector uncertainLowerBnds;
  /// distribution upper bounds for all uncertain variables
  /// (collected from \c nuv_upper_bounds, \c lnuv_upper_bounds,
  /// \c uuv_upper_bounds, \c luuv_upper_bounds, \c tuv_lower_bounds,
  /// and \c buv_upper_bounds specifications in \ref VarUV, and derived
  /// for gamma, gumbel, frechet, weibull and histogram specifications)
  RealVector uncertainUpperBnds;
  /// labels for all uncertain variables (collected from \c nuv_descriptors,
  /// \c lnuv_descriptors, \c uuv_descriptors, \c luuv_descriptors, \c
  /// tuv_descriptors, \c buv_descriptors, \c gauv_descriptors, \c
  /// guuv_descriptors, \c fuv_descriptors, \c wuv_descriptors, and \c
  /// huv_descriptors specifications in \ref VarUV)
  StringArray uncertainLabels;

  // State arrays

  /// initial values for the continuous state variables array (from
  /// the \c csv_initial_state specification in \ref VarSV)
  RealVector continuousStateVars;
  /// the continuous state lower bounds array (from the \c
  /// csv_lower_bounds specification in \ref VarSV)
  RealVector continuousStateLowerBnds;
  /// the continuous state upper bounds array (from the \c
  /// csv_upper_bounds specification in \ref VarSV)
  RealVector continuousStateUpperBnds;
  /// initial values for the discrete state variables array (from
  /// the \c dsv_initial_state specification in \ref VarSV)
  IntVector discreteStateVars;
  /// the discrete state lower bounds array (from the \c
  /// dsv_lower_bounds specification in \ref VarSV)
  IntVector discreteStateLowerBnds;
  /// the discrete state upper bounds array (from the \c
  /// dsv_upper_bounds specification in \ref VarSV)
  IntVector discreteStateUpperBnds;
  /// the continuous state labels array (from the \c csv_descriptors
  /// specification in \ref VarSV)
  StringArray continuousStateLabels;
  /// the discrete state labels array (from the \c dsv_descriptors
  /// specification in \ref VarSV)
  StringArray discreteStateLabels;

private:

  //
  //- Heading: Private functions
  //

  DataVariablesRep();  ///< default constructor
  ~DataVariablesRep(); ///< destructor

  /// write a DataVariablesRep object to an ostream
  void write(ostream& s) const;

  /// read a DataVariablesRep object from a packed MPI buffer
  void read(MPIUnpackBuffer& s);
  /// write a DataVariablesRep object to a packed MPI buffer
  void write(MPIPackBuffer& s) const;

  //
  //- Heading: Private data members
  //

  /// number of handle objects sharing dataVarsRep
  int referenceCount;
};


inline DataVariablesRep::~DataVariablesRep()
{ }


/// Handle class for variables specification data.

/** The DataVariables class is used to provide a memory management
    handle for the data in DataVariablesRep.  It is populated by
    IDRProblemDescDB::variables_kwhandler() and is queried by the
    ProblemDescDB::get_<datatype>() functions.  A list of
    DataVariables objects is maintained in
    ProblemDescDB::dataVariablesList, one for each variables
    specification in an input file. */

class DataVariables
{
  //
  //- Heading: Friends
  //

  // comparison function
  //friend bool data_variables_id_compare(const DataVariables& dv,
  //                                      const void* id);

  // the problem description database
  //friend class ProblemDescDB;
  // the NIDR derived problem description database
  //friend class NIDRProblemDescDB;

public:

  //
  //- Heading: Constructors, destructor, operators
  //

  DataVariables();                                ///< constructor
  DataVariables(const DataVariables&);            ///< copy constructor
  ~DataVariables();                               ///< destructor

  DataVariables operator=(const DataVariables&); ///< assignment operator
  bool operator==(const DataVariables&);          ///< equality operator

  //
  //- Heading: Member methods
  //

  /// write a DataVariables object to an ostream
  void write(ostream& s) const;

  /// read a DataVariables object from a packed MPI buffer
  void read(MPIUnpackBuffer& s);
  /// write a DataVariables object to a packed MPI buffer
  void write(MPIPackBuffer& s) const;

  // Get Functions (composite variable counts only):

  /// return total number of design variables
  size_t design() { return dataVarsRep->numContinuousDesVars +
    dataVarsRep->numDiscreteDesVars; }
  /// return total number of uncertain variables
  size_t uncertain() { return dataVarsRep->numNormalUncVars +
    dataVarsRep->numLognormalUncVars + dataVarsRep->numUniformUncVars +
    dataVarsRep->numLoguniformUncVars + dataVarsRep->numTriangularUncVars +
    dataVarsRep->numExponentialUncVars + dataVarsRep->numBetaUncVars +
    dataVarsRep->numGammaUncVars + dataVarsRep->numGumbelUncVars +
    dataVarsRep->numFrechetUncVars + dataVarsRep->numWeibullUncVars +
    dataVarsRep->numHistogramUncVars + dataVarsRep->numIntervalUncVars; }
  /// return total number of state variables
  size_t state() { return dataVarsRep->numContinuousStateVars +
    dataVarsRep->numDiscreteStateVars; }
  /// return total number of continuous variables
  size_t num_continuous_variables() { return dataVarsRep->numContinuousDesVars +
    uncertain() + dataVarsRep->numContinuousStateVars; }
  /// return total number of discrete variables
  size_t num_discrete_variables()   { return dataVarsRep->numDiscreteDesVars +
    dataVarsRep->numDiscreteStateVars; }
  /// return total number of variables
  size_t num_variables() { return design() + uncertain() + state(); }

//private:

  //
  //- Heading: Data
  //

  /// pointer to the body (handle-body idiom)
  DataVariablesRep* dataVarsRep;
};


inline bool DataVariables::operator==(const DataVariables& data_vars)
{
  Cerr << "Warning:  This is not a full equality operator for the "
       << "DataVariables class" << endl;
  return (data_vars.dataVarsRep->idVariables == dataVarsRep->idVariables);
}


/// global comparison function for DataVariables
inline bool data_variables_id_compare(const DataVariables& dv, const void* id)
{ return ( *(const String*)id == dv.dataVarsRep->idVariables ); }


/// MPIPackBuffer insertion operator for DataVariables
inline MPIPackBuffer& operator<<(MPIPackBuffer& s, const DataVariables& data)
{ data.write(s); return s;}


/// MPIUnpackBuffer extraction operator for DataVariables
inline MPIUnpackBuffer& operator>>(MPIUnpackBuffer& s, DataVariables& data)
{ data.read(s); return s;}


/// ostream insertion operator for DataVariables
inline ostream& operator<<(ostream& s, const DataVariables& data)
{ data.write(s); return s;}


inline void DataVariables::write(ostream& s) const
{ dataVarsRep->write(s); }


inline void DataVariables::read(MPIUnpackBuffer& s)
{ dataVarsRep->read(s); }


inline void DataVariables::write(MPIPackBuffer& s) const
{ dataVarsRep->write(s); }

} // namespace Dakota

#endif
