/*  _______________________________________________________________________

    DAKOTA: Design Analysis Kit for Optimization and Terascale Applications
    Copyright (c) 2006, Sandia National Laboratories.
    This software is distributed under the GNU General Public License.
    For more information, see the README file in the top Dakota directory.
    _______________________________________________________________________ */

//- Class:        DirectApplicInterface
//- Description:  Derived class for the case when analysis code simulators are
//-               linked into the code and may be invoked directly
//- Owner:        Mike Eldred
//- Version: $Id: DirectApplicInterface.H 5146 2008-07-02 02:41:18Z mseldre $

#ifndef DIRECT_APPLIC_INTERFACE_H
#define DIRECT_APPLIC_INTERFACE_H

#include "ApplicationInterface.H"
//#include "DakotaVariables.H"
//#include "DakotaActiveSet.H"
//#include "DakotaResponse.H"
//#ifndef OSF
//#include <pthread.h>
//#endif

#ifdef DAKOTA_MATLAB
#include "engine.h"
//struct engine;  //forward declaration for engine pointer
//struct mxArray; //forward declaration for mxArray argument
#endif
#ifdef DAKOTA_PYTHON
#include <Python.h>  //eventually use forward declaration of PyObject
#endif

namespace Dakota {

/// Derived application interface class which spawns simulation codes
/// and testers using direct procedure calls.

/** DirectApplicInterface uses a few linkable simulation codes and several
    internal member functions to perform parameter to response mappings. */

class DirectApplicInterface: public ApplicationInterface
{
public:

  //
  //- Heading: Constructor and destructor
  //

  DirectApplicInterface(const ProblemDescDB& problem_db); ///< constructor
  ~DirectApplicInterface();                               ///< destructor

  //
  //- Heading: Virtual function redefinitions
  //

  void derived_map(const Variables& vars, const ActiveSet& set,
		   Response& response, int fn_eval_id);
  void derived_map_asynch(const ParamResponsePair& pair);

  void derived_synch(PRPList& prp_list);
  void derived_synch_nowait(PRPList& prp_list);

  int  derived_synchronous_local_analysis(const int& analysis_id);

  const StringArray& analysis_drivers() const;

  //void clear_bookkeeping(); // clears threadIdMap

protected:

  //
  //- Heading: New virtual fns (redefined by derived interface plug-ins)
  //

  /// execute the input filter portion of a direct evaluation invocation
  virtual int derived_map_if(const String& if_name);
  /// execute an analysis code portion of a direct evaluation invocation
  virtual int derived_map_ac(const String& ac_name);
  /// execute the output filter portion of a direct evaluation invocation
  virtual int derived_map_of(const String& of_name);

  //
  //- Heading: Methods
  //

  /// convenience function for local test simulators which sets
  /// variable attributes and zeros response data
  void set_local_data(const Variables& vars, const ActiveSet& set,
		      const Response& response);
  /// convenience function for local test simulators which overlays
  /// response contributions from multiple analyses using MPI_Reduce
  void overlay_response(Response& response);

  //
  //- Heading: Data
  //

  String iFilterName; ///< name of the direct function input filter
  String oFilterName; ///< name of the direct function output filter

  // map of pthread id's to function evaluation id's for asynch evaluations
  //std::map<pthread_t, int> threadIdMap;

  // data used by direct fns is class scope to allow common utility usage
  bool gradFlag;  ///< signals use of fnGrads in direct simulator functions
  bool hessFlag;  ///< signals use of fnHessians in direct simulator functions

  size_t numFns;  ///< number of functions in fnVals
  size_t numVars; ///< total number of continuous and discrete variables
  size_t numACV;  ///< total number of continuous variables
  size_t numADV;  ///< total number of discete variables
  size_t numDerivVars; ///< number of active derivative variables

  // Keeping a copy of the data passed to derived_map allows common prototypes
  // with SysCall and ForkApplicInterface in virtual functions (since SysCall
  // and Fork don't need vars/asv/response outside of write_parameters/
  // read_responses.
  //Variables directFnVars; ///< class scope variables object
  RealVector xC; ///< continuous variables used within direct simulator fns
  IntVector  xD; ///< discrete variables used within direct simulator fns
  StringArray xCLabels; ///< continuous variable labels
  StringArray xDLabels; ///< discrete variable labels
  //ActiveSet directFnActSet; // class scope ActiveSet object
  ShortArray directFnASV; ///< class scope active set vector
  UIntArray  directFnDVV; ///< class scope derivative variables vector
  //Response  directFnResponse; // class scope response object
  RealVector      fnVals;    ///< response fn values within direct simulator fns
  RealMatrix      fnGrads;   ///< response fn gradients w/i direct simulator fns
  RealMatrixArray fnHessians;///< response fn Hessians  w/i direct simulator fns

  /// the set of analyses within each function evaluation (from the
  /// analysis_drivers interface specification)
  StringArray analysisDrivers;
  /// the index of the active analysis driver within analysisDrivers
  size_t analysisDriverIndex;
  /// the set of optional analysis components used by the analysis drivers
  /// (from the analysis_components interface specification)
  String2DArray analysisComponents;

#ifdef DAKOTA_MATLAB
  /// pointer to the MATLAB engine used for direct evaluations
  engine *matlabEngine;
#endif

private:

  //
  //- Heading: Simulators and test functions
  //

  int cantilever();  ///< the cantilever UQ/OUU test function
  int cyl_head();    ///< the cylinder head constrained optimization test fn
  int rosenbrock();  ///< the Rosenbrock optimization and least squares test fn
  int generalized_rosenbrock(); ///< n-dimensional Rosenbrock (Schittkowski)
  int extended_rosenbrock();    ///< n-dimensional Rosenbrock (Nocedal/Wright)

  int log_ratio();         ///< the log_ratio UQ test function
  int short_column();      ///< the short_column UQ/OUU test function
  int steel_column_cost(); ///< the steel_column_cost UQ/OUU test function
  int steel_column_perf(); ///< the short_column_perf UQ/OUU test function

  int text_book();     ///< the text_book constrained optimization test function
  int text_book1();    ///< portion of text_book() evaluating the objective fn
  int text_book2();    ///< portion of text_book() evaluating constraint 1
  int text_book3();    ///< portion of text_book() evaluating constraint 2
  int text_book_ouu(); ///< the text_book_ouu OUU test function
  int multimodal();    ///< multimodal UQ test function 

#ifdef DAKOTA_SALINAS
  int salinas(); ///< direct interface to the SALINAS structural dynamics code
#endif // DAKOTA_SALINAS
#ifdef DAKOTA_MODELCENTER
  int mc_api_run(); ///< direct interface to ModelCenter via API, HKIM 4/3/03
#endif // DAKOTA_MODELCENTER
#ifdef DAKOTA_MATLAB
  int matlab_engine_run(); ///< direct interface to Matlab via API, BMA 11/28/05
  /// check that the dakota_matlab strucutre has the specified field_name and
  /// add if necessary; free structure memory in preparation for new alloc
  int matlab_field_prep(mxArray* dakota_matlab, const char* field_name);
#endif // DAKOTA_MATLAB
#ifdef DAKOTA_PYTHON
  int python_run(); ///< direct interface to Python via API, BMA 07/02/07
  bool userNumpyFlag; ///< whether the user requested numpy data structures
  /// convert arrays of integer types to Python
  template<class ArrayT> 
  bool python_convert_int(const ArrayT & src, PyObject** dst);
  /// convert RealVector to Python list or numpy array
  bool python_convert(const RealVector& src, PyObject** dst);
  /// convert RealVector + IntVector to Python mixed list or numpy double array
  bool python_convert(const RealVector& c_src, const IntVector& d_src,
		      PyObject** dst);
  /// convert labels
  bool python_convert(const StringArray& src,PyObject** dst);
  /// convert all labels to single list
  bool python_convert(const StringArray& c_src, const StringArray& d_src,
		      PyObject** dst);
  /// convert python list of int or float to RealVector
  bool python_convert(PyObject *pyv, RealBaseVector &rv, const int& dim);
  /// convert python list of lists of int or float to RealMatrix
  bool python_convert(PyObject *pym, RealMatrix &rm);
  /// convert python list of lists of lists of int or float to RealMatrixArray
  bool python_convert(PyObject *pyma, RealMatrixArray &rma);
#endif // DAKOTA_PYTHON
};


/** This code provides the derived function used by 
    ApplicationInterface::serve_analyses_synch(). */
inline int DirectApplicInterface::
derived_synchronous_local_analysis(const int& analysis_id)
{ 
  analysisDriverIndex = analysis_id-1;
  return derived_map_ac(analysisDrivers[analysisDriverIndex]);
}


inline const StringArray& DirectApplicInterface::analysis_drivers() const
{ return analysisDrivers; }


//inline void DirectApplicInterface::clear_bookkeeping()
//{ threadIdMap.clear(); }

} // namespace Dakota

#endif
