//------------------------------------------------------------------------
// P.J. Williams
// Sandia National Laboratories
// pwillia@sandia.gov
// Last Modified 01/29/2004 
//------------------------------------------------------------------------

#ifdef HAVE_CONFIG_H
#include "OPT++_config.h"
#endif

#include <typeinfo>
#ifdef HAVE_STD
#include <cstring>
#include <ctime>
#else
#include <string.h>
#include <time.h>
#endif

#include "OptDHNIPS.h"
#include "precisio.h"
#include "OptppArray.h"
#include "cblas.h"
#include "ioformat.h"

using NEWMAT::Real;
using NEWMAT::FloatingPointPrecision;
using NEWMAT::ColumnVector;
using NEWMAT::Matrix;
using NEWMAT::DiagonalMatrix;
using NEWMAT::SymmetricMatrix;

namespace OPTPP {

void OptDHNIPS::reset()
{
   NLP2* nlp2 = nlprob2();
   int   n    = nlp2->getDim();
   nlp2->reset();
   OptimizeClass::defaultReset(n);
   indices = 0;
   HCk_ = 0;
}

void OptDHNIPS::nonLinearConstraintIndices(const ColumnVector& types)
{ 
       for(int i = 1; i <= types.Nrows(); i++){
          if(types(i) == NLeqn || types(i) == NLineq)
	        indices.append(i);
       }
       return;
}

void OptDHNIPS::initHessian()
{ 
  NLP2* nlp2 = nlprob2();
  hessl      = nlp2->getHess();

  if(nlp2->hasConstraints()){
     CompoundConstraint* constraints = nlp2->getConstraints();
     int nlncons = constraints->getNumOfNLCons();

     if(nlncons){

       SymmetricMatrix Htmp(nlp2->getDim());
       Htmp  = 0.0; 

       ColumnVector type = constraints->getConstraintType();
       /* 
	* Map the Lagrange multipliers for the nonlinear constraints
        * to the corresponding constraints.
        */ 
       nonLinearConstraintIndices(type);


       /* 
        * Set initial constraint Hessian approximations to zero,
        * which guarantees a positive definite inital Hessian.
        * See M. Goldsmith (1999), "Sequential Quadratic Programming
        * Methods Based on Indefinite Hessian Approximations", p.66 
        */
       for(int i = 1; i <= nlncons; i++)
	   HCk_.append(Htmp);
     }
  }
  return;
}

SymmetricMatrix OptDHNIPS::updateH(SymmetricMatrix&Hk, int k)
{ 
  if(k == 0){
      initHessian();
      Hk = hessl;
      return Hk;
  }

  Real mcheps  = FloatingPointPrecision::Epsilon();
  Real sqrteps = sqrt(mcheps);

  NLP2* nlp2   = nlprob2();
  
  // Compute the Hessian of the objective function at the current point
  hessl       = nlp2->evalH();

  /*
   * Compute quasi-Newton approximation to the Hessian of each constraint 
   * NOTE: This is a very naive implementation which increases storage
   * costs by (nlncons + 1)(ndim*ndim).  Alternative methods include LBFGS, 
   * LSR1 or sparse methods.
   *
   * Compact representation of LBFGS, Nocedal and Wright p. 230
   * Bk = B - [BSk Yk]M^{-1}[BSk Yk]^T,
   * where B = B0, Sk = [ s0,....,sk-1], Yk = [y0,...,yk-1],
   * M = [ Sk^TBSk   Lk ; Lk^T   -Dk],
   * (Lk)ij = s_{i-1}^Ty_{j-1} for i > j,  Dk = diag{s^Ty} j = 0:k-1
   * NOTE: Requires nlncons factorizations of k x k matrix
   * and storage of 2 m x m matrices. Modifications needed for k > m.
   *
   * Compact representation of LSR1, Nocedal and Wright p. 232
   * Bk = B + (Yk - BSk)(Dk + Lk + Lk^T -Sk^TBSk)^{-1}(Yk - BSk)^T,
   * NOTE: Requires (nlncons) factorizations of k x k matrix
   * and storage of 2 ndim x k matrices.
   */
  if(nlp2->hasConstraints()){
      
     CompoundConstraint* constraints = nlp2->getConstraints();

     bool evaluate;
     int j, ndim, nlncons; 
     real rts, yts, rmnrm, rnrm, snrm, ynrm, maxres, restol, gamma;

     ndim    = nlp2->getDim();
     nlncons = constraints->getNumOfNLCons();

     ColumnVector xc, yk, sk, res, Bsk, multipliers;
     Matrix cg, cgprev, Htmp(ndim,ndim);

     multipliers = y & z;
     gamma       = 1.0e8;

     // Compute the current step 
     xc     = nlp2->getXc();
     sk     = xc - xprev; 

     cg     = getConstraintGradient();
     cgprev = constraintGradientPrev;

     for(j = 1; j <= nlncons; j++){
        
        yk   = cg.Column(indices[j-1]) - cgprev.Column(indices[j-1]); 

        yts  = Dot(sk,yk);
        snrm = Norm2(sk);
        ynrm = Norm2(yk);

        res    = yk - HCk_[j-1]*sk;
	rts    = Dot(res,sk);
        rnrm   = Norm2(res);
	rmnrm  = (res*res.t()).Norm1(); 
        maxres = res.NormInfinity();
	restol = gamma*fabs(rts/ndim)*(1 + HCk_[j-1].Norm1());

	evaluate = true;
     
        /* 
         * For more information about updating safeguards for
	 * the SR1 Hessian approximation; see A. R. Conn,
	 * N. I. M. Gould, and PH. L. Toint (1991), "Convergence of 
	 * quasi-Newton matrices generated by the symmetric rank one 
	 * update", Mathematical Programming 50:177-195.  and 
	 * M. Goldsmith (1999), "Sequential Quadratic Programming
         * Methods Based on Indefinite Hessian Approximations", pp. 60-62.
	 */
        if(fabs(rts) <= sqrteps*snrm*rnrm || restol < rnrm*rnrm){
	    evaluate = false;
            if (debug_) {
                *optout << "UpdateH: y-Hs = " << e(maxres,12,4) 
	  	        << " is too small\n";
                *optout << "UpdateH: The SR1 update is skipped\n";
            }
        }

	if(evaluate){
          // Perfom SR1 update 
          Htmp = HCk_[j-1] + (res * res.t()) / rts;
          HCk_[j-1] << Htmp;
          Htmp.Release();
	}
        hessl -= multipliers(indices[j-1])*HCk_[j-1];
     }
  }
  Hk = hessl;
  return Hk; 
}


void OptDHNIPS::printStatus(char *title) // set Message
{
  NLP2* nlp2 = nlprob2();

  *optout << "\n\n=========  " << title << "  ===========\n\n";
  *optout << "Optimization method       = " << method   << "\n";
  *optout << "Dimension of the problem  = " << nlp2->getDim()  << "\n"; *optout << "No. equalities            = " << me      << "\n";
  *optout << "No. inequalities          = " << mi      << "\n";
  *optout << "Merit Function (0= NormFmu, 1 = Argaez, 2 = Vanderbei) = " 
         << mfcn      << "\n";
  *optout << "Return code               = " << ret_code << " (" 
         << mesg << ")\n";
  *optout << "No. iterations taken      = " << iter_taken  << "\n";
  *optout << "No. function evaluations  = " << nlp2->getFevals() << "\n";
  *optout << "No. gradient evaluations  = " << nlp2->getGevals() << "\n";

  if (debug_) {
    *optout << "\nHessian of the Lagrangian";
    FPrint(optout, hessl);

//  Compute eigenvalues of Hessian
    DiagonalMatrix D;
    SVD(hessl, D);
    *optout << "\nEigenvalues of Hessian";
    FPrint(optout, D);
  }

  nlp2->fPrintState(optout, title);
  fPrintMultipliers(optout, title);

  tol.printTol(optout);
}

} // namespace OPTPP
