/*  _______________________________________________________________________

    DAKOTA: Design Analysis Kit for Optimization and Terascale Applications
    Copyright (c) 2006, Sandia National Laboratories.
    This software is distributed under the GNU General Public License.
    For more information, see the README file in the top Dakota directory.
    _______________________________________________________________________ */

//- Class:       NonDExpansion
//- Description: Implementation code for NonDExpansion class
//- Owner:       Mike Eldred

#include "system_defs.h"
#include "NonDExpansion.H"
#include "NonDLHSSampling.H"
#include "NonDQuadrature.H"
#include "NonDSparseGrid.H"
#ifdef DAKOTA_QUADRATURE
#include "sparse_grid_mixed.H"
#endif // DAKOTA_QUADRATURE
#include "DakotaModel.H"
#include "DakotaResponse.H"
#include "ProblemDescDB.H"
#include "RecastModel.H"
#include "BasisPolyApproximation.H"


namespace Dakota {

NonDExpansion::NonDExpansion(Model& model): NonD(model),
  expansionCoeffsApproach(-1), numUncertainQuant(0), numSamplesOnModel(0),
  numSamplesOnExpansion(probDescDB.get_int("method.samples"))
{
  // This constructor is called for a standard letter-envelope iterator 
  // instantiation.

  const short& active_view = iteratedModel.current_variables().view().first;
  if (active_view == MERGED_ALL || active_view == MIXED_ALL) {
    numDesignVars = probDescDB.get_sizet("variables.continuous_design");
    numStateVars  = probDescDB.get_sizet("variables.continuous_state");
    if (active_view == MERGED_ALL) {
      numDesignVars += probDescDB.get_sizet("variables.discrete_design");
      numStateVars  += probDescDB.get_sizet("variables.discrete_state");
    }
    if (numContinuousVars != numDesignVars + numUncertainVars + numStateVars) {
      Cerr << "\nError: bad number of active variables in NonDExpansion."<<endl;
      abort_handler(-1);
    }
    // expand ProbabilityTransformation::corrMatrixX to include design + state
    // vars.  TO DO: propagate through model recursion?
    natafTransform.reshape_correlation_matrix(numDesignVars, numUncertainVars,
					      numStateVars);
  }

  // Check for proper variables and response function definitions. 
  if ( !(numDesignVars || numUncertainVars || numStateVars) ||
       !numResponseFunctions ) {
    Cerr << "\nError: number of basis variables and response functions must be "
	 << "nonzero in NonDExpansion." << endl;
    abort_handler(-1);
  }

  // initialize meanStats, stdDevStats, and finalStatistics
  meanStats.reshape(numFunctions);
  stdDevStats.reshape(numFunctions);
  initialize_final_statistics();

  // ----------------------------------------
  // Determine Wiener-Askey or Wiener-Hermite
  // ----------------------------------------
  // use extended u-space definitions in Nataf transformations
  extendedUSpace = true; // default for PCE/SC unless correlation support issues
  initialize_random_variable_types(); // need ranVarTypesX/U below

  // check for correlations that are not supported for (1) use in extended
  // u-space and (2) use by Der Kiureghian & Liu for basic u-space (bounded
  // normal, bounded lognormal, loguniform, triangular, and beta vars).
  size_t i, j;
  if (natafTransform.x_correlation()) {

    const Pecos::ShortArray&    x_types = natafTransform.x_types();
    const Pecos::ShortArray&    u_types = natafTransform.u_types();
    const Pecos::RealSymMatrix& x_corr  = natafTransform.x_correlation_matrix();

    // check extendedUSpace first, since it affects beta logic below.  Prior
    // to expanding upon Der Kiureghian & Liu, the uniform, exponential, beta,
    // and gamma vars cannot be correlated with anything undergoing a nonlinear
    // tranformation.   Note: loop below must check all columns, despite
    // symmetry, since only columns are checked for nonlinear tranformations.
    for (i=numDesignVars; i<numDesignVars+numUncertainVars; i++)
      if (u_types[i] == UNIFORM || u_types[i] == EXPONENTIAL ||
	  u_types[i] == BETA    || u_types[i] == GAMMA)
	for (j=numDesignVars; j<numDesignVars+numUncertainVars; j++)
	  if (i != j && fabs(x_corr(i, j)) > 1.e-25 && x_types[j] != u_types[j])
	    { extendedUSpace = false; break; }

    if (!extendedUSpace) {
      Cerr << "\nWarning: extended u-space has been deactivated in "
	   << "NonDExpansion\n         due to correlations involving "
	   << "nonlinear variable transformations.\n\n";
      // update ProbabilityTransformation::ranVarTypesX/U
      initialize_random_variable_types();
    }

    // bounded normal, bounded lognormal, loguniform, triangular, and
    // non-extended beta are undergoing nonlinear transformations and are not
    // supported by Der Kiureghian & Liu: they may not be correlated with
    // anything (even other variable types undergoing linear transformations).
    bool distribution_error = false;
    for (i=numDesignVars; i<numDesignVars+numUncertainVars; i++)
      if ( x_types[i] == BOUNDED_NORMAL || x_types[i] == BOUNDED_LOGNORMAL ||
	   x_types[i] == LOGUNIFORM     || x_types[i] == TRIANGULAR ||
	   ( !extendedUSpace && x_types[i] == BETA ) )
	for (j=numDesignVars; j<numDesignVars+numUncertainVars; j++)
	  if (i != j && fabs(x_corr(i, j)) > 1.e-25)
	    { distribution_error = true; break; }

    if (distribution_error) {
      Cerr << "Error: correlated bounded normal, correlated bounded lognormal, "
	   << "correlated loguniform,\n       correlated triangular, and "
	   << "correlated non-extended beta distributions\n       are not "
	   << "currently supported in NonDExpansion." << endl;
      abort_handler(-1);
    }
  }

  // Check for suitable distribution types.
  // LHS distribution checks are performed in NonDSampling.  The following
  // checks mirror NonDReliability due to use of u-space transformations.
  if (numHistogramVars || numIntervalVars) {
    Cerr << "Error: histogram and interval distributions are not currently "
	 << "supported in NonDExpansion." << endl;
    abort_handler(-1);
  }
#ifndef HAVE_GSL
  if ( numFrechetVars || numWeibullVars ||
       ( !extendedUSpace && ( numBetaVars || numGammaVars ) ) ) {
    Cerr << "\nError: GSL library is required to support frechet, weibull, "
	 << "non-extended beta,\n       and non-extended gamma distribution "
	 << "transformations in NonDExpansion." << endl;
    abort_handler(-1);
  }
#endif
}


NonDExpansion::~NonDExpansion()
{ }


void NonDExpansion::construct_g_u_model(Model& g_u_model)
{
  size_t i;
  Sizet2DArray vars_map, primary_resp_map, secondary_resp_map;
  vars_map.reshape(numContinuousVars);
  for (i=0; i<numContinuousVars; i++) {
    vars_map[i].reshape(1);
    vars_map[i][0] = i;
  }
  primary_resp_map.reshape(numFunctions);
  for (i=0; i<numFunctions; i++) {
    primary_resp_map[i].reshape(1);
    primary_resp_map[i][0] = i;
  }
  // Nataf is a nonlinear tranformation for all variables except Normals.
  // Nonlinear mappings require special ASV logic for transforming Hessians.
  bool nonlinear_vars_map
    = ( (!extendedUSpace && numUncertainVars != numNormalVars) ||
	( extendedUSpace && numUncertainVars != numNormalVars + numUniformVars +
	  numExponentialVars + numBetaVars + numGammaVars ) );
  // There is no additional response mapping beyond that required by the
  // nonlinear variables mapping.
  BoolDequeArray nonlinear_resp_map(numFunctions, BoolDeque(1, false));
  g_u_model.assign_rep(new RecastModel(iteratedModel, vars_map,
    nonlinear_vars_map, vars_u_to_x_mapping, set_u_to_x_mapping,
    primary_resp_map, secondary_resp_map, 0, nonlinear_resp_map,
    resp_x_to_u_mapping, NULL), false);
  // Populate random variable distribution parameters for (extended) u-space.
  // *** Note ***: For use with REGRESSION approaches, variable ordering in
  // get_parameter_sets() does not use x_types/u_types as in NonDQuadrature/
  // NonDSparseGrid and thus a possibility for future ordering errors exists.
  // Currently, Design/State -> Uniform is handled separately via global bounds
  // and no current x->u selection in NonD::initialize_random_variable_types()
  // causes an ordering discrepancy.  If a future mapping causes an ordering
  // inconsistency, this could be handled above via the RecastModel vars_map.
  RealVector c_l_bnds, c_u_bnds;
  bool global_bounds = (numDesignVars || numStateVars);
  if (global_bounds) {
    // [-1,1] are standardized bounds for design, state, uniform, and beta
    c_l_bnds.reshape(numContinuousVars); c_l_bnds = -1.;
    c_u_bnds.reshape(numContinuousVars); c_u_bnds =  1.;
  }
  const Pecos::ShortArray& x_types = natafTransform.x_types();
  const Pecos::ShortArray& u_types = natafTransform.u_types();
  size_t num_g_u_nuv = (extendedUSpace) ?
    std::count(u_types.begin(), u_types.end(), (short)NORMAL) :
    numUncertainVars;
  if (num_g_u_nuv) {
    RealDenseVector nuv_means(num_g_u_nuv, false),
      nuv_std_devs(num_g_u_nuv, false), nuv_l_bnds(num_g_u_nuv, false),
      nuv_u_bnds(num_g_u_nuv, false);
    for (i=0; i<num_g_u_nuv; i++) {
      nuv_means[i]  =  0.;      nuv_std_devs[i] = 1.;
      nuv_l_bnds[i] = -DBL_MAX; nuv_u_bnds[i]   = DBL_MAX;
    }
    g_u_model.normal_means(nuv_means);
    g_u_model.normal_std_deviations(nuv_std_devs);
    g_u_model.normal_lower_bounds(nuv_l_bnds);
    g_u_model.normal_upper_bounds(nuv_u_bnds);
    if (global_bounds)
      for (i=0; i<num_g_u_nuv; i++) {
	c_l_bnds[i+numDesignVars] = -10.; // 10 std devs
	c_u_bnds[i+numDesignVars] =  10.; // 10 std devs
      }
  }
  if (extendedUSpace) {
    size_t num_g_u_uuv =
        std::count(u_types.begin(), u_types.end(), (short)UNIFORM) - 
        std::count(x_types.begin(), x_types.end(), (short)DESIGN) - 
        std::count(x_types.begin(), x_types.end(), (short)STATE),
      num_g_u_euv  =
        std::count(u_types.begin(), u_types.end(), (short)EXPONENTIAL),
      num_g_u_buv  = std::count(u_types.begin(), u_types.end(), (short)BETA),
      num_g_u_gauv = std::count(u_types.begin(), u_types.end(), (short)GAMMA);
    if (num_g_u_uuv) {
      RealDenseVector uuv_l_bnds(num_g_u_uuv, false),
	              uuv_u_bnds(num_g_u_uuv, false);
      for (i=0; i<num_g_u_uuv; i++)
	{ uuv_l_bnds[i] = -1.; uuv_u_bnds[i] =  1.; }
      g_u_model.uniform_lower_bounds(uuv_l_bnds);
      g_u_model.uniform_upper_bounds(uuv_u_bnds);
    }
    if (num_g_u_euv) {
      RealDenseVector euv_betas(num_g_u_euv, false);
      for (i=0; i<num_g_u_euv; i++)
	euv_betas[i] = 1.;
      g_u_model.exponential_betas(euv_betas);
      if (global_bounds)
	for (i=0; i<num_g_u_euv; i++) {
	  size_t offset_i = i + numDesignVars + num_g_u_nuv + num_g_u_uuv;
	  c_l_bnds[offset_i] =  0.;
	  c_u_bnds[offset_i] = 10.; // 10 std devs for beta = 1
	}
    }
    if (num_g_u_buv) {
      RealDenseVector buv_l_bnds(num_g_u_buv, false),
	              buv_u_bnds(num_g_u_buv, false);
      for (i=0; i<num_g_u_buv; i++)
	{ buv_l_bnds[i] = -1.; buv_u_bnds[i] =  1.; }
      g_u_model.beta_alphas(iteratedModel.beta_alphas());
      g_u_model.beta_betas(iteratedModel.beta_betas());
      g_u_model.beta_lower_bounds(buv_l_bnds);
      g_u_model.beta_upper_bounds(buv_u_bnds);
    }
    if (num_g_u_gauv) {
      const RealDenseVector& gamma_alphas = iteratedModel.gamma_alphas();
      g_u_model.gamma_alphas(gamma_alphas);
      RealDenseVector gauv_betas(num_g_u_gauv, false);
      for (i=0; i<num_g_u_gauv; i++)
	gauv_betas[i] = 1.;
      g_u_model.gamma_betas(gauv_betas);
      if (global_bounds)
	for (i=0; i<num_g_u_gauv; i++) {
	  size_t offset_i = i + numDesignVars + num_g_u_nuv + num_g_u_uuv + 
	    num_g_u_euv + num_g_u_buv;
	  c_l_bnds[offset_i] =  0.;
	  c_u_bnds[offset_i] = 10.*sqrt(gamma_alphas[i]); // 10 std devs, beta=1
	}
    }
  }
  if (global_bounds) {
    // uncertain bounds not currently used, since ACTIVE, not ACTIVE_UNIFORM
    g_u_model.continuous_lower_bounds(c_l_bnds);
    g_u_model.continuous_upper_bounds(c_u_bnds);
  }
}


void NonDExpansion::
construct_quadrature(Iterator& u_space_sampler, Model& g_u_model,
		     const UShortArray& quad_order)
{
  u_space_sampler.assign_rep(new NonDQuadrature(g_u_model, quad_order), false);
  if (quad_order.length() == 1)
    numSamplesOnModel = (int)pow((Real)quad_order[0],(int)numContinuousVars);
  else if (quad_order.length() == numContinuousVars) {
    numSamplesOnModel = 1;
    for (size_t i=0; i<numContinuousVars; i++)
      numSamplesOnModel *= quad_order[i];
  }
  else {
    Cerr << "Error: length of quadrature_order specification is inconsistent "
	 << "with active continuous variables in NonDExpansion." << endl;
    abort_handler(-1);
  }
}


void NonDExpansion::
construct_sparse_grid(Iterator& u_space_sampler, Model& g_u_model,
		      const UShortArray& sparse_grid_level)
{
#ifdef DAKOTA_QUADRATURE
  // construct NonDSparseGrid instance
  NonDSparseGrid* nond_sparse = new NonDSparseGrid(g_u_model,sparse_grid_level);
  u_space_sampler.assign_rep(nond_sparse, false);

  // update numSamplesOnModel
  const Pecos::ShortArray& u_types = natafTransform.u_types();
  IntArray rules;
  nond_sparse->initialize_rules(u_types, rules);
  numSamplesOnModel = webbur::sparse_grid_mixed_size(numContinuousVars,
    sparse_grid_level[0], rules);
#endif // DAKOTA_QUADRATURE
}


void NonDExpansion::
construct_lhs(Iterator& u_space_sampler, Model& g_u_model, int orig_seed)
{
  if (numSamplesOnModel <= 0) {
    Cerr << "Error: bad expansion construction specification in NonDExpansion."
	 << endl;
    abort_handler(-1);
  }

  /*
  if (exp_samples > 0 &&
      probDescDB.get_string("method.nond.expansion_sample_type")
      == "incremental_lhs")
    // TO DO: define and pass previous_samples ?
    u_space_sampler.assign_rep(new NonDIncremLHSSampling(g_u_model,
      numSamplesOnModel, orig_seed, ACTIVE), false);
  else
  */
  u_space_sampler.assign_rep(new NonDLHSSampling(g_u_model, numSamplesOnModel,
    orig_seed, ACTIVE), false);
}


void NonDExpansion::construct_expansion_sampler()
{
  if (totalLevelRequests) {
    bool exp_sampling = false;
    for (size_t i=0; i<numFunctions; i++)
      if ( requestedProbLevels[i].length() || requestedGenRelLevels[i].length()
	   || ( requestedRespLevels[i].length() &&
		respLevelTarget != RELIABILITIES ) )
	{ exp_sampling = true; break; }
    if (exp_sampling && !numSamplesOnExpansion) {
      Cerr << "\nError: number of samples must be specified for evaluating "
	   << "probabilities." << endl;
      abort_handler(-1);
    }
    // even if !exp_sampling, expansionSampler needed for moment projections
    int orig_seed = probDescDB.get_int("method.random_seed");
    expansionSampler.assign_rep(new NonDLHSSampling(uSpaceModel,
      numSamplesOnExpansion, orig_seed, UNCERTAIN), false);
    //expansionSampler.sampling_reset(numSamplesOnExpansion, true, false);
    NonD* exp_sampler_rep = (NonD*)expansionSampler.iterator_rep();
    exp_sampler_rep->requested_levels(requestedRespLevels, requestedProbLevels,
      requestedRelLevels, requestedGenRelLevels, respLevelTarget, cdfFlag);
  }
}


void NonDExpansion::initialize_expansion()
{
  // update ranVar info to capture any distribution param insertions
  initialize_random_variable_parameters();
  natafTransform.trans_correlations();

  // now that labels have flowed down at run-time from any higher level
  // recursions, propagate them up the instantiate-on-the-fly Model
  // recursion so that they are correct when they propagate back down.
  uSpaceModel.update_from_subordinate_model(); // recurse_flag = true

  // store the current design/state vars in u-space
  size_t i, j, cntr = 0;
  RealVector initial_pt_x;
  if (numDesignVars || numStateVars || !subIteratorFlag) {
    initial_pt_x = iteratedModel.continuous_variables();
    if (numUncertainQuant) { // reset uncertain values to means
      const Pecos::RealVector& x_means = natafTransform.x_means();
      for (i=numDesignVars; i<numDesignVars + numUncertainVars; i++)
	initial_pt_x[i] = x_means[i];
    }
    trans_X_to_U(initial_pt_x, initialPtU);
  }

  // initialize finalStatValues and finalStatGrads
  const ShortArray& final_asv = finalStatistics.active_set_request_vector();
  const UIntArray&  final_dvv = finalStatistics.active_set_derivative_vector();
  size_t num_final_stats     = final_asv.length(),
         num_final_grad_vars = final_dvv.length();
  bool final_stat_flag = false, final_grad_flag = false;
  for (i=0; i<num_final_stats; i++) {
    if (final_asv[i] & 1) final_stat_flag = true;
    if (final_asv[i] & 2) final_grad_flag = true;
  }
  if (final_stat_flag) {
    finalStatValues.reshape(num_final_stats);
    finalStatValues = 0.;
  }
  else
    finalStatValues.clear();
  if (final_grad_flag) {
    finalStatGrads.reshape_2d(num_final_stats, num_final_grad_vars);
    finalStatGrads = 0.;
  }
  else
    finalStatGrads.clear();
}


void NonDExpansion::compute_expansion()
{
  Iterator& u_space_sampler = uSpaceModel.subordinate_iterator();
  NonD* u_space_sampler_rep = (NonD*)u_space_sampler.iterator_rep();
  // pass x-space data so that u-space Models can perform inverse transforms
  u_space_sampler_rep->initialize_random_variables(natafTransform);

  // if required statistical sensitivities are not covered by All variables
  // mode for augmented design variables, then the simulations must evaluate
  // response sensitivities.
  bool expansion_grad = false, dist_param_deriv = false,
    all_vars = (numDesignVars || numStateVars);
  if (!finalStatGrads.empty()) {
    size_t i, num_outer_cv = secondaryACVarMapTargets.length();
    for (i=0; i<num_outer_cv; i++)
      if (secondaryACVarMapTargets[i] != NO_TARGET) // dist param insertion
	{ dist_param_deriv = true; break; }
    expansion_grad = (all_vars) ? dist_param_deriv : true;
  }
  u_space_sampler_rep->distribution_parameter_derivatives(dist_param_deriv);
  if (dist_param_deriv)
    u_space_sampler.variable_mappings(primaryACVarMapIndices,
      primaryADVarMapIndices,	secondaryACVarMapTargets,
      secondaryADVarMapTargets);
  Array<Approximation>& poly_approxs = uSpaceModel.approximations();
  const ShortArray& final_asv = finalStatistics.active_set_request_vector();
  const UIntArray&  final_dvv = finalStatistics.active_set_derivative_vector();
  size_t num_final_grad_vars  = final_dvv.length();
  if (expansion_grad) {
    size_t i, j, cntr = 0;
    ActiveSet sampler_set;
    // define ASV for u_space_sampler
    ShortArray sampler_asv(numFunctions, 0);
    for (i=0; i<numFunctions; i++) {
      BasisPolyApproximation* poly_approx_rep
	= (BasisPolyApproximation*)poly_approxs[i].approx_rep();
      bool expansion_coeff_flag = false, expansion_grad_flag = false,
           mean_grad_flag       = false, std_dev_grad_flag   = false;
      size_t rl_len = requestedRespLevels[i].length(),
	     pl_len = requestedProbLevels[i].length(),
	     bl_len = requestedRelLevels[i].length(),
	     gl_len = requestedGenRelLevels[i].length();
      // map final_asv value bits into expansion_coeff_flag requirements
      for (j=0; j<2+rl_len+pl_len+bl_len+gl_len; j++)
	if (final_asv[cntr+j] & 1)
	  { expansion_coeff_flag = true; break; }
      // map final_asv gradient bits into moment grad requirements
      if (final_asv[cntr++] & 2)
	mean_grad_flag    = true;
      if (final_asv[cntr++] & 2)
	std_dev_grad_flag = true;
      if (respLevelTarget == RELIABILITIES)
	for (j=0; j<rl_len; j++)
	  if (final_asv[cntr+j] & 2)
	    { mean_grad_flag = std_dev_grad_flag = true; break; }
      cntr += rl_len + pl_len;
      for (j=0; j<bl_len; j++)
	if (final_asv[cntr+j] & 2)
	  { mean_grad_flag = std_dev_grad_flag = true; break; }
      cntr += bl_len + gl_len;
      // map moment grad requirements into expansion_{coeff,grad}_flag reqmts
      // (refer to *PolyApproximation::get_*_gradient() implementations)
      if (all_vars) {
	if (std_dev_grad_flag)
	  expansion_coeff_flag = true;
	size_t deriv_index, num_deriv_vars = final_dvv.length();
	for (j=0; j<num_deriv_vars; j++) {
	  deriv_index = final_dvv[j] - 1; // OK since we are in an "All" view
	  if (deriv_index >= numDesignVars &&
	      deriv_index <  numDesignVars + numUncertainVars) { // random var
	    if (mean_grad_flag || std_dev_grad_flag)
	      expansion_grad_flag = true;
	  }
	  else if (mean_grad_flag)
	    expansion_coeff_flag = true;
	}
      }
      else { // distinct variables
	if (mean_grad_flag)
	  expansion_grad_flag = true;
	if (std_dev_grad_flag)
	  expansion_coeff_flag = expansion_grad_flag = true;
      }
      // map expansion_{coeff,grad}_flag requirements into ASV and
      // BasisPolyApproximation settings
      if (expansion_coeff_flag)
	sampler_asv[i] |= 1;
      if (expansion_grad_flag)
	sampler_asv[i] |= 2;
      poly_approx_rep->expansion_coefficient_flag(expansion_coeff_flag);
      poly_approx_rep->expansion_gradient_flag(expansion_grad_flag);
    }
    sampler_set.request_vector(sampler_asv);
    // final_dvv: NestedModel::derived_compute_response() maps the top-level
    // optimizer deriv vars to the sub-iterator deriv vars in
    // NestedModel::set_mapping() and then sets this DVV within finalStats
    // using subIterator.response_results_active_set().  NonDPCE/NonDSC then
    // maps the finalStats DVV to the default set for u_space_sampler, which
    // may then be augmented for correlations in NonD::set_u_to_x_mapping().
    if (all_vars) {
      UIntArray filtered_final_dvv; // retain insertion targets only
      for (i=0; i<num_final_grad_vars; i++) {
	unsigned int dvv_i = final_dvv[i];
	if (dvv_i > numDesignVars && dvv_i <= numDesignVars+numUncertainVars)
	  filtered_final_dvv.push_back(dvv_i);
      }
      sampler_set.derivative_vector(filtered_final_dvv);
    }
    else
      sampler_set.derivative_vector(final_dvv);
    u_space_sampler.active_set(sampler_set);
  }
  else {
    // override gradient/Hessian responses specification, if present
    ActiveSet sampler_set = u_space_sampler.active_set();
    sampler_set.request_values(1);
    u_space_sampler.active_set(sampler_set);
    // reset BasisPolyApproximation::expansionGradFlag to false
    for (size_t i=0; i<numFunctions; i++) {
      BasisPolyApproximation* poly_approx_rep
	= (BasisPolyApproximation*)poly_approxs[i].approx_rep();
      poly_approx_rep->expansion_coefficient_flag(true);
      poly_approx_rep->expansion_gradient_flag(false);
    }
  }

#ifdef DERIV_DEBUG
  // numerical verification of analytic Jacobian/Hessian routines
  RealDenseVector rdv_x, rdv_u;
  copy_data(iteratedModel.continuous_variables(), rdv_x);
  natafTransform.trans_X_to_U(rdv_x, rdv_u);
  natafTransform.verify_trans_jacobian_hessian(rdv_u);//(rdv_x);
  natafTransform.verify_design_jacobian(rdv_u);
#endif // DERIV_DEBUG

  // Build orthogonal/interpolation polynomial approximations for all response
  // fns.  If OUU (multiple calls to quantify_uncertainty()), an expansion
  // constructed over the full range of all variables does not need to be
  // reconstructed on subsequent calls.
  bool full_approx = ( all_vars && ( primaryACVarMapIndices.empty() ||
    numDesignVars + numStateVars == primaryACVarMapIndices.length() ) );
  if (!full_approx || !numUncertainQuant)
    uSpaceModel.build_approximation();
}


/** Calculate analytic and numerical statistics from the expansion and
    log results within final_stats for use in OUU. */
void NonDExpansion::compute_statistics()
{
  // -----------------------------
  // Calculate analytic statistics
  // -----------------------------
  const ShortArray& final_asv = finalStatistics.active_set_request_vector();
  const UIntArray&  final_dvv = finalStatistics.active_set_derivative_vector();
  bool all_vars = (numDesignVars || numStateVars), beta_mappings = false,
    final_stat_flag = !finalStatValues.empty();
  size_t i, j, cntr = 0, num_final_grad_vars = final_dvv.length();

  // initialize expGradsMeanX
  if (!subIteratorFlag && expGradsMeanX.empty())
    expGradsMeanX.reshape_2d(numFunctions, numContinuousVars);

  // loop over response fns and compute/store analytic stats/stat grads
  Array<Approximation>& poly_approxs = uSpaceModel.approximations();
  for (i=0; i<numFunctions; i++) {
    BasisPolyApproximation* poly_approx_rep
      = (BasisPolyApproximation*)poly_approxs[i].approx_rep();

    // store mean/std dev grads in finalStatGrads if needed for beta mappings
    bool beta_mapping_flag = false, beta_mapping_grad_flag = false;
    if (respLevelTarget == RELIABILITIES && !requestedRespLevels[i].empty()) {
      size_t rl_len = requestedRespLevels[i].length();
      for (j=0; j<rl_len; j++) {
	short asv_j = final_asv[cntr+2+j];
	if (asv_j & 3) beta_mapping_flag      = true;
	if (asv_j & 2) beta_mapping_grad_flag = true;
      }
    }
    if (!requestedRelLevels[i].empty()) {
      size_t bl_len = requestedRelLevels[i].length(),
	     pl_len = requestedProbLevels[i].length();
      for (j=0; j<bl_len; j++) {
	short asv_j = final_asv[cntr+2+pl_len+j];
	if (asv_j & 1) beta_mapping_flag      = true;
	if (asv_j & 2) beta_mapping_grad_flag = true;
      }
    }
    if (beta_mapping_flag)
      beta_mappings = true;

    // *** analytic mean
    if (final_asv[cntr] & 1 || beta_mapping_flag) {
      meanStats[i] = (all_vars) ? 
	poly_approx_rep->get_mean(initialPtU) :
	poly_approx_rep->get_mean();
      if (final_stat_flag) finalStatValues[cntr] = meanStats[i];
    }
    // *** analytic mean gradient
    if (final_asv[cntr] & 2 || beta_mapping_grad_flag)
      finalStatGrads[cntr] = (all_vars) ?
	poly_approx_rep->get_mean_gradient(initialPtU, final_dvv) :
	poly_approx_rep->get_mean_gradient();
    cntr++;

    // *** analytic std deviation
    if (final_asv[cntr] & 1 || beta_mapping_flag) {
      const Real& variance = (all_vars) ?
	poly_approx_rep->get_variance(initialPtU) :
	poly_approx_rep->get_variance();
      stdDevStats[i] = sqrt(variance);
      if (final_stat_flag) finalStatValues[cntr] = stdDevStats[i];
    }
    // *** analytic std deviation gradient
    if (final_asv[cntr] & 2 || beta_mapping_grad_flag) {
      const RealBaseVector& variance_grad = (all_vars) ?
	poly_approx_rep->get_variance_gradient(initialPtU, final_dvv) :
	poly_approx_rep->get_variance_gradient();
      const Real& sigma = stdDevStats[i];
      for (j=0; j<num_final_grad_vars; j++)
	finalStatGrads[cntr][j] = variance_grad[j]/2./sigma;
    }
    cntr++;

    // expansion sensitivities are defined from the coefficients and basis
    // polynomial derivatives.  They are computed for the means of the
    // uncertain varables and are intended to serve as importance factors.
    if (!subIteratorFlag) {
      const RealBaseVector& exp_grad_u_rbv
	= poly_approxs[i].get_gradient(initialPtU);
      RealDenseVector exp_grad_u_pv, exp_grad_x_pv;
      copy_data(exp_grad_u_rbv, exp_grad_u_pv);
      const UIntArray& cv_ids = iteratedModel.continuous_variable_ids();
      natafTransform.trans_grad_U_to_X(exp_grad_u_pv, exp_grad_x_pv,
				       natafTransform.x_means(),cv_ids,cv_ids);
      copy_data(exp_grad_x_pv, expGradsMeanX[i]);
    }

    cntr += requestedRespLevels[i].length() + requestedProbLevels[i].length()
         +  requestedRelLevels[i].length()  + requestedGenRelLevels[i].length();
  }

  // ------------------------------
  // Calculate numerical statistics
  // ------------------------------
  // Estimate CDF/CCDF statistics by sampling on the expansion
  if (totalLevelRequests) {
    NonDSampling* exp_sampler_rep
      = (NonDSampling*)expansionSampler.iterator_rep();

    bool exp_sampling = false;
    size_t i, j, k, cntr = 0;
    for (i=0; i<numFunctions; i++)
      if ( requestedProbLevels[i].length() || requestedGenRelLevels[i].length()
	   || ( requestedRespLevels[i].length() &&
		respLevelTarget != RELIABILITIES ) )
	{ exp_sampling = true; break; }

    if (exp_sampling) { // sample on expansion to generate probability mappings
      // pass x-space data so that u-space Models can perform inverse transforms
      exp_sampler_rep->initialize_random_variables(natafTransform);
      // since expansionSampler uses an UNCERTAIN sampling mode, we must set the
      // unsampled variables to their u-space values.
      if (numDesignVars || numStateVars)
	uSpaceModel.continuous_variables(initialPtU);
      expansionSampler.run_iterator(); // quiet mode
    }

    if (beta_mappings)
      exp_sampler_rep->moments(meanStats, stdDevStats); // for beta mappings
    if (beta_mappings || exp_sampling) {
      exp_sampler_rep->
	compute_distribution_mappings(expansionSampler.all_responses());
      exp_sampler_rep->update_final_statistics();
    }
    if (!subIteratorFlag && exp_sampling)
      exp_sampler_rep->compute_correlations(expansionSampler.all_variables(),
					    expansionSampler.all_responses());
    const RealVector& sampler_final_stats
      = expansionSampler.response_results().function_values();
    const ShortArray& final_asv = finalStatistics.active_set_request_vector();
    size_t num_final_grad_vars
      = finalStatistics.active_set_derivative_vector().length();
    for (i=0; i<numFunctions; i++) {//level stats provided from expansionSampler
      size_t rl_len = requestedRespLevels[i].length(),
	     pl_len = requestedProbLevels[i].length(),
	     bl_len = requestedRelLevels[i].length(),
	     gl_len = requestedGenRelLevels[i].length();

      bool beta_mapping_grad_flag = false;
      if (respLevelTarget == RELIABILITIES)
	for (j=0; j<rl_len; j++)
	  if (final_asv[cntr+2+j] & 2)
	    { beta_mapping_grad_flag = true; break; }
      for (j=0; j<bl_len; j++)
	if (final_asv[cntr+2+pl_len+j] & 2)
	  { beta_mapping_grad_flag = true; break; }
      RealBaseVector empty_rbv;
      const RealBaseVector& mean_grad    = (beta_mapping_grad_flag) ?
	finalStatGrads[cntr]   : empty_rbv;
      const RealBaseVector& std_dev_grad = (beta_mapping_grad_flag) ?
	finalStatGrads[cntr+1] : empty_rbv;
      cntr += 2;

      for (j=0; j<rl_len; j++, cntr++) {
	if (final_asv[cntr] & 1)
	  finalStatValues[cntr] = sampler_final_stats[cntr];
	if (final_asv[cntr] & 2) {
	  switch (respLevelTarget) {
	  case PROBABILITIES: // TO DO: z->p sampling sensitivity analysis
	    Cerr << "\nError: analytic response probability sensitivity not "
		 << "yet supported." << endl;
	    abort_handler(-1);
	    break;
	  case RELIABILITIES: {
	    const Real& std_dev = stdDevStats[i];
	    if (std_dev > 1.e-25) {
	      const Real& z_bar = requestedRespLevels[i][j];
	      const Real& mean  = meanStats[i];
	      for (k=0; k<num_final_grad_vars; k++) {
		//Real ratio = (meanStats[i] - z_bar)/stdDevStats[i];
		Real dratio_dx = (std_dev*mean_grad[k] - (mean - z_bar)*
				  std_dev_grad[k]) / pow(std_dev, 2);
		finalStatGrads[cntr][k] = (cdfFlag) ? dratio_dx : -dratio_dx;
	      }
	    }
	    else
	      finalStatGrads[cntr] = 0.;
	    break;
	  }
	  case GEN_RELIABILITIES: // TO DO: z->p->beta* sampling SA
	    Cerr << "\nError: analytic response generalized reliability "
		 << "sensitivity not yet supported." << endl;
	    abort_handler(-1);
	    break;
	  }
	}
      }
      for (j=0; j<pl_len; j++, cntr++) {
	if (final_asv[cntr] & 1)
	  finalStatValues[cntr] = sampler_final_stats[cntr];
	if (final_asv[cntr] & 2) {
	  // TO DO: p->z sampling sensitivity analysis
	  Cerr << "\nError: analytic response level sensitivity not yet "
	       << "supported for mapping from probability." << endl;
	  abort_handler(-1);
	}
      }
      for (j=0; j<bl_len; j++, cntr++) {
	if (final_asv[cntr] & 1)
	  finalStatValues[cntr] = sampler_final_stats[cntr];
	if (final_asv[cntr] & 2) {
	  const Real& beta_bar = requestedRelLevels[i][j];
	  for (k=0; k<num_final_grad_vars; k++)
	    finalStatGrads[cntr][k] = (cdfFlag) ?
	      mean_grad[k] - beta_bar * std_dev_grad[k] :
	      mean_grad[k] + beta_bar * std_dev_grad[k];
	}
      }
      for (j=0; j<gl_len; j++, cntr++) {
	if (final_asv[cntr] & 1)
	  finalStatValues[cntr] = sampler_final_stats[cntr];
	if (final_asv[cntr] & 2) {
	  // TO DO: beta*->p->z sampling sensitivity analysis
	  Cerr << "\nError: analytic response level sensitivity not yet "
	       << "supported for mapping from generalized reliability." << endl;
	  abort_handler(-1);
	}
      }
    }
  }
}


void NonDExpansion::update_final_statistics()
{
  if (!finalStatValues.empty())
    finalStatistics.function_values(finalStatValues);
  if (!finalStatGrads.empty()) {
    // Augmented design vars:
    // > All vars: transform dg/du to dg/dx -> provides desired dg/ds for x = s
    // > Distinct vars: PCE/SC approximations for dg/ds are formed
    // Inserted design vars:
    // > All and Distinct views for the subIterator are equivalent
    // > PCE/SC approximations for dg/ds are formed
    // > Alternative: All view could force an artificial cdv augmentation
    // Mixed augmented/inserted design vars:
    // > All vars: bookkeep the two dg/ds approaches
    // > Distinct vars: PCE/SC approximations for dg/ds are formed

    // for all_variables, finalStatGrads are in standardized design space
    // -> transform to the original design space
    if (numDesignVars || numStateVars) {
      // this approach is more efficient but less general.  If we can assume
      // that the DVV only contains design/state vars, then we know they are
      // uncorrelated and the jacobian matrix is diagonal with terms 2./range.
      const UIntArray& cv_ids = iteratedModel.continuous_variable_ids();
      const UIntArray& final_dvv
	= finalStatistics.active_set_derivative_vector();
      size_t num_final_grad_vars = final_dvv.length();
      Real factor;
      const Pecos::RealVector& x_l_bnds = natafTransform.x_lower_bounds();
      const Pecos::RealVector& x_u_bnds = natafTransform.x_upper_bounds();

      for (size_t j=0; j<num_final_grad_vars; j++) {
	size_t deriv_j = cv_ids.index(final_dvv[j]); //final_dvv[j]-1;
	if ( deriv_j <  numDesignVars ||
	     deriv_j >= numDesignVars+numUncertainVars ) {
	  // augmented design variable sensitivity
	  factor = 2. / (x_u_bnds(deriv_j) - x_l_bnds(deriv_j));
	  size_t num_final_stats = finalStatGrads.num_rows();
	  for (size_t i=0; i<num_final_stats; i++)
	    finalStatGrads[i][j] *= factor; // see jacobian_dZ_dX()
	}
	// else inserted design variable sensitivity: no scaling required
      }
      
      // This approach is more general, but is overkill for this purpose
      // and incurs additional copying overhead.
      /*
      RealDenseVector initial_pt_x_pv, fn_grad_u, fn_grad_x;
      copy_data(initial_pt_x, initial_pt_x_pv);
      RealDenseMatrix jacobian_ux;
      natafTransform.jacobian_dU_dX(initial_pt_x_pv, jacobian_ux);
      for (i=0; i<num_final_stats; i++) {
	copy_data(finalStatGrads[i], fn_grad_u);
	natafTransform.trans_grad_U_to_X(fn_grad_u, fn_grad_x, jacobian_ux,
	                                 final_dvv);
	copy_data(fn_grad_x, finalStatGrads[i]);
      }
      */
    }

    // For distinct vars, nothing additional is needed since u_space_sampler
    // has been configured to compute dg/ds at each of the sample points.
    // uSpaceModel.build_approximation() -> BasisPolyApproximation::
    // find_coefficients() then constructs PCE/SC approximations of these
    // gradients, and BasisPolyApproximation::get_<mean,variance>_gradient()
    // are used above to generate dmu/ds, dsigma/ds, and dbeta/ds.

    finalStatistics.function_gradients(finalStatGrads);
  }
}


void NonDExpansion::print_results(ostream& s)
{
  s.setf(ios::scientific);
  s << setprecision(write_precision);
  const StringArray& fn_labels = iteratedModel.response_labels();
  size_t i;

  s << "-------------------------------------------------------------------"
    << "\nStatistics derived analytically from polynomial expansion:\n";

  s << "\nMoments for each response function:\n";
  for (i=0; i<numFunctions; i++) {
    s << fn_labels[i] << ":  Mean = " << meanStats[i]
      << "  Std. Dev. = " << stdDevStats[i] << "  Coeff. of Variation = ";
    if (fabs(meanStats[i]) > 1.e-25)
      s << stdDevStats[i]/meanStats[i] << '\n';
    else
      s << "Undefined\n";
  }
  if (!subIteratorFlag) {
    s << "\nSensitivities for each response function evaluated at uncertain "
      << "variable means:\n";
    for (i=0; i<numFunctions; i++) {
      s << fn_labels[i] << ":\n";
      expGradsMeanX.write_row_vector(s, i, true, true, true);
    }
  }

  if (totalLevelRequests) {
    bool exp_sampling = false;
    for (i=0; i<numFunctions; i++)
      if ( requestedProbLevels[i].length() || requestedGenRelLevels[i].length()
	   || ( requestedRespLevels[i].length() &&
		respLevelTarget != RELIABILITIES ) )
	{ exp_sampling = true; break; }
    s << "\nStatistics based on ";
    if (exp_sampling)
      s << numSamplesOnExpansion << " samples performed on polynomial "
	<< "expansion:\n";
    else
      s << "projection of analytic moments:\n";
    NonDSampling* exp_sampler_rep
      = (NonDSampling*)expansionSampler.iterator_rep();
    exp_sampler_rep->print_distribution_mappings(s);
    if (!subIteratorFlag && exp_sampling)
      exp_sampler_rep->print_correlations(s);
  }

  s << "-------------------------------------------------------------------"
    << endl;
}

} // namespace Dakota
