/*  _______________________________________________________________________

    DAKOTA: Design Analysis Kit for Optimization and Terascale Applications
    Copyright (c) 2006, Sandia National Laboratories.
    This software is distributed under the GNU General Public License.
    For more information, see the README file in the top Dakota directory.
    _______________________________________________________________________ */

//- Class:       MergedConstraints
//- Description: Container class of linear constraints and bounds employing 
//-              the "merged" approach to variables usage.
//- Owner:       Mike Eldred
//- Version: $Id: MergedConstraints.H 5003 2008-05-01 22:43:46Z mseldre $

#ifndef MERGED_CONSTRAINTS_H
#define MERGED_CONSTRAINTS_H

#include "DakotaConstraints.H"
#include "DataVariables.H"


namespace Dakota {

/// Derived class within the Constraints hierarchy which employs
/// the merged data view.

/** Derived variable constraints classes take different views of the
    design, uncertain, and state variable types and the continuous and
    discrete domain types.  The MergedConstraints derived class
    combines continuous and discrete domain types but separates
    design, uncertain, and state variable types.  The result is merged
    design bounds arrays (mergedDesignLowerBnds,
    mergedDesignUpperBnds), uncertain distribution bounds arrays
    (uncertainLowerBnds, uncertainUpperBnds), and merged state
    bounds arrays (mergedStateLowerBnds, mergedStateUpperBnds).  The
    branch and bound strategy uses this approach (see
    Variables::get_variables(problem_db) for variables type
    selection; variables type is passed to the Constraints
    constructor in Model). */

class MergedConstraints: public Constraints
{
public:

  //
  //- Heading: Constructor and destructor
  //

  /// default constructor
  MergedConstraints();
  /// standard constructor
  MergedConstraints(const ProblemDescDB& problem_db,
		    const pair<short,short>& view);
  /// destructor
  ~MergedConstraints();

  //
  //- Heading: Virtual function redefinitions
  //

  const RealVector& continuous_lower_bounds() const;
  void continuous_lower_bounds(const RealVector& c_l_bnds);
  const RealVector& continuous_upper_bounds() const;
  void continuous_upper_bounds(const RealVector& c_u_bnds);

  //const IntVector& discrete_lower_bounds() const;
  //void discrete_lower_bounds(const IntVector& d_l_bnds);
  //const IntVector& discrete_upper_bounds() const;
  //void discrete_upper_bounds(const IntVector& d_u_bnds);

  const RealVector& inactive_continuous_lower_bounds() const;
  void inactive_continuous_lower_bounds(const RealVector& i_c_l_bnds);
  const RealVector& inactive_continuous_upper_bounds() const;
  void inactive_continuous_upper_bounds(const RealVector& i_c_u_bnds);

  //const IntVector& inactive_discrete_lower_bounds() const;
  //void inactive_discrete_lower_bounds(const IntVector& i_d_l_bnds);
  //const IntVector& inactive_discrete_upper_bounds() const;
  //void inactive_discrete_upper_bounds(const IntVector& i_d_u_bnds);

  RealVector all_continuous_lower_bounds() const;
  void all_continuous_lower_bounds(const RealVector& a_c_l_bnds);
  RealVector all_continuous_upper_bounds() const;
  void all_continuous_upper_bounds(const RealVector& a_c_u_bnds);

  //IntVector all_discrete_lower_bounds() const;
  //void all_discrete_lower_bounds(const IntVector& a_d_l_bnds);
  //IntVector all_discrete_upper_bounds() const;
  //void all_discrete_upper_bounds(const IntVector& a_d_u_bnds);

  void write(ostream& s) const;
  void read(istream& s);

protected:

  void copy_rep(const Constraints* con_rep);

  void reshape_rep(const Sizet2DArray& vars_comps);

private:

  //
  //- Heading: Private data members
  //

  /// a design lower bounds array merging continuous and discrete
  /// domains (integer values promoted to reals)
  RealVector mergedDesignLowerBnds;
  /// a design upper bounds array merging continuous and discrete
  /// domains (integer values promoted to reals)
  RealVector mergedDesignUpperBnds;
  /// the uncertain distribution lower bounds array (no discrete
  /// uncertain to merge)
  RealVector uncertainLowerBnds;
  /// the uncertain distribution upper bounds array (no discrete
  /// uncertain to merge)
  RealVector uncertainUpperBnds;
  /// a state lower bounds array merging continuous and discrete
  /// domains (integer values promoted to reals)
  RealVector mergedStateLowerBnds;
  /// a state upper bounds array merging continuous and discrete
  /// domains (integer values promoted to reals)
  RealVector mergedStateUpperBnds;
};


inline MergedConstraints::MergedConstraints()
{ }


inline MergedConstraints::~MergedConstraints()
{ }


inline const RealVector& MergedConstraints::continuous_lower_bounds() const
{
  switch (variablesView.first) {
  case MERGED_DISTINCT_DESIGN:
    return mergedDesignLowerBnds; break;
  case MERGED_DISTINCT_UNCERTAIN:
    return uncertainLowerBnds;    break;
  case MERGED_DISTINCT_STATE:
    return mergedStateLowerBnds;  break;
  }
  return emptyRealVector;
}


inline void MergedConstraints::
continuous_lower_bounds(const RealVector& c_l_bnds)
{
  switch (variablesView.first) {
  case MERGED_DISTINCT_DESIGN:
    mergedDesignLowerBnds = c_l_bnds; break;
  case MERGED_DISTINCT_UNCERTAIN:
    uncertainLowerBnds    = c_l_bnds; break;
  case MERGED_DISTINCT_STATE:
    mergedStateLowerBnds  = c_l_bnds; break;
  }
}


inline const RealVector& MergedConstraints::continuous_upper_bounds() const
{
  switch (variablesView.first) {
  case MERGED_DISTINCT_DESIGN:
    return mergedDesignUpperBnds; break;
  case MERGED_DISTINCT_UNCERTAIN:
    return uncertainUpperBnds;    break;
  case MERGED_DISTINCT_STATE:
    return mergedStateUpperBnds;  break;
  }
  return emptyRealVector;
}


inline void MergedConstraints::
continuous_upper_bounds(const RealVector& c_u_bnds)
{
  switch (variablesView.first) {
  case MERGED_DISTINCT_DESIGN:
    mergedDesignUpperBnds = c_u_bnds; break;
  case MERGED_DISTINCT_UNCERTAIN:
    uncertainUpperBnds    = c_u_bnds; break;
  case MERGED_DISTINCT_STATE:
    mergedStateUpperBnds  = c_u_bnds; break;
  }
}


inline const RealVector& MergedConstraints::
inactive_continuous_lower_bounds() const
{
  switch (variablesView.second) {
  case MERGED_DISTINCT_DESIGN:
    return mergedDesignLowerBnds; break;
  case MERGED_DISTINCT_UNCERTAIN:
    return uncertainLowerBnds;    break;
  case MERGED_DISTINCT_STATE:
    return mergedStateLowerBnds;  break;
  }
  return emptyRealVector;
}


inline void MergedConstraints::
inactive_continuous_lower_bounds(const RealVector& i_c_l_bnds)
{
  switch (variablesView.second) {
  case MERGED_DISTINCT_DESIGN:
    mergedDesignLowerBnds = i_c_l_bnds; break;
  case MERGED_DISTINCT_UNCERTAIN:
    uncertainLowerBnds    = i_c_l_bnds; break;
  case MERGED_DISTINCT_STATE:
    mergedStateLowerBnds  = i_c_l_bnds; break;
  }
}


inline const RealVector& MergedConstraints::
inactive_continuous_upper_bounds() const
{
  switch (variablesView.second) {
  case MERGED_DISTINCT_DESIGN:
    return mergedDesignUpperBnds; break;
  case MERGED_DISTINCT_UNCERTAIN:
    return uncertainUpperBnds;    break;
  case MERGED_DISTINCT_STATE:
    return mergedStateUpperBnds;  break;
  }
  return emptyRealVector;
}


inline void MergedConstraints::
inactive_continuous_upper_bounds(const RealVector& i_c_u_bnds)
{
  switch (variablesView.second) {
  case MERGED_DISTINCT_DESIGN:
    mergedDesignUpperBnds = i_c_u_bnds; break;
  case MERGED_DISTINCT_UNCERTAIN:
    uncertainUpperBnds    = i_c_u_bnds; break;
  case MERGED_DISTINCT_STATE:
    mergedStateUpperBnds  = i_c_u_bnds; break;
  }
}


inline RealVector MergedConstraints::all_continuous_lower_bounds() const
{
  RealVector all_c_l_bnds;
  aggregate_data(mergedDesignLowerBnds, uncertainLowerBnds,
		 mergedStateLowerBnds,  all_c_l_bnds);
  return all_c_l_bnds;
}


inline void MergedConstraints::
all_continuous_lower_bounds(const RealVector& a_c_l_bnds)
{
  separate_data(a_c_l_bnds, mergedDesignLowerBnds, uncertainLowerBnds,
		mergedStateLowerBnds);
}


inline RealVector MergedConstraints::all_continuous_upper_bounds() const
{
  RealVector all_c_u_bnds;
  aggregate_data(mergedDesignUpperBnds, uncertainUpperBnds,
		 mergedStateUpperBnds,  all_c_u_bnds);
  return all_c_u_bnds;
}


inline void MergedConstraints::
all_continuous_upper_bounds(const RealVector& a_c_u_bnds)
{
  separate_data(a_c_u_bnds, mergedDesignUpperBnds, uncertainUpperBnds,
		mergedStateUpperBnds);
}

} // namespace Dakota

#endif
