/*  _________________________________________________________________________
 *
 *  Coliny: A Library of COLIN optimizers
 *  Copyright (c) 2003, Sandia National Laboratories.
 *  This software is distributed under the GNU Lesser General Public License.
 *  For more information, see the README.html file in the top Coliny directory.
 *  _________________________________________________________________________
 */

/**
 * \file OptSolverUtil.h
 *
 *  This file defines a variety of utility functions that are useful for
 *  running OptSolvers in a straight-forward manner.  The first set
 *  of functions are templated on the optimizer type.  The ColinySolver
 *  class is used by the remaining functions to allow a user to specify 
 *  a solver using predefined names.
 */

#ifndef colin_OptSolverUtil_h
#define colin_OptSolverUtil_h

#include <acro_config.h>
#include <colin/OptSolver.h>
#include <colin/MasterSlaveOptExec.h>

namespace colin {

template <class A, class B>
double dist(const A& a, const B& b)
{
double ans = 0.0;
for (unsigned int i=0; i<a.size(); i++)
  ans += (a[i]-b[i])*(a[i]-b[i]);
return sqrt(ans);
}


//
// Core solver routine that applies a COLIN solver in a generic fashion
//
template <class DomainT, class ResponseT, class CommT>
bool solve(OptSolverHandle<DomainT,ResponseT>& opt, OptProblem<DomainT,ResponseT>& prob, 
			int seed,
			bool debug_flag, 
			bool verbose_flag, 
			DomainT& best_point, real& best_value, 
#if defined(ACRO_HAVE_MPI)
			CommT comm)
#else
			CommT )
#endif
{
try {
  #if defined(ACRO_HAVE_MPI)
  /// TODO: how use the communicator here?
  int processor_rank = utilib::uMPI::rank;
  int num_processors = utilib::uMPI::size;
  #else
  int processor_rank=0, num_processors=1;
  #endif

  if (verbose_flag) {
     if (prob.enforcing_bounds()) {
	if (prob.num_real_params() > 0) {
	   utilib::pvector<real> lb,ub;
	   prob.get_real_bounds(lb,ub);
	   if (processor_rank == 0) {
              ucout << "real lb: " << lb << std::endl;
              ucout << "real ub: " << ub << std::endl;
	      }
	   }
	/*** TODO - fix this...
	if (prob.num_int_params() > 0) {
	   vector<int> lb,ub;
	   prob.get_int_bounds(lb,ub);
           cout << "int lb: " << lb << std::endl;
           cout << "int ub: " << ub << std::endl;
	   }
     cout << "nint: " << prob.num_int_params() << std::endl;
     cout << "nbit: " << prob.num_binary() << std::endl;
     ***/
        }
     if (processor_rank == 0)
        ucout << "nreal: " << prob.num_real_params() << std::endl;
     }
  if (processor_rank == 0) {
     //PM_LCG rng(seed);
     //opt.set_rng(&rng);
     if (debug_flag) {
	opt->debug = 10000000;
        opt->set_parameter("output_level","verbose");
        }
     if (seed >= 0)
        opt->set_parameter("seed",seed);
     ucout << utilib::Flush;
     std::cout << std::flush;
     opt->reset();
     if (verbose_flag) {
        ucout << std::endl;
        opt->write_parameters(ucout);
        opt->write_parameter_values(ucout);
        }
     }
  ucout << utilib::Flush;
  std::cout << std::flush;
  double run_start = CPUSeconds();
  if (num_processors == 1)
     opt->minimize();
  else {
     if (debug_flag)
        ucout << "Starting Parallel Execution" << std::endl;
     colin::MasterSlaveOptExec(opt,prob,1,false,debug_flag);
     }
  double run_end = CPUSeconds();
  if (processor_rank == 0) {
     ucout << "-----------------------------------------------------------------------" << std::endl;
     ucout << "Solver Statistics" << std::endl;
     ucout << "-----------------------------------------------------------------------" << std::endl;
     opt->print_stats(ucout);
     if (prob.numObjectives() == 1) {
        best_point << opt.opt_response().point;
        best_value = opt.opt_response().value();
        #ifdef ACRO_VALIDATING
        if (std::fabs(opt.opt_response().value() - opt.opt_response().response.function_value()) > 1e-8)
           EXCEPTION_MNGR(std::runtime_error,"Solver Error: Discrepancy between the OptResponse value and the final value of the OptResponse object")
        #endif
        }
     ucout << "Solver-Time: " << (run_end-run_start) << std::endl;
     }
  }
catch (std::exception& err) {
  if (verbose_flag) {
     ucout << err.what() << std::endl;
     }
  throw;
  }
return true;
}


//
// Core solver routine that applies a COLIN solver with an initial point 
// in a generic fashion.
//
template <class DomainT, class ResponseT>
bool solve(OptSolverHandle<DomainT,ResponseT>& opt, OptProblem<DomainT,ResponseT>& prob, DomainT& point,
			int seed,
			bool debug_flag, 
			bool verbose_flag, 
			DomainT& best_point, real& best_value)
{
prob.init_real_params(point.size());
opt.set_problem(prob);
opt.set_initial_point(point);
#if defined(ACRO_HAVE_MPI)
return solve(opt,prob,seed,debug_flag,verbose_flag,best_point,best_value,utilib::uMPI::comm);
#else
int dummy=0;
return solve(opt,prob,seed,debug_flag,verbose_flag,best_point,best_value,dummy);
#endif
}


//
// Core solver routine that applies a COLIN solver to optimize globally 
// in a generic fashion.
//
template <class DomainT, class ResponseT>
bool solve(OptSolverHandle<DomainT,ResponseT>& opt, OptProblem<DomainT,ResponseT>& prob,
			int seed,
			bool debug_flag, 
			bool verbose_flag, 
			DomainT& best_point, real& best_value)
{
opt.set_problem(prob);
#if defined(ACRO_HAVE_MPI)
return solve(opt,prob,seed,debug_flag,verbose_flag,best_point,best_value,utilib::uMPI::comm);
#else
int dummy=0;
return solve(opt,prob,seed,debug_flag,verbose_flag,best_point,best_value,dummy);
#endif
}



/*
//
// Core solver routine that applies a COLIN solver with an initial point 
// in a generic fashion.
//
template <class ProblemT, class DomainT>
bool solve(const char* opt, ProblemT& prob, DomainT& point,
			int seed,
			std::istream& is, 
			bool debug_flag, 
			bool verbose_flag, 
			DomainT& best_point, real& best_value)
{
ColinySolver<ProblemT,DomainT> solver(opt);
solver.read_parameter_values(is);
bool status = solver.minimize(prob,point,seed,debug_flag,verbose_flag,best_point,best_value);
return status;
}


//
// Core solver routine that applies a COLIN solver to optimize globally 
// in a generic fashion.
//
template <class ProblemT, class DomainT>
bool solve(const char* opt, ProblemT& prob,
			int seed,
			std::istream& is, 
			bool debug_flag, 
			bool verbose_flag, 
			DomainT& best_point, real& best_value)
{
ColinySolver<ProblemT,DomainT> solver(opt);
solver.read_parameter_values(is);
bool status = solver.minimize(prob,seed,debug_flag,verbose_flag,best_point,best_value);
return status;
}


//
// Core solver routine that applies a COLIN solver with an initial point 
// in a generic fashion.
//
template <class ProblemT, class DomainT>
bool solve(const char* opt, ProblemT& prob, DomainT& point,
			int seed,
			std::istream& is, 
			bool debug_flag, 
			bool verbose_flag, 
			DomainT& opt_point,
			DomainT& best_point, real& best_value)
{
ColinySolver<ProblemT,DomainT> solver;
solver.initialize(opt);
solver.read_parameter_values(is);
bool status = solver.minimize(prob,point,seed,debug_flag,verbose_flag,opt_point,best_point,best_value);
return status;
}


//
// Core solver routine that applies a COLIN solver to optimize globally 
// in a generic fashion.
//
template <class ProblemT, class DomainT>
bool solve(const char* opt, ProblemT& prob,
			int seed,
			std::istream& is, 
			bool debug_flag, 
			bool verbose_flag, 
			DomainT& opt_point,
			DomainT& best_point, real& best_value)
{
ColinySolver<ProblemT,DomainT> solver;
solver.initialize(opt);
solver.read_parameter_values(is);
bool status = solver.minimize(prob,seed,debug_flag,verbose_flag,opt_point,best_point,best_value);
return status;
}
*/

}

#endif
