/*  _______________________________________________________________________

    DAKOTA: Design Analysis Kit for Optimization and Terascale Applications
    Copyright (c) 2006, Sandia National Laboratories.
    This software is distributed under the GNU General Public License.
    For more information, see the README file in the top Dakota directory.
    _______________________________________________________________________ */

//- Class:       Dakota::Array<T>
//- Description: An array class template that provides additional functionality
//-              that is specific to Dakota's needs.  Inheritance is used to
//-              build upon the capabilities provided by a third party array 
//-              class template.
//- Owner:       Mike Eldred
//- Modified by: Mario Alleva
//- Checked by:
//- Version: $Id

#ifndef DAKOTA_ARRAY_H
#define DAKOTA_ARRAY_H

#include "template_defs.h"
#include "global_defs.h"
#include "DakotaBinStream.H"
#include "MPIPackBuffer.H"


namespace Dakota {

/// Template class for the Dakota bookkeeping array

/** An array class template that provides additional functionality
    that is specific to Dakota's needs.  The Array class adds
    additional functionality needed by Dakota to the inherited base
    array class.  The Array class can inherite from either the
    STL or RW vector classes. */

template<class T>
class Array : public DAKOTA_BASE_ARRAY<T>
{
public:

  //
  //- Heading: Constructors and destructor
  //

  /// Default constructor
  Array();
  /// Constructor which takes an initial size
  Array(size_t size);
  /// Constructor which takes an initial size and an initial value
  Array(size_t size, const T& initial_val);
  /// Copy constructor
  Array(const Array<T>& a);
  /// Constructor which copies size entries from T*
  Array(const T* p, size_t size);
  /// Destructor
  ~Array();

  //
  //- Heading: Assignment operators
  //

  /// Normal const assignment operator.
  Array<T>& operator=(const Array<T>& a);

  /// Normal assignment operator.
  Array<T>& operator=(Array<T>& a);

  /// Sets all elements in self to the value ival.
  Array<T>& operator=(const T& ival);

  //
  //- Heading: Conversion operators
  //

  /// Converts the Array to a standard C-style array. Use with care!
  operator T*() const;

  //
  //- Heading: Indexing operators
  //

#ifdef HAVE_PARTIAL_ORDERING
  // Quiets errors occurring for instances of array[int], e.g.:
  //   more than one operator [] matches these operands:
  //   built-in operator "pointer-to-object[integer]"
  //   function "Array<T>::operator[](size_t) [with T=int]"

  /// alternate bounds-checked indexing operator for int indices
  T& operator[](int i);
  /// alternate bounds-checked const indexing operator for int indices
  const T& operator[](int i) const;
  // For completeness sake, could add:
  //T& operator() (int i);
  //const T& operator() (int i) const;
#endif // HAVE_PARTIAL_ORDERING

  // Many of the methods below add functionality to STL vectors to mimic old
  // functions from the RWTValVector class in order to avoid source changes in
  // the main DAKOTA code.  Note that the bounds-checked and nonbounds-checked
  // operators in Array are different from STL vectors. 

  /// Index operator, returns the ith value of the array
  T& operator[](size_t i); 
  /// Index operator const, returns the ith value of the array
  const T& operator[](size_t i) const;

  /// Index operator, not bounds checked
  T& operator() (size_t i); 
  /// Index operator const, not bounds checked
  const T& operator() (size_t i) const;

  //
  //- Heading: Member functions 
  //

  /// Reads an Array from an istream
  void read(istream& s);
  /// Writes an Array to an output stream
  void write(ostream& s) const;
  /// Writes an Array and associated label array to an output stream
  void write(ostream& s, const Array<String>& label_array) const;
  /// Writes an Array and associated label array to an output stream in
  /// aprepro format
  void write_aprepro(ostream& s, const Array<String>& label_array) const;
  /// Writes an Array to an output stream in annotated format
  void write_annotated(ostream& s, bool write_len) const;

  /// Reads an Array from a binary input stream
  void read(BiStream& s);
  /// Writes an Array to a binary output stream
  void write(BoStream& s) const;

  /// Reads an Array from a buffer after an MPI receive
  void read(MPIUnpackBuffer& s);
  /// Writes an Array to a buffer prior to an MPI send
  void write(MPIPackBuffer& s) const;

  /// Returns size of array
  size_t length() const;
  /// Resizes array to size sz
  void reshape(size_t sz);

  /// Returns the index of the first array item which matches the object a
  size_t index(const T& a) const;
  /// Checks if the array contains an object which matches the object a
  bool contains(const T& a) const;
  /// Returns the number of items in the array matching the object a
  size_t count(const T& a) const;

  /// Returns pointer T* to continuous data
  const T* data() const;

private:

  //
  //- Heading: Data
  //
};


template <class T>
inline Array<T>::Array()
{ }


template <class T>
inline Array<T>::Array(size_t size): DAKOTA_BASE_ARRAY<T>(size)
{ }


template <class T> 
inline Array<T>::Array(size_t size, const T& initial_val):
  DAKOTA_BASE_ARRAY<T>(size, initial_val)
{ }


template <class T>
inline Array<T>::Array(const Array<T>& a):
  DAKOTA_BASE_ARRAY<T>(a)
{ }


/** Assigns size values from p into array. */
template <class T>
inline Array<T>::Array(const T* p, size_t size):
  DAKOTA_BASE_ARRAY<T>(size)
{
  for (size_t i=0; i<size; i++)
    DAKOTA_BASE_ARRAY<T>::operator[](i) = p[i];
}


template <class T>
inline Array<T>::~Array()
{ }


template <class T>
inline Array<T>& Array<T>::operator=(const Array<T>& a)
{
  if (this != &a) // verify that these are different objects
    DAKOTA_BASE_ARRAY<T>::operator=(a); // base class assignment operator
  return *this;
}


template <class T>
inline Array<T>& Array<T>::operator=(Array<T>& a)
{
  if (this != &a) // verify that these are different objects
    DAKOTA_BASE_ARRAY<T>::operator=(a); // base class assignment operator
  return *this;
}


/** Assigns all values of array to the value passed in as ival. 
    For the Rogue Wave case, utilizes base class operator=(ival),
    while for the ANSI case, uses the STL assign() method.  */
template <class T>
inline Array<T>& Array<T>::operator=(const T& ival)
{
  // set the values in the array using the base class assignment operator
  assign(DAKOTA_BASE_ARRAY<T>::size(), ival);
  return *this;
}


/** The operator() returns a c style pointer to the data within the array. 
    Calls the data() method.  USE WITH CARE. */
// I would like to use the const version but linking with the FORTRAN
// and passing the array prevents const!
//template<class T> Array<T>::operator const T*() const
template <class T>
inline Array<T>::operator T*() const
{ return (T*)data(); }


#ifdef HAVE_PARTIAL_ORDERING
template <class T> 
inline T& Array<T>::operator[](int i)
{
#ifdef HAVE_VECTOR_AT
  return DAKOTA_BASE_ARRAY<T>::at(i);
#else
  return DAKOTA_BASE_ARRAY<T>::operator[](i);
#endif // HAVE_VECTOR_AT
}


template <class T> 
inline const T& Array<T>::operator[](int i) const
{
#ifdef HAVE_VECTOR_AT
  return DAKOTA_BASE_ARRAY<T>::at(i);
#else
  return DAKOTA_BASE_ARRAY<T>::operator[](i);
#endif // HAVE_VECTOR_AT
}
#endif // HAVE_PARTIAL_ORDERING


/** Index operator; calls the STL method at() which is bounds
    checked. Mimics the RW vector class.  Note: the at() method is not
    supported by the __GNUC__ STL implementation or by builds
    omitting exceptions (e.g., SIERRA). */
template <class T>
inline T& Array<T>::operator[](size_t i)
{ 
#ifdef HAVE_VECTOR_AT
  return DAKOTA_BASE_ARRAY<T>::at(i);
#else
  return DAKOTA_BASE_ARRAY<T>::operator[](i);
#endif // HAVE_VECTOR_AT
}


/** A const version of the index operator; calls the STL method at()
    which is bounds checked. Mimics the RW vector class.  Note: the
    at() method is not supported by the __GNUC__ STL implementation
    or by builds omitting exceptions (e.g., SIERRA). */
template <class T>
inline const T& Array<T>::operator[](size_t i) const
{ 
#ifdef HAVE_VECTOR_AT
  return DAKOTA_BASE_ARRAY<T>::at(i);
#else
  return DAKOTA_BASE_ARRAY<T>::operator[](i);
#endif // HAVE_VECTOR_AT
}


/** Non bounds check index operator, calls the STL operator[] which is not
    bounds checked. Needed to mimic the RW vector class */ 
template <class T>
inline T& Array<T>::operator()(size_t i)
{ return DAKOTA_BASE_ARRAY<T>::operator[](i); }


/** A const version of the non-bounds check index operator, calls the
    STL operator[] which is not bounds checked. Needed to mimic the RW
    vector class */ 
template <class T>
inline const T& Array<T>::operator()(size_t i) const 
{ return DAKOTA_BASE_ARRAY<T>::operator[](i); }


template <class T>
void Array<T>::read(istream& s)
{
  size_t len = DAKOTA_BASE_ARRAY<T>::size();
  for (register size_t i=0; i<len; i++)
    s >> DAKOTA_BASE_ARRAY<T>::operator[](i);
}


template <class T>
void Array<T>::write(ostream& s) const
{
  s.setf(ios::scientific);
  s << setprecision(write_precision);

  size_t len = DAKOTA_BASE_ARRAY<T>::size();
  for (register size_t i=0; i<len; i++)
    s << "                     " << setw(write_precision+7)
      << DAKOTA_BASE_ARRAY<T>::operator[](i) << '\n';
}


/// global istream extraction operator for Vector
template <class T>
inline istream& operator>>(istream& s, Array<T>& data)
{ data.read(s); return s; }


/// global ostream insertion operator for Array
template <class T>
inline ostream& operator<<(ostream& s, const Array<T>& data)
{ data.write(s); return s; }


template <class T>
void Array<T>::write(ostream& s, const Array<String>& label_array) const
{
  s.setf(ios::scientific);
  s << setprecision(write_precision);

  size_t len = DAKOTA_BASE_ARRAY<T>::size();
  if (label_array.size() != len) {
    Cerr << "Error: size of label_array in Array<T>::write() does not equal "
	 << "length of Array." << endl;
    abort_handler(-1);
  }
  for (register size_t i=0; i<len; i++)
    s << "                     " << setw(write_precision+7) 
      << DAKOTA_BASE_ARRAY<T>::operator[](i) << ' ' << label_array[i] << '\n';
}


template <class T>
void Array<T>::
write_aprepro(ostream& s, const Array<String>& label_array) const
{
  s.setf(ios::scientific);
  s << setprecision(write_precision);

  size_t len = DAKOTA_BASE_ARRAY<T>::size();
  if (label_array.size() != len) {
    Cerr << "Error: size of label_array in Array<T>::write() does not equal "
	 << "length of Array." << endl;
    abort_handler(-1);
  }
  for (register size_t i=0; i<len; i++)
    s << "                    { " << setw(15) << setiosflags(ios::left) 
      << label_array[i].data() << resetiosflags(ios::adjustfield) << " = " 
      << setw(write_precision+7) << DAKOTA_BASE_ARRAY<T>::operator[](i)<<" }\n";
}


template <class T>
void Array<T>::write_annotated(ostream& s, bool write_len) const
{
  s.setf(ios::scientific);
  s << setprecision(write_precision);

  size_t len = DAKOTA_BASE_ARRAY<T>::size();
  if (write_len)
    s << len << ' ';
  for (register size_t i=0; i<len; i++)
    s << DAKOTA_BASE_ARRAY<T>::operator[](i) << ' ';
}


template <class T>
void Array<T>::read(BiStream& s)
{
  size_t len;
  s >> len;
  DAKOTA_BASE_ARRAY<T>::resize(len);
  for (register size_t i=0; i<len; i++)
    s >> DAKOTA_BASE_ARRAY<T>::operator[](i);
}


template <class T>
void Array<T>::write(BoStream& s) const
{
  size_t len = DAKOTA_BASE_ARRAY<T>::size();
  s << len;
  for (register size_t i=0; i<len; i++)
    s << DAKOTA_BASE_ARRAY<T>::operator[](i);
}


/// global BiStream extraction operator for Array
template <class T>
inline BiStream& operator>>(BiStream& s, Array<T>& data)
{ data.read(s); return s; }


/// global BoStream insertion operator for Array
template <class T>
inline BoStream& operator<<(BoStream& s, const Array<T>& data)
{ data.write(s); return s; }


template <class T>
void Array<T>::read(MPIUnpackBuffer& s)
{
  size_t len;
  s >> len;
  DAKOTA_BASE_ARRAY<T>::resize(len);
  for (register size_t i=0; i<len; i++)
    s >> DAKOTA_BASE_ARRAY<T>::operator[](i);
}


template <class T>
void Array<T>::write(MPIPackBuffer& s) const
{
  size_t len = DAKOTA_BASE_ARRAY<T>::size();
  s << len;
  for (register size_t i=0; i<len; i++)
    s << DAKOTA_BASE_ARRAY<T>::operator[](i);
}


/// global MPIUnpackBuffer extraction operator for Array
template <class T>
inline MPIUnpackBuffer& operator>>(MPIUnpackBuffer& s, Array<T>& data)
{ data.read(s); return s; }


/// global MPIPackBuffer insertion operator for Array
template <class T>
inline MPIPackBuffer& operator<<(MPIPackBuffer& s, const Array<T>& data)
{ data.write(s); return s; }


template <class T>
inline size_t Array<T>::length() const
{ return DAKOTA_BASE_ARRAY<T>::size(); }


template <class T>
inline void Array<T>::reshape(size_t sz)
{ DAKOTA_BASE_ARRAY<T>::resize(sz); }


template <class T> 
size_t Array<T>::index(const T& a) const
{
  // should be more efficient than find() + distance()
  size_t len = DAKOTA_BASE_ARRAY<T>::size();
  for (size_t i=0; i<len; i++)
    if (DAKOTA_BASE_ARRAY<T>::operator[](i) == a)
      return i;
  return _NPOS;
}


template <class T> 
bool Array<T>::contains(const T& a) const
{
  return (std::find(DAKOTA_BASE_ARRAY<T>::begin(),DAKOTA_BASE_ARRAY<T>::end(),a)
	  != DAKOTA_BASE_ARRAY<T>::end()) ? true : false;

  /*
  size_t len = DAKOTA_BASE_ARRAY<T>::size();
  for (size_t i=0; i<len; i++)
    if (DAKOTA_BASE_ARRAY<T>::operator[](i) == a)
      return true;
  return false;
  */
}


template <class T> 
size_t Array<T>::count(const T& a) const
{
#ifdef HAVE_STD_COUNT
  // All other platforms use the ANSI C++ version of the count() algorithm
  return std::count(DAKOTA_BASE_ARRAY<T>::begin(),
		    DAKOTA_BASE_ARRAY<T>::end(), a);
#else
  // These platforms use an older version of the count() algorithm
  size_t cnt = 0;
  std::count(DAKOTA_BASE_ARRAY<T>::begin(),
	     DAKOTA_BASE_ARRAY<T>::end(), a, cnt);
  return cnt;
#endif
}


/** Returns a C style pointer to the data within the array. USE WITH
    CARE. Needed to mimic RW vector class, is used in the operator(). 
    Uses the STL front method. */
template <class T>
inline const T* Array<T>::data() const 
{
  return (DAKOTA_BASE_ARRAY<T>::empty()) ? NULL :
    (T*)&DAKOTA_BASE_ARRAY<T>::front();
}

} // namespace Dakota

#endif  // DAKOTA_ARRAY_H
