/*  _________________________________________________________________________
 *
 *  Coliny: A Library of COLIN optimizers
 *  Copyright (c) 2003, Sandia National Laboratories.
 *  This software is distributed under the GNU Lesser General Public License.
 *  For more information, see the README.html file in the top Coliny directory.
 *  _________________________________________________________________________
 */

//
// PLGO.cpp
//

#include <acro_config.h>
#ifdef ACRO_USING_PLGO

#include <coliny/Factory.h>
#include <coliny/PLGO.h>

using namespace std;
using namespace utilib;

namespace coliny {


PLGO::PLGO()
 : step_tolerance(1e-4),
   initial_step(1.0)
{
#if defined(ACRO_HAVE_MPI)
if (uMPI::running() && (uMPI::size == 1))
   solver = new plgo::PLGOSerialSolver;
else {
   solver = new plgo::PLGOParallelSolver;
   }
#else
solver = new plgo::PLGOSerialSolver;
#endif

func.solver = this;
func.problem = &problem;
solver->set_func(&func);

augment_parameters(solver->parameters(),true);

disable_parameter("accuracy");
disable_parameter("ftol");

solver->base()->set_parameter("absTolerance",1e-5);
solver->base()->set_parameter("lipshitzConstant",1.0);
}

void PLGO::reset()
{
if (!problem)
   return;

#if defined(ACRO_HAVE_MPI)
if (uMPI::size > 1)
   #ifdef UTILIB_MPI_COMM_IS_POINTER
   set_parameter("mpicomm",static_cast<void*>(uMPI::comm));
   #else
   set_parameter("mpicomm",uMPI::comm);
   #endif
#endif

if (problem.has_all_bounds())
   problem.get_real_bounds(func.lower,func.upper);

x.resize(problem.num_real_params());

ucout << utilib::Flush;
cout.flush();
solver->reset();
}


void PLGO::minimize()
{
if (problem.num_real_params() == 0) {
   best().termination_info = "No-Real-Params";
   return;
   }
if (!problem.has_all_bounds()) {
   best().termination_info = "Missing-Bound-Constraints";
   return;
   }

opt_init();

int tmp_neval = max(max_neval-problem.neval(),0);
if (max_neval_curr != 0)
   tmp_neval = min(tmp_neval, max_neval_curr);

try {
  solver->minimize(best().termination_info, x, best().value());
}
STD_CATCH(;)

best().point << x;
if (problem.numConstraints() > 0)
   problem.Eval(x, best().response, colin::mode_f | colin::mode_cf);
else
   problem.Eval(x, best().response, colin::mode_f);
compute_response_info(best().response,problem.state->constraint_lower_bounds,  problem.state->constraint_upper_bounds, best().value(),best().constraint_violation);
}

} 

typedef colin::OptSolver<utilib::BasicArray<double>,colin::                     AppResponse_Utilib> base_t;
FACTORY_REGISTER(plgo, base_t*, return new coliny::PLGO();, "An alias to PLGO")
FACTORY_REGISTER(plipshitzian, base_t*, return new coliny::PLGO();, "A parallel Lipshitzian global optimizer")

#endif
