/*  _______________________________________________________________________

    DAKOTA: Design Analysis Kit for Optimization and Terascale Applications
    Copyright (c) 2001, Sandia National Laboratories.
    This software is distributed under the GNU General Public License.
    For more information, see the README file in the top Dakota directory.
    _______________________________________________________________________ */

//- Class:	 NonDSparseGrid
//- Description: Wrapper class for C++ code from packages/quadrature/sparse_grid
//- Owner:       Mike Eldred
//- Revised by:  
//- Version:

#ifndef NOND_SPARSE_GRID_H
#define NOND_SPARSE_GRID_H

#include "data_types.h"
#include "NonDIntegration.H"

namespace Dakota {


/// Derived nondeterministic class that generates N-dimensional
/// Smolyak sparse grids for numerical evaluation of expectation
/// integrals over independent standard random variables.

/** This class is used by NonDPolynomialChaos and
    NonDStochCollocation, but could also be used for general numerical
    integration of moments.  It employs 1-D Clenshaw-Curtis and Gaussian
    quadrature rules within Smolyak sparse grids. */

class NonDSparseGrid: public NonDIntegration
{
public:

  //
  //- Heading: Constructors and destructor
  //

  // alternate constructor for instantiations "on the fly"
  NonDSparseGrid(Model& model, const UShortArray& levels);

  //
  //- Heading: Member functions
  //

  /// set rules based on u_types
  void initialize_rules(const Pecos::ShortArray& u_types, IntArray& rules);

  /// return sparseGridLevel
  const UShortArray& sparse_grid_level() const;
  /// return integrationRules
  const IntArray& integration_rules() const;

  // return sparseGridIndexMap
  //const IntArraySizetMap& sparse_grid_index_map() const;

  /// return ProbabilityTransformation::ranVarTypesU
  const Pecos::ShortArray& integrated_variable_types() const;

  /// converts sparse grid level to integration order for closed rules
  void level_to_order_closed(const unsigned short& level,
			     unsigned short& order) const;
  /// converts an array of sparse grid levels to an array of integration orders
  /// for closed rules
  void level_to_order_closed(const UShortArray& levels,
			     UShortArray& orders) const;
  /// converts sparse grid level to integration order for open rules
  void level_to_order_open(const unsigned short& level,
			   unsigned short& order) const;
  /// converts an array of sparse grid levels to an array of integration orders
  /// for open rules
  void level_to_order_open(const UShortArray& levels,
			   UShortArray& orders) const;

protected:

  //
  //- Heading: Constructors and destructor
  //

  NonDSparseGrid(Model& model); ///< constructor
  ~NonDSparseGrid();            ///< destructor

  //
  //- Heading: Virtual function redefinitions
  //

  void get_parameter_sets(const Model& model);

  void check_input();

  void sampling_reset(int min_samples, int rec_samples, bool all_data_flag,
		      bool stats_flag);

private:

  //
  //- Heading: Convenience functions
  //

  //
  //- Heading: Data
  //

  /// the user specification for the sparse grid levels
  UShortArray sparseGridLevelSpec;
  /// the actual sparse grid levels, as modified by external requirements
  /// communicated through sampling_reset()
  UShortArray sparseGridLevel;

  /// integer codes for webbur routine integration rule options
  IntArray integrationRules;

  // maps indices and bases from webbur::sparse_grid_mixed_index() to
  // collocation point index
  //IntArraySizetMap sparseGridIndexMap;
};


inline const UShortArray& NonDSparseGrid::sparse_grid_level() const
{ return sparseGridLevel; }


inline const IntArray& NonDSparseGrid::integration_rules() const
{ return integrationRules; }


//inline const IntArraySizetMap& NonDSparseGrid::sparse_grid_index_map() const
//{ return sparseGridIndexMap; }


inline const Pecos::ShortArray& NonDSparseGrid::
integrated_variable_types() const
{ return natafTransform.u_types(); }


// NOTE: these mappings are intended for 0-based indices, i.e. the j index set


/** Adapted from webbur::level_to_order_closed() for DAKOTA data types. */
inline void NonDSparseGrid::
level_to_order_closed(const unsigned short& level, unsigned short& order) const
{ order = (level) ? (unsigned short)pow(2., (int)level) + 1 : 1; }


/** Adapted from webbur::level_to_order_closed() for DAKOTA data types. */
inline void NonDSparseGrid::
level_to_order_closed(const UShortArray& levels, UShortArray& orders) const
{
  size_t i, num_levels = levels.length();
  if (orders.length() != num_levels)
    orders.reshape(num_levels);
  for (i=0; i<num_levels; i++)
    orders[i] = (levels[i]) ? (unsigned short)pow(2., (int)levels[i]) + 1 : 1;
}


/** Adapted from webbur::level_to_order_open() for DAKOTA data types. */
inline void NonDSparseGrid::
level_to_order_open(const unsigned short& level, unsigned short& order) const
{ order = (unsigned short)pow(2., (int)level + 1) - 1; }


/** Adapted from webbur::level_to_order_open() for DAKOTA data types. */
inline void NonDSparseGrid::
level_to_order_open(const UShortArray& levels, UShortArray& orders) const
{
  size_t i, num_levels = levels.length();
  if (orders.length() != num_levels)
    orders.reshape(num_levels);
  for (i=0; i<num_levels; i++)
    orders[i] = (unsigned short)pow(2., (int)levels[i] + 1) - 1;
}

} // namespace Dakota

#endif
