/*!\file VecToMPISerial.cpp
 * \brief gather a Petsc Vector spread across the cluster, onto node 0, and then broadcast to all nodes. 
 */

#include "../petscincludes.h"
#include "../../../shared/shared.h"

int VecToMPISerial(double** pgathered_vector, Vec vector){
	
	int i;
	extern int num_procs; 
	extern int my_rank;

	/*Petsc*/
	MPI_Status status;
	PetscInt lower_row,upper_row; 
	int range;
	int * idxn=NULL; 
	int buffer[3];
	
	/*intermediary results*/
	double* local_vector=NULL;

	/*input*/
	int vector_size;
	
	/*Output*/
	double* gathered_vector=NULL; //Global vector holding the final assembled vector on all nodes.

	VecGetSize(vector,&vector_size);
	if(vector_size==0){
		*pgathered_vector=NULL;
		return 1;
	}

	/*Allocate gathered vector on all nodes .*/
	gathered_vector=(double*)xmalloc(vector_size*sizeof(double));
	
	/*Allocate local vectors*/
	VecGetOwnershipRange(vector,&lower_row,&upper_row);
	upper_row--;
	range=upper_row-lower_row+1;    

	if (range){
		idxn=(int*)xmalloc(range*sizeof(int)); 
		for (i=0;i<range;i++){
			*(idxn+i)=lower_row+i;
		} 
		local_vector=(double*)xmalloc(range*sizeof(double));
		/*Extract values from MPI vector to serial local_vector on each node*/
		VecGetValues(vector,range,idxn,local_vector); 
	}

	/*Now each node holds its local_vector containing range rows. 
	 * We send this local_vector  to the gathered_vector on node 0*/
	for (i=1;i<num_procs;i++){
		if (my_rank==i){ 
			buffer[0]=my_rank;
			buffer[1]=lower_row;
			buffer[2]=range;
			MPI_Send(buffer,3,MPI_INT,0,1,MPI_COMM_WORLD);  
			if (range)MPI_Send(local_vector,range,MPI_DOUBLE,0,1,MPI_COMM_WORLD); 
		}
		if (my_rank==0){
			MPI_Recv(buffer,3,MPI_INT,i,1,MPI_COMM_WORLD,&status); 
			#ifdef _DEBUG_
				_printf_("Received from node %i: %i-%i\n",buffer[0],buffer[1],buffer[2]); 
			#endif
			if (buffer[2])MPI_Recv(gathered_vector+buffer[1],buffer[2],MPI_DOUBLE,i,1,MPI_COMM_WORLD,&status);
		}
	}

	if (my_rank==0){ 
		//Still have the local_vector on node 0 to take care of.
		if (range)memcpy(gathered_vector+lower_row,local_vector,range*sizeof(double));
	}

	/*Now, broadcast gathered_vector from node 0 to other nodes: */
	MPI_Bcast(gathered_vector,vector_size,MPI_DOUBLE,0,MPI_COMM_WORLD);

	/*Assign output pointers: */
	*pgathered_vector=gathered_vector;
	
	/*free ressources: */
	xfree((void**)&idxn);
	xfree((void**)&local_vector);
	
	return 1;
}
