/*  _______________________________________________________________________

    DAKOTA: Design Analysis Kit for Optimization and Terascale Applications
    Copyright (c) 2006, Sandia National Laboratories.
    This software is distributed under the GNU General Public License.
    For more information, see the README file in the top Dakota directory.
    _______________________________________________________________________ */

#include <cstdlib>
#include <iostream>
#include <fstream>
#include <vector>
#include <string>
#ifdef HAVE_CONFIG_H
#include "dakota_config.h"
#endif // HAVE_CONFIG_H
#ifdef HAVE_STD
#include <cmath>
#else
#include <math.h>
#endif // HAVE_STD
using namespace std;


int main(int argc, char** argv)
{
  // This test problem is an OUU example from Applied Research Associates
  // (42nd AIAA SDM conference, April 2001).

  ifstream fin(argv[1]);
  if (!fin) {
    cerr << "\nError: failure opening " << argv[1] << endl;
    exit(-1);
  }
  size_t i, num_vars, num_fns, num_deriv_vars;
  string vars_text, fns_text, dvv_text;

  // Get the parameter vector and ignore the labels
  fin >> num_vars >> vars_text;
  vector<double> x(num_vars);
  for (i=0; i<num_vars; i++) {
    fin >> x[i];
    fin.ignore(256, '\n');
  }

  // Get the ASV vector and ignore the labels
  fin >> num_fns >> fns_text;
  vector<int> ASV(num_fns);
  for (i=0; i<num_fns; i++) {
    fin >> ASV[i];
    fin.ignore(256, '\n');
  }

  // Get the DVV vector and ignore the labels
  fin >> num_deriv_vars >> dvv_text;
  vector<int> DVV(num_deriv_vars);
  for (i=0; i<num_deriv_vars; i++) {
    fin >> DVV[i];
    fin.ignore(256, '\n');
  }

  if (num_vars != 4 && num_vars != 6) {
    cerr << "Error: Wrong number of variables in cantilever test fn." << endl;
    exit(-1);
  }
  if (num_fns != 3) {
    cerr << "Error: wrong number of response functions in cantilever test fn."
         << endl;
    exit(-1);
  }

  // Compute the cross-sectional area, stress, and displacement of the
  // cantilever beam.  This simulator is unusual in that it supports both
  // the case of design variable insertion and the case of design variable
  // augmentation.  It does not support mixed insertion/augmentation.
  double w = (num_vars == 4) ? 2.5  : x[0]; // beam width
  double t = (num_vars == 4) ? 2.5  : x[1]; // beam thickness
  double R = (num_vars == 4) ? x[0] : x[2]; // yield strength
  double E = (num_vars == 4) ? x[1] : x[3]; // Young's modulus
  double X = (num_vars == 4) ? x[2] : x[4]; // horizontal load
  double Y = (num_vars == 4) ? x[3] : x[5]; // vertical load

  // Compute the results and output them directly to argv[2] (the NO_FILTER
  // option is used).  Response tags are optional; output them for ease of
  // results readability.
  double D0 = 2.2535, L = 100., area = w*t, w_sq = w*w, t_sq = t*t,
    R_sq = R*R, X_sq = X*X, Y_sq = Y*Y;
  double stress = 600.*Y/w/t_sq + 600.*X/w_sq/t;
  double D1 = 4.*pow(L,3)/E/area, D2 = pow(Y/t_sq, 2)+pow(X/w_sq, 2);
  double D3 = D1/sqrt(D2)/D0,     D4 = D1*sqrt(D2)/D0;

  // limit state >= 0
  //double g_stress = R  - stress;
  //double g_disp   = D0 - disp;

  // inequality constraint <= 0
  //double g_stress = stress/R - 1.0;
  //double g_disp   = disp/D0  - 1.0;

  ofstream fout(argv[2]); // do not instantiate until ready to write results
  if (!fout) {
    cerr << "\nError: failure creating " << argv[2] << endl;
    exit(-1);
  }
  fout.precision(15); // 16 total digits
  fout.setf(ios::scientific);
  fout.setf(ios::right);

  // **** f:
  if (ASV[0] & 1)
    fout << "                     " << area << '\n';

  // **** c1:
  if (ASV[1] & 1)
    fout << "                     " << stress/R - 1. << '\n';

  // **** c2:
  if (ASV[2] & 1)
    fout << "                     " << D4 - 1. << '\n';

  // **** df/dx:
  if (ASV[0] & 2) {
    fout << "[ ";
    if (num_vars == 6) { // normal design + uncertain case
      for (size_t i=0; i<num_deriv_vars; i++) {
	switch (DVV[i]) {
	case 1:  fout << t << ' '; break; // design var derivative
	case 2:  fout << w << ' '; break; // design var derivative
	default: fout << "0. ";    break; // uncertain var derivative
	}
      }
    }
    else if (num_vars == 4) // uncertain vars only
      for (size_t i=0; i<num_deriv_vars; i++)
	fout << "0. "; // uncertain var derivative
    fout << "]\n";
  }

  // **** dc1/dx:
  if (ASV[1] & 2) {
    fout << "[ ";
    if (num_vars == 6) { // normal design + uncertain case
      for (size_t i=0; i<num_deriv_vars; i++) {
	switch (DVV[i]) {
	case 1: fout << -600.*(Y/t + 2.*X/w)/w_sq/t/R << ' '; break; // des var
	case 2: fout << -600.*(2.*Y/t + X/w)/w/t_sq/R << ' '; break; // des var
	case 3: fout << -stress/R_sq  << ' '; break; // uncertain var deriv
	case 4: fout << "0. ";                break; // uncertain var deriv
	case 5: fout << 600./w_sq/t/R << ' '; break; // uncertain var deriv
	case 6: fout << 600./w/t_sq/R << ' '; break; // uncertain var deriv
	}
      }
    }
    else if (num_vars == 4) { // uncertain vars only
      for (size_t i=0; i<num_deriv_vars; i++) {
	switch (DVV[i]) {
	case 1: fout << -stress/R_sq  << ' '; break; // uncertain var deriv
	case 2: fout << "0. ";                break; // uncertain var deriv
	case 3: fout << 600./w_sq/t/R << ' '; break; // uncertain var deriv
	case 4: fout << 600./w/t_sq/R << ' '; break; // uncertain var deriv
	}
      }
    }
    fout << "]\n";
  }

  // **** dc2/dx:
  if (ASV[2] & 2) {
    fout << "[ ";
    if (num_vars == 6) { // normal design + uncertain case
      for (size_t i=0; i<num_deriv_vars; i++) {
	switch (DVV[i]) {
	case 1: fout << -D3*2.*X_sq/w_sq/w_sq/w - D4/w << ' '; break; // des var
	case 2: fout << -D3*2.*Y_sq/t_sq/t_sq/t - D4/t << ' '; break; // des var
	case 3: fout << "0. ";                 break; // unc var deriv
	case 4: fout << -D4/E          << ' '; break; // unc var deriv
	case 5: fout << D3*X/w_sq/w_sq << ' '; break; // unc var deriv
	case 6: fout << D3*Y/t_sq/t_sq << ' '; break; // unc var deriv
	}
      }
    }
    else if (num_vars == 4) { // uncertain vars only
      for (size_t i=0; i<num_deriv_vars; i++) {
	switch (DVV[i]) {
	case 1: fout << "0. ";                 break; // unc var deriv
	case 2: fout << -D4/E          << ' '; break; // unc var deriv
	case 3: fout << D3*X/w_sq/w_sq << ' '; break; // unc var deriv
	case 4: fout << D3*Y/t_sq/t_sq << ' '; break; // unc var deriv
	}
      }
    }
    fout << "]\n";
  }

  // **** d^2f/dx^2:
  if (ASV[0] & 4) {
    fout << "[[ ";
    if (num_vars == 6) { // normal design + uncertain case
      for (size_t i=0; i<num_deriv_vars; i++) {
	int var_index_i = DVV[i] - 1;
	for (size_t j=0; j<num_deriv_vars; j++) {
	  int var_index_j = DVV[j] - 1;
	  if ( (var_index_i == 0 && var_index_j == 1) ||
	       (var_index_i == 1 && var_index_j == 0) )
	    fout << "1. ";
	  else
	    fout << "0. ";
	}
      }
    }
    else if (num_vars == 4) // uncertain vars only
      for (size_t i=0; i<num_deriv_vars; i++)
	for (size_t j=0; j<num_deriv_vars; j++)
	  fout << "0. ";
    fout << "]]\n";
  }

  // **** d^2c1/dx^2:
  if (ASV[1] & 4) {
    fout << "[[ ";
    if (num_vars == 6) { // normal design + uncertain case
      for (size_t i=0; i<num_deriv_vars; i++) {
	int var_index_i = DVV[i] - 1;
	for (size_t j=0; j<num_deriv_vars; j++) {
	  int var_index_j = DVV[j] - 1;
	  if (var_index_i == 0 && var_index_j == 0)          // d^2g/dw^2
	    fout << 1200.*(Y/t + 3.*X/w)/w_sq/area/R << ' ';
	  else if (var_index_i == 1 && var_index_j == 1)     // d^2g/dt^2
	    fout << 1200.*(3.*Y/t + X/w)/t_sq/area/R << ' ';
	  else if (var_index_i == 2 && var_index_j == 2)     // d^2g/dR^2
	    fout << 2.*stress/pow(R, 3) << ' ';
	  else if ( (var_index_i == 0 && var_index_j == 1) ||
		    (var_index_i == 1 && var_index_j == 0) ) // d^2g/dwdt
	    fout << 1200.*(Y/t + X/w)/w_sq/t_sq/R << ' ';
	  else if ( (var_index_i == 0 && var_index_j == 2) ||
		    (var_index_i == 2 && var_index_j == 0) ) // d^2g/dwdR
	    fout << 600.*(Y/t + 2.*X/w)/w_sq/t/R_sq << ' ';
	  else if ( (var_index_i == 0 && var_index_j == 4) ||
		    (var_index_i == 4 && var_index_j == 0) ) // d^2g/dwdX
	    fout << -1200./w_sq/w/t/R << ' ';
	  else if ( (var_index_i == 0 && var_index_j == 5) ||
		    (var_index_i == 5 && var_index_j == 0) ) // d^2g/dwdY
	    fout << -600./w_sq/t_sq/R << ' ';
	  else if ( (var_index_i == 1 && var_index_j == 2) ||
		    (var_index_i == 2 && var_index_j == 1) ) // d^2g/dtdR
	    fout << 600.*(2.*Y/t + X/w)/w/t_sq/R_sq << ' ';
	  else if ( (var_index_i == 1 && var_index_j == 4) ||
		    (var_index_i == 4 && var_index_j == 1) ) // d^2g/dtdX
	    fout << -600./w_sq/t_sq/R << ' ';
	  else if ( (var_index_i == 1 && var_index_j == 5) ||
		    (var_index_i == 5 && var_index_j == 1) ) // d^2g/dtdY
	    fout << -1200./w/t_sq/t/R << ' ';
	  else if ( (var_index_i == 2 && var_index_j == 4) ||
		    (var_index_i == 4 && var_index_j == 2) ) // d^2g/dRdX
	    fout << -600./w_sq/t/R_sq << ' ';
	  else if ( (var_index_i == 2 && var_index_j == 5) ||
		    (var_index_i == 5 && var_index_j == 2) ) // d^2g/dRdY
	    fout << -600./w/t_sq/R_sq << ' ';
	  else
	    fout << "0. ";
	}
      }
    }
    else if (num_vars == 4) { // uncertain vars only
      for (size_t i=0; i<num_deriv_vars; i++) {
	int var_index_i = DVV[i] - 1;
	for (size_t j=0; j<num_deriv_vars; j++) {
	  int var_index_j = DVV[j] - 1;
	  if (var_index_i == 0 && var_index_j == 0)          // d^2g/dR^2
	    fout << 2.*stress/pow(R, 3) << ' ';
	  else if ( (var_index_i == 0 && var_index_j == 2) ||
		    (var_index_i == 2 && var_index_j == 0) ) // d^2g/dRdX
	    fout << -600./w_sq/t/R_sq << ' ';
	  else if ( (var_index_i == 0 && var_index_j == 3) ||
		    (var_index_i == 3 && var_index_j == 0) ) // d^2g/dRdY
	    fout << -600./w/t_sq/R_sq << ' ';
	  else
	    fout << "0. ";
	}
      }
    }
    fout << "]]\n";
  }

  // **** d^2c2/dx^2:
  if (ASV[2] & 4) {
    double D5 = 1./sqrt(D2)/D0, D6 = -D1/2./D0/pow(D2,1.5);
    double D7 = sqrt(D2)/D0,    D8 =  D1/2./D0/sqrt(D2);
    double dD2_dX = 2.*X/w_sq/w_sq, dD3_dX = D6*dD2_dX, dD4_dX = D8*dD2_dX;
    double dD2_dY = 2.*Y/t_sq/t_sq, dD3_dY = D6*dD2_dY, dD4_dY = D8*dD2_dY;
    fout << "[[ ";
    if (num_vars == 6) { // normal design + uncertain case
      double dD1_dw = -D1/w, dD2_dw = -4.*X_sq/w_sq/w_sq/w,
	dD3_dw = D5*dD1_dw + D6*dD2_dw, dD4_dw = D7*dD1_dw + D8*dD2_dw;
      double dD1_dt = -D1/t, dD2_dt = -4.*Y_sq/t_sq/t_sq/t,
	dD3_dt = D5*dD1_dt + D6*dD2_dt, dD4_dt = D7*dD1_dt + D8*dD2_dt;
      for (size_t i=0; i<num_deriv_vars; i++) {
	int var_index_i = DVV[i] - 1;
	for (size_t j=0; j<num_deriv_vars; j++) {
	  int var_index_j = DVV[j] - 1;
	  if (var_index_i == 0 && var_index_j == 0)            // d^2g/dw^2
	    fout << D3*10.*X_sq/pow(w_sq,3)
	      - 2.*X_sq/w_sq/w_sq/w*dD3_dw + D4/w_sq - dD4_dw/w << ' ';
	  else if (var_index_i == 1 && var_index_j == 1)       // d^2g/dt^2
	    fout << D3*10.*Y_sq/pow(t_sq,3)
	      - 2.*Y_sq/t_sq/t_sq/t*dD3_dt + D4/t_sq - dD4_dt/t << ' ';
	  else if (var_index_i == 3 && var_index_j == 3) {     // d^2g/dE^2
	    double dD1_dE = -D1/E, dD4_dE = D7*dD1_dE;
	    fout << D4/E/E - dD4_dE/E << ' ';
	  }
	  else if (var_index_i == 4 && var_index_j == 4)       // d^2g/dX^2
	    fout << D3/w_sq/w_sq + X/w_sq/w_sq*dD3_dX << ' ';
	  else if (var_index_i == 5 && var_index_j == 5)       // d^2g/dY^2
	    fout << D3/t_sq/t_sq + Y/t_sq/t_sq*dD3_dY << ' ';
	  else if ( (var_index_i == 0 && var_index_j == 1) ||
		    (var_index_i == 1 && var_index_j == 0) )   // d^2g/dwdt
	    fout << -2.*X_sq/w_sq/w_sq/w*dD3_dt - dD4_dt/w << ' ';
	  else if ( (var_index_i == 0 && var_index_j == 3) ||
		    (var_index_i == 3 && var_index_j == 0) )   // d^2g/dwdE
	    fout << -dD4_dw/E << ' ';
	  else if ( (var_index_i == 0 && var_index_j == 4) ||
		    (var_index_i == 4 && var_index_j == 0) )   // d^2g/dwdX
	    fout << -4.*X*D3/w_sq/w_sq/w + X/w_sq/w_sq*dD3_dw << ' ';
	  else if ( (var_index_i == 0 && var_index_j == 5) ||
		    (var_index_i == 5 && var_index_j == 0) )   // d^2g/dwdY
	    fout << Y/t_sq/t_sq*dD3_dw << ' ';
	  else if ( (var_index_i == 1 && var_index_j == 3) ||
		    (var_index_i == 3 && var_index_j == 1) )   // d^2g/dtdE
	    fout << -dD4_dt/E << ' ';
	  else if ( (var_index_i == 1 && var_index_j == 4) ||
		    (var_index_i == 4 && var_index_j == 1) )   // d^2g/dtdX
	    fout << X/w_sq/w_sq*dD3_dt << ' ';
	  else if ( (var_index_i == 1 && var_index_j == 5) ||
		    (var_index_i == 5 && var_index_j == 1) )   // d^2g/dtdY
	    fout << -4.*Y*D3/t_sq/t_sq/t + Y/t_sq/t_sq*dD3_dt << ' ';
	  else if ( (var_index_i == 3 && var_index_j == 4) ||
		    (var_index_i == 4 && var_index_j == 3) )   // d^2g/dEdX
	    fout << -dD4_dX/E << ' ';
	  else if ( (var_index_i == 3 && var_index_j == 5) ||
		    (var_index_i == 5 && var_index_j == 3) )   // d^2g/dEdY
	    fout << -dD4_dY/E << ' ';
	  else if ( (var_index_i == 4 && var_index_j == 5) ||
		    (var_index_i == 5 && var_index_j == 4) )   // d^2g/dXdY
	    fout << X/w_sq/w_sq*dD3_dY << ' ';
	  else
	    fout << "0. ";
	}
      }
    }
    else if (num_vars == 4) { // uncertain vars only
      for (size_t i=0; i<num_deriv_vars; i++) {
	int var_index_i = DVV[i] - 1;
	for (size_t j=0; j<num_deriv_vars; j++) {
	  int var_index_j = DVV[j] - 1;
	  if (var_index_i == 1 && var_index_j == 1) {          // d^2g/dE^2
	    double dD1_dE = -D1/E, dD4_dE = D7*dD1_dE;
	    fout << D4/E/E - dD4_dE/E << ' ';
	  }
	  else if (var_index_i == 2 && var_index_j == 2)       // d^2g/dX^2
	    fout << D3/w_sq/w_sq + X/w_sq/w_sq*dD3_dX << ' ';
	  else if (var_index_i == 3 && var_index_j == 3)       // d^2g/dY^2
	    fout << D3/t_sq/t_sq + Y/t_sq/t_sq*dD3_dY << ' ';
	  else if ( (var_index_i == 1 && var_index_j == 2) ||
		    (var_index_i == 2 && var_index_j == 1) )   // d^2g/dEdX
	    fout << -dD4_dX/E << ' ';
	  else if ( (var_index_i == 1 && var_index_j == 3) ||
		    (var_index_i == 3 && var_index_j == 1) )   // d^2g/dEdY
	    fout << -dD4_dY/E << ' ';
	  else if ( (var_index_i == 2 && var_index_j == 3) ||
		    (var_index_i == 3 && var_index_j == 2) )   // d^2g/dXdY
	    fout << X/w_sq/w_sq*dD3_dY << ' ';
	  else
	    fout << "0. ";
	}
      }
    }
    fout << "]]\n";
  }

  fout.flush();
  fout.close();  
  return 0;
}
