/*  _______________________________________________________________________

    DAKOTA: Design Analysis Kit for Optimization and Terascale Applications
    Copyright (c) 2006, Sandia National Laboratories.
    This software is distributed under the GNU General Public License.
    For more information, see the README file in the top Dakota directory.
    _______________________________________________________________________ */

//- Class:       ParamStudy
//- Description: Implementation code for the ParamStudy class
//- Owner:       Mike Eldred

#include "system_defs.h"
#include "ParamStudy.H"
#include "ProblemDescDB.H"

static const char rcsId[]="@(#) $Id: ParamStudy.C 5003 2008-05-01 22:43:46Z mseldre $";


namespace Dakota {

ParamStudy::ParamStudy(Model& model): PStudyDACE(model),
  initialPoint(model.continuous_variables()),
  finalPoint(probDescDB.get_drv("method.parameter_study.final_point")),
  pStudyType(probDescDB.get_short("method.parameter_study.type"))
{
  // Extract specification from ProblemDescDB, perform sanity checking, and
  // compute/estimate maxConcurrency.
  switch (pStudyType) {
  case -1: // list_parameter_study
    copy_data(probDescDB.get_drv("method.parameter_study.list_of_points"),
	      listOfPoints, 0, numContinuousVars);
    maxConcurrency *= listOfPoints.length();
    break;
  case 1: // vector_parameter_study (finalPoint & stepLength specification)
    stepLength = probDescDB.get_real("method.parameter_study.step_length");
    if (stepLength <= 0.) {
      Cerr << "\nError: Parameter Study step_length must be positive." << endl;
      abort_handler(-1);
    }
    compute_vector_steps(); // numSteps estimate needed for maxConcurrency
                            // (numSteps recomputed in extract_trends())
    maxConcurrency *= numSteps + 1;
    break;
  case 2: // vector_parameter_study (finalPoint & numSteps specification)
    numSteps = probDescDB.get_int("method.parameter_study.num_steps");
    if ( numSteps < 0 ) {
      Cerr << "\nError: Parameter Study num_steps must be nonnegative." << endl;
      abort_handler(-1);
    }
    maxConcurrency *= numSteps + 1;
    break;
  case 3: // vector_parameter_study (stepVector & numSteps specification)
    stepVector = probDescDB.get_drv("method.parameter_study.step_vector");
    numSteps   = probDescDB.get_int("method.parameter_study.num_steps");
    if (stepVector.length() != numContinuousVars) {
      Cerr << "\nError: Parameter Study step_vector length must be " 
           << numContinuousVars << '.' << endl;
      abort_handler(-1);
    }
    if ( numSteps < 0 ) {
      Cerr << "\nError: Parameter Study num_steps must be nonnegative." << endl;
      abort_handler(-1);
    }
    maxConcurrency *= numSteps + 1;
    break;
  case 4: // centered_parameter_study
    percentDelta = probDescDB.get_real("method.parameter_study.percent_delta");
    deltasPerVariable
      = probDescDB.get_int("method.parameter_study.deltas_per_variable");
    if ( percentDelta <= 0. )
      Cerr << "\nWarning: Parameter Study percent_delta should be greater than "
           << "zero.\n";
    if ( deltasPerVariable < 0 ) {
      Cerr << "\nError: Parameter Study deltas_per_variable must be "
           << "nonnegative." << endl;
      abort_handler(-1);
    }
    maxConcurrency *= 2*deltasPerVariable*numContinuousVars + 1;
    break;
  case 5: // multidim_parameter_study
    variablePartitions = probDescDB.get_dia("method.partitions");
    if (variablePartitions.length() != numContinuousVars) {
      Cerr << "\nError: Parameter Study partitions length must be " 
           << numContinuousVars << '.' << endl;
      abort_handler(-1);
    }
    for (size_t i=0; i<numContinuousVars; i++) {
      maxConcurrency *= variablePartitions[i] + 1;
      if (variablePartitions[i] < 0) {
        Cerr << "\nError: values for Parameter Study partitions must be "
             << "nonnegative." << endl;
        abort_handler(-1);
      }
    }
    break;
  default:
    Cerr << "\nError: bad pStudyType in ParamStudy constructor.\n       "
         << "pStudyType = " << pStudyType << endl;
    abort_handler(-1);
  }
}


ParamStudy::~ParamStudy() { }


void ParamStudy::extract_trends()
{
  // Capture any changes in initialPoint resulting from the strategy layer's
  // passing of best variable info between iterators.  If no such variable 
  // passing has occurred, then this reassignment is merely repetitive of the 
  // one in the ParamStudy constructor.  If there is a final_point 
  // specification, then stepVector and numSteps must be (re)computed.
  initialPoint = iteratedModel.continuous_variables();

  // Run the selected algorithm
  size_t num_evals = 1;
  switch (pStudyType) {
  case -1: // list_parameter_study
    num_evals = listOfPoints.length();
    break;
  case 1: case 2: // vector_parameter_study (finalPoint & stepLength/numSteps)
    // (re)compute stepVector & numSteps for updated initialPoint
    compute_vector_steps();
    // fall through
  case 3: // vector_parameter_study (stepVector & numSteps specification)
    num_evals = numSteps + 1;
    break;
  case 4: // centered_parameter_study
    num_evals = 2*deltasPerVariable*numContinuousVars + 1;
    break;
  case 5: // multidim_parameter_study
    for (size_t i=0; i<numContinuousVars; i++)
      num_evals *= variablePartitions[i] + 1;
    break;
  }

  if (allVariables.length() != num_evals)
    allVariables.reshape(num_evals);
  const pair<short,short>& view = iteratedModel.current_variables().view();
  for (size_t i=0; i<num_evals; i++)
    if (allVariables[i].is_null())
      allVariables[i] = Variables(view); // minimal instantiate-on-the-fly
  if (outputLevel > SILENT_OUTPUT && pStudyType >= 1 && pStudyType <= 4)
    allHeaders.reshape(num_evals);

  switch (pStudyType) {
  case -1: // list_parameter_study
    if (outputLevel > SILENT_OUTPUT)
      Cout << "\nList parameter study for " << listOfPoints.length()
	   << " samples\n\n";
    sample(listOfPoints);
    break;
  case 1: // vector_parameter_study (finalPoint & stepLength specification)
    if (outputLevel > SILENT_OUTPUT)
      Cout << "\nVector parameter study from\n" << initialPoint << "to\n"
           << finalPoint << "with steps of total length " << stepLength<<"\n\n";
    vector_loop(initialPoint, stepVector, numSteps);
    break;
  case 2: // vector_parameter_study (finalPoint & numSteps specification)
    if (outputLevel > SILENT_OUTPUT)
      Cout << "\nVector parameter study from\n" << initialPoint << "to\n"
           << finalPoint << "using " << numSteps << " steps\n\n";
    vector_loop(initialPoint, stepVector, numSteps);
    break;
  case 3: // vector_parameter_study (stepVector & numSteps specification)
    if (outputLevel > SILENT_OUTPUT)
      Cout << "\nVector parameter study for " << numSteps
           << " steps starting from\n" << initialPoint
           << "with a step vector of\n" << stepVector << '\n';
    vector_loop(initialPoint, stepVector, numSteps);
    break;
  case 4: // centered_parameter_study
    if (outputLevel > SILENT_OUTPUT)
      Cout << "\nCentered parameter study for " << deltasPerVariable
           << " steps per side of " << percentDelta << "% each\nwith the "
           << "following center point:\n" << initialPoint << '\n';
    centered_loop(initialPoint, percentDelta, deltasPerVariable);
    break;
  case 5: // multidim_parameter_study
    if (outputLevel > SILENT_OUTPUT)
      Cout << "\nMultidimensional parameter study for variable partitions of\n"
           << variablePartitions << '\n';
    multidim_loop(variablePartitions);
    break;
  default:
    Cerr << "\nError: bad pStudyType in ParamStudy::extract_trends().\n       "
         << "pStudyType = " << pStudyType << endl;
    abort_handler(-1);
  }
}


void ParamStudy::compute_vector_steps()
{
  // vector_parameter_study: compute stepVector and numSteps from initialPoint,
  // finalPoint, and either numSteps or stepLength (pStudyType is 1 or 2).

  size_t i, final_pt_len = finalPoint.length();
  if (final_pt_len != numContinuousVars) {
    Cerr << "\nError: Parameter Study final_point must be of dimension "
         << numContinuousVars << '.' << endl;
    abort_handler(-1);
  }
  stepVector.reshape(numContinuousVars);

  if (pStudyType == 1) {
    // Define stepVector & numSteps from initialPoint, finalPoint, & stepLength
    Real sum_of_squares = 0.;
    for (i=0; i<numContinuousVars; i++)
      sum_of_squares += pow( finalPoint[i]-initialPoint[i], 2);
    if (sum_of_squares > 0.) {
      Real distance = sqrt(sum_of_squares);
      for (i=0; i<numContinuousVars; i++)
        stepVector[i] = (stepLength/distance)*(finalPoint[i]-initialPoint[i]);
      // Truncate with cast - won't go beyond finalPoint.
      numSteps = (int)(distance/stepLength);
    }
    else { // finalPoint is equal to initialPoint
      numSteps = 0;
      stepVector = 0.;
    }
  }
  else if (pStudyType == 2) {
    // Define stepVector from initialPoint, finalPoint, & numSteps
    if (numSteps)
      for (i=0; i<numContinuousVars; i++)
        stepVector[i] = (finalPoint[i]-initialPoint[i])/numSteps;
    else // numSteps == 0 (numSteps<0 case checked above)
      stepVector = 0.; // prevent division by zero
  }
  else {
    Cerr << "\nError: bad pStudyType in ParamStudy::compute_vector_steps().\n"
         << "       pStudyType = " << pStudyType << endl;
    abort_handler(-1);
  }
}


void ParamStudy::sample(const RealVectorArray& list_of_points)
{
  // populate allVariables
  size_t num_samples = list_of_points.length();
  for (size_t i=0; i<num_samples; i++)
    allVariables[i].continuous_variables(list_of_points[i]);
  // perform the evaluations
  evaluate_parameter_sets(iteratedModel, false, true);
}


void ParamStudy::
vector_loop(const RealVector& start, const RealVector& step_vect, 
	    const int& num_steps)
{
  // start     = initial point for the parameter study (start is not really
  //             needed since X is available, but use this API for generality)
  // step_vect = a hard step in X defining magnitude & direction of each step
  // num_steps = number of steps to sample in the study (the number of fn.
  //             evaluations in the study is num_steps + 1 since the initial
  //             point is also evaluated but is not counted as a "step")

  // In the finalPoint/stepLength specification case, the distance between
  // initial and final points may not be evenly divisible by stepLength, in
  // which case the "% along vector" output can't simply use the loop index.
  // In this case, additional variables are needed for use inside the loop.
  // Identification of the direction with max change assures well-behaved
  // numerics.
  size_t i, j, num_vars = start.length(), max_step_index = 0;
  Real max_step = 0.0;
  if (outputLevel > SILENT_OUTPUT && pStudyType == 1) {
    for (j=0; j<num_vars; j++) {
      Real step = fabs(finalPoint[j] - start[j]);
      if (step > max_step) {
        max_step = step;
        max_step_index = j;
      }
    }
  }

  // Loop over num_steps
  RealVector c_vars;
  for (i=0; i<=num_steps; i++) {

    // store each parameter set in allVariables
    if (i == 0)
      c_vars = start;
    else
      for (j=0; j<num_vars; j++)
        c_vars[j] += step_vect[j];
    allVariables[i].continuous_variables(c_vars);

    // store each output header in allHeaders
    if (outputLevel > SILENT_OUTPUT) {
      ostringstream h_stream;
      h_stream.setf(ios::scientific);
      if (asynchFlag)
	h_stream << "\n\n";
      if (num_steps == 0) // Allow num_steps == 0 case
	h_stream << ">>>>> Initial_point only (no steps)\n";
      else if (pStudyType == 1)
	h_stream << ">>>>> Vector parameter study evaluation for "
		 << 100.*fabs(c_vars[max_step_index] - start[max_step_index])
	            /max_step << "% along vector\n";
      else
	h_stream << ">>>>> Vector parameter study evaluation for "
                 << i*100./num_steps << "% along vector\n";
      allHeaders[i] = h_stream.str();
    }
  }

  // evaluate the parameter sets
  evaluate_parameter_sets(iteratedModel, false, true);
}


void ParamStudy::
centered_loop(const RealVector& start, const Real& percent_delta, 
              const int& deltas_per_variable)
{
  int i;
  size_t j, k, num_vars = start.length(), cntr = 0;

  // Always evaluate center point, even if deltas_per_variable = 0
  if (outputLevel > SILENT_OUTPUT) {
    if (asynchFlag)
      allHeaders[cntr] =
        "\n\n>>>>> Centered parameter study evaluation for center point\n";
    else
      allHeaders[cntr] =
        ">>>>> Centered parameter study evaluation for center point\n";
  }
  allVariables[cntr++].continuous_variables(start);

  // Evaluate +/- deltas for each variable
  Real step;
  RealVector c_vars = start;
  for (k=0; k<num_vars; k++) {

    // This check is primarily for start[i]==0.  It also checks for %delta
    // less than the machine epsilon, which should not occur in practice.
    if (fabs(percent_delta/100.) < DBL_EPSILON) {  // offset
      Cerr << "\nError: offset near machine precision.\n       "
           << "Use larger offset value (% delta)" << endl;
      abort_handler(-1);
    }
    else if (fabs(start[k]) < DBL_MIN ) { 
      Cerr << "\nWarning: starting point is less than DBL_MIN or is equal\n"
           << "         to zero.  Using a default step vector.\n";
      step = percent_delta/100.*(0.01*((start[k] >= 0.) ? 1. : -1.));
    }
    else
      step = percent_delta/100.*start[k];

    // Loop over deltas_per_variable
    for (i=-deltas_per_variable; i<=deltas_per_variable; i++) {
      if (i) { // skip the center

        // store each output header in allHeaders
        if (outputLevel > SILENT_OUTPUT) {
          ostringstream h_stream;
          if (asynchFlag)
            h_stream << "\n\n";
          if (i<0)
            h_stream << ">>>>> Centered parameter study evaluation for x["
                     << k+1 << "] - " << -i << "delta:\n";
          else
            h_stream << ">>>>> Centered parameter study evaluation for x["
                     << k+1 << "] + " << i << "delta:\n";
          allHeaders[cntr] = h_stream.str();
        }

	// compute parameter set by adding multiple of step to start
	c_vars[k] = start[k] + i*step;
        allVariables[cntr++].continuous_variables(c_vars);
      }
    }
    c_vars[k] = start[k];
  }

  // evaluate each of the parameter sets in allVariables
  evaluate_parameter_sets(iteratedModel, false, true);
}


void ParamStudy::multidim_loop(const IntArray& var_partitions)
{
  // Perform a multidimensional parameter study based on the number of 
  // partitions specified for each variable.

  size_t i, j, num_vars = var_partitions.length(), num_evals = 1;
  for (i=0; i<num_vars; i++)
    num_evals *= var_partitions[i] + 1;

  // Get bounded region and check for case of default bounds (upper/lower = 
  // +/-DBL_MAX) since this (1) is probably not what the user wants, and 
  // (2) will overflow step_vect
  const RealVector& c_l_bnds = iteratedModel.continuous_lower_bounds();
  const RealVector& c_u_bnds = iteratedModel.continuous_upper_bounds();
  //const IntVector& d_l_bnds = iteratedModel.discrete_lower_bounds();
  //const IntVector& d_u_bnds = iteratedModel.discrete_upper_bounds();
  for (i=0; i<num_vars; i++) {
    if (c_l_bnds[i] <= -DBL_MAX || c_u_bnds[i] >= DBL_MAX) {
      Cerr << "\nError: multidim_parameter_study requires specification of "
	   << "variable bounds." << endl;
      abort_handler(-1);
    }
  }

  RealVector step_vect(num_vars, 0.);
  for (i=0; i<num_vars; i++)
    if (var_partitions[i])
      step_vect[i] = (c_u_bnds[i] - c_l_bnds[i])/var_partitions[i];
  RealVector c_vars(num_vars);
  SizetArray multidim_indices(num_vars, 0);
  for (i=0; i<num_evals; i++) {
    for (j=0; j<num_vars; j++)
      c_vars[j] = c_l_bnds[j] + multidim_indices[j] * step_vect[j];
    allVariables[i].continuous_variables(c_vars);
    // increment the multidimensional index set
    size_t increment_index = 0;
    multidim_indices[increment_index]++;
    while (increment_index < num_vars &&
	   multidim_indices[increment_index] > var_partitions[increment_index]){
      multidim_indices[increment_index] = 0;
      increment_index++;
      if (increment_index < num_vars)
	multidim_indices[increment_index]++;
    }
  }

  // evaluate each of the parameter sets in allVariables
  evaluate_parameter_sets(iteratedModel, false, true);
}

} // namespace Dakota
