/*  _______________________________________________________________________

    DAKOTA: Design Analysis Kit for Optimization and Terascale Applications
    Copyright (c) 2006, Sandia National Laboratories.
    This software is distributed under the GNU General Public License.
    For more information, see the README file in the top Dakota directory.
    _______________________________________________________________________ */

//- Class:       SOLBase
//- Description: Implementation code for the SOLBase class
//- Owner:       Mike Eldred
//- Checked by:

#include "system_defs.h"
#include "DakotaModel.H"
#include "DakotaResponse.H"
#include "SOLBase.H"
#include "DakotaMinimizer.H"
#include "DataMethod.H"

static const char rcsId[]="@(#) $Id: SOLBase.C 5217 2008-08-15 23:49:08Z dmgay $";

#define NPOPTN2_F77 F77_FUNC(npoptn2,NPOPTN2)
extern "C" void NPOPTN2_F77( char* option_string );


namespace Dakota {

SOLBase*   SOLBase::solInstance(NULL);
Minimizer* SOLBase::optLSqInstance(NULL);


SOLBase::SOLBase(Model& model)
{
  // Prevent nesting of an instance of a Fortran iterator within another
  // instance of the same iterator (which would result in data clashes since
  // Fortran does not support object independence).  Recurse through all
  // sub-models and test each sub-iterator for NPSOL/NLSSOL presence.
  // Note: This check is performed for DOT, CONMIN, and SOLBase, but not
  //       for LHS (since it is only active in pre-processing) or SOL
  //       user-functions mode (since there is no model in this case).
  Iterator sub_iterator = model.subordinate_iterator();
  if (!sub_iterator.is_null() && 
       ( sub_iterator.method_name().ends("sol_sqp") ||
	 sub_iterator.uses_method().ends("sol_sqp") ) )
    sub_iterator.method_recourse();
  ModelList& sub_models = model.subordinate_models();
  for (ModelLIter ml_iter = sub_models.begin();
       ml_iter != sub_models.end(); ml_iter++) {
    sub_iterator = ml_iter->subordinate_iterator();
    if (!sub_iterator.is_null() && 
	 ( sub_iterator.method_name().ends("sol_sqp") ||
	   sub_iterator.uses_method().ends("sol_sqp") ) )
      sub_iterator.method_recourse();
  }

  // Use constructor to populate only the problem_db attributes inherited by 
  // NPSOL/NLSSOL from SOLBase (none currently).  For attributes inherited by
  // NPSOL/NLSSOL from the Iterator hierarchy that are needed in SOLBase,
  // it's a bit cleaner/more flexible to have them passed through member
  // function parameter lists rather than re-extracted from problem_db.
  //const ProblemDescDB& problem_db = model.problem_description_db();
}


void SOLBase::
allocate_arrays(const int& num_cv, const size_t& num_nln_con,
		const RealMatrix& lin_ineq_coeffs,
		const RealMatrix& lin_eq_coeffs)
{
  // NPSOL directly handles equality constraints and 1- or 2-sided inequalities
  size_t num_lin_ineq_con = lin_ineq_coeffs.num_rows(),
         num_lin_eq_con   = lin_eq_coeffs.num_rows(),
         num_lin_con      = num_lin_ineq_con + num_lin_eq_con;

  // The Fortran optimizers' need for a nonzero array size is handled with 
  // nlnConstraintArraySize & linConstraintArraySize.
  nlnConstraintArraySize = (num_nln_con) ? num_nln_con : 1;
  linConstraintArraySize = (num_lin_con) ? num_lin_con : 1;

  // Matrix memory passed to Fortran must be contiguous
  linConstraintMatrixF77 = new double[linConstraintArraySize * num_cv];
  upperFactorHessianF77  = new double[num_cv * num_cv];
  constraintJacMatrixF77 = new double[nlnConstraintArraySize * num_cv];

  // Populate linConstraintMatrixF77 with linear coefficients from PDDB. Loop
  // order is reversed (j, then i) since Fortran matrix ordering is reversed 
  // from C ordering (linConstraintMatrixF77 is column-major order: columns
  // arranged head to tail).
  size_t i, j, cntr = 0;
  for (j=0; j<num_cv; j++) { // loop over columns
    for (i=0; i<num_lin_ineq_con; i++) // loop over inequality rows
      linConstraintMatrixF77[cntr++] = lin_ineq_coeffs[i][j];
    for (i=0; i<num_lin_eq_con; i++) // loop over equality rows
      linConstraintMatrixF77[cntr++] = lin_eq_coeffs[i][j];
  }

  boundsArraySize = num_cv + num_lin_con + num_nln_con;
  cLambda.reshape(boundsArraySize);          // clambda[bnd_size]
  constraintState.reshape(boundsArraySize);  // istate[bnd_size]
}


void SOLBase::deallocate_arrays()
{
  // Delete double* matrix allocations
  delete [] linConstraintMatrixF77;
  delete [] upperFactorHessianF77;
  delete [] constraintJacMatrixF77;
}


void SOLBase::allocate_workspace(const int& num_cv, const int& num_nln_con,
                                 const int& num_lin_con, const int& num_lsq)
{
  // see leniw/lenw discussion in "Subroutine npsol" section of NPSOL manual
  // for workspace size requirements.
  intWorkSpaceSize  = 3*num_cv + num_lin_con + 2*num_nln_con;
  if (num_lin_con == 0 && num_nln_con == 0)
    realWorkSpaceSize = 20*num_cv;
  else if (num_nln_con == 0)
    realWorkSpaceSize = 2*num_cv*num_cv + 20*num_cv + 11*num_lin_con;
  else
    realWorkSpaceSize = 2*num_cv*num_cv + num_cv*num_lin_con 
      + 2*num_cv*num_nln_con + 20*num_cv + 11*num_lin_con + 21*num_nln_con;
 
  // in subroutine nlssol() in nlssolsubs.f, subroutine nlloc() adds the
  // following to the result from nploc().
  realWorkSpaceSize += 3*num_lsq + num_lsq*num_cv;

  realWorkSpace.reshape(realWorkSpaceSize);  // work[lwork]
  intWorkSpace.reshape(intWorkSpaceSize);    // iwork[liwork]
}


void SOLBase::set_options(bool speculative_flag, bool vendor_num_grad_flag, 
                          short output_lev,      const int& verify_lev,
                          const Real& fn_prec,   const Real& linesrch_tol,
                          const int& max_iter,   const Real& constr_tol,
                          const Real& conv_tol,  const String& grad_type,
                          const Real& fdss)
{
  // Set NPSOL options (see "Optional Input Parameters" section of NPSOL manual)

  // The subroutine npoptn2 in file npoptn_wrapper.f accepts a string of 
  // length 72 (the max that NPSOL accepts) which is then passed along to
  // the npoptn routine in NPSOL. Therefore, strings passed to npoptn2
  // need to be of length 72, including the \0 terminating character.
 
  // Each of NPSOL's settings is an optional parameter in dakota.input.nspec, 
  // but always assigning them is OK since they are always defined, either from
  // dakota.in or from the default specified in the DataMethod constructor.
  // However, this approach may not use the NPSOL default:

  // If speculative_flag is set and numerical_gradients are used, then check 
  // method_source for dakota setting (behaves like speculative) or vendor 
  // setting (doesn't behave like speculative approach) and output info message
  if (speculative_flag) {
    Cerr << "\nWarning: speculative setting is ignored by SOL methods.";
    if (vendor_num_grad_flag)
      Cerr << "\n         A value-based line search will be used for vendor"
	   << "\n         numerical gradients.\n";
    else
      Cerr << "\n         A gradient-based line-search will be used for "
	   << "\n         analytic and dakota numerical gradients.\n";
  }

  // For each option, set the data width to 72-(length of strings)-1 = 26.
  // Pad the right hand end of the string with white space to ensure success
  // for compilers that have problems with ios formatting (e.g., janus, Linux).
  char verify_string[72];
  ostringstream verify_stream;
  verify_stream <<   "Verify Level                = " << setiosflags(ios::left)
                << setw(26) << verify_lev << "               ";
  string verify_s(verify_stream.str());
  copy(verify_s.begin(), verify_s.end(), verify_string);
  NPOPTN2_F77(verify_string);

  // Default NPSOL function precision is frequently tighter than DAKOTA's 
  // 11-digit precision.  Scaling back NPSOL's precision prevents wasted fn. 
  // evals. which are seeking to discern between design points that appear
  // identical in DAKOTA's precision. Unfortunately, NPSOL's definition of
  // Function Precision is not as simple as number of significant figures (see
  // "Optional Input Parameters" section of NPSOL manual); appropriate precision
  // is linked to the scale of f.  Default ProblemDescDB::functionPrecision is
  // 1e-10, but this may not always be appropriate depending on the scale of f.
  char fnprec_string[72];
  ostringstream fnprec_stream;
  fnprec_stream <<   "Function Precision          = " << setiosflags(ios::left)
                << setw(26) << fn_prec << "               ";
  string fnprec_s(fnprec_stream.str());
  copy(fnprec_s.begin(), fnprec_s.end(), fnprec_string);
  NPOPTN2_F77(fnprec_string);

  char lstol_string[72];
  ostringstream lstol_stream;
  lstol_stream <<    "Linesearch Tolerance        = " << setiosflags(ios::left)
               << setw(26) << linesrch_tol << "               ";
  string lstol_s(lstol_stream.str());
  copy(lstol_s.begin(), lstol_s.end(), lstol_string);
  NPOPTN2_F77(lstol_string);

  char maxiter_string[72];
  ostringstream maxiter_stream;
  maxiter_stream <<  "Major Iteration Limit       = " << setiosflags(ios::left)
                 << setw(26) << max_iter << "               ";
  string maxiter_s(maxiter_stream.str());
  copy(maxiter_s.begin(), maxiter_s.end(), maxiter_string);
  NPOPTN2_F77(maxiter_string);

  if (output_lev > NORMAL_OUTPUT) {
    NPOPTN2_F77(
    "Major Print Level           = 20                                       ");
    Cout << "\nNPSOL option settings:\n----------------------\n" 
         << verify_string << '\n' << "Major Print Level           = 20\n" 
         << fnprec_string << '\n' << lstol_string << '\n' << maxiter_string 
	 << '\n';
  }
  else
    NPOPTN2_F77(
    "Major Print Level           = 10                                       ");

  // assign a nondefault linear/nonlinear constraint tolerance if a valid
  // value has been set in dakota.in; otherwise utilize the NPSOL default.
  if (constr_tol > 0.0) {
    char ct_tol_string[72];
    ostringstream ct_tol_stream;
    ct_tol_stream << "Feasibility Tolerance       = " << setiosflags(ios::left)
		  << setw(26) << constr_tol << "               ";
    string ct_tol_s(ct_tol_stream.str());
    copy(ct_tol_s.begin(), ct_tol_s.end(), ct_tol_string);
    NPOPTN2_F77(ct_tol_string);
    if (output_lev > NORMAL_OUTPUT)
      Cout << ct_tol_string << '\n';
  }

  // conv_tol is an optional parameter in dakota.input.nspec, but
  // defining our own default (in the DataMethod constructor) and
  // always assigning it applies some consistency across methods.
  // Therefore, the NPSOL default is not used.
  char ctol_string[72];
  ostringstream ctol_stream;
  ctol_stream <<     "Optimality Tolerance        = " << setiosflags(ios::left)
	      << setw(26) << conv_tol << "               ";
  string ctol_s(ctol_stream.str());
  copy(ctol_s.begin(), ctol_s.end(), ctol_string);
  NPOPTN2_F77(ctol_string);
  if (output_lev > NORMAL_OUTPUT)
    Cout << ctol_string << "\nNOTE: NPSOL's convergence tolerance is not a "
	 << "relative tolerance.\n      See \"Optimality tolerance\" in "
         << "Optional Input Parameters section of \n      NPSOL manual for "
         << "description.\n";

  // Set Derivative Level = 3 for user-supplied gradients (analytic, dakota 
  // numerical, or mixed analytic/dakota numerical gradients).  This does NOT
  // take advantage of NPSOL's internal mixed gradient capability, but is 
  // simpler, parallelizable, and more consistent with the other optimizers.
  // Inactive functions have been observed in NPSOL's internal mixed gradient
  // mode, so support for NPSOL's internal mixed gradients might add efficiency.
  if ( grad_type == "analytic" || grad_type == "mixed" || 
       ( grad_type == "numerical" && !vendor_num_grad_flag ) ) {
    // user-supplied gradients: Derivative Level = 3
    NPOPTN2_F77(
    "Derivative Level            = 3                                        ");
    if (output_lev > NORMAL_OUTPUT)
      Cout << "Derivative Level            = 3\n";
  }
  else if (grad_type == "none") {
    Cerr << "\nError: gradient type = none is invalid with SOL methods.\n"
         << "Please select numerical, analytic, or mixed gradients." << endl;
    abort_handler(-1);
  }
  else { // vendor numerical gradients: Derivative Level = 0. No forward/central
         // intervalType control, since NPSOL switches automatically.
    NPOPTN2_F77(
    "Derivative Level            = 0                                        ");

    char fdss_string[72], cfdss_string[72];
    ostringstream fdss_stream, cfdss_stream;
    fdss_stream <<   "Difference Interval         = " << setiosflags(ios::left)
                << setw(26) << fdss << "               ";
    string fdss_s(fdss_stream.str());
    copy(fdss_s.begin(), fdss_s.end(), fdss_string);
    NPOPTN2_F77(fdss_string);
  
    // Set "Central Difference Interval" to fdss as well.
    // It may be desirable to set central FDSS to fdss/2. (?)
    cfdss_stream <<  "Central Difference Interval = " << setiosflags(ios::left)
                 << setw(26) << fdss << "               ";
    string cfdss_s(cfdss_stream.str());
    copy(cfdss_s.begin(), cfdss_s.end(), cfdss_string);
    NPOPTN2_F77(cfdss_string);
  
    if (output_lev > NORMAL_OUTPUT)
      Cout << "Derivative Level            = 0\n" << fdss_string << '\n'
	   << cfdss_string << "\nNOTE: NPSOL's finite difference interval uses "
	   << "a unit offset to remove the\n      need for a minimum step "
	   << "specification (see \"Difference interval\" in\n      Optional "
           << "Input Parameters section of NPSOL manual).\n"
	   << "Interval type ignored since NPSOL automatically selects\n"
	   << "and switches between forward and central differences.\n\n";
  }
}


void SOLBase::
augment_bounds(RealVector& augmented_l_bnds, RealVector& augmented_u_bnds,
	       const RealVector& lin_ineq_l_bnds,
	       const RealVector& lin_ineq_u_bnds,
	       const RealVector& lin_eq_targets,
	       const RealVector& nln_ineq_l_bnds,
	       const RealVector& nln_ineq_u_bnds,
	       const RealVector& nln_eq_targets)
{
  // Construct augmented_l_bnds & augmented_u_bnds from variable bounds,
  // linear inequality bounds and equality targets, and nonlinear inequality
  // bounds and equality targets.  Arrays passed in are assumed to already 
  // contain the variable bounds and are augmented with linear and nonlinear
  // constraint bounds.  Note: bounds above or below NPSOL's "Infinite bound
  // size" (see bl/bu in "Subroutine npsol" section and Infinite bound size in
  // "Optional Input Parameters" section of NPSOL manual) are ignored. 
  // DAKOTA's default bounds for no user specification are set in
  // ProblemDescDB::responses_kwhandler for nonlinear constraints and in
  // Constraints::manage_linear_constraints for linear constraints.

  size_t num_cv       = augmented_l_bnds.length(),
         num_lin_ineq = lin_ineq_l_bnds.length(),
         num_lin_eq   = lin_eq_targets.length(),
         num_nln_ineq = nln_ineq_l_bnds.length(),
         num_nln_eq   = nln_eq_targets.length();
  if (boundsArraySize != num_cv + num_lin_ineq + num_lin_eq +
                         num_nln_ineq + num_nln_eq) {
    Cerr << "Error: bad boundsArraySize in SOLBase::augment_bounds." << endl;
    abort_handler(-1);
  }

  augmented_l_bnds.reshape(boundsArraySize); // retains variables data
  augmented_u_bnds.reshape(boundsArraySize); // retains variables data
  size_t i, cntr = num_cv;
  for (i=0; i<num_lin_ineq; i++) { // linear inequality
    augmented_l_bnds[cntr] = lin_ineq_l_bnds[i];
    augmented_u_bnds[cntr] = lin_ineq_u_bnds[i];
    cntr++;
  }
  for (i=0; i<num_lin_eq; i++) { // linear equality
    augmented_l_bnds[cntr] = lin_eq_targets[i];
    augmented_u_bnds[cntr] = lin_eq_targets[i];
    cntr++;
  }
  for (i=0; i<num_nln_ineq; i++) { // nonlinear inequality
    augmented_l_bnds[cntr] = nln_ineq_l_bnds[i];
    augmented_u_bnds[cntr] = nln_ineq_u_bnds[i];
    cntr++;
  }
  for (i=0; i<num_nln_eq; i++) { // nonlinear equality
    augmented_l_bnds[cntr] = nln_eq_targets[i];
    augmented_u_bnds[cntr] = nln_eq_targets[i];
    cntr++;
  }
}


void SOLBase::
constraint_eval(int& mode, int& ncnln, int& n, int& nrowj, int* needc,
                double* x, double* c, double* cjac, int& nstate)
{
  // This routine is called before objective_eval, but is only called if
  // there are nonlinear constraints (numNonlinearConstraints != 0)
  
  // Constraint requests are governed by "mode" and "needc":
  //   SOL mode=2: get active values and gradients using needc
  //   SOL mode=1: get active gradients using needc
  //   SOL mode=0: get active values using needc

  // Adjust SOL's mode requests for case of vendor numerical gradients.
  // [SOL's requests are valid since Derivative Level=0 means that SOME
  // elements of the objFunctionGradient & Jacobian are unavailable. However,
  // Dakota's numerical gradient mode currently assumes that NO gradient data
  // is available, which means that some SOL requests need to be modified.
  // If mixed _vendor_ gradients are supported in a future Dakota version,
  // these adjustments should be removed.]
  int i;
  short asv_request = mode + 1;
  //if ( !(solInstance->derivLevel & 2) && (asv_request & 2) ) { // more general
  if (optLSqInstance->vendorNumericalGradFlag && asv_request & 2) {
    asv_request -= 2; // downgrade request
    Cout << "SOL has requested user-supplied constraint gradients for case of "
         << "vendor numerical gradients.\n";
    if (asv_request)
      Cout << "Request will be downgraded to function values alone.\n" << endl;
    else
      Cout << "Request will be ignored and no evaluation performed.\n" << endl;
  }

  if (asv_request) {
    // Generate local_asv from asv_request (not mode) to accomodate special case
    // behavior. If needc[i]>0, then make asv_request request for constraint i.
    // For local_asv[0], assume the same request for the obj. fn.  NOTE: This 
    // would be wasteful in a future implementation for mixed gradients, since
    // it is possible to have requests for FD evaluation of constraint values
    // only or of the obj. fn. only (Derivative Level = 0, 1, or 2).
    ShortArray local_asv(solInstance->constrOffset + ncnln);
    for (i=0; i<solInstance->constrOffset; i++)
      local_asv[i] = asv_request;
    for (i=0; i<ncnln; i++)
      local_asv[i+solInstance->constrOffset] = (needc[i] > 0) ? asv_request : 0;
      // NOTE: SOL does not appear to use needc[i] for active set requests
      // (SOL always needs all gradients, even for constraints which are
      // inactive by a lot). So far, needc[i]=0 has only been observed for
      // partial finite difference requests in the case of mixed gradients.

    // Update model variables from x for use in compute_response()
    RealVector local_des_vars(n);
    copy_data(x, n, local_des_vars);
    optLSqInstance->iteratedModel.continuous_variables(local_des_vars);

    optLSqInstance->activeSet.request_vector(local_asv);
    optLSqInstance->
      iteratedModel.compute_response(optLSqInstance->activeSet);
    solInstance->fnEvalCntr++;
  }
  
  // Follow asv_request request (special case exceptions handled above).
  // Could follow local_asv entry-by-entry, but the design below is 
  // easier since the Response structure matches that needed by SOL.
  const Response& local_response
    = optLSqInstance->iteratedModel.current_response();
  if (asv_request & 1) {
    // Direct use of the array class assignment operator works
    // fine in DOTOptimizer, but causes major memory problems in npsol!
    // So use a more "brute force" approach to assign the constraint values.
    const RealVector& local_fn_vals = local_response.function_values();
    for (i=0; i<ncnln; i++)
      c[i] = local_fn_vals[i+solInstance->constrOffset];
  }

  if (asv_request & 2) {
    const RealMatrix& local_fn_grads = local_response.function_gradients();

    // No need to concatenate active gradients (as in DOT).  Response population
    // mechanism sets inactive constraint entries to zero which works fine with
    // SOL so long as the gradients are either exclusively user-supplied or
    // exclusively SOL-computed (with finite differences).  In the former case
    // (Der. Level = 3), *all* of cjac is specified and inactive entries are
    // ignored, and in the latter case (Der. Level = 0), this code is not used
    // and cjac is left at its special value (see "Constant Jacobian elements"
    // section in NPSOL manual). For a future implementation of mixed _vendor_
    // gradients (some user-supplied, some not), unavailable cjac elements must
    // *not* be set to zero since SOL uses detection of the special value to
    // determine which cjac entries require finite differencing.
      
    // Loop order is reversed (j, then i) since Fortran matrix ordering is
    // reversed from C ordering. Gradients are not mixed, so assign each
    // entry of cjac (inactive entries will be zero, but SOL ignores them).
    int j, cntr = 0;
    for (j=0; j<n; j++)
      for (i=solInstance->constrOffset; i<solInstance->constrOffset+ncnln; i++)
	cjac[cntr++] = local_fn_grads[i][j];
  }
}

} // namespace Dakota
