/*  _______________________________________________________________________

    DAKOTA: Design Analysis Kit for Optimization and Terascale Applications
    Copyright (c) 2006, Sandia National Laboratories.
    This software is distributed under the GNU General Public License.
    For more information, see the README file in the top Dakota directory.
    _______________________________________________________________________ */

//- Class:        DakotaConstraints
//- Description:  Class implementation
//- Owner:        Mike Eldred

#include "system_defs.h"
#include "ProblemDescDB.H"
#include "DakotaConstraints.H"
#include "AllConstraints.H"
#include "MergedConstraints.H"
#include "DistinctConstraints.H"

static const char rcsId[]="@(#) $Id";

namespace Dakota {


/** This constructor is the one which must build the base class data
    for all derived classes.  get_constraints() instantiates a derived
    class letter and the derived constructor selects this base class
    constructor in its initialization list (to avoid recursion in the
    base class constructor calling get_constraints() again).  Since
    the letter IS the representation, its rep pointer is set to NULL
    (an uninitialized pointer causes problems in ~Constraints). */
Constraints::
Constraints(BaseConstructor, const ProblemDescDB& problem_db,
	    const pair<short,short>& view): variablesView(view),
  numNonlinearIneqCons(
    problem_db.get_sizet("responses.num_nonlinear_inequality_constraints")),
  nonlinearIneqConLowerBnds(
    problem_db.get_drv("responses.nonlinear_inequality_lower_bounds")),
  nonlinearIneqConUpperBnds(
    problem_db.get_drv("responses.nonlinear_inequality_upper_bounds")),
  numNonlinearEqCons(
    problem_db.get_sizet("responses.num_nonlinear_equality_constraints")),
  nonlinearEqConTargets(
    problem_db.get_drv("responses.nonlinear_equality_targets")), 
  numLinearIneqCons(0), numLinearEqCons(0),
  linearIneqConLowerBnds(
    problem_db.get_drv("method.linear_inequality_lower_bounds")),
  linearIneqConUpperBnds(
    problem_db.get_drv("method.linear_inequality_upper_bounds")),
  linearEqConTargets(
    problem_db.get_drv("method.linear_equality_targets")),
  constraintsRep(NULL), referenceCount(1)
{
#ifdef REFCOUNT_DEBUG
  Cout << "Constraints::Constraints(BaseConstructor) called to build base "
       << "class data for letter object." << endl;
#endif
}


/** The default constructor: constraintsRep is NULL in this case (a
    populated problem_db is needed to build a meaningful Constraints
    object).  This makes it necessary to check for NULL in the copy
    constructor, assignment operator, and destructor. */
Constraints::Constraints(): constraintsRep(NULL), referenceCount(1)
{
#ifdef REFCOUNT_DEBUG
  Cout << "Constraints::Constraints() called to build empty "
       << "variable constraints object." << endl;
#endif
}


/** The envelope constructor only needs to extract enough data to
    properly execute get_constraints, since the constructor
    overloaded with BaseConstructor builds the actual base class data
    inherited by the derived classes. */
Constraints::
Constraints(const ProblemDescDB& problem_db, const pair<short,short>& view):
  referenceCount(1) // not used since this is the envelope, not the letter
{
#ifdef REFCOUNT_DEBUG
  Cout << "Constraints::Constraints(ProblemDescDB&, const pair<short,short>&) "
       << "called to instantiate envelope." << endl;
#endif

  // Set the rep pointer to the appropriate variable constraints type
  constraintsRep = get_constraints(problem_db, view);
  if ( !constraintsRep ) // bad type or insufficient memory
    abort_handler(-1);
}


/** Initializes constraintsRep to the appropriate derived type, as given
    by the variables view. */
Constraints* Constraints::
get_constraints(const ProblemDescDB& problem_db, const pair<short,short>& view)
{
#ifdef REFCOUNT_DEBUG
  Cout << "Envelope instantiating letter in get_constraints(ProblemDescDB&)."
       << endl;
#endif

  int active_view = view.first;
  if (active_view == MERGED_DISTINCT_DESIGN ||
      active_view == MERGED_DISTINCT_UNCERTAIN ||
      active_view == MERGED_DISTINCT_STATE)
    return new MergedConstraints(problem_db, view);
  else if (active_view == MIXED_ALL)
    return new AllConstraints(problem_db, view);
  // The AllMerged case has been deleted since this combination is unlikely
  // (integer relaxation is only envisioned for optimization).
  //else if (active_view == MERGED_ALL)
  //  return new AllMergedConstraints(problem_db, view);
  else if (active_view == MIXED_DISTINCT_DESIGN ||
	   active_view == MIXED_DISTINCT_UNCERTAIN ||
	   active_view == MIXED_DISTINCT_STATE)
    return new DistinctConstraints(problem_db, view);
  else {
    Cerr << "Constraints active view " << active_view
	 << " not currently supported in derived Constraints classes." << endl;
    return NULL;
  }
}


/** Envelope constructor for instantiations on the fly. */
Constraints::Constraints(const pair<short,short>& view):
  referenceCount(1) // not used since this is the envelope, not the letter
{
#ifdef REFCOUNT_DEBUG
  Cout << "Constraints::Constraints(const pair<short,short>&) called "
       << "to instantiate envelope." << endl;
#endif

  // Set the rep pointer to the appropriate variable constraints type
  constraintsRep = get_constraints(view);
  if ( !constraintsRep ) // bad type or insufficient memory
    abort_handler(-1);
}


/** Initializes constraintsRep to the appropriate derived type, as
    given by the variables view. The default derived class constructors
    are invoked. */
Constraints* Constraints::get_constraints(const pair<short,short>& view) const
{
#ifdef REFCOUNT_DEBUG
  Cout << "Envelope instantiating letter in get_constraints(pair<short,short>&)"
       << endl;
#endif

  int active_view = view.first;
  Constraints* con;
  if (active_view == MERGED_DISTINCT_DESIGN ||
      active_view == MERGED_DISTINCT_UNCERTAIN ||
      active_view == MERGED_DISTINCT_STATE)
    con = new MergedConstraints();
  else if (active_view == MIXED_ALL)
    con = new AllConstraints();
  // The AllMerged case has been deleted since this combination is unlikely
  // (discrete variable relaxation is only envisioned for optimization; perhaps
  // global surrogate-based branch and bound would require it).
  //else if (active_view == MERGED_ALL)
  //  return new AllMergedConstraints(problem_db, view);
  else if (active_view == MIXED_DISTINCT_DESIGN ||
	   active_view == MIXED_DISTINCT_UNCERTAIN ||
	   active_view == MIXED_DISTINCT_STATE)
    con = new DistinctConstraints();
  else {
    Cerr << "Constraints active view " << active_view
	 << " not currently supported in derived Constraints classes." << endl;
    con = NULL;
  }
  if (con)
    con->variablesView = view;
  return con;
}


/** Copy constructor manages sharing of constraintsRep and incrementing
    of referenceCount. */
Constraints::Constraints(const Constraints& con)
{
  // Increment new (no old to decrement)
  constraintsRep = con.constraintsRep;
  if (constraintsRep) // Check for an assignment of NULL
    constraintsRep->referenceCount++;

#ifdef REFCOUNT_DEBUG
  Cout << "Constraints::Constraints(Constraints&)" << endl;
  if (constraintsRep)
    Cout << "constraintsRep referenceCount = " << constraintsRep->referenceCount
	 << endl;
#endif
}


/** Assignment operator decrements referenceCount for old constraintsRep,
    assigns new constraintsRep, and increments referenceCount for new
    constraintsRep. */
Constraints Constraints::operator=(const Constraints& con)
{
  if (constraintsRep != con.constraintsRep) { // normal case: old != new
    // Decrement old
    if (constraintsRep) // Check for NULL
      if ( --constraintsRep->referenceCount == 0 ) 
	delete constraintsRep;
    // Assign and increment new
    constraintsRep = con.constraintsRep;
    if (constraintsRep) // Check for NULL
      constraintsRep->referenceCount++;
  }
  // else if assigning same rep, then do nothing since referenceCount
  // should already be correct

#ifdef REFCOUNT_DEBUG
  Cout << "Constraints::operator=(Constraints&)" << endl;
  if (constraintsRep)
    Cout << "constraintsRep referenceCount = " << constraintsRep->referenceCount
	 << endl;
#endif

  return *this; // calls copy constructor since returned by value
}


/** Destructor decrements referenceCount and only deletes constraintsRep
    when referenceCount reaches zero. */
Constraints::~Constraints()
{ 
  // Check for NULL pointer 
  if (constraintsRep) {
    --constraintsRep->referenceCount;
#ifdef REFCOUNT_DEBUG
    Cout << "constraintsRep referenceCount decremented to " 
         << constraintsRep->referenceCount << endl;
#endif
    if (constraintsRep->referenceCount == 0) {
#ifdef REFCOUNT_DEBUG
      Cout << "deleting constraintsRep" << endl;
#endif
      delete constraintsRep;
    }
  }
}


void Constraints::read(istream& s)
{
  if (constraintsRep)
    constraintsRep->read(s); // envelope fwd to letter
  else { // letter lacking redefinition of virtual fn.!
    Cerr << "Error: Letter lacking redefinition of virtual read function.\n"
         << "No default defined at base class." << endl;
    abort_handler(-1);
  }
}


void Constraints::write(ostream& s) const
{
  if (constraintsRep)
    constraintsRep->write(s); // envelope fwd to letter
  else { // letter lacking redefinition of virtual fn.!
    Cerr << "Error: Letter lacking redefinition of virtual write function.\n"
         << "No default defined at base class." << endl;
    abort_handler(-1);
  }
}


/** Deep copies are used for history mechanisms such as bestVariables
    and data_pairs since these must catalogue copies (and should not
    change as the representation within currentVariables changes). */
Constraints Constraints::copy() const
{
  // the envelope class instantiates a new envelope and a new letter and copies
  // current attributes into the new objects.

#ifdef REFCOUNT_DEBUG
  Cout << "Constraints::copy() called to generate a deep copy with no "
       << "representation sharing." << endl;
#endif

  Constraints con; // new envelope: referenceCount=1, constraintsRep=NULL

  if (constraintsRep) {
    // new letter: allocate a constraintsRep
    con.constraintsRep = get_constraints(constraintsRep->variablesView);

    // Copy attributes used by all letters (so code not replicated in copy_rep)
    con.constraintsRep->variablesView      = constraintsRep->variablesView;
    con.constraintsRep->numNonlinearIneqCons
      = constraintsRep->numNonlinearIneqCons;
    con.constraintsRep->numNonlinearEqCons = constraintsRep->numNonlinearEqCons;
    con.constraintsRep->nonlinearIneqConLowerBnds
      = constraintsRep->nonlinearIneqConLowerBnds;
    con.constraintsRep->nonlinearIneqConUpperBnds
      = constraintsRep->nonlinearIneqConUpperBnds;
    con.constraintsRep->nonlinearEqConTargets
      = constraintsRep->nonlinearEqConTargets;
    con.constraintsRep->numLinearIneqCons  = constraintsRep->numLinearIneqCons;
    con.constraintsRep->numLinearEqCons    = constraintsRep->numLinearEqCons;
    con.constraintsRep->linearIneqConCoeffs
      = constraintsRep->linearIneqConCoeffs;
    con.constraintsRep->linearEqConCoeffs  = constraintsRep->linearEqConCoeffs;
    con.constraintsRep->linearIneqConLowerBnds
      = constraintsRep->linearIneqConLowerBnds;
    con.constraintsRep->linearIneqConUpperBnds
      = constraintsRep->linearIneqConUpperBnds;
    con.constraintsRep->linearEqConTargets = constraintsRep->linearEqConTargets;

    // Copy derived class attributes
    con.copy_rep(constraintsRep);
  }

  return con;
}


void Constraints::copy_rep(const Constraints* con_rep)
{
  if (constraintsRep)
    constraintsRep->copy_rep(con_rep);
  else { // letter lacking redefinition of virtual fn.!
    Cerr << "Error: Letter lacking redefinition of virtual copy_rep function.\n"
         << "       No default defined at base class." << endl;
    abort_handler(-1);
  }
}


/** Resizes the linear and nonlinear constraint arrays at the base
    class.  Does NOT currently resize the derived bounds arrays. */
void Constraints::
reshape(const size_t& num_nln_ineq_cons, const size_t& num_nln_eq_cons,
	const size_t& num_lin_ineq_cons, const size_t& num_lin_eq_cons)
      //const Sizet2DArray& vars_comps)
{
  if (constraintsRep) // envelope
    constraintsRep->reshape(num_nln_ineq_cons, num_nln_eq_cons,
			    num_lin_ineq_cons, num_lin_eq_cons);
  else { // base class implementation for letter

    // Reshape attributes used by all letters
    numNonlinearIneqCons = num_nln_ineq_cons;
    nonlinearIneqConLowerBnds.reshape(num_nln_ineq_cons);
    nonlinearIneqConUpperBnds.reshape(num_nln_ineq_cons);

    numNonlinearEqCons = num_nln_eq_cons;
    nonlinearEqConTargets.reshape(num_nln_eq_cons);

    numLinearIneqCons = num_lin_ineq_cons;
    linearIneqConLowerBnds.reshape(num_lin_ineq_cons);
    linearIneqConUpperBnds.reshape(num_lin_ineq_cons);
    linearIneqConCoeffs.reshape(num_lin_ineq_cons);
    //size_t num_vars = num_cv + num_dv;
    //linearIneqConCoeffs.reshape_2d(num_lin_ineq_cons, num_vars);

    numLinearEqCons = num_lin_eq_cons;
    linearEqConTargets.reshape(num_lin_eq_cons);
    linearEqConCoeffs.reshape(num_lin_eq_cons);
    //linearEqConCoeffs.reshape_2d(num_lin_eq_cons, num_vars);

    // reshape attributes specific to letter classes
    //reshape_rep(vars_comps);
  }
}


/** Resizes the derived bounds arrays. */
void Constraints::reshape(const Sizet2DArray& vars_comps)
{
  if (constraintsRep) { // envelope
    //constraintsRep->variablessComponents = vars_comps;
    constraintsRep->reshape_rep(vars_comps);
  }
  else { // letter
    //variablesComponents = vars_comps;
    reshape_rep(vars_comps);
  }
}


void Constraints::reshape_rep(const Sizet2DArray& vars_comps)
{
  if (constraintsRep)
    constraintsRep->reshape_rep(vars_comps);
  else { // letter lacking redefinition of virtual fn.!
    Cerr << "Error: Letter lacking redefinition of virtual reshape_rep function"
	 << ".\n       No default defined at base class." << endl;
    abort_handler(-1);
  }
}


/** Convenience function called from derived class constructors.  The
    number of variables active for applying linear constraints is
    currently defined to be the number of active continuous variables
    plus the number of active discrete variables (the most general
    case), even though very few optimizers can currently support mixed
    variable linear constraints. */
void Constraints::manage_linear_constraints(const ProblemDescDB& problem_db)
{
  const RealVector& linear_ineq_cons
    = problem_db.get_drv("method.linear_inequality_constraints");
  const RealVector& linear_eq_cons
    = problem_db.get_drv("method.linear_equality_constraints");
  size_t lin_ineq_len = linear_ineq_cons.length(),
         lin_eq_len   = linear_eq_cons.length();
  // invoke virtual functions to get the number of active variables
  // to which the linear constraints are applied.
  size_t num_vars = continuous_lower_bounds().length()
                  + discrete_lower_bounds().length();

  if (lin_ineq_len || lin_eq_len) { // check sanity of inputs
    // check on num_vars is embedded so that, if there are no linear
    // constraints, the error is managed downstream within the iterators
    if (num_vars == 0) {
      Cerr << "Error: no active variable bounds in Constraints::"
	   << "manage_linear_constraints()." << endl;
      abort_handler(-1);
    }
    else if (lin_ineq_len%num_vars || lin_eq_len%num_vars) {
      Cerr << "Error: number of terms in linear constraint specification not "
	   << "evenly\n       divisible by " << num_vars << " variables."<<endl;
      abort_handler(-1);
    }
  }

  if (lin_ineq_len) {
    numLinearIneqCons = lin_ineq_len/num_vars;
    copy_data(linear_ineq_cons, linearIneqConCoeffs, numLinearIneqCons,
	      num_vars);

    size_t i, len_lower_bnds = linearIneqConLowerBnds.length(),
      len_upper_bnds = linearIneqConUpperBnds.length();
    if (!len_lower_bnds) {
      linearIneqConLowerBnds.reshape(numLinearIneqCons);
      linearIneqConLowerBnds = -DBL_MAX; // default lower bounds
    }
    else if (len_lower_bnds != numLinearIneqCons) {
      Cerr << "Error: length of linear inequality lower bounds specification "
           << "not equal to\n       number of linear inequality constraints."
           << endl;
      abort_handler(-1);
    }
    if (!len_upper_bnds) {
      linearIneqConUpperBnds.reshape(numLinearIneqCons);
      linearIneqConUpperBnds = 0.0; // default upper bounds
    }
    else if (len_upper_bnds != numLinearIneqCons) {
      Cerr << "Error: length of linear inequality upper bounds specification "
           << "not equal to\n       number of linear inequality constraints."
           << endl;
      abort_handler(-1);
    }
    // Sanity check on bounds (prevents subtle specification error resulting
    // from specifying positive lower bounds and relying on the default upper
    // bounds of 0).
    for (i=0; i<numLinearIneqCons; i++) {
      if (linearIneqConLowerBnds[i] > linearIneqConUpperBnds[i]) {
	Cerr << "Error: linear inequality lower bound values must be less than "
	     << "or equal to\n       linear inequality upper bound values."
	     << endl;
	abort_handler(-1);
      }
    }
  }
  if (lin_eq_len) {
    numLinearEqCons = lin_eq_len/num_vars;
    copy_data(linear_eq_cons, linearEqConCoeffs, numLinearEqCons, num_vars);

    size_t len_targets = linearEqConTargets.length();
    if (!len_targets) {
      linearEqConTargets.reshape(numLinearEqCons);
      linearEqConTargets = 0.0;
    }
    else if (len_targets != numLinearEqCons) {
      Cerr << "Error: length of linear equality targets specification not "
           << "equal to\n       number of linear equality constraints." << endl;
      abort_handler(-1);
    }
  }
}


const RealVector& Constraints::continuous_lower_bounds() const
{
  if (!constraintsRep) { // letter lacking redefinition of virtual fn.!
    Cerr << "Error: Letter lacking redefinition of virtual continuous_lower_"
         << "bounds() function.\n       No default defined at base class."
         << endl;
    abort_handler(-1);
  }

  return constraintsRep->continuous_lower_bounds(); // envelope -> letter
}


void Constraints::continuous_lower_bounds(const RealVector& c_l_bnds)
{
  if (constraintsRep)
    constraintsRep->continuous_lower_bounds(c_l_bnds); // envelope -> letter
  else { // letter lacking redefinition of virtual fn.!
    Cerr << "Error: Letter lacking redefinition of virtual continuous_lower_"
         << "bounds(RealVector) function.\n       "
	 << "No default defined at base class." << endl;
    abort_handler(-1);
  }
}


const RealVector& Constraints::continuous_upper_bounds() const
{
  if (!constraintsRep) { // letter lacking redefinition of virtual fn.!
    Cerr << "Error: Letter lacking redefinition of virtual continuous_upper_"
         << "bounds() function.\n       No default defined at base class."
         << endl;
    abort_handler(-1);
  }
  
  return constraintsRep->continuous_upper_bounds(); // envelope -> letter
}


void Constraints::continuous_upper_bounds(const RealVector& c_u_bnds)
{
  if (constraintsRep)
    constraintsRep->continuous_upper_bounds(c_u_bnds); // envelope -> letter
  else { // letter lacking redefinition of virtual fn.!
    Cerr << "Error: Letter lacking redefinition of virtual continuous_upper_"
         << "bounds(RealVector) function.\n       "
	 << "No default defined at base class." << endl;
    abort_handler(-1);
  }
}


const IntVector& Constraints::discrete_lower_bounds() const
{
  if (constraintsRep)
    return constraintsRep->discrete_lower_bounds(); // envelope -> letter
  else // default definition for letters lacking redefinition (merged types)
    return emptyIntVector;
}


void Constraints::discrete_lower_bounds(const IntVector& d_l_bnds)
{
  if (constraintsRep)
    constraintsRep->discrete_lower_bounds(d_l_bnds); // envelope -> letter
  else if (!d_l_bnds.empty()) {
    // default definition for letters lacking redefinition (merged types)
    Cerr << "Error: There are no active discrete lower bounds in this "
	 << "Constraints view.\n       Setting this array is erroneous."<< endl;
    abort_handler(-1);
  }
}


const IntVector& Constraints::discrete_upper_bounds() const
{
  if (constraintsRep)
    return constraintsRep->discrete_upper_bounds(); // envelope -> letter
  else // default definition for letters lacking redefinition (merged types)
    return emptyIntVector;
}


void Constraints::discrete_upper_bounds(const IntVector& d_u_bnds)
{
  if (constraintsRep)
    constraintsRep->discrete_upper_bounds(d_u_bnds); // envelope -> letter
  else if (!d_u_bnds.empty()) {
    // default definition for letters lacking redefinition (merged types)
    Cerr << "Error: There are no active discrete upper bounds in this "
	 << "Constraints view.\n       Setting this array is erroneous."<< endl;
    abort_handler(-1);
  }
}


const RealVector& Constraints::inactive_continuous_lower_bounds() const
{
  if (constraintsRep) // envelope -> letter
    return constraintsRep->inactive_continuous_lower_bounds();
  else // default definition for letters lacking redefinition of virtual fn.
    return emptyRealVector;
}


void Constraints::
inactive_continuous_lower_bounds(const RealVector& i_c_l_bnds)
{
  if (constraintsRep) // envelope -> letter
    constraintsRep->inactive_continuous_lower_bounds(i_c_l_bnds);
  else if (!i_c_l_bnds.empty()) {
    // default definition for letters lacking redefinition of virtual fn.
    Cerr << "Error: There are no inactive continuous lower bounds in this "
	 << "Constraints view.\n       Setting this array is erroneous."<< endl;
    abort_handler(-1);
  }
}


const RealVector& Constraints::inactive_continuous_upper_bounds() const
{
  if (constraintsRep) // envelope -> letter
    return constraintsRep->inactive_continuous_upper_bounds();
  else // default definition for letters lacking redefinition of virtual fn.
    return emptyRealVector;
}


void Constraints::
inactive_continuous_upper_bounds(const RealVector& i_c_u_bnds)
{
  if (constraintsRep) // envelope -> letter
    constraintsRep->inactive_continuous_upper_bounds(i_c_u_bnds);
  else if (!i_c_u_bnds.empty()) { 
    // default definition for letters lacking redefinition of virtual fn.
    Cerr << "Error: There are no inactive continuous upper bounds in this "
	 << "Constraints view.\n       Setting this array is erroneous."<< endl;
    abort_handler(-1);
  }
}


const IntVector& Constraints::inactive_discrete_lower_bounds() const
{
  if (constraintsRep) // envelope -> letter
    return constraintsRep->inactive_discrete_lower_bounds();
  else // default definition for letters lacking redefinition of virtual fn.
    return emptyIntVector;
}


void Constraints::inactive_discrete_lower_bounds(const IntVector& i_d_l_bnds)
{
  if (constraintsRep) // envelope -> letter
    constraintsRep->inactive_discrete_lower_bounds(i_d_l_bnds);
  else if (!i_d_l_bnds.empty()) {
    // default definition for letters lacking redefinition of virtual fn.
    Cerr << "Error: There are no inactive discrete lower bounds in this "
	 << "Constraints view.\n       Setting this array is erroneous."<< endl;
    abort_handler(-1);
  }
}


const IntVector& Constraints::inactive_discrete_upper_bounds() const
{
  if (constraintsRep) // envelope -> letter
    return constraintsRep->inactive_discrete_upper_bounds();
  else // default definition for letters lacking redefinition of virtual fn.
    return emptyIntVector;
}


void Constraints::inactive_discrete_upper_bounds(const IntVector& i_d_u_bnds)
{
  if (constraintsRep) // envelope -> letter
    constraintsRep->inactive_discrete_upper_bounds(i_d_u_bnds);
  else if (!i_d_u_bnds.empty()) {
    // default definition for letters lacking redefinition of virtual fn.
    Cerr << "Error: There are no inactive discrete upper bounds in this "
	 << "Constraints view.\n       Setting this array is erroneous."<< endl;
    abort_handler(-1);
  }
}


RealVector Constraints::all_continuous_lower_bounds() const
{
  if (!constraintsRep) { // letter lacking redefinition of virtual fn.!
    Cerr << "Error: Letter lacking redefinition of virtual all_continuous_lower"
	 << "_bounds() get function.\n       No default defined at base class."
	 << endl;
    abort_handler(-1);
  }
  
  // envelope -> letter 
  return constraintsRep->all_continuous_lower_bounds();
}


void Constraints::all_continuous_lower_bounds(const RealVector& a_c_l_bnds)
{
  if (constraintsRep) // envelope -> letter 
    constraintsRep->all_continuous_lower_bounds(a_c_l_bnds);
  else { // letter lacking redefinition of virtual fn.!
    Cerr << "Error: Letter lacking redefinition of virtual all_continuous_lower"
	 << "_bounds() set function.\n       No default defined at base class."
	 << endl;
    abort_handler(-1);
  }
}


RealVector Constraints::all_continuous_upper_bounds() const
{
  if (!constraintsRep) { // letter lacking redefinition of virtual fn.!
    Cerr << "Error: Letter lacking redefinition of virtual all_continuous_upper"
	 << "_bounds() get function.\n       No default defined at base class."
	 << endl;
    abort_handler(-1);
  }
  
  // envelope -> letter
  return constraintsRep->all_continuous_upper_bounds();
}


void Constraints::all_continuous_upper_bounds(const RealVector& a_c_u_bnds)
{
  if (constraintsRep) // envelope -> letter
    constraintsRep->all_continuous_upper_bounds(a_c_u_bnds);
  else { // letter lacking redefinition of virtual fn.!
    Cerr << "Error: Letter lacking redefinition of virtual all_continuous_upper"
	 << "_bounds() set function.\n       No default defined at base class."
	 << endl;
    abort_handler(-1);
  }
}


IntVector Constraints::all_discrete_lower_bounds() const
{
  if (constraintsRep) // envelope -> letter
    return constraintsRep->all_discrete_lower_bounds();
  else // default definition for letters lacking redefinition (merged types)
    return emptyIntVector;
}


void Constraints::all_discrete_lower_bounds(const IntVector& a_d_l_bnds)
{
  if (constraintsRep) // envelope -> letter
    constraintsRep->all_discrete_lower_bounds(a_d_l_bnds);
  else if (!a_d_l_bnds.empty()) {
    // default definition for letters lacking redefinition (merged types)
    Cerr << "Error: all discrete lower bounds is not supported in this "
	 << "variables view.\n       Setting this array is erroneous." << endl;
    abort_handler(-1);
  }
}


IntVector Constraints::all_discrete_upper_bounds() const
{
  if (constraintsRep) // envelope -> letter
    return constraintsRep->all_discrete_upper_bounds();
  else // default definition for letters lacking redefinition (merged types)
    return emptyIntVector;
}


void Constraints::all_discrete_upper_bounds(const IntVector& a_d_u_bnds)
{
  if (constraintsRep) // envelope -> letter
    constraintsRep->all_discrete_upper_bounds(a_d_u_bnds);
  else if (!a_d_u_bnds.empty()) {
    // default definition for letters lacking redefinition (merged types)
    Cerr << "Error: all discrete upper bounds is not supported in this "
	 << "variables view.\n       Setting this array is erroneous." << endl;
    abort_handler(-1);
  }
}

} // namespace Dakota
