/*  _________________________________________________________________________
 *
 *  Coliny: A Library of COLIN optimizers
 *  Copyright (c) 2003, Sandia National Laboratories.
 *  This software is distributed under the GNU Lesser General Public License.
 *  For more information, see the README.html file in the top Coliny directory.
 *  _________________________________________________________________________
 */

/**
 * \file EAgeneric.h
 *
 * Defines the coliny::EAgeneric class.
 */

#ifndef coliny_EAgeneric_h
#define coliny_EAgeneric_h

#include <acro_config.h>
#include <coliny/EApoint.h>
#include <coliny/PEAbase.h>

namespace coliny {


///
template <class DomainT, class InfoT, class OpClassT>
class EAgeneric_name
{
protected:

  ///
  const char* get_name() { return "EAgeneric"; }
};

///
/// An evolutionary algorithm with a generic problem representation
///
template <class DomainT, class InfoT, class OpClassT>
class EAgeneric : public PEAbase<EApoint<DomainT,InfoT>, DomainT > ,
		  public EAgeneric_name<DomainT,InfoT,OpClassT>
{
  #if !defined(DOXYGEN)
  typedef EApoint<DomainT,InfoT > point_t;
  typedef PEAbase<point_t,DomainT> base_t;
  #if !defined(SOLARIS)
  using base_t::popsize_;
  #endif
  #endif

public:

  ///
  EAgeneric()
	{
	this->opt_name = this->get_name();
	search_ops = new OpClassT;
	#if 0
	ParameterSet::create_parameter("distinct_initial_population",
	  init_unique,
          "<bool>","true",
          "If true, then the initial population is composed of distinct points");
	#endif
	augment_parameters(*search_ops);
	}

  ///
  void reset()
	{
	if (! (this->problem)) return;
	if (search_ops->check_domain(this->problem,this->best().termination_info))
	   return;
	if (popsize_ == 0) popsize_ = 100;
	search_ops->debug = this->debug;
	// Resize the best point
        search_ops->initialize(this->problem,popsize_,this->best().point,this->xover_rate,this->mutation_rate);
	base_t::reset(); 
        if (this->xover_rate == 0) {
           //search_ops->set_parameter("binary_xovertype","none");
           search_ops->set_parameter("intarray_xover_type","none");
           search_ops->set_parameter("realarray_xover_type","none");
           }
	search_ops->reset();
   	this->init_population(popsize_);
        randomize();
	}

  ///
  void minimize()
	{
	if (search_ops->check_domain(this->problem,this->best().termination_info)) 
	   return;
	base_t::minimize();
	}

  ///
  void write(std::ostream& os) const
	{
	base_t::write(os);
	search_ops->write(os);
	}

protected:

  ///
  OpClassT* search_ops;

  ///
  void initialize_best_point()
	{ search_ops->initialize_point(this->best().point, best_point_info); }

  ///
  void set_op_rng(utilib::AnyRNG& rng_)
	{
	search_ops->set_rng(rng_);
	}

  ///
  void randomize()
	{
	unsigned int i=0, m = 0;
	//
	// Read up to popsize_ elements from this file
	//
	if (this->init_filename != "") {
   	   std::ifstream ifstr(this->init_filename.c_str());
   	   if (ifstr) {
      	      search_ops->read_point(ifstr,this->population(i).pt.point,m);
	      if (ifstr) {
       	         this->population(i).reset();
                 i++;
		 }
      	      while (i < popsize_) {
      	        search_ops->read_point(ifstr,this->population(i).pt.point,m);
		if (!ifstr) break;
	        this->population(i).reset();
       	        i++;
       	        }
	      if (m)
		ucout << "\n\n**** Projecting " << m
			<< " out-of-bounds value(s) in file \""
			<< this->init_filename
			<< "\"\n**** onto the nearest bound(s). ****\n\n"
			<< std::endl;
      	      }
	   else
    	      EXCEPTION_MNGR(std::runtime_error,"EAgeneric::randomize - bad initial point file name.");
	   DEBUGPR(1,ucout << "Initial population initialized with " << i << " points from the file " << this->init_filename.c_str() << std::endl;);
   	   }
	//
	// Generate remaining points randomly, possibly testing for uniqueness
	//
	// TODO - pass an entire array of points.  This would facilitate more
	// effective randomization strategies.
	///
	for (; i<popsize_; i++) {
	  int ctr=0;
	  bool flag = false;
	  while (!flag) {
	    search_ops->randomize( this->population(i).pt.point, this->population(i).pt.info );
	    if ((flag == 0) || !init_unique || (ctr == 20)) {
	       flag=true;
	       continue;
	       }
	    flag=true;
            for (unsigned int k=0; k<i; k++)
              if (this->population(i).pt.point == this->population(k).pt.point) {
                 flag=false;
                 break;
                 }
	    ctr++;
	    }
	  this->population(i).reset();
	  }
	}

  ///
  void apply_xover(EAindividual<point_t,DomainT>& parent1,
		   EAindividual<point_t,DomainT>& parent2,
                   EAindividual<point_t,DomainT>& child)
	{
	if (parent1==parent2) {
	   child.copy(parent1);
	   return;
           }

	point_t& p1 = parent1.pt;
	point_t& p2 = parent2.pt;
	point_t& c1 = child.pt;

	int status = search_ops->apply_xover(p1.point,p1.info,
				p2.point,p2.info,c1.point,c1.info);
	OUTPUTPR(4, search_ops->write(ucout); );

        if (status == 1) 
           child.copy(parent1);
        else if (status == 2)
           child.copy(parent2);
        else
           child.reset();
	}

  ///
  bool apply_mutation(EAindividual<point_t,DomainT>& point, int parent_ndx)
	{
	point_t& p1 = point.pt;
	bool status = search_ops->apply_mutation(p1.point,p1.info, parent_ndx);
	OUTPUTPR(4, search_ops->write(ucout); );
        point.reset();
	return status;
#if 0
        //
        // Print the progress of the best point in the population
        //
        if (eval_flag &&
            (Debug_search_steps) && 
            (parent_ndx != -1) && (parent_ndx < num_min)) {
           ucout << "[" << endl << "Curr_best_point: ";
           population(parent_ndx).write_point(ucout);
           ucout << "  New_point: ";
           point.write_point(ucout);
           ucout << endl << "]" << endl;
           }
#endif
	}

  ///
  InfoT best_point_info;

  ///
  bool init_unique;

  ///
  void init(const unsigned int popsize)
	{
	EAbase<EApoint<DomainT,InfoT>, DomainT>::init(popsize);
	for (unsigned int i=0; i<this->pop_array.size(); i++)
	  search_ops->initialize_point( this->pop_array[i]->pt.point, 
					this->pop_array[i]->pt.info );
	for (unsigned int i=0; i<this->trial_array.size(); i++)
	  search_ops->initialize_point( this->trial_array[i]->pt.point, 
					this->trial_array[i]->pt.info );
        search_ops->initialize_point( this->min_pt.pt.point,
				      this->min_pt.pt.info);
        search_ops->initialize_point( this->true_min_pt.pt.point,
				      this->true_min_pt.pt.info);
        search_ops->initialize_point( this->max_pt.pt.point,
				      this->max_pt.pt.info);
        search_ops->initialize_point( this->true_max_pt.pt.point,
				      this->true_max_pt.pt.info);
	}
};

} // namespace coliny

#endif
