/*  _______________________________________________________________________

    DAKOTA: Design Analysis Kit for Optimization and Terascale Applications
    Copyright (c) 2006, Sandia National Laboratories.
    This software is distributed under the GNU General Public License.
    For more information, see the README file in the top Dakota directory.
    _______________________________________________________________________ */

//- Class:       NonDStochCollocation
//- Description: Implementation code for NonDStochCollocation class
//- Owner:       Mike Eldred

#include "system_defs.h"
#include "NonDStochCollocation.H"
#include "DakotaModel.H"
#include "DakotaResponse.H"
#include "ProblemDescDB.H"
#include "DataFitSurrModel.H"
#include "InterpPolyApproximation.H"


namespace Dakota {

NonDStochCollocation::NonDStochCollocation(Model& model): NonDExpansion(model)
{
  // This constructor is called for a standard letter-envelope iterator 
  // instantiation.

  // -------------------
  // Recast g(x) to G(u)
  // -------------------
  Model g_u_model;
  construct_g_u_model(g_u_model);

  // -------------------------
  // Construct u_space_sampler
  // -------------------------
  // LHS/Incremental LHS/Quadrature/SparseGrid samples in u-space
  // generated using active sampling view:
  Iterator u_space_sampler;
  const UShortArray& quad_order_spec
    = probDescDB.get_dusa("method.nond.quadrature_order");
  const UShortArray& sparse_grid_level_spec
    = probDescDB.get_dusa("method.nond.sparse_grid_level");
  if (!quad_order_spec.empty()) {
    construct_quadrature(u_space_sampler, g_u_model, quad_order_spec);
    expansionCoeffsApproach = QUADRATURE;
  }
  else if (!sparse_grid_level_spec.empty()) {
    construct_sparse_grid(u_space_sampler, g_u_model, sparse_grid_level_spec);
    expansionCoeffsApproach = SPARSE_GRID;
  }
  // iteratedModel concurrency is defined by the number of samples
  // used in constructing the expansion
  if (numSamplesOnModel) // optional with default = 0
    maxConcurrency *= numSamplesOnModel;

  // --------------------------------
  // Construct G-hat(u) = uSpaceModel
  // --------------------------------
  // G-hat(u) uses an orthogonal polynomial approximation over the
  // active/uncertain variables (using same view as iteratedModel/g_u_model:
  // not the typical All view for DACE).  No correction is employed.
  // *** Note: for PCBDO with polynomials over {u}+{d}, change view to All.
  String approx_type = "global_interpolation_polynomial", sample_reuse,
    corr_type;
  short expansion_order = -1, corr_order = -1;
  uSpaceModel.assign_rep(new DataFitSurrModel(u_space_sampler, g_u_model,
    g_u_model.current_variables().view(),
    g_u_model.current_response().active_set(), approx_type, expansion_order,
    corr_type, corr_order, sample_reuse), false);
  // if all variables mode, initialize key to random variable subset
  bool all_vars = (numDesignVars || numStateVars);
  BoolDeque random_vars_key;
  size_t i;
  if (all_vars) {
    random_vars_key.resize(numContinuousVars);
    for (i=0; i<numContinuousVars; i++)
      random_vars_key[i]
	= (i >= numDesignVars && i < numDesignVars + numUncertainVars);
  }
  // The expansion_order specification is passed through the DataFitSurrModel
  // ctor.  Any additional specification data needed by InterpPolyApproximation
  // must be passed through by diving through the hierarchy.
  Array<Approximation>& interp_poly_approxs = uSpaceModel.approximations();
  for (i=0; i<numFunctions; i++) {
    InterpPolyApproximation* interp_poly_approx_rep
      = (InterpPolyApproximation*)interp_poly_approxs[i].approx_rep();
    if (interp_poly_approx_rep) {
      interp_poly_approx_rep->solution_approach(expansionCoeffsApproach);
      interp_poly_approx_rep->integration_iterator(u_space_sampler);
      if (all_vars)
	interp_poly_approx_rep->random_variables_key(random_vars_key);
    }
  }

  // -------------------------------
  // Construct expSampler, if needed
  // -------------------------------
  construct_expansion_sampler();

  // uSpaceModel concurrency is defined by the number of samples used
  // in evaluating the PC expansion
  uSpaceModel.init_communicators(numSamplesOnExpansion);
}


NonDStochCollocation::~NonDStochCollocation()
{
  uSpaceModel.free_communicators(numSamplesOnExpansion);
}


void NonDStochCollocation::quantify_uncertainty()
{
  initialize_expansion();
  compute_expansion();

  compute_statistics();
  update_final_statistics();

  numUncertainQuant++;

#ifdef DEBUG
  if (subIteratorFlag)
    print_results(Cout);
#endif // DEBUG
}

} // namespace Dakota
