/*  _______________________________________________________________________

    DAKOTA: Design Analysis Kit for Optimization and Terascale Applications
    Copyright (c) 2006, Sandia National Laboratories.
    This software is distributed under the GNU General Public License.
    For more information, see the README file in the top Dakota directory.
    _______________________________________________________________________ */

//- Class:        DistinctVariables
//- Description:  Class implementation
//- Owner:        Mike Eldred

#include "DistinctVariables.H"
#include "ProblemDescDB.H"
#include "DakotaBinStream.H"

static const char rcsId[]="@(#) $Id";


namespace Dakota {

/** In this class, the distinct approach is used (design, uncertain, and
    state variable types and continuous and discrete domain types are 
    distinct).  Most iterators/strategies use this approach. */
DistinctVariables::
DistinctVariables(const ProblemDescDB& problem_db,
		  const pair<short,short>& view):
  Variables(BaseConstructor(), problem_db, view),
  continuousDesignVars(
    problem_db.get_drv("variables.continuous_design.initial_point")),
  uncertainVars(problem_db.get_drv("variables.uncertain.initial_point")),
  continuousStateVars(
    problem_db.get_drv("variables.continuous_state.initial_state")),
  discreteDesignVars(
    problem_db.get_div("variables.discrete_design.initial_point")),
  discreteStateVars(
    problem_db.get_div("variables.discrete_state.initial_state")),
  continuousDesignLabels(
    problem_db.get_dsa("variables.continuous_design.labels")), 
  uncertainLabels(problem_db.get_dsa("variables.uncertain.labels")), 
  continuousStateLabels(
    problem_db.get_dsa("variables.continuous_state.labels")), 
  discreteDesignLabels(problem_db.get_dsa("variables.discrete_design.labels")),
  discreteStateLabels(problem_db.get_dsa("variables.discrete_state.labels"))
{
  // construct continuousVarTypes/discreteVarTypes and
  // continuousVarIds/inactiveContinuousVarIds/allContinuousVarIds
  build_types_ids();

#ifdef REFCOUNT_DEBUG
  Cout << "Letter instantiated: variablesView active = " << variablesView.first
       << " inactive = " << variablesView.second << endl;
#endif
}


void DistinctVariables::reshape_rep(const Sizet2DArray& vars_comps)
{
  size_t num_cdv = vars_comps[0][1], num_ddv = vars_comps[0][2],
         num_uv  = vars_comps[1][0],
         num_csv = vars_comps[2][1], num_dsv = vars_comps[2][2];

  continuousDesignVars.reshape(num_cdv);
  discreteDesignVars.reshape(num_ddv);
  uncertainVars.reshape(num_uv);
  continuousStateVars.reshape(num_csv);
  discreteStateVars.reshape(num_dsv);

  continuousDesignLabels.reshape(num_cdv);
  discreteDesignLabels.reshape(num_ddv);
  uncertainLabels.reshape(num_uv);
  continuousStateLabels.reshape(num_csv);
  discreteStateLabels.reshape(num_dsv);

  build_types_ids();
}


void DistinctVariables::build_types_ids()
{
  size_t i, offset, num_cdv = continuousDesignVars.length(),
    num_ddv = discreteDesignVars.length(), num_uv = uncertainVars.length(),
    num_csv = continuousStateVars.length(),
    num_dsv = discreteStateVars.length(), num_acv = num_cdv + num_uv + num_csv,
    num_adv = num_ddv + num_dsv, acv_cntr = 0, adv_cntr = 0;

  allContinuousVarIds.reshape(num_acv);
  allContinuousVarTypes.reshape(num_acv);
  allDiscreteVarTypes.reshape(num_adv);
  for (i=0; i<num_cdv; i++) {
    allContinuousVarTypes[acv_cntr] = "continuous_design";
    allContinuousVarIds[acv_cntr++] = i + 1;
  }
  for (i=0; i<num_ddv; i++)
    allDiscreteVarTypes[adv_cntr++] = "discrete_design";
  uncertain_var_types(acv_cntr);
  offset = num_cdv + num_ddv + 1;
  for (i=0; i<num_uv; i++)
    allContinuousVarIds[i + num_cdv] = i + offset;
  offset += num_uv;
  for (i=0; i<num_csv; i++) {
    allContinuousVarTypes[acv_cntr] = "continuous_state";
    allContinuousVarIds[acv_cntr++] = i + offset;
  }
  for (i=0; i<num_dsv; i++)
    allDiscreteVarTypes[adv_cntr++] = "discrete_state";

  // Don't bleed over any logic about supported view combinations.
  // Rather, keep this class general and encapsulated.

  // Initialize continuousVarTypes/discreteVarTypes/continuousVarIds
  if (variablesView.first == MIXED_DISTINCT_DESIGN) {
    copy_data_partial(allContinuousVarTypes, 0, num_cdv, continuousVarTypes);
    copy_data_partial(allContinuousVarIds,   0, num_cdv, continuousVarIds);
    copy_data_partial(allDiscreteVarTypes,   0, num_ddv, discreteVarTypes);
  }
  else if (variablesView.first == MIXED_DISTINCT_UNCERTAIN) {
    copy_data_partial(allContinuousVarTypes, num_cdv, num_uv,
		      continuousVarTypes);
    copy_data_partial(allContinuousVarIds,   num_cdv, num_uv, continuousVarIds);
    // discreteVarTypes remains empty
  }
  else if (variablesView.first == MIXED_DISTINCT_STATE) {
    copy_data_partial(allContinuousVarTypes, num_cdv + num_uv, num_csv,
		      continuousVarTypes);
    copy_data_partial(allContinuousVarIds, num_cdv + num_uv, num_csv,
		      continuousVarIds);
    copy_data_partial(allDiscreteVarTypes, num_ddv, num_dsv, discreteVarTypes);
  }

  // Initialize inactiveContinuousVarIds
  if (variablesView.second == MIXED_DISTINCT_DESIGN)
    copy_data_partial(allContinuousVarIds, 0, num_cdv,
		      inactiveContinuousVarIds);
  else if (variablesView.second == MIXED_DISTINCT_UNCERTAIN)
    copy_data_partial(allContinuousVarIds, num_cdv, num_uv,
		      inactiveContinuousVarIds);
  else if (variablesView.second == MIXED_DISTINCT_STATE)
    copy_data_partial(allContinuousVarIds, num_cdv + num_uv, num_csv,
		      inactiveContinuousVarIds);
}


void DistinctVariables::read(istream& s)
{
  // ASCII version.  Invokes Vector::read(istream&).

  continuousDesignVars.read(s, continuousDesignLabels);
  discreteDesignVars.read(s, discreteDesignLabels);
  uncertainVars.read(s, uncertainLabels);
  continuousStateVars.read(s, continuousStateLabels);
  discreteStateVars.read(s, discreteStateLabels);
}


void DistinctVariables::write(ostream& s) const
{
  // ASCII version.  Invokes Vector::write(ostream&)

  continuousDesignVars.write(s, continuousDesignLabels);
  discreteDesignVars.write(s, discreteDesignLabels);
  uncertainVars.write(s, uncertainLabels);
  continuousStateVars.write(s, continuousStateLabels);
  discreteStateVars.write(s, discreteStateLabels);
}


void DistinctVariables::write_aprepro(ostream& s) const
{
  // ASCII version.  Invokes Vector::write_aprepro(ostream&)

  continuousDesignVars.write_aprepro(s, continuousDesignLabels);
  discreteDesignVars.write_aprepro(s, discreteDesignLabels);
  uncertainVars.write_aprepro(s, uncertainLabels);
  continuousStateVars.write_aprepro(s, continuousStateLabels);
  discreteStateVars.write_aprepro(s, discreteStateLabels);
}


void DistinctVariables::read_annotated(istream& s)
{
  // ASCII version for neutral file I/O.

  continuousDesignVars.read_annotated(s, continuousDesignLabels);
  discreteDesignVars.read_annotated(s, discreteDesignLabels);
  uncertainVars.read_annotated(s, uncertainLabels);
  continuousStateVars.read_annotated(s, continuousStateLabels);
  discreteStateVars.read_annotated(s, discreteStateLabels);
  // types/ids not required
}


void DistinctVariables::write_annotated(ostream& s) const
{
  // ASCII version for neutral file I/O.

  continuousDesignVars.write_annotated(s, continuousDesignLabels);
  discreteDesignVars.write_annotated(s, discreteDesignLabels);
  uncertainVars.write_annotated(s, uncertainLabels);
  continuousStateVars.write_annotated(s, continuousStateLabels);
  discreteStateVars.write_annotated(s, discreteStateLabels);
  // types/ids not required
}


void DistinctVariables::write_tabular(ostream& s) const
{
  // ASCII version for tabular file I/O.

  continuousDesignVars.write_tabular(s);
  discreteDesignVars.write_tabular(s);
  uncertainVars.write_tabular(s);
  continuousStateVars.write_tabular(s);
  discreteStateVars.write_tabular(s);
}


void DistinctVariables::read(BiStream& s)
{
  // Binary version.  Invokes Vector::read(BiStream&).

  continuousDesignVars.read(s, continuousDesignLabels);
  discreteDesignVars.read(s, discreteDesignLabels);
  uncertainVars.read(s, uncertainLabels);
  continuousStateVars.read(s, continuousStateLabels);
  discreteStateVars.read(s, discreteStateLabels);
  // types/ids not required
}


void DistinctVariables::write(BoStream& s) const
{
  // Binary version.  Invokes Vector::write(BoStream&).

  continuousDesignVars.write(s, continuousDesignLabels); 
  discreteDesignVars.write(s, discreteDesignLabels);
  uncertainVars.write(s, uncertainLabels);
  continuousStateVars.write(s, continuousStateLabels);
  discreteStateVars.write(s, discreteStateLabels);
  // types/ids not required
}


void DistinctVariables::read(MPIUnpackBuffer& s)
{
  // MPI buffer version.  Invokes Vector::read(MPIUnpackBuffer&).

  continuousDesignVars.read(s, continuousDesignLabels);
  discreteDesignVars.read(s, discreteDesignLabels);
  uncertainVars.read(s, uncertainLabels);
  continuousStateVars.read(s, continuousStateLabels);
  discreteStateVars.read(s, discreteStateLabels);
  // types/ids not required
}


void DistinctVariables::write(MPIPackBuffer& s) const
{
  // MPI buffer version.  Invokes Vector::write(MPIPackBuffer&)

  continuousDesignVars.write(s, continuousDesignLabels);
  discreteDesignVars.write(s, discreteDesignLabels);
  uncertainVars.write(s, uncertainLabels);
  continuousStateVars.write(s, continuousStateLabels);
  discreteStateVars.write(s, discreteStateLabels);
  // types/ids not required
}


void DistinctVariables::copy_rep(const Variables* vars_rep)
{
  // copy derived class attributes
  continuousDesignVars  = ((DistinctVariables*)vars_rep)->continuousDesignVars;
  discreteDesignVars    = ((DistinctVariables*)vars_rep)->discreteDesignVars;
  uncertainVars         = ((DistinctVariables*)vars_rep)->uncertainVars;
  continuousStateVars   = ((DistinctVariables*)vars_rep)->continuousStateVars;
  discreteStateVars     = ((DistinctVariables*)vars_rep)->discreteStateVars;

  continuousDesignLabels
    = ((DistinctVariables*)vars_rep)->continuousDesignLabels;
  uncertainLabels       = ((DistinctVariables*)vars_rep)->uncertainLabels;
  continuousStateLabels = ((DistinctVariables*)vars_rep)->continuousStateLabels;
  discreteDesignLabels  = ((DistinctVariables*)vars_rep)->discreteDesignLabels;
  discreteStateLabels   = ((DistinctVariables*)vars_rep)->discreteStateLabels;

  continuousVarTypes  = ((DistinctVariables*)vars_rep)->continuousVarTypes;
  if (variablesView.first != MIXED_DISTINCT_UNCERTAIN)
    discreteVarTypes  = ((DistinctVariables*)vars_rep)->discreteVarTypes;
  allDiscreteVarTypes = ((DistinctVariables*)vars_rep)->allDiscreteVarTypes;
  continuousVarIds    = ((DistinctVariables*)vars_rep)->continuousVarIds;
  inactiveContinuousVarIds
    = ((DistinctVariables*)vars_rep)->inactiveContinuousVarIds;
}


/// equality operator for DistinctVariables
/** Checks each array using operator== from data_types.C. Labels are ignored. */
bool operator==(const DistinctVariables& vars1, const DistinctVariables& vars2)
{
  return (vars2.continuousDesignVars == vars1.continuousDesignVars && 
	  vars2.discreteDesignVars   == vars1.discreteDesignVars   &&
	  vars2.uncertainVars        == vars1.uncertainVars        &&
	  vars2.continuousStateVars  == vars1.continuousStateVars  &&
	  vars2.discreteStateVars    == vars1.discreteStateVars);
  // labels/types/ids not required
}

} // namespace Dakota
