/*
 * cielothermalsteady.c:
 */

#include "../include/cielo.h"
#include "../modules.h"
#include "./parallel.h"

#undef __FUNCT__ 
#define __FUNCT__ "cielothermalsteady"

int main(int argc,char* *argv){
	
	/*I/O: */
	FILE* fid=NULL;
	char* inputfilename=NULL;
	char* outputfilename=NULL;
	char* lockname=NULL;
	char* analysis_type_t="thermalsteady";
	char* analysis_type_m="melting";

	/*Intermediary: */
	
	#if defined(_PARALLEL_) && defined(_HAVE_PETSC_)
	FemModel femmodel_t;
	FemModel femmodel_m;	Vec* t_g=NULL;
	#endif
	ParameterInputs* inputs=NULL;

	#if !defined(_PARALLEL_) || (defined(_PARALLEL_) && !defined(_HAVE_PETSC_))
		_printf_("%s%s\n",__FUNCT__," error message: parallel executable was compiled without support of parallel libraries!");
		return 1;
	#else

		/*Initialize MPI environment: */
		PetscInitialize(&argc,&argv,(char *)0,"");  

		/*Size and rank: */
		MPI_Comm_rank(MPI_COMM_WORLD,&my_rank);  
		MPI_Comm_size(MPI_COMM_WORLD,&num_procs); 

		/*Some checks on size of cluster*/
		if (num_procs<=1){
			_printf_("\nSize of MPI COMM WORLD is 1, needs to be at least 2. Include more nodes\n"); 
			PetscFinalize(); 
			return 0;
		}

		/*Recover dbdir, input file name and output file name: */
		dbdir=argv[1];
		inputfilename=argv[2];
		outputfilename=argv[3];
		lockname=argv[4];

		
		
		/*Read and create thermal finite element model: */
		fid=fopen(inputfilename,"rb");
		if(fid==NULL){
			_printf_("%s%s\n",__FUNCT__," error message: could not open file ",inputfilename," for binary reading"); 
			return 0;
		}
		if(!CreateFemModel(&femmodel_t,fid,analysis_type_t)){
			_printf_("%s\n",__FUNCT__," error message: could not read melting finite element model!\n");
			return 0;
		}
		fclose(fid);

		/*Read and create melting finite element model: */
		fid=fopen(inputfilename,"rb");
		if(!CreateFemModel(&femmodel_m,fid,analysis_type_m)){
			_printf_("%s\n",__FUNCT__," error message: could not read thermal finite element model!\n");
			return 0;
		}
		fclose(fid);

		/*Initialize inputs: */
		inputs=NewParameterInputs();
			
		ParameterInputsAddFromMat(inputs,WorkspaceParamsGetVelocity(femmodel_t.workspaceparams),femmodel_t.workspaceparams->gsize,"velocity");
		ParameterInputsAddFromMat(inputs,WorkspaceParamsGetPressure(femmodel_t.workspaceparams),femmodel_t.workspaceparams->gsize,"pressure");

		ParameterInputsAddFromMat(inputs,&femmodel_t.batchparams->dt,1,"dt");
		
		/*Call computational core: */
		if(!cielothermal_core(&t_g,inputs,&femmodel_t)){
			_printf_("%s\n",__FUNCT__," error message: could not run computational core!\n");
			return 0;
		}
		
		/*Write results to disk: */
		OutputThermal(t_g,femmodel_t.tpartition,outputfilename,analysis_type_t);

		/*Write lock file if requested: */
		if (femmodel_t.batchparams->waitonlock){
			WriteLockFile(lockname);
		}
					
		cleanup_and_return:
		_printf_("done.\n");
		
		/*Synchronize everyone before exiting: */
		MPI_Barrier(MPI_COMM_WORLD);

		/*Close MPI libraries: */
		PetscFinalize(); 
		
		return 0; //unix success return;

	#endif
}
